/*******************************************************************************
* Copyright 2004-2008 LSI Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppLnx26p_sysdep.c
SUMMARY         %description%
VERSION         %version: 12 %
UPDATE DATE     %date_modified: Fri May 16 13:11:40 2008 %
PROGRAMMER      %created_by:    jwendel %


DESCRIPTION:
     This source file contains MPP Linux system dependent functions. A system 
     dependent function is a function that are called by MPP common APIs and 
     must be supplied/implemented by a platform implementation.
INCLUDE FILES:

NOTES: The error numbers for this file is ranging from 850 to 880.

RESTRICTIONS:

SEE ALSO:

REFERENCE:

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/


/***  INCLUDES  ***/  
#include "MPP_Sysdep.h"
#include "MPP_RdacInfo.h"
#include "MPP_Common.h"
#include "MPP_ProtoTypes.h"
#include "copyright.h"
#include "mppLnx26p_shared.h"

atomic_t		mpp_CurrentlyAllocatedMemory;

/*** GLOBALS ***/

/***  LOCALS  ***/
static void mppLnx_WakeupThread(  struct semaphore *sem ); 

/*
* the /proc/mpp directory. It is the root directory of mpp driver's device
* topology
*/
struct proc_dir_entry * proc_mpp = NULL; 
/*
* HBA driver's identification. It is used to build controller path directory
*/
char           * mppLnx_host_procName = NULL;  
extern spinlock_t mpp_hot_plug_spin_lock;

#define		WWN_LENGTH 16

/***  MACRO DEFINITIONS  ***/
/*
* the size of a proc directory name.
*/
#define  MPPLNX_PROC_NAMESIZE 128


/***  PROCEDURES  ***/

/*******************************************************************************
* PROCEDURE
* NAME:    mppLnx_zmalloc 
* SUMMARY: System dependent kernel memory allocation function  
* SCOPE:   public   
* DESCRIPTION:
* 	This function allocates kernel memory using the Linux platform specific
*	allocation kernel function kmalloc. It also memsets the entire memory 
*	allocated to zero. The debug messages are put in to help ensure that all 
*	memory malloced, is being set free. 
*	It takes the size of memory chunk being requested and the type of allocation
*	GFP_KERNEL Or GFP_ATOMIC as the flag. In GFP_KERNEL, the alloc function
*	will go to sleep until it can allocate the requested memory.
*	GFP_ATOMIC will return if it cannot allocate memory. 
*
* RESTRICTIONS:
* RETURNS: pointer to allocated memory or NULL (if memory could not be allocated )
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  mppLnx_zfree
****************************************************************************/

void *
mppLnx_zmalloc( size_t size, int flag) 
{
	void *memory;

	memory = kmalloc(size, flag);
	if(	memory ) {
#ifdef MPP_DEBUG
		MPP_DEBUGPRINT((MPP_ALLOC_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_zmalloc: BZEROing the memory allocated \n"));
#endif
		memset(memory, 0, size);
		atomic_add( size, &mpp_CurrentlyAllocatedMemory ); 
#ifdef MPP_DEBUG
		MPP_DEBUGPRINT((MPP_ALLOC_DEBUG+MPP_DEBUG_LEVEL_3, 
			"mppLnx_zmalloc: memory 0x%lx, size %d, total %d\n",	
			memory,	size, atomic_read(&mpp_CurrentlyAllocatedMemory)));
#endif
	} else {
#ifdef MPP_DEBUG
		MPP_DEBUGPRINT((MPP_ALLOC_DEBUG+MPP_DEBUG_LEVEL_3, 
			"mppLnx_zmalloc: Failed to allocate,	size %d\n",	
		size));
#endif
	}
	return(memory);


}


/*******************************************************************************
* PROCEDURE
* NAME:    mppLnx_zfree
* SUMMARY: System dependent kernel memory de-allocation function  
* SCOPE:   public 
* DESCRIPTION:
* 	This function de-allocates kernel memory using the Linux platform specific
*	allocation kernel function kfree. 
*	The debug messages are put in to help ensure that all 
*	memory malloced, is being set free. 
*	It takes the pointer to the memory being deallocated and the  size of
*	memory chunk being deallocated as arguments. However kfree on Linux only 
*	needs a pointer to the memory being de-allocated.
*
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  mppLnx_zmalloc
****************************************************************************/

void 
mppLnx_zfree( void *buf, size_t size)
{
	atomic_sub( size, &mpp_CurrentlyAllocatedMemory);
#ifdef MPP_DEBUG
	MPP_DEBUGPRINT((MPP_ALLOC_DEBUG+MPP_DEBUG_LEVEL_3, 
		"mppLnx_zfree: freeing 0x%lx, size %d, total	%d\n", 
	buf, size, atomic_read(&mpp_CurrentlyAllocatedMemory)));
#endif
	kfree(buf);

}

/*******************************************************************************
* PROCEDURE
* NAME:		mpp_Lock
* SUMMARY:	MPP	Driver Lock function
* SCOPE:	public 
* DESCRIPTION: This routine allows a debug build of the	MPP driver to print debug
*		messages while calling a spin lock function.
* RESTRICTIONS
* RETURNS:      void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:     	
****************************************************************************/
void mpp_Lock(spinlock_t *lock, unsigned long level)
{
	MPP_DEBUGPRINT((MPP_DEBUG_LEVEL_4, "Preparing to lock MPP mutex 0x%lx", lock));
	spin_lock_irqsave(lock, level);

}

/*******************************************************************************
* PROCEDURE
* NAME:		mpp_Unlock
* SUMMARY:	MPP	Driver Unlock function
* SCOPE:	public 
* DESCRIPTION: This routine allows a debug build of the	MPP driver to print debug
*		messages while calling a spin Unlock function.
* RESTRICTIONS:
* RETURNS:      void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:     	
****************************************************************************/
void mpp_Unlock(spinlock_t *lock, unsigned long level)
{
	MPP_DEBUGPRINT((MPP_DEBUG_LEVEL_4, "Preparing to Unlock MPP mutex 0x%lx", lock));
	spin_unlock_irqrestore(lock, level );
}

/*******************************************************************************
* PROCEDURE
* NAME:		mpp_DebugPrint
* SUMMARY:	MPP	Driver Debug Messages Print
* SCOPE:	public 
* DESCRIPTION: This routine allows a debug build of the	MPP driver to print debug
*		messages.  The routine supports	various	debug levels.  The debug
*		levels are set in the mpp_Debug	cell.
* RESTRICTIONS:
* RETURNS:      void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:     	
****************************************************************************/

void 
mpp_DebugPrint(LWORD DebugPrintLevel, TINY *DebugMessage, ...)
{
    TINY mpp_PrintBuffer[DEBUG_BUFFER_LENGTH];
	va_list	ap;

	memset(mpp_PrintBuffer, 0, DEBUG_BUFFER_LENGTH);

	va_start(ap, DebugMessage);

		
	if ( (DebugPrintLevel & mpp_Debug) == DebugPrintLevel )	
	{
		vsprintf((mpp_PrintBuffer),	DebugMessage, ap);

		printk (KERN_DEBUG "%s", mpp_PrintBuffer);
	}

	va_end(ap);

} /* end mpp_DebugPrint()*/


/****************************************************************************
* PROCEDURE
* NAME:    mpp_ErrorLog
* SUMMARY: MPP Error message logging function 
* SCOPE:   public 
* DESCRIPTION: This routine allows a MPP driver to log messages. This routine
*	supports various error levels.The error levels are set in the mpp_ErrorLevel cell. 
*	The number argument is the message number of the message.
*
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/



void 
mpp_ErrorLog(LWORD Level, LWORD Number, TINY *LogMessage, ...)
{
	TINY 		mpp_ErrorLogBuffer[DEBUG_BUFFER_LENGTH];
	va_list		ap;
	LINT 		length;


	memset(mpp_ErrorLogBuffer, 0, DEBUG_BUFFER_LENGTH);

	va_start(ap, LogMessage);

	if ( Level >= mpp_ErrorLevel ) {
		sprintf( mpp_ErrorLogBuffer, "%d [%s.mpp]",Number, MPP_MESSAGESHEADER );
		length = strlen(mpp_ErrorLogBuffer);
		vsprintf(mpp_ErrorLogBuffer+length,	LogMessage,	ap);		
		switch(Level) {
			case 0:
			case 1:
				printk( KERN_INFO "%s", mpp_ErrorLogBuffer);
				break;
			case 2:
				printk(KERN_NOTICE "%s", mpp_ErrorLogBuffer);
				break;
			case 3:
				printk(KERN_WARNING "%s", mpp_ErrorLogBuffer);
				break;
			case 4:
			case 5:
			default:
				printk(KERN_WARNING "%s", mpp_ErrorLogBuffer);
				break;
		}
	}

	va_end(ap);

}

/****************************************************************************/

/****************************************************************************
* PROCEDURE
* NAME:    mppLnx_WakeupThread
* SUMMARY: MPP function that wakes up sleeper routine
* SCOPE:   public 
* DESCRIPTION: This routine wakes up the routine that sleeps on a wait queue.
*
*
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/
static void 
mppLnx_WakeupThread( struct semaphore *sem) 
{
	if (sem != NULL) {
		up(sem);
	}
}
/****************************************************************************
* PROCEDURE
* NAME:    mpp_ThreadDelay
* SUMMARY: MPP function that introduces delay 
* SCOPE:   public 
* DESCRIPTION: This routine provides delay by using sleep and wakeup
*		mechanism instead of busy waiting like udelay.
*	
*		DelayTime 	= No. of DelayResolution Units
*		DelayReolustion = delay in secs, millisecs ..etc
*
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:  This is not MultiThread safe
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/
void 
mpp_ThreadDelay(ULONG DelayTime,ULONG DelayResolution)
{

	struct timer_list	mppLnx_timer;
	struct semaphore sem;

	init_timer( &mppLnx_timer );
	sema_init(&sem, 0);
	mppLnx_timer.function	= (void (*)(unsigned long)) mppLnx_WakeupThread;
	mppLnx_timer.data		= (unsigned long) &sem;
	
	if((( DelayTime * HZ )/ DelayResolution) < 10 )
	{
		mppLnx_timer.expires = jiffies + 10;
	}
	else
	{
		mppLnx_timer.expires = jiffies + ((DelayTime *HZ)/DelayResolution);
	}

	add_timer ( &mppLnx_timer );

	down(&sem);

	del_timer(&mppLnx_timer);
	return;
}

/****************************************************************************
* PROCEDURE
* NAME:    mpp_GetTime
* SUMMARY: MPP function that returns system time 
* SCOPE:   public 
* DESCRIPTION: This routine provides the MPP driver with the current system 
*	time in jiffies. jiffies is the number of clock ticks since the 
*	Operating System was booted.
* 
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/

void
mpp_GetTime(unsigned long *time)
{
	*time = jiffies / HZ;

}

/****************************************************************************
* PROCEDURE
* NAME:    mpp_IssueFailover
* SUMMARY: MPP System Dependent routine to issue a failover command to controller
* SCOPE:   public 
* DESCRIPTION: This function is just a dummy function in the Linux Upper Level driver.
*	The real functionality for this function is provided by the Linux Virtual
*	HBA driver. Since the Upper Level driver is loaded before the virtual hba 
*	driver, and we want to avoid circular dependencies, we use inter-module communication
*	to call the actual virtual hba function.	
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/

void 
mpp_IssueFailover (RdacDeviceInformation_t *RdacInfo,
                   BYTE Controller,   
                   BYTE Path,
                   MPP_HANDLE FailoverHandle,
                   BYTE *Page2C,
                   BYTE *modeSelectCdb,
                   MPPCMN_FAILOVER_METHOD FailoverMethod,
                   BOOL       ForceQuiescence,
                   BOOL       noHoldInReset)
{
   /*
   * The function pointer mppLnx_IssueFailover will be retrieved from the mppVhba driver module through symbol_get() kernel API.
    * the name of the function pointer must be same as that of the exported symbol name of the mppVhba driver module.
    * The exported symbols of the mppVhba driver module is defined in mppLnx26p_vhbamisc.c
   */
   LINT                 LockLevel;
	void (*mppLnx_IssueFailover) ( RdacDeviceInformation_t *, BYTE, BYTE, MPP_HANDLE, BYTE *, BYTE *, MPPCMN_FAILOVER_METHOD, BOOL, BOOL);

	mppLnx_IssueFailover = (void *)symbol_get(mppLnx_IssueFailover);

	if(mppLnx_IssueFailover)
   {
		(*mppLnx_IssueFailover) (RdacInfo, Controller, Path, FailoverHandle, Page2C, modeSelectCdb, FailoverMethod, ForceQuiescence, noHoldInReset);
      symbol_put(mppLnx_IssueFailover);
   }
	else
   {
      MPP_LOCK(RdacInfo->Lock, LockLevel);
      RdacInfo->ControllerInfo[Controller]->FailoverInProgress = FALSE;
      MPP_UNLOCK(RdacInfo->Lock, LockLevel);
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 850, "mpp_IssueFailover : symbol_get did not return a function pointer\n"));	
   }

}

/****************************************************************************
* PROCEDURE
* NAME:    mppSys_ValidatePath
* SUMMARY: MPP System Dependent routine to issue a Path Validation (inquiry) command. 
* SCOPE:   public 
* DESCRIPTION: This function is just a dummy function in the Linux Upper Level driver.
*	The real functionality for this function is provided by the Linux Virtual
*	HBA driver. Since the Upper Level driver is loaded before the virtual hba 
*	driver, and we want to avoid circular dependencies, we use inter-module communication
*	to call the actual virtual hba function.	
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/
LWORD
mppSys_ValidatePath (RdacDeviceInformation_t *RdacInfo, LWORD Controller, LWORD Path)
{
	LWORD returnvalue=0;
   /*
   * The function pointer mppLnx_validatepath will be retrieved from the mppVhba driver module through symbol_get() kernel API.
   * the name of the function pointer must be same as that of the exported symbol name of the mppVhba driver module.
   * The exported symbols of the mppVhba driver module is defined in mppLnx26p_vhbamisc.c
   */

	LWORD (*mppLnx_validatepath) ( RdacDeviceInformation_t *, LWORD, LWORD);

	mppLnx_validatepath = (void *) symbol_get(mppLnx_validatepath);

	if(mppLnx_validatepath)
   {
		returnvalue = (*mppLnx_validatepath) (RdacInfo, Controller, Path);
      symbol_put(mppLnx_validatepath);
   }
	else
   {
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 855, "mppSys_ValidatePath : symbol_get did not return a function pointer\n"));
   }

	
	return (returnvalue);
}

/****************************************************************************
* PROCEDURE
* NAME:    mppSys_PrintSenseBuffer
* SUMMARY: MPP System Dependent routine to print sense buffer
* SCOPE:   public 
* DESCRIPTION: This function is just a dummy function in the Linux Upper Level driver.
*	The real functionality for this function is provided by the Linux Virtual
*	HBA driver. Since the Upper Level driver is loaded before the virtual hba 
*	driver, and we want to avoid circular dependencies, we use inter-module communication
*	to call the actual virtual hba function.	
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/
VOID
mppSys_PrintSenseBuffer (RdacDeviceInformation_t *RdacInfo, LWORD Controller,
			 LWORD Path, void *SenseBuffer, ULONG SerialNumber)	
{

	LWORD (*mppLnx_printsensebuffer) ( RdacDeviceInformation_t *, LWORD, LWORD , ULONG);
  /*
   * The function pointer mppLnx_printsensebuffer will be retrieved from the mppVhba driver module through symbol_get() kernel API.
   * the name of the function pointer must be same as that of the exported symbol name of the mppVhba driver module.
   * The exported symbols of the mppVhba driver module is defined in mppLnx26p_vhbamisc.c
   */
	mppLnx_printsensebuffer = (void *)symbol_get(mppLnx_printsensebuffer);

	if(mppLnx_printsensebuffer)
   {
		(*mppLnx_printsensebuffer) (RdacInfo, Controller, Path, SerialNumber);
      symbol_put(mppLnx_printsensebuffer);
   }
	else
   {
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 857, "mppSys_PrintSenseBuffer : symbol_get did not return a function pointer\n"));	

   }
	
	return ;
}



/*******************************************************************************
* PROCEDURE
* NAME:     mpp_makeVirtualTarget
* SUMMARY:  
* SCOPE:    public 
* ARGUMENTS:
*       targ - Instance Id
* DESCRIPTION: 
* RESTRICTIONS:
* RETURNS: 
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/

MPP_DIR_HANDLE
mpp_makeVirtualTarget(targ_t targ)
{
   struct proc_dir_entry * vTargetProc;
   if(NULL == proc_mpp)
   {
      /* create the /proc/mpp directory */
      proc_mpp = proc_mkdir(MPPLNX_PROC_NAME, 0);
      if (NULL == proc_mpp)
      {
       	 MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 851, "Failed to make the proc directory entry\n"));
         return NULL;
      }
   }
   /*
   * create proc_dir_entry for the virtual target 
   */
   vTargetProc = proc_mkdir((char *)mpp_ModuleArray[(int)targ].RdacInfo->ModuleName,
      proc_mpp);
   if(NULL == vTargetProc )
   {
       	 MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 852, "Failed to makethe  virtual target proc_dir_entry\n"));
   }
	return vTargetProc;

}

/*******************************************************************************
* PROCEDURE
* NAME:     mpp_makeVirtualLun
* SUMMARY:  On Linux, this function is just a dummy function.
* SCOPE:    public 
* ARGUMENTS: 
* DESCRIPTION: mpp_makeVirtualLun creates the nodes that are associated with the virtual 
* LUN and MPP_HANDLE arguments. On Linux, it is important to set the *PBusStatus argument
* to MPP_PSEUDO_BUS_DISABLED. 
* RESTRICTIONS:
* RETURNS: 
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:  
****************************************************************************/

MPP_HANDLE
mpp_makeVirtualLun(MPP_HANDLE DeviceVertex, RdacDeviceInformation_t *RdacInfo, LWORD Lun, LWORD *PBusStatus)
{
	RdacInfo->RdacLuns[Lun].Present = TRUE;
	RdacInfo->RdacLuns[Lun].ReportedPresent = TRUE;
	RdacInfo->RdacLuns[Lun].ReportedMissing = FALSE;
	MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp_InsertLun: %s:%d inserted, CurrentOwningPath %d, PreferredPath %d, UTM %d\n",
	RdacInfo->ModuleName, Lun, RdacInfo->RdacLuns[Lun].CurrentOwningPath, RdacInfo->RdacLuns[Lun].PreferredPath,
	RdacInfo->RdacLuns[Lun].UTMLun));
	*PBusStatus = MPP_PSEUDO_BUS_DISABLED;
	return((MPP_HANDLE)0);
}

/*******************************************************************************
* PROCEDURE
* NAME:     mpp_CreateControllerVertex
* SUMMARY:  Create Linux proc directory for the given controller path
            /proc/mpp/array_name/controller/path
* SCOPE:    public 
* DESCRIPTION:
   This function is a system dependent function. It is called by the MPP common APIs.
   It creates a Linux /proc file system directory for the given controller path.
   A controller path represents an I_T nexus from a host to a storage array.
*  @RdacInfo A RdacDeviceInformation_t object that repesents a storage array
*  @ControllerIndex Indicates a controller's slot in a storage array
*  @path The index of the controler path
*  @DeviceAddress The Linux physical address of the path's I_T nenux. The DeviceAddress 
      object contains host id (HBA ID), channel, and target ID.
* RESTRICTIONS:
* RETURNS: A pointer to the proc_dir object that represents the controller path's
           /proc directory or NULL if it fails to create this directory entry
* ERRNO:
* NOTES:

  The caller in the common code holds the lock. This function cannot call MPP_LOCK
  again. Doing so will cause system-locking-up.

* EXAMPLES:
* SEE ALSO:
****************************************************************************/

MPP_DIR_HANDLE
mpp_CreateControllerVertex ( RdacDeviceInformation_t *RdacInfo, BYTE ControllerIndex, BYTE Path, ControllerAddress_t *DeviceAddress)
{
   struct proc_dir_entry       * proc_array = NULL;
   struct proc_dir_entry       * proc_controller = NULL;
   struct proc_dir_entry       * proc_this_path = NULL;
   char                 * pathName = NULL;
   char                 * controllerName = NULL;
   int                  hostNo = DeviceAddress->HostId;
   int                  channelId = DeviceAddress->ChannelId;
   int                  targetId = DeviceAddress->TargetId;
  


   proc_array = RdacInfo->ModuleHandle;

   proc_controller = RdacInfo->ControllerInfo[ControllerIndex]->ControllerSlotHandle;

   if(NULL == proc_controller)
   {
      /* the memory must be freed at the proc dir remove time
      */
      controllerName = MPP_KMEM_ALLOC(MPPLNX_PROC_NAMESIZE);
      sprintf(controllerName,"controller%s",
         ControllerIndex ? MPP_CONTROLLER_B_PATH:MPP_CONTROLLER_A_PATH);
      proc_controller = proc_mkdir(controllerName,proc_array);
      if(NULL == proc_controller)
      {
         /* error */
         MPP_FREE(controllerName, MPPLNX_PROC_NAMESIZE);
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 853, "Failed in making proc dir /proc/mpp/%s/%d\n", (char *) RdacInfo->ModuleName, ControllerIndex));
         return NULL;
      }
      else
      {
         RdacInfo->ControllerInfo[ControllerIndex]->ControllerSlotHandle = proc_controller ;
         MPP_FREE(controllerName, MPPLNX_PROC_NAMESIZE);
      }

   }

   /* 
    * Create ControllerPath PathId. This is new field. This is supposed to be in this format
    * (77-HostId-ChannelId-TargetId) each 8 bits apart 
    */
   RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[Path].PathId = 
	(0x77 << 24) + (DeviceAddress->HostId << 16) + (DeviceAddress->ChannelId << 8) + DeviceAddress->TargetId;

   /*
   * create the directory of /proc/mpp/arrayname/controlller/path
   */
   pathName = MPP_KMEM_ALLOC(MPPLNX_PROC_NAMESIZE);
   
   /*
   * prepare the this dir's proc name
   */
   sprintf(pathName,"%s_h%dc%dt%d",mppLnx_host_procName,hostNo,channelId,targetId);

   proc_this_path = proc_mkdir(pathName, proc_controller);
   if(NULL == proc_this_path)
   {
      MPP_FREE(pathName, MPPLNX_PROC_NAMESIZE);
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 854, " Failed to create controller path %s proc_dir_entry\n", pathName));
   }
   else {
      MPP_FREE(pathName, MPPLNX_PROC_NAMESIZE);
   }
   return proc_this_path;

}


/****************************************************************************
* PROCEDURE
* NAME:     mpp_CreateLunLink
* SUMMARY:  mpp_CreateLunLink updates the OS device directories and object graphs, if applicable,
* to indicate the existence of the path to the logical unit. 
* SCOPE:    private 
* DESCRIPTION: Since mknod facility will make the /dev entry, this routine can just be a dummy 
* function for linux.
* RESTRICTIONS:
* RETURNS: 
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/

void
mpp_CreateLunLink(MPP_HANDLE DeviceVertex, RdacDeviceInformation_t *RdacInfo, BYTE ControllerIndex, BYTE	PathIndex, LWORD	Lun)
{
		RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[PathIndex].PathStartState = MPP_STARTED;
		return;
}

/*******************************************************************************
* PROCEDURE
* NAME:     mpp_RemoveControllerDirectory
* SUMMARY:  Remove the identified controller's proc directory
* SCOPE:    public 
* DESCRIPTION:
*  This function is a system dependent function. It is called by the MPP common APIs.
*  It removes the identified controller's /proc  directory.
*  @DeviceVertex A MPP_HANDLE that represents the vertex of the controller
*  directory. It is not used in the Linux mpp driver.
*  @RdacInfo A RdacDeviceInformation_t object that repesents a storage array
*  @ControllerIndex Indicates a controller's slot in a storage array
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:

  The caller in the common code holds the lock. This function cannot call MPP_LOCK
  again. Doing so will cause system-locking-up.

* EXAMPLES:
* SEE ALSO:
****************************************************************************/

void 
mpp_RemoveControllerDirectory ( MPP_HANDLE DeviceVertex, RdacDeviceInformation_t *RdacInfo, BYTE Controller) 
{
   char * proc_name = NULL;
   struct proc_dir_entry   * parent;
   struct proc_dir_entry   * this_dir = NULL;

   parent = RdacInfo->ModuleHandle;
   this_dir = RdacInfo->ControllerInfo[Controller]->ControllerSlotHandle;

   RdacInfo->ControllerInfo[Controller]->ControllerSlotHandle = NULL;

   proc_name = (char *)this_dir->name;

   remove_proc_entry(proc_name, parent);

   return;

}

/*******************************************************************************
* PROCEDURE
* NAME:     mpp_RemovePathDirectory
* SUMMARY:  Removes a controller path directory from the Linux /proc file system
* SCOPE:    public 
* DESCRIPTION:
   This function is a system dependent function. It is called by the MPP common APIs.
   It removes a controller path Linux /proc directory. A controller path /proc directory
   is located in /proc/mpp/arrayName/Controller/path.
   @DeviceVertex A MPP_HANDLE that represents the vertex of the controller
         directory. It is not used in the Linux mpp driver.
*  @RdacInfo A RdacDeviceInformation_t object that repesents a storage array
*  @ControllerIndex Indicates a controller's slot in a storage array
*  @path The index of the controller path
* RESTRICTIONS:
* RETURNS: void 
* ERRNO:
* NOTES:

  The caller in the common code holds the lock. This function cannot call MPP_LOCK
  again. Doing so will cause system-locking-up.

* EXAMPLES:
* SEE ALSO:
****************************************************************************/

void
mpp_RemovePathDirectory ( MPP_HANDLE DeviceVertex, RdacDeviceInformation_t	*RdacInfo, BYTE	Controller,	BYTE Path)
{
   struct proc_dir_entry * this_path;
   struct proc_dir_entry * controller_proc;
   char           * proc_name;
   
   controller_proc = RdacInfo->ControllerInfo[Controller]->ControllerSlotHandle;
   this_path = RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].DirectoryVertex;

   RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].DirectoryVertex = NULL;

   proc_name = (char *)this_path->name;
   remove_proc_entry(proc_name, controller_proc);

   return;
}

/*******************************************************************************
* PROCEDURE
* NAME:     mpp_RemoveModule    
* SUMMARY:  Remove system resources for the storage array module
* SCOPE:    public 
* DESCRIPTION:
* RESTRICTIONS:
* RETURNS: 
* ERRNO:
* NOTES:
*       remove /proc/mpp/arrayName and /proc/mpp proc directory

  
 The caller in the common code holds the lock. This function cannot call MPP_LOCK
  again. Doing so will cause system-locking-up.

* EXAMPLES:
* SEE ALSO:
****************************************************************************/

void
mpp_RemoveModule(MPP_HANDLE DeviceVertex, RdacDeviceInformation_t *RdacInfo)
{

   RdacDeviceInformation_t *rdacInfo;
   ArrayModuleEntry_t *ModPtr;
   BYTE i=0;

   /* remove /proc/mpp/arrayName */
   remove_proc_entry((char *)RdacInfo->ModuleName, proc_mpp);

   RdacInfo->ModuleHandle = NULL;

   ModPtr = mpp_ModuleArray;
   for(i=0; i<mpp_MaxArrayModules; ++i, ++ModPtr) {
        rdacInfo = ModPtr->RdacInfo;
        if( rdacInfo )
            return;
   }


   return;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppSys_DeviceAlreadyDiscovered
* SUMMARY:  MPP function that goes through RdacInfo data structure to find if a device has already 
* been discovered
* SCOPE:    local
*
* DESCRIPTION: We need to send VPD Inquiry pages to the storage array to determine the type of a LUN
* presented to the detect function of an Upper level driver. A Scsi_Device could have already been 
* discovered by another device driver. In this case we do not want to send a VPD Page Inquiry as 
* this would have already been done and is expensive. So the solution is to go through the RdacInfo
* data structure and look for the device and check and see if it has already been discovered. The 
* device can only be a Physical Device. We do not go through the entire list of luns on an array but
* instead index on the lun number. This Lun number is passed to mpp BuildRdacInfo from within which
* this function will be called.
*
* RESTRICTIONS:
* 
* RETURNS: TRUE on finding a match for the device in the RdacInfo data structure, else returns FALSE.
*/
/*******************************************************************************/

LWORD 
mppSys_DeviceAlreadyDiscovered (MPP_HANDLE Devicepnt, LWORD Lun ) 
{

	RdacDeviceInformation_t	*pRdacDevice;
	RdacLunInformation_t	*pRdacLun;
	LunPathInfo_t		*pLunPathInfo;
	LunPathObjects_t	*pLunPath;
	ArrayModuleEntry_t		*mppModule = NULL;
	int LunPath = 0, LunCont = 0, numModules = 0;
	unsigned char			nullWWN[WWN_LENGTH];

	memset(nullWWN, 0, WWN_LENGTH);

    	MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppSys_DeviceAlreadyDiscovered: called with vertex 0x%x, Lun %d\n", Devicepnt,Lun));
	if( Lun >= mpp_MaxLunsPerArray ) {
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 856, "Lun number(%d) exceeds Max(%d)\n",
			Lun, mpp_MaxLunsPerArray));
		return(FALSE);
	}


	for ( mppModule  = mpp_ModuleArray, numModules = 0; 
		numModules < mpp_MaxArrayModules; mppModule++, numModules++)
	{
        	if (! (pRdacDevice = mppModule->RdacInfo))
		{
			/* There is no array at this location as yet */
			continue;
		}
	      
		/* index on the Lun number passed to this function  */ 
		pRdacLun = &pRdacDevice->RdacLuns[Lun];

		if ( !bcmp((BYTE *) pRdacLun->WWN, (BYTE *) nullWWN, 16) )
		{
			/* That means that memory for the struct has been allocated and bzeroed but the lun has not been inserted as yet */

			continue;   
		}
		  
		/* Go through each of the controllers */	
		for( LunCont = 0; LunCont < 2; LunCont++)
		{
			pLunPath = pRdacLun->LunControllers[LunCont];

			/* Go through each path */
			/* We loop through the entire set of luns as otherwise on systems like linux we can have a problem. if there are 4 paths to a controller and of paths 0, 1, 2, 3, path 2 goes down, the NumberOfLunObjects field goes to 3 but only loop indexes 0, 1, 2 will be iterated through the next time. This is incorrect as path 3 will be left out of the loop.*/ 
			for( LunPath = 0; LunPath < mpp_MaxPathsPerController; LunPath++)
			{
				pLunPathInfo = &pLunPath->LunPaths[LunPath];
				if (!pLunPathInfo->LunPathDevice ||     
					pLunPathInfo->PathRemoveState)
				{
					/* Path or Physical device does not exist or is in state of removal */
					continue;
				}
				if ( pLunPathInfo->LunPathDevice == Devicepnt)
				{
					/* This is a physical device, return TRUE */
					MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
						"mppCmn_DeviceAlreadyDiscovered: already found device 0x%x, returning TRUE\n", Devicepnt));

					return TRUE;
				}
			}
		}

	}
	return FALSE;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppSys_AreControllerPathsFailed
* SUMMARY:  Checks whether or not all paths to the specified controller are failed 
* SCOPE:    public
*
* DESCRIPTION:This function is a system dependent function invoked by the mpp common 
* function. During the time of performing IO error analysis, the common functions give 
* the system dependent a chance to check path states of all paths to the specified controller. 
* This function should return TRUE if all paths are in failed state. Otherwise, returns FALSE.
*
* RESTRICTIONS: This function will return false even in the MPPCMN_FAILED_NEED_CHECK and
                MPPCMN_FAILED_CHECKING states.
*
* @RdacInfo - represents the storage array
  @ControllerIndex - Represents a controller in the redundant controller array
  @LUN - the virtual LUN number where an IO is addressed when the common function asks the
            path states
* RETURNS: This function should return TRUE if all paths are in failed, failed-need-checking
         or failed-checking states. Otherwise, returns FALSE
*/
/*******************************************************************************/
LWORD
mppSys_AreControllerPathsFailed(RdacDeviceInformation_t *RdacInfo, 
                                BYTE ControllerIndex, LWORD Lun)
{
   LINT                 LockLevel;
   BYTE                 Path;
   LWORD                stateIndex;
   MPPCMN_PATH_STATE    pathState;
   LWORD                retValue = TRUE;
   
   MPP_LOCK(RdacInfo->Lock, LockLevel);
   
   for(Path=0;Path< mpp_MaxPathsPerController;Path++)
   {
      if(RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[Path].Present
         && RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[Path].LunPathDevice)
      {
         stateIndex = RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[Path].PathStateIndex;
         pathState  = RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[Path].PathState[stateIndex];
         if((pathState == MPPCMN_OPTIMAL) ||
            (pathState == MPPCMN_OPTIMAL_NEED_CHECK) ||
            (pathState == MPPCMN_OPTIMAL_CHECKING) ||
            (pathState == MPPCMN_FAILED_NEED_CHECK) ||
            (pathState == MPPCMN_FAILED_CHECKING) )
         {
            retValue = FALSE;
            break;
         }
      }
   }
   MPP_UNLOCK(RdacInfo->Lock, LockLevel);
   return retValue;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_CopyRdacInfoToUser
* SUMMARY:  Sysdep function to copy rdacinfo to user buffer
* SCOPE:    local
*
*
* DESCRIPTION: This is a system dependent function to copy the RdacInfo
* 		data structures to the user buffers . This routine is
*		called from the ioctl.
*
* RESTRICTIONS:
* 
* RETURNS: 0
*
* ERRNO:
*
* NOTES: 
* 	ioctl status returned by the call to copy_to_user()
* 
********************************************************************************/
SINT
mppSys_CopyRdacInfoToUser( void *RdacInfoAdrs, RdacDeviceInformation_t *RdacInfo)
{
	LWORD			 	RdacDevInfoBlock_Size=0;
	LWORD				RdacCtlrInfoBlock_Size=0;
	LWORD				RdacLunPathObjBlock_Size=0;
	LWORD 				Lun;
	BYTE				*ptr=NULL;

	int ioctl_status = 0;

	RdacDevInfoBlock_Size 		= MPP_RDACDEVINFO_BLOCKSIZE( mpp_MaxLunsPerArray ) ; 
	RdacCtlrInfoBlock_Size 		= MPP_RDACCTLRINFO_BLOCKSIZE( mpp_MaxPathsPerController ) ; 
	RdacLunPathObjBlock_Size 	= MPP_RDACLUNPATHOBJ_BLOCKSIZE( mpp_MaxPathsPerController ) ; 

	ptr = (BYTE *)RdacInfoAdrs;

	/* copy the Device info and lun info data structures */
	ioctl_status = copy_to_user(ptr, (BYTE *)RdacInfo, RdacDevInfoBlock_Size);
	if( ioctl_status ) {
		ioctl_status = -EFAULT;
		return ioctl_status;
	}
	ptr += RdacDevInfoBlock_Size;	
	
	/* copy the Controller info and controller path data structures */
	ioctl_status = copy_to_user(ptr, (BYTE *)RdacInfo->ControllerInfo[0], RdacCtlrInfoBlock_Size);
	if( ioctl_status ) {
		ioctl_status = -EFAULT;
		return ioctl_status;
	}
	ptr += RdacCtlrInfoBlock_Size;

	/* copy the lunpath object and lun path info data structures */
	for(Lun=0; Lun < mpp_MaxLunsPerArray ;Lun++)
	{
		ioctl_status = copy_to_user(ptr, (BYTE *)RdacInfo->RdacLuns[Lun].LunControllers[0], RdacLunPathObjBlock_Size);
		if( ioctl_status ) {
			ioctl_status = -EFAULT;
			return ioctl_status;
		}
		ptr += RdacLunPathObjBlock_Size;


	}
	
	MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, 
		"mppLnx_CopyRdacInfoToUser() :: RdacInfoSize = %d\n", 
		RdacDevInfoBlock_Size+RdacCtlrInfoBlock_Size+(mpp_MaxLunsPerArray*RdacLunPathObjBlock_Size)));

	return ioctl_status;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppSys_GetUtilSysdepData
* SUMMARY:  Function that fills in the Virtual TargetId and VirtualHostId info.
* SCOPE:    local
*
* DESCRIPTION: This Function finds the first lun on an array which is present 
* and gets the host_no of the MPP Virtual HBA driver from the PseudoLunObject 
* for that lun. It also fills in the VirtualTargetId of the array.
*
* RESTRICTIONS:
* 
* RETURNS: None
*/
/*******************************************************************************/
void
mppSys_GetUtilSysdepData(RdacDeviceInformation_t *RdacInfo, mpp_utilStatusInfo_t *utilStatusInfo )
{
        RdacLunInformation_t    *pRdacLun;
        LWORD i;
   	int (*mppLnx_delete_PdeviceEntry) (struct scsi_device * ) = NULL;

	/* symbol_get and symbol_put are used to determine if the mppVhba driver is loaded or not.
	  Ideally, the scsi_host pointer, scsi_device->host should have been NULL for all the virtual devices
	  after removal of the mppVhba, but this cannot be guaranteed. Hence, this other method to determine
	  the presence or absence of the mppVhba has to be used - CR129397*/	

        for(i=0; i<mpp_MaxLunsPerArray; i++) {
                pRdacLun = &RdacInfo->RdacLuns[i];
	 	if ( pRdacLun && pRdacLun->PseudoLunObject ) {	
      			mppLnx_delete_PdeviceEntry = (void *) symbol_get(mppLnx_delete_PdeviceEntry);
		}
                if ( mppLnx_delete_PdeviceEntry ) {
	        		utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualHostId = pRdacLun->PseudoLunObject->host->host_no;
         			symbol_put(mppLnx_delete_PdeviceEntry);
                        	break;
		}
        }

        utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualTargetId = RdacInfo->VirtualTargetId;
	MPP_DEBUGPRINT((MPP_IOCTL_DEBUG+MPP_DEBUG_LEVEL_4, "mppSys_GetUtilSysdepData: VirtualHostId %d VirtualTargetId %d\n",utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualHostId, utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualTargetId ));


}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mpp_RemoveLunLink
* SUMMARY:  Function that removes a Lun Link (path) for the Linux mpp driver.
* SCOPE:    system dependent
*
* DESCRIPTION: This function is invoked while a physical path to a device is ready to be removed. This function will remove
* the physical device entry in the /proc/mpp/array_name/controllerX/host_target/ directory
*
* RESTRICTIONS:
* 
* RETURNS: None
*/
/*******************************************************************************/
void mpp_RemoveLunLink( RdacDeviceInformation_t *RdacInfo, 
                          BYTE Controller, 
                          BYTE Path, LWORD Lun)
{
  /*
   * we need to call int mppLnx_delete_PdeviceEntry(struct scsi_device * pdevice)
   */
   int (*mppLnx_delete_PdeviceEntry) (struct scsi_device * );

   struct scsi_device * psd = RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice;
   
   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp_RemoveLunLink- removing array %s:%d:%d:%d\n", RdacInfo->ModuleName, Controller,Path,Lun));

   if(psd != NULL )
   {
      mppLnx_delete_PdeviceEntry = (void *) symbol_get(mppLnx_delete_PdeviceEntry);
      if(mppLnx_delete_PdeviceEntry) 
      {
      	 (*mppLnx_delete_PdeviceEntry) (psd);
         RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice = NULL;
         symbol_put(mppLnx_delete_PdeviceEntry);
      }
      else
      {  
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 858, "mpp_RemoveLunLink: ERROR symbol_get did not return a function pointer\n"));
      }
   }
   else {
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 859, "mpp_RemoveLunLink: ERROR did not find pdevice for %s:%d:%d:%d\n", RdacInfo->ModuleName, Controller,Path,Lun));

   }

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp_RemoveLunLink: removed array %s:%d:%d:%d\n", RdacInfo->ModuleName, Controller,Path,Lun));
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mpp_RemoveVirtualLun
* SUMMARY:  Function that removes a virtual device from the Linux kernel
* SCOPE:    Linux system dependent
*
* DESCRIPTION: This function is invoked when a virtual device is ready to be removed. The condition to remove a virtual device
is that all physical paths to the virtual device have been removed. This function will remove the virtual device entry in 
/proc/mpp/array_name/. And then it schedules a task to remove the virtual device object from the Linux kernel from another thread.
* RESTRICTIONS:
* 
* RETURNS: None
*/
/*******************************************************************************/
void mpp_RemoveVirtualLun (RdacDeviceInformation_t *RdacInfo, LWORD Lun)
{
   /*
   * we need to call scsi_remove_device here.
   * the scsi_remove_device will call  mppLnx_slave_destroy(struct scsi_device * sdev)
   * to clean up the virtual lun proc directory
   */
   struct scsi_device * vsd = RdacInfo->RdacLuns[Lun].PseudoLunObject;
   mpp_hot_plug_event * mpp_hot_plug_info = NULL;
   unsigned long iflags; 

   int (*mppLnx_delete_VdeviceEntry) (void * vde );
   void (*mppLnx_wakeup_pathvalidate_thread) ( void );

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_RemoveVirtualLun: removing array %s lun%d\n", RdacInfo->ModuleName, Lun));
   
   if(NULL == vsd)
   {
        MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 860, "mppLnx_RemoveVirtualLun: ERROR did not find virtual scsi device %s lun %d\n", RdacInfo->ModuleName, Lun));
        /* Case where virtual host is not there but rmmod of mppVhba is called*/
        RdacInfo->RdacLuns[Lun].Present = 0;
        return;
   }

   if(vsd != NULL)
   {
      mpp_hot_plug_info = kmalloc(sizeof(mpp_hot_plug_event), GFP_ATOMIC);
      if (!mpp_hot_plug_info) {
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 861, "mppLnx_RemoveVirtualLun: ERROR could not malloc mpp_hot_plug_event\n"));
          return;
      }
      memset(mpp_hot_plug_info,0,sizeof(mpp_hot_plug_event));
      mpp_hot_plug_info->device = RdacInfo->RdacLuns[Lun].PseudoLunObject;
      mpp_hot_plug_info->hot_plug_lock = &mpp_hot_plug_spin_lock; 
      mpp_hot_plug_info->event_type = DEL_DEVICE;
      spin_lock_irqsave(mpp_hot_plug_info->hot_plug_lock, iflags);
      list_add_tail(&(mpp_hot_plug_info->list), &(mpp_hot_plug_list));
      spin_unlock_irqrestore(mpp_hot_plug_info->hot_plug_lock, iflags);
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_RemoveVirtualLun - added hot plug event array %s lun %d\n", RdacInfo->ModuleName, Lun));

      RdacInfo->RdacLuns[Lun].PseudoLunObject = NULL;
      RdacInfo->RdacLuns[Lun].Present = 0;
      RdacInfo->RdacLuns[Lun].ReportedMissing = TRUE;

      mppLnx_delete_VdeviceEntry = (void *) symbol_get(mppLnx_delete_VdeviceEntry);
      if(mppLnx_delete_VdeviceEntry)
      {
        if(NULL != RdacInfo->RdacLuns[Lun].PlatformPrivateData)
         (*mppLnx_delete_VdeviceEntry) ( RdacInfo->RdacLuns[Lun].PlatformPrivateData);
         // Lock needed here
         RdacInfo->RdacLuns[Lun].PlatformPrivateData = NULL;
         symbol_put(mppLnx_delete_VdeviceEntry);
      }
      else
      {
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 862, "mppLnx_RemoveVirtualLun: ERROR symbol_get did not return a function pointer for mppLnx_delete_VdeviceEntry\n"));
      }

   }
   else {
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 863, "mppLnx_RemoveVirtualLun: ERROR did not find scsi_device for virtual lun  array %s lun %d\n", RdacInfo->ModuleName, Lun));

   }

      mppLnx_wakeup_pathvalidate_thread = (void *) symbol_get(mppLnx_wakeup_pathvalidate_thread);
      if(mppLnx_wakeup_pathvalidate_thread)
      {
         (*mppLnx_wakeup_pathvalidate_thread) ( );
         symbol_put(mppLnx_wakeup_pathvalidate_thread);
      }
      else
      {
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 864, "mppLnx_RemoveVirtualLun: ERROR symbol_get did not return a function pointer for mppLnx_wakeup_pathvalidate_thread\n"));
      }

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_RemoveVirtualLun- removed array %s lun %d\n", RdacInfo->ModuleName, Lun));

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppSys_IsOkayToClearRdacLuns
* SUMMARY:  Function that tells the common code where or not clear up the 
RdacDeviceInformation_t memory for the virtual device
* SCOPE:    Linux system dependent
*
* DESCRIPTION: This function is invoked from the common code when the virtual device
 is ready to be removed. The common code asks the system dependent code where or not
to reset the RdacDeviceInformation_t object for this virtual device.
The Linux implementation will set the memory during slave_destroy time. This function just 
return FALSE
*
* RESTRICTIONS:
* 
* RETURNS: None
*/
/*******************************************************************************/
LWORD  
mppSys_IsOkayToClearRdacLuns(IN RdacDeviceInformation_t *RdacInfo, IN LWORD Lun)
{
   LWORD retV;
   /*
   *If it is UTM LUN, we would like to let the common code to clear up the RdacLuns
   *since UTM lun doesn't have virtual device
   */
   if( RdacInfo->RdacLuns[Lun].UTMLun )
   {
      retV = 1;
   }else
   {
      retV = 0;
   }

	return(retV);
}
/**************************************************************************
* PROCEDURE
*
* NAME:     mppSys_IsConfiguredDevice()
* SUMMARY:  Verify the given physical device, which is represented by the
   "MPP_HANDLE DeviceVertex", is a configured device in the storage array.
   This function issues "Mode Sense" command with Page 2C to the controller and
   verify whether or not the "Lun" is a configured LUN in the given storage partition.
*
* INPUT: DeviceVertex- A physical device object whose configured-status will be verified
*        DeviceAddress -A ControllerAddress_t object, which represents
                  the DeviceVertex's physical connections in a platform. It is not used
                  at the present code.
         Lun - the Lun number of the DeviceVertex.
* RETURNS: TRUE if the DeviceVertex is a configured LUN; Otherwise,  FALSE
* NOTES: this function used to be in the common code. The error code IDs are inherited 
*        from the common code and will not be modified for consistent purpose.
*************************************************************************/
BOOL
mppSys_IsConfiguredDevice(MPP_HANDLE DeviceVertex, ControllerAddress_t *DeviceAddress, LWORD Lun)
{

   BYTE                       *cdb;
   BYTE                       *page2C;
   BYTE                       MaxLuns;
   BOOL                       isPage2CSubPage1 = FALSE;
   LWORD                      PageSize;
   BYTE                       cdblen = 10;
   ModePage2CSubPage_t        *modePage2CSubPage;
   ModePage2C_t               *modePage2C;
   BOOL                       isConfigured;
   RdacDeviceInformation_t    *RdacInfo;
   BYTE                       ControllerIndex;
   BYTE                       PathIndex;
   LWORD                      lunIndex;

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppSys_IsConfiguredDevice: vertex 0x%x, ControllerAddress 0x%lx, Lun %d\n",
	    DeviceVertex, DeviceAddress, Lun));

   if(mpp_GetMaxLunsSupported(DeviceVertex, &MaxLuns))
   {

	   // Anything above 31 Luns must use the SubPage for 0x2C
      if(MaxLuns > 0x1f)
      {
		   isPage2CSubPage1 = TRUE;
         page2C = (BYTE *) MPP_INT_KMEM_ALLOC(sizeof(ModePage2CSubPage_t)+10);
         if(!page2C)
         {
            MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 866, "Unable to allocate memory\n"));
            return FALSE;
         }
         PageSize = sizeof(ModePage2CSubPage_t);

         cdb = ((BYTE *) page2C) + PageSize;
      }
      else
      {
         isPage2CSubPage1 = FALSE;
         page2C = (BYTE *) MPP_INT_KMEM_ALLOC(sizeof(ModePage2C_t)+10);
         if(!page2C)
         {
            MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 867, "Unable to allocate memory\n"));
            return FALSE;
         }
         PageSize = sizeof(ModePage2C_t);
         cdb = ((BYTE *) page2C) + PageSize;
		}
   }
   else
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 868, "Failed to get GetMaxLunsSupported\n"));
      return FALSE;
   }


   /*
   *preparing the CDB
   */
   *cdb = SCSI_MODE_SENSE;
   *(cdb+1) = 0x08; /*DBD bit on*/
   *(cdb+2) = (BYTE)(0x2c); /*PC = 00b (current) and Page code = 0x2c*/
   if (isPage2CSubPage1)
   {
      *(cdb+3) = 0x01;
   }
   else
   {
      *(cdb+3) = 0x00;
   }

   *(cdb+7) = (BYTE) (PageSize >> 8) &0x00ff;
   *(cdb+8) = (BYTE) PageSize & 0x00ff;
   if(mpp_SynchronousIo(DeviceVertex, cdb,cdblen, page2C,PageSize, MPPCMN_XFER_DEV_HOST) != MPP_NO_ERROR)
   {
      /*
      * log error
      */
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 869, "Failed to issue mode sense\n"));
      MPP_FREE(page2C, PageSize+10);
      return FALSE;
   }

   RdacInfo = mpp_MatchExistingControllerPath(DeviceAddress, &ControllerIndex, &PathIndex);

   if (isPage2CSubPage1)
   {
      modePage2CSubPage = (ModePage2CSubPage_t *)page2C;
      isConfigured = modePage2CSubPage->ControllerLunTable[Lun] & 0x03 ? TRUE:FALSE;
      if(RdacInfo)
      {
         for(lunIndex=0; lunIndex <mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
         {
            if(lunIndex <= MaxLuns)
            {
               RdacInfo->RdacLuns[lunIndex].NotConfigured = modePage2CSubPage->ControllerLunTable[lunIndex] & 0x03 ? 0:1;
            }else
            {
               RdacInfo->RdacLuns[lunIndex].NotConfigured = 1;
            }
         }
      }
   }
   else
   {
      modePage2C = (ModePage2C_t*)page2C;
      isConfigured = modePage2C->ControllerLunTable[Lun] & 0x03 ? TRUE:FALSE;
      if(RdacInfo)
      {
         for(lunIndex=0; lunIndex <mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
         {
            if(lunIndex <= MaxLuns)
            {
               RdacInfo->RdacLuns[lunIndex].NotConfigured = modePage2C->ControllerLunTable[lunIndex] & 0x03 ? 0:1;
            }else
            {
               RdacInfo->RdacLuns[lunIndex].NotConfigured = 1;
            }
         }
      }
   }

   MPP_FREE(page2C, PageSize+10);
   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "mppSys_IsConfiguredDevice: vertex 0x%x, ControllerAddress 0x%lx, Lun %d, isConfigured %d\n",
	    DeviceVertex, DeviceAddress, Lun, isConfigured));

   return isConfigured;
}

/**************************************************************************
* PROCEDURE
*
* NAME:     mppSys_QueueRequest
* SUMMARY:  System-dependent wrapper for queueing worker thread requests.
* SCOPE:   public
* DESCRIPTION: This function is just a dummy function in the Linux Upper Level driver.
*	The real functionality for this function is provided by the Linux Virtual
*	HBA driver. Since the Upper Level driver is loaded before the virtual hba
*	driver, and we want to avoid circular dependencies, we use inter-module communication
*	to call the actual virtual hba function.
* RESTRICTIONS:
* RETURNS:
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/

LWORD
mppSys_QueueRequest(
	IN MPPCMN_QUEUE_REQUEST_CALLBACK RequestCallback,
	IN void *Context, 
	IN LWORD DelaySecs)
{
	LWORD returnvalue=0;
	/*
	* The function pointer mppLnx_QueueRequest will be retrieved from the mppVhba driver module through symbol_get() kernel API.
	* the name of the function pointer must be same as that of the exported symbol name of the mppVhba driver module.
	* The exported symbols of the mppVhba driver module is defined in mppLnx26_vhbamisc.c
	* 
	* 01/28/08 - CR136027.  Added new argument that can be used to specify whether the
	* callee wants to delay the queued request.  For now Linux will ignore this option.
	*/

	LWORD (*mppLnx_QueueRequest) ( MPPCMN_QUEUE_REQUEST_CALLBACK, void *, LWORD);

	mppLnx_QueueRequest = (void *) symbol_get(mppLnx_QueueRequest);

	if(mppLnx_QueueRequest)
	{
		returnvalue = (*mppLnx_QueueRequest) (RequestCallback, Context, DelaySecs);
		symbol_put(mppLnx_QueueRequest);
	}
	else
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 865, "mppSys_QueueRequest : symbol_get did not return a function pointer\n"));
	}

	return returnvalue;
}
