/*******************************************************************************
* Copyright 2004-2008 LSI Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppLnx26p_vhba.c
SUMMARY         %description%
VERSION         %version: 23 %
UPDATE DATE     %date_modified: Thu Sep 04 10:03:19 2008 %
PROGRAMMER      %created_by:    bmoger %


DESCRIPTION:
   The Linux MPP virtual HBA driver is a lower level driver in the Linux SCSI subsystem. 
   A lower level driver must implement struct scsi_host_template data struct. This source 
   file contains all functions that are required by the Linux Scsi middle level. The 
   responsibilities of the functions in this file include:
    1. Initialization of the module
    2. Registration with the SCSI middle level
    3. Receive SCSI commands for any virtual devices from the middle level
    4. Provides all required function interfaces to the middle level
    5. Clearly clean system resources and unregister with the Kernel at the module unloading time.

INCLUDE FILES:

NOTES: The error number for this file ranges from 900 to 940.

RESTRICTIONS:

SEE ALSO:

REFERENCE:

  349-1043860 Linux MPP driver for Linux 2.6 Kernel
  349-1046490 On-The-Fly Path Validation

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/

#define __SRCmppLnx26p_vhba_c

/***  Linux kernel INCLUDES  ***/  
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/genhd.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/device.h>
#include <linux/interrupt.h>
#include <scsi/scsi_host.h>
#include <scsi/scsicam.h>
#include <scsi/scsi_tcq.h>
#include <scsi/scsi_eh.h>

#include <asm/io.h>

#include <linux/blkdev.h>


#include <linux/stat.h>

/***  MPP commomn API INCLUDES  ***/  
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "MPP_Sysdep.h"
#include "mppLnx_version.h"
#include "mppCmn_s2tos3.h"

/***  Linux MPP Virtual HBA INCLUDES  ***/  
#include "mppLnx26p_vhba.h"
#include "mppLnx26p_shared.h"
/***  CONSTANT DEFINITIONS  ***/

/***  MACRO DEFINITIONS  ***/
#define         YES_NO(a)               a == 1 ? 'Y' : 'N'
#define         A_OR_B(a)               a == 0 ? 'A' : 'B'
/*
* A macro used in mppLnx_proc_info for maintaining write-buffer position.
*/
#define  VHBA_PROC_INFO_POS  {   \
         pos = pos + len;  \
         if(pos > (MPPLNX_PROC_INFO_BUFFSIZE -256)) \
         { \
            isTruncated = 1; \
            goto proc_info_output_stop; \
         } \
   };
/*
* Mpp status string used for debug-print
*/
char * mppStatusStrings[] = 
{
      "MPP_GOOD_STATUS",
      "MPP_CHECK_CONDITION",
      "MPP_SCSI_BUSY",
      "MPP_SCSI_RESERVATION",
      "MPP_UNSUPPORTED_SCSI_STATUS",
      "MPP_INVALID_REQUEST",
      "MPP_SELECTION_TIMEOUT",
      "MPP_HARDWARE_ERROR",
      "MPP_COMMAND_TIMEOUT",
      "MPP_COMMAND_ABORTED",
      "MPP_UNRECOGNIZED_OS_STATUS",
      "MPP_REQUEST_SENSE_FAILED"
};
/*
* mpp action string used for debug-print
*/
char * mppActionStrings[] = 
{
      "MPP_NO_ERROR",
      "MPP_RETURN_ERROR",
      "MPP_RETRY_DEC_COUNT",
      "MPP_RETRY_NO_DEC",
      "MPP_RETRY_ALTERNATE_CONTROLLER",
      "MPP_FAILOVER",
      "MPP_FAILOVER_CONTROLLER",
      "MPP_FAILOVER_TO_CURRENT"
};
/***  TYPE DEFINITIONS  ***/

/***  LOCALS  ***/
/*
* proxy command timeout
* the value of a proxy command timeout is dependent on physical HBAs.
* We set the default value to 25 seconds.
* The value will be reset based on a physical HBA
*/
int mppLnx_proxy_cmnd_timeout;
static char * QLA_DRIVER_PROC_NAME_PREFIX     = "qla";

extern mppLnx_VHBAContext_t mppLnxVHBAContext;

/*
* 2.6 kernel device/driver related variables
*/
/*
* Whether or not the virtual host has been initialized
*/
static int mppLnx_is_virtual_hba_initialized = 0;
/*
* the device name of the primary device
*/
#define MPPLNX_MPP_PRIMARY_DEVICE_NAME "mppvirtual"

static struct device mppLnx_primary_device = {
   .bus_id = MPPLNX_MPP_PRIMARY_DEVICE_NAME,
   .release = mppLnx_primary_device_release,
};
/*
* the virtual mpp bus
*/
#define MPPLNX_MPP_VBUS_NAME  "mppvbus"

static struct bus_type mppLnx_mppv_bus = {
   .name = MPPLNX_MPP_VBUS_NAME,
   .match = mppLnx_mppvbus_match,
};
/*
*the virtual bus driver
*/
#define MPPLNX_MPP_VBUS_DRIVER_NAME "mppvdb"

static struct device_driver mppLnx_mppvbus_driver = {
   .name = MPPLNX_MPP_VBUS_DRIVER_NAME,
   .bus = &mppLnx_mppv_bus,
   .probe = mppLnx_mppvbus_driver_probe,
   .remove = mppLnx_mppvbus_driver_remove,
};

#define MPPLNX_VADAPTER_NAME  "mppadapter0"

static struct device mppLnx_mppvadapter0 = {
   .bus_id = MPPLNX_VADAPTER_NAME,
   .bus = &mppLnx_mppv_bus,
   .parent = &mppLnx_primary_device,
   .release = mppLnx_mppvadapter0_release,
};

/*
* the dispatch-and-completion thread related declaration
*/
mppLnx_dpcQueueContext_t   mppLnxDpcQueueContext;


struct semaphore mppLnx_dpc_sem;

/*
* fail back scan thread
*/
struct semaphore mppLnx_detect_sem;
struct semaphore mppLnx_failback_sem;
mppLnx_failbackScanContext_t mppLnxFailbackScanContext;

/*
* path validation scan thread which performs PRRegistration checks 
*/
struct semaphore mppLnx_pathvalidate_sem;
mppLnx_pathvalidateContext_t mppLnxPathValidateContext;

/*
* syncIO worker thread and its queue
*/
struct semaphore mppLnx_syncIOWorker_sem;
mppLnx_workerQueueContext_t mppLnxWorkerQueueContextContext;

/*
* mpp_wq thread to take care of only Failover cmnd
*/
struct workqueue_struct *mppLnx_wq = NULL;

/*
* virtual device creation handling
*/
int mppLnx_workTaskRefCnt = 0;
int mppLnx_startVhbaModRemove = 0;
/*
* The Scsi middle level does not make its scsi_done() function (lower level command
* completion function) and its time out function public. We must get them from a struct scsi_cmnd
* and save them for later use.
*/
void (*middle_level_scsi_done)(struct scsi_cmnd *) = NULL;
void (*middle_level_timeoutfunc)(struct scsi_cmnd *) = NULL;
/***  PROCEDURES  ***/
static void mppLnx_set_command_timeout(struct scsi_host_template *pshtp);
static void mppLnx_vhba_setupHostTemplate(struct scsi_host_template * pshtp, struct Scsi_Host *pshp);
static void mppLnx_SetTAS_All_Arrays(void);
static int mppLnx_vhba_regVirtualHost(struct Scsi_Host ** vshp,struct Scsi_Host * pshp,int hostnumber, int fromVscan);
static int mppLnx_get_min_queue_depth(struct scsi_device * vsdp);

static void mppLnx_checkpath_timer_init(void);
static void mppLnx_checkpath_timer_function( unsigned long data);
static void mppLnx_checkpath_timer_exit(void);

static void mppLnx_cleanup_requests(int phase);
static BOOL mppLnx_needsInitFailover(void);
static void mppLnx_delayHostScan(struct Scsi_Host *vhost);
static void mppLnx_genVdAddEvent(
   RdacDeviceInformation_t *rdacInfo, 
   struct scsi_device * sdp,
   int tid, int lun);
static BOOL mppLnx_readyAddVd(mppLnx_vdAddEvent_t * vdAddEvent);

/* 
* The virtual HBA driver initialization and exit. The Scsi middle level source file
* "scsi_module.c will take care of this lower level driver's initialization/registration
* and un-registration.
*/
static struct scsi_host_template  mppLnx_vhost_template = {
   .module	                           = THIS_MODULE,
   .name	                              = MPPLNX_PROC_NAME,
   .info	                              = mppLnx_info,
   .queuecommand	                     = mppLnx_queuecommand,
   .change_queue_depth                 = mppLnx_change_queue_depth,
   .eh_device_reset_handler	         = mppLnx_eh_device_reset_handler,
   .eh_bus_reset_handler	            = mppLnx_eh_bus_reset_handler,
   .eh_host_reset_handler	            = mppLnx_eh_host_reset_handler,
   .slave_alloc	                     = mppLnx_slave_alloc,
   .slave_configure	                  = mppLnx_slave_configure,
   .slave_destroy	                     = mppLnx_slave_destroy,
   .bios_param	                        = NULL,
   .proc_info		                     = mppLnx_proc_info,
   .proc_name	                        = MPPLNX_PROC_NAME,
   .this_id	                           = -1,
};
/*
* the virtual HBA host
*/
static struct Scsi_Host        * mppLnx_vhba_virtual_host_fc = NULL;      // virtual linux scsi host for fibre channel
static struct Scsi_Host        * mppLnx_vhba_virtual_host_iscsi = NULL;   // virtual linux scsi host for iscsi
static struct Scsi_Host        * mppLnx_vhba_virtual_host_sas = NULL;     // virtual linux scsi host for sas
static struct Scsi_Host        * mppLnx_vhba_virtual_host_ib = NULL;      // virtual linux scsi host for infiniband

/*
 * Timer function to call mppCmn_CheckPathGood() every Interval.
 */
#define 				MPPLNX_CHECK_PATH_TIME_INTERVAL 	( 1 * HZ )


#define           MPPLNX_CLEAN_FIRST_PHASE      1
#define           MPPLNX_CLEAN_SECOND_PHASE     2

#define           MPPLNX_DEVICE_SCAN_WAITTIME   mppLnx_Config_Parameters.mpp_ControllerIoWaitTime
#define           MPPLNX_BOOT_DISCOVERY_FAILOVER_WAIT   mppLnx_Config_Parameters.mpp_ArrayIoWaitTime

struct timer_list		mppLnx_checkpath_timer;

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_init 
* SUMMARY:  The virtual HBA driver's module initialization function.
* SCOPE:    public
* DESCRIPTION: This function is assigned to kernel module API module_init(). This
               function is the initialization function of the virtual HBA driver module. 
               It will perform the following tasks:
                  register the primary device with the kernel
                  register the virtual bus with the kernel
                  register the virtual bus driver with the kernel
                  register the virtual adapter with the kernel
*
* RESTRICTIONS:
* RETURNS: 0 if the initialization is successful. Otherwise, 1
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
static int __init mppLnx_vhba_init(void)
{
   int error;

   /*
   * Create the mpp_wq worker thread CR132259
   */
   mppLnx_wq = create_singlethread_workqueue("mpp_wq");
   if(!mppLnx_wq)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 902,
         "mppLnx_wq failed to create \n"));
      return 1;
   }

   error = device_register(&mppLnx_primary_device);
   if(error)
   {
      goto cleanup_primary;
   }

   error = bus_register(&mppLnx_mppv_bus);
   if(error)
   {
      goto cleanup_bus;
   }

   error = driver_register(&mppLnx_mppvbus_driver);
   if(error)
   {
      goto cleanup_driver;
   }

   error = device_register(&mppLnx_mppvadapter0);
   if(error)
   {
      goto cleanup_adapter;
   }

   return 0;

cleanup_adapter:
   driver_unregister(&mppLnx_mppvbus_driver);

cleanup_driver:
   bus_unregister(&mppLnx_mppv_bus);

cleanup_bus:
   device_unregister(&mppLnx_primary_device);

cleanup_primary:
   destroy_workqueue(mppLnx_wq);
   return 1;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_exit
* SUMMARY:  The virtual HBA driver's module exit function
* SCOPE:    public
* DESCRIPTION: This function is assigned to kernel module API module_exit(). This
               function is the exiting function of the virtual HBA driver module. 
               It should perform the reverse function as the mppLnx_vhba_init() function
               It will perform the following tasks:

                  unregister the virtual adapter with the kernel
                  unregister the virtual bus driver with the kernel
                  unregister the virtual bus with the kernel
                  unregister the primary device with the kernel
                  kill the work queue
*
* RESTRICTIONS:
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
static void __exit mppLnx_vhba_exit(void)
{
   device_unregister(&mppLnx_mppvadapter0);
   driver_unregister(&mppLnx_mppvbus_driver);
   bus_unregister(&mppLnx_mppv_bus);
   device_unregister(&mppLnx_primary_device);
   if(mppLnx_wq != NULL)
      destroy_workqueue(mppLnx_wq);
   return;

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_checkpath_timer_init
* SUMMARY:  Timer initialization function for checking path every interval.
* SCOPE:    public
* DESCRIPTION: Timer initialization function for checking path every interval .
*
* RESTRICTIONS:
* RETURNS: VOID
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
static void mppLnx_checkpath_timer_init(void)
{
	init_timer ( &mppLnx_checkpath_timer );

	/* call the "function " when timer expires */
	mppLnx_checkpath_timer.function = mppLnx_checkpath_timer_function; 

	mppLnx_checkpath_timer.data     = 0;

	mppLnx_checkpath_timer.expires  = jiffies + MPPLNX_CHECK_PATH_TIME_INTERVAL;  /* expires every interval */

	add_timer ( &mppLnx_checkpath_timer );
	return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_checkpath_timer_function
* SUMMARY:  Timer  function for checking path every interval.
* SCOPE:    public
* DESCRIPTION: Timer  function for checking path every interval .
*
* RESTRICTIONS:
* RETURNS: VOID
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
static void mppLnx_checkpath_timer_function(unsigned long data)
{
	
       mppCmn_CheckPathStatus();	
	
	mppLnx_checkpath_timer.expires  = jiffies + MPPLNX_CHECK_PATH_TIME_INTERVAL ;  
	
	add_timer ( &mppLnx_checkpath_timer );
	return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_checkpath_timer_exit
* SUMMARY:  Clean up function for check path timer.
* SCOPE:    public
* DESCRIPTION: Clean up function for check path timer.
*
* RESTRICTIONS:
* RETURNS: VOID
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
static void mppLnx_checkpath_timer_exit(void)
{
	del_timer ( &mppLnx_checkpath_timer );

	return;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_primary_device_release
* SUMMARY:  The virtual HBA primary device release callback function
* SCOPE:    public
* DESCRIPTION: This function is assigned to the primary device's release function.
                The kernel will invoke this function at the time of the primary 
                device is being removed.
*
* RESTRICTIONS:
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*******************************************************************************/


void mppLnx_primary_device_release(struct device * dev)
{
   /*
   * don't have anything to do for now.
   * but it is required by the kernel
   */
   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_mppvadapter0_release
* SUMMARY:  The virtual adapter device release callback function
* SCOPE:    public
* DESCRIPTION: This function is assigned to the virtual adapter device's release function.
                The kernel will invoke this function at the time of the virtual adapter 
                device is being removed.
*
* RESTRICTIONS:
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*******************************************************************************/


void mppLnx_mppvadapter0_release(struct device * dev)
{
   /*
   * don't have anything to do for now.
   * but it is required by the kernel
   */
}

/*******************************************************************************
* PROCEDURE
*
* NAME:      mppLnx_mppvbus_match 
* SUMMARY:  Checks whether or not the device and the device driver passed in are 
*           matched.
* SCOPE:    public
* DESCRIPTION: The kernel invokes this callback function when a device is about
*               to bind to a driver.
*
* @dev The device object which is about to bind to a bus device driver
* @drv  The device driver object for which the device would like to bind to
* RESTRICTIONS:
* RETURNS: 1 if the device and device driver are matched. Otherwise, 0
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
int  mppLnx_mppvbus_match(struct device * dev, struct device_driver * drv)
{
   int retvalue = 0;
   /* 
   * check whether or not the device is mppadapter0 and the driver is "mppvdb"
   */
   
   if((strcmp(dev->bus_id, MPPLNX_VADAPTER_NAME) == 0) && 
         (strcmp(drv->name, MPPLNX_MPP_VBUS_DRIVER_NAME) == 0) )
   {
      retvalue = 1;
   }

   return retvalue;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:      mppLnx_register_virtual_hosts 
* SUMMARY:   Create or Modify virtual hosts based on the physical hosts detected.
*            
* SCOPE:    public
* DESCRIPTION: This routine is called from mppLnx_mppvbus_driver_probe() during vhba
*              load time via initrd or mppLnx_vhbaScanHost() during hot_add or dynamic
*              device discovery.             
*              This routines walks through all the physical hosts detected and calls
*              the  mppLnx_vhba_regVirtualHost() for the physical host interface type.
*
* RESTRICTIONS:
* RETURNS: 
*           
* ERRNO:
*
* NOTES:
*******************************************************************************/
void mppLnx_register_virtual_hosts(int fromVscan)
{
   int     vtarget =0;
   int     vhost_sas=0;
   int     vhost_ib=0;
   int     vhost_fc=0;
   int     vhost_iscsi=0;
   int     phost_sas_no=-1;
   int     phost_ib_no=-1;
   int     phost_fc_no=-1;
   int     phost_iscsi_no=-1;
   int     error=0;
   int     lun=0;
   RdacDeviceInformation_t 	   *rdacInfo=NULL;
   struct Scsi_Host            *pshp=NULL;
   mppLnx_PhostEntry_t         *phostEntry=NULL;
   mppLnx_PtargetEntry_t       *pTargetEntry = NULL;
   struct  list_head           *hostList;
   struct  list_head           *targetList;
 

   /* look up array module list and register individual host interfaces */
   for(vtarget = 0; vtarget < mppLnx_Config_Parameters.mpp_MaxArrayModules; vtarget++)
   {
      if( (rdacInfo = mpp_ModuleArray[vtarget].RdacInfo) == NULL)
      {
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
            "no storage array at virtual target %d\n",vtarget));
         continue;
      }else
      {
         switch( rdacInfo->ArrayType )
         {
             case MPPCMN_TARGET_FC:
                    vhost_fc++;
                    break;
             case MPPCMN_TARGET_IB:
                    vhost_ib++;
                    break;
             case MPPCMN_TARGET_ISCSI:
                    vhost_iscsi++;
                    break;
             case MPPCMN_TARGET_SAS:
                    vhost_sas++;
                    break;
             default:
                    break;
         }

         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                             "mppLnx_register_virtual_hosts :: from %d array modules detected fc %d : sas %d : iscsi %d : ib %d\n",
                             fromVscan,vhost_fc,vhost_sas,vhost_iscsi,vhost_ib));
      }
   }

   if( NULL == mppLnxVHBAContext.phostList )
   {
       MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,"mppLnx_register_virtual_hosts :: No physical hosts detected.\n"));
       return;
   }
  
   for(hostList = mppLnxVHBAContext.phostList->list.next; 
         hostList != &(mppLnxVHBAContext.phostList->list); 
         hostList =hostList->next)
   {
      phostEntry = list_entry(hostList, mppLnx_PhostEntry_t, list);
      
      /* should not hit this case DVS */
      if( phostEntry == NULL ) break;
     
      /* no targets detected for this host - should not hit this case DVS */
      if( phostEntry->targetList == NULL ) break;
 
      /*
       * find its target
       */
      for(targetList = phostEntry->targetList->list.next;
            targetList != &(phostEntry->targetList->list);
            targetList =targetList->next)
      {
          pTargetEntry = list_entry(targetList, mppLnx_PtargetEntry_t, list);

          /* should not hit this case DVS */
          if( pTargetEntry == NULL ) break;

          /* find the first lun that has valid rdacInfo data structure */
          for(lun=0; lun<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lun++)
          {
             if( pTargetEntry->pdevices[lun] == NULL ) continue;

             rdacInfo = pTargetEntry->pdevices[lun]->rdacInfo;

             if( rdacInfo == NULL ) continue;
            
          } /* for all luns on pTargetEntry */

      } /* for all targets on phostEntry */

       if( rdacInfo == NULL ) continue; 

       switch( rdacInfo->ArrayType )
       {
           case MPPCMN_TARGET_FC:
                  /* initialize virtual fc host */
                  phost_fc_no = phostEntry->hostNo;
                  error=0; 

                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                                 "mppLnx_register_virtual_hosts :: before mppLnx_vhba_regVirtualHost_fc %d phost_no = %d\n",error,phost_fc_no));

                  error = mppLnx_vhba_regVirtualHost(&mppLnx_vhba_virtual_host_fc,pshp,phost_fc_no,fromVscan);
                  if ( error )
                     printk(KERN_WARNING "mppLnx_register_virtual_hosts :: 1 after mppLnx_vhba_regVirtualHost_fc error %d \n",error);
                  break;
           case MPPCMN_TARGET_IB:
                  /* initialize virtual ib host */
                  phost_ib_no = phostEntry->hostNo;
                  error=0;

                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                                 "mppLnx_register_virtual_hosts :: before mppLnx_vhba_regVirtualHost_ib %d phost_no = %d\n",error,phost_ib_no));

                  error = mppLnx_vhba_regVirtualHost(&mppLnx_vhba_virtual_host_ib,pshp,phost_ib_no,fromVscan);
                  if ( error )
                     printk(KERN_WARNING "mppLnx_register_virtual_hosts :: 1 after mppLnx_vhba_regVirtualHost_ib error %d \n",error);
                  break;
           case MPPCMN_TARGET_ISCSI:
                  /* initialize virtual iscsi host */
                  phost_iscsi_no = phostEntry->hostNo;
                  error=0;

                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                                 "mppLnx_register_virtual_hosts :: before mppLnx_vhba_regVirtualHost_iscsi %d phost_no = %d\n",error,phost_iscsi_no));

                  error = mppLnx_vhba_regVirtualHost(&mppLnx_vhba_virtual_host_iscsi,pshp,phost_iscsi_no,fromVscan);
                  if ( error )
                     printk(KERN_WARNING "mppLnx_register_virtual_hosts :: 1 after mppLnx_vhba_regVirtualHost_iscsi error %d \n",error);
                  break;
           case MPPCMN_TARGET_SAS:
                  /* initialize virtual sas host */
                  phost_sas_no = phostEntry->hostNo;
                  error=0;

                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                                 "mppLnx_register_virtual_hosts :: before mppLnx_vhba_regVirtualHost_sas %d phost_no = %d\n",error,phost_sas_no));

                  error = mppLnx_vhba_regVirtualHost(&mppLnx_vhba_virtual_host_sas,pshp,phost_sas_no,fromVscan);
                  if ( error )
                     printk(KERN_WARNING "mppLnx_register_virtual_hosts :: 1 after mppLnx_vhba_regVirtualHost_sas error %d \n",error);
                  break;
           default:
                  /* unrecognized host type */
                  break;
       }

       MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                         "mppLnx_register_virtual_hosts :: from %d physical hosts detected fc %d : sas %d : iscsi %d : ib %d\n",
                          fromVscan, phost_fc_no,phost_sas_no,phost_iscsi_no,phost_ib_no));
 
    } /* for all physical hosts detected */      


   return;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:      mppLnx_mppvbus_driver_probe 
* SUMMARY:   Probe virtual scsi host that is connected to the mppadapter0 virtual
*            adapter device
* SCOPE:    public
* DESCRIPTION: When the virtual adapter, mppadapter0, is added to the kernel, 
               this mppLnx_mppvbus_driver_probe() callback function, which is the 
               mpp virtual bus device driver's probe() function, will be invoked by
               the kernel. This function will first check whether or not 
               ArrayModuleEntry (from common code) has any storage array devices are 
               discovered. If yes, this function will invoke mppLnx_init_virtual_hba() 
               to create/add/scan all virtual targets and virtual devices. If there 
               is no virtual devices are discovered, it will delay the initialization 
               of the virtual scsi host. End user needs to perform hot_add from user 
               space to initiate the virtual scsi host creation. The interface is 
               mppLnx_vhbaScanHost().
*
* @dev The device object which is the mppadaper0 object
* RESTRICTIONS:
* RETURNS: 0 if the probe() is successful. Otherwise, 1. This function will always 
*           return 0 to allow the driver bind to the mppadapter0 device.
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
int  mppLnx_mppvbus_driver_probe(struct device * dev)
{
   
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,"Entering mppLnx_mppvbus_driver_probe()\n"));

   /*
   *check if the upper level driver is loaded. If not, don't try
   *to load this driver
   */
   if(mpp_ModuleArray == NULL)
   {
      /*
      * if the upper level driver is not loaded, we will not get here because the virtual
      * HBA driver requires symbols from the upper level driver. The kernel will report 
      * unknown symbol in module error before getting here
      */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,900, 
         "mpp:vhba:mppLnx_mppvbus_driver_probe: the upper level driver is not loaded\n"));
      
      printk(KERN_WARNING "The mpp virtual HBA driver requires the mpp Upper level driver to present\n");
      mppLnx_is_virtual_hba_initialized = 0;
      return 0;
   }
   
    /*
    * initialize internal data struct mppLnx_VHBAContext and create
    * proc node
    */
    mppLnx_initVHBAContext();
   /*
   * check whether or not there are Engenio devices. 
   * 
   */
   if(NULL == mppLnxVHBAContext.phostList)
   {
      mppLnx_is_virtual_hba_initialized = 0;
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,906,
         "No supported SCSI devices are discovered. Please run \"hot_add\" after supported storage arrays are attached to the Linux host.\n"));
      return 0;
   }

   /* initialize common virtual hba tasks for all types of interfaces fc, iscsi ,sas and ib */
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,"mppLnx_mppvbus_driver_probe :: before init_virtual_hba\n"));
   mppLnx_init_virtual_hba(0);
   mppLnx_is_virtual_hba_initialized = 1;
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,"mppLnx_mppvbus_driver_probe :: after init_virtual_hba\n"));

   mppLnx_register_virtual_hosts(0); 

    MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,"driver probe exiting. return 0\n"));
   return 0;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhbaScanHost
* SUMMARY:  create the virtual HBA Host if it is not created during the driver loading time
* SCOPE:    public
* DESCRIPTION: During the virtual HBA driver loading time, we may not create the virtual 
   scsi host if no Engenio's device is discovered or the HBA cables are not connected.
   This function is called when a user initiates a  space program (mppUtil) to rescan
   the virtual bus.
*
* RESTRICTIONS:
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*******************************************************************************/
void mppLnx_vhbaScanHost(int fromWorkQ)
{
   int                        vTarget =0;
   RdacDeviceInformation_t 	   *rdacInfo;

   if(!fromWorkQ && mppLnx_workTaskRefCnt)
   {
      MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
            "mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
      /*
      *if it is invoked from the mppUtil and the virtual device creation work tasks are
      * going on, don't do anything. Let the work tasks to do the job.
      */
      return;
   }

   if(mppLnx_is_virtual_hba_initialized == 0)
   {
      /*
      * initialize internal data struct mppLnx_VHBAContext and create
      * proc node
      */
      mppLnx_initVHBAContext();
      if(NULL != mppLnxVHBAContext.phostList)
      { 
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_2,"mppLnx_vhbaScanHost :: before init_virtual_hba \n"));
         mppLnx_init_virtual_hba(fromWorkQ);
         mppLnx_is_virtual_hba_initialized = 1;
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_2,"mppLnx_vhbaScanHost :: after init_virtual_hba \n"));
      }
      else 
      {
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,"mppLnx_vhbaScanHost :: No physical hosts detected.\n"));
         return;
      }
   }

   mppLnx_register_virtual_hosts(fromWorkQ); 

   for(vTarget = 0; vTarget < mppLnx_Config_Parameters.mpp_MaxArrayModules; vTarget++)
   {
      if( NULL != (rdacInfo = mpp_ModuleArray[vTarget].RdacInfo) )
      {
         mppLnx_checkPdevEntry(rdacInfo);
      }
   }

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_mppvbus_driver_remove
* SUMMARY:  The mpp virtual bus driver remove() callback function
* SCOPE:    public
* DESCRIPTION: This function is assigned to the virtual bus driver remove function.
               It is invoked when the mpp virtual adapter mppadapter0 is about to be 
               removed. This function will perform the reverse operation of 
               the mppLnx_mppvbus_driver_probe(). The tasks include:
                  1. stop all running thread
                  2. clean-up resources that are allocated during 
                    mppLnx_mppvbus_driver_probe()
                  3. remove the virtual scsi host from the middle level
*
* RESTRICTIONS:
* RETURNS:  0 indicates a successful remove. 1 otherwise. The kernel doesn't use 
            the return value.
*
* ERRNO:
*
* NOTES:
*******************************************************************************/

int  mppLnx_mppvbus_driver_remove(struct device * dev)
{
   /*
   * Here we should do the reverse operation of the mppLnx_mppvbus_driver_probe()
   */
   if(mppLnx_is_virtual_hba_initialized)
   {

      mppLnx_release_virtual_host();
   }
   else  /* The case when virutal host is not initiazed but rmmod of mppVhba is called */
   {
      /* If the workqueue is there for adding the virtual devices then clean it*/
           if (mppLnx_workTaskRefCnt)
           {
              mppLnx_startVhbaModRemove = 1;
              msleep(2*HZ);
              flush_scheduled_work();
           }
           /* This will clean the Physical LUN from the proc*/
           mppLnx_cleanVHBAContext();
   }
   return 0;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:      mppLnx_proc_info 
* SUMMARY:  Returns the virtual HBA's state information to the user space via
 /proc file system.
* SCOPE:    public
*
*
* DESCRIPTION:This function is an interface to the mpp virtual bus driver's proc 
   file /proc/scsi/mpp/n, where "n" is this virtual HBA driver's host number. The 
   host number is assigned by the Scsi middle level and is dependent on Scsi host 
   registration order.
   This function will be assigned to struct scsi_host_template data struct's "proc_info" 
   function pointer.
   This function is invoked when an end user types "cat /proc/scsi/mpp/n".
   The virtual HBA driver will not take input from the /proc file. The function 
   reports the following information to the caller.
      Mpp Driver verion information. This includes mpp driver version and driver
      build time
      Virtual HBA attributes
      Physical HBA driver name of each attached physical HBA
      Physical HBA host attributes such as supported maximum target id, supported
      maximum LUNs
      Each managed storage array attributes/states
*
*
* writeto1_read0 : decides the direction of the dataflow and the meaning of the
*         variables
* buffer: If inout==FALSE data is being written to it else read from it
*         (ptr to a page buffer)
* *start: If inout==FALSE start of the valid data in the buffer
* offset: If inout==FALSE starting offset from the beginning of all
*         possible data to return.
* length: If inout==FALSE max number of bytes to be written into the buffer
*         else number of bytes in "buffer"
*      @writeto1_read0: 1 -> data coming from user space towards driver
*                            (e.g. "echo some_string > /proc/scsi/mpp/2")
*                       0 -> user what data from this driver
*                            (e.g. "cat /proc/scsi/mpp/2")
* RESTRICTIONS:
*        The virtual HBA driver will not take input from the /proc file.
* 
* RETURNS:number of chars output to buffer past offset.
*
* ERRNO:
*
* NOTES:Driven from drivers/scsi/scsi_proc.c which interfaces to proc_fs
* offset is 0 or greater than 0. But it should be less than length
*    |<------------             lenght     ------------------->|
*    ----------------------------------------------------------
*    |                            |                            |
*    ----------------------------------------------------------
*                               offset
*
* EXAMPLES:
*
* SEE ALSO:
*
* proc_info interface is changed
* Description:
*   Return information to handle /proc support for the driver.

* Returns:
*         < 0:  error. errno value.
*         >= 0: sizeof data returned.
*************************************************************************/

int mppLnx_proc_info(
                     struct Scsi_Host * host, 
                     char * buffer,
                     char ** start, 
                     off_t offset, 
                     int length, 
                     int writeto1_read0)

{
   static char                   mppLnx_proc_info_buff[MPPLNX_PROC_INFO_BUFFSIZE];
   int                           len=0, begin=0, pos=0;
   int				               controllerCount, pathCount, lunCount;
	unsigned char			         nullWWN[WWN_LENGTH];
	RdacDeviceInformation_t 	   *rdacInfo;
	RdacControllerInformation_t	*controllerInfo;
	RdacControllerPath_t		      *controllerPath;
	RdacLunInformation_t		      *lunInfo;
	LunPathObjects_t		         *lunPath;
	LunPathInfo_t			         *lunPathInfo;
	char				               strBuf[16], strBuf2[16];
   char                          asciiWWN[34];
   int                           vTarget = 0;
   int                           isTruncated = 0;
   struct Scsi_Host              *this_SHp = NULL;
   int				 i;
   LWORD			stateIndex;
   int                           vQueueDepth = 0;
   struct scsi_device          *vSDp=NULL;
   struct scsi_device          *pSDp=NULL;
   MPPCMN_ARRAY_TYPE           arrayType; 

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_proc_info() offset %d length %d \n",
      offset,length));
   memset(mppLnx_proc_info_buff, 0, MPPLNX_PROC_INFO_BUFFSIZE);
   /**
   * we do not support write to the proc
   */
	if (writeto1_read0)
   {
		
		return -EINVAL;
	}
  
	begin = 0;
	len = sprintf(&mppLnx_proc_info_buff[0], 
      "\nLinux MPP driver. Version:%s Build:%s\n",
	    MPPLNX_VERSION, MPPLNX_BUILDTIMESTAMP);

   /*
   * MPP Configuration Parameters
   */
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
      "\nMPP Configuration Parameters\n");

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	VendorId		   = %s\n",mppLnx_Config_Parameters.mpp_VendorId);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
	"	ProductId		   = %s\n",mppLnx_Config_Parameters.mpp_ProductId);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	MaxLunsPerArray		   = %d\n",mppLnx_Config_Parameters.mpp_MaxLunsPerArray );

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	MaxPathsPerController	   = %d\n",mppLnx_Config_Parameters.mpp_MaxPathsPerController); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	MaxArrayModules	 	   = %d\n",mppLnx_Config_Parameters.mpp_MaxArrayModules); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	ErrorLevel		   = %d\n",mppLnx_Config_Parameters.mpp_ErrorLevel); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	DebugLevel		   = 0x%08x\n",mppLnx_Config_Parameters.mpp_Debug); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	NotReadyWaitTime	   = %d\n", mppLnx_Config_Parameters.mpp_NotReadyWaitTime);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	BusyWaitTime		   = %d\n", mppLnx_Config_Parameters.mpp_BusyWaitTime);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	QuiescenceWaitTime	   = %d\n",mppLnx_Config_Parameters.mpp_QuiescenceWaitTime);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	InquiryWaitTime		   = %d\n",mppLnx_Config_Parameters.mpp_InquiryWaitTime );

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	ScanInterval		   = %d\n",mppLnx_Config_Parameters.mpp_ScanInterval); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	InquiryInterval		   = %d\n",mppLnx_Config_Parameters.mpp_InquiryInterval); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	SelectionTimeoutRetryCount = %d\n",mppLnx_Config_Parameters.mpp_SelectionTimeoutRetryCount); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	CommandTimeoutRetryCount   = %d\n",mppLnx_Config_Parameters.mpp_CommandTimeoutRetryCount); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	UaRetryCount		   = %d\n",mppLnx_Config_Parameters.mpp_UaRetryCount); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	SyncRetryCount		   = %d\n",mppLnx_Config_Parameters.mpp_SyncRetryCount); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	SynchTimeout		   = %d\n",mppLnx_Config_Parameters.mpp_SynchTimeout); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	FailoverQuiescenceTime	   = %d\n",mppLnx_Config_Parameters.mpp_FailOverQuiescenceTime); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	FailoverTimeout		   = %d\n",mppLnx_Config_Parameters.mpp_FailoverTimeout); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	FailBackToCurrentAllowed   = %d\n",mppLnx_Config_Parameters.mpp_FailBackToCurrentAllowed); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	HoldAltInReset		   = %d\n",mppLnx_Config_Parameters.mpp_HoldAltInReset); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	DoUARetry		   = %d\n",mppLnx_Config_Parameters.mpp_DoUARetry); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	ControllerIoWaitTime	   = %d\n",mppLnx_Config_Parameters.mpp_ControllerIoWaitTime); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	ArrayIoWaitTime		   = %d\n",mppLnx_Config_Parameters.mpp_ArrayIoWaitTime); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	DisableLUNRebalance	   = %d\n",mppLnx_Config_Parameters.mpp_DisableLUNRebalance); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "	ArrayFailoverWaitTime	   = %d\n",mppLnx_Config_Parameters.mpp_MaxArrayFailoverLength); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       IdlePathCheckingInterval    = %d\n",mppLnx_Config_Parameters.mpp_IdlePathCheckingInterval); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       RecheckFailedPathWaitTime   = %d\n",mppLnx_Config_Parameters.mpp_RecheckFailedPathWaitTime); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       FailedPathCheckingInterval  = %d\n",mppLnx_Config_Parameters.mpp_FailedPathCheckingInterval); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       PrintSenseBuffer            = %d\n",mppLnx_Config_Parameters.mpp_PrintSenseBuffer); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
	"	S2toS3Key		   = 0x");
   for(i=0;i<8;i++)
   {
   	VHBA_PROC_INFO_POS
   	len = sprintf(&mppLnx_proc_info_buff[pos],
		"%02x",mppLnx_Config_Parameters.mpp_S2ToS3Key[i]);
   }
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
	"\n");

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
       "       LunFailoverDelay             = %d\n",mppLnx_Config_Parameters.mpp_LunFailoverDelay);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
       "       ClassicModeFailover          = %d\n",mppLnx_Config_Parameters.mpp_ClassicModeFailover);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
       "       AVTModeFailover              = %d\n",mppLnx_Config_Parameters.mpp_AVTModeFailover);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "\nmppLnx_proxy_cmnd_timeout  = %d\n",(int)(mppLnx_proxy_cmnd_timeout/HZ)); 

   /*
   * the queue depth of a virtual device. We assume all virtual device has the same queue depth
   * as it reported by the qla driver. This is a request by CR 89445.
   */

   for(vTarget = 0; vTarget < mppLnx_Config_Parameters.mpp_MaxArrayModules; vTarget++)
   {
      if( (rdacInfo = mpp_ModuleArray[vTarget].RdacInfo) == NULL)
      {
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
            "no storage array at virtual target %d\n",vTarget));
         continue;
      }else
      {

         for(lunCount = 0; lunCount < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunCount++)
	      {
		      lunInfo = &rdacInfo->RdacLuns[lunCount];
		      if(!lunInfo->Present || lunInfo->PseudoLunObject == NULL)
            {
			      continue;
		      }
            vQueueDepth = lunInfo->PseudoLunObject->queue_depth;
            break; //break the inner loop
         }

         break; //break the vTarget loop
      }
   }

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "virtual device queue depth = %d\n",vQueueDepth); 

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
	"\n");

   this_SHp = host;

   /*
   * Virutal HBA host information
   */
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
      "Virtual HBA host information\n");

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host number 	  	   = %d\n",this_SHp->host_no);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       base 			   = %ld\n",    this_SHp->base);

   VHBA_PROC_INFO_POS;
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       can_queue 		   = %d\n",    this_SHp->can_queue);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       cmd_per_lun 		   = %d\n",    this_SHp->cmd_per_lun);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       dma_channel 		   = %d\n",    this_SHp->dma_channel);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host_blocked 		   = %d\n",    this_SHp->host_blocked);
   
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host_self_blocked 	   = %d\n",    this_SHp->host_self_blocked);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host_failed 		   = %d\n",    this_SHp->host_failed);

 
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       io_port		   = %ld\n",    this_SHp->io_port);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       last_reset 		   = %ld\n",    this_SHp->last_reset);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_channel 		   = %d\n",    this_SHp->max_channel);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_channel 		   = %d\n",    this_SHp->max_channel);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_cmd_len 		   = %d\n",    this_SHp->max_cmd_len);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_lun		   = %d\n",    this_SHp->max_lun);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_id			   = %d\n",    this_SHp->max_id);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_sectors 		   = %d\n",    this_SHp->max_sectors);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       sg_tablesize 		   = %d\n",    this_SHp->sg_tablesize);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       this_id 		   = %d\n",    this_SHp->this_id);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       unchecked_isa_dma 	   = %d\n",    this_SHp->unchecked_isa_dma);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       unique_id 		   = %d\n",    this_SHp->unique_id);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       use_clustering              = %d\n",    this_SHp->use_clustering);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       dma_boundary		      = %ld\n\n",    this_SHp->dma_boundary);


   /*
   * physical HBA parameters
   * We get the physical HBA host and host template
   * Set certain fields from the physical HBA's template to ours
   */

   /* get any virtual device from this virtual host */
   shost_for_each_device( vSDp, host)
   {
       if(vSDp != NULL)
       {
          break;
       }
       else 
       {
         /////////////////////// Handle this case DVS
         isTruncated = 1; 
         goto proc_info_output_stop;
       }

   }
   if(vSDp == NULL)
   {
         isTruncated = 0; 
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL,929, 
             "mpp:vhba:mppLnx_proc_info: vSDp is NULL\n")); 
         goto proc_info_output_stop;
   }

   /* get any physical device corresponding to the virtual device */
   pSDp = mppLnx_getAnyPathByVDev(vSDp);
   if ( pSDp == NULL )
   {
         /////////////////////// Handle this case DVS
         isTruncated = 0; 
         goto proc_info_output_stop;

   }

   /* get the array type information from rdac info */
   if( (rdacInfo = mpp_ModuleArray[vSDp->id].RdacInfo) == NULL)
   {
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
            "no storage array at virtual target %d\n",vTarget));
         /////////////////////// Handle this case DVS
         isTruncated = 0; 
         goto proc_info_output_stop;
   }
   arrayType = rdacInfo->ArrayType;

   /* get the physical host from the physical device */
   this_SHp = pSDp->host;
   
   /* print the information from this physical host */
   
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
      "Physical HBA host information\n");

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host number 	  	   = %d\n",this_SHp->host_no);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       base 			   = %ld\n",    this_SHp->base);

   VHBA_PROC_INFO_POS;
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       can_queue                   = %d\n",this_SHp->can_queue);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       cmd_per_lun 		   = %d\n",    this_SHp->cmd_per_lun);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       dma_channel 		   = %d\n",    this_SHp->dma_channel);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host_blocked 		   = %d\n",    this_SHp->host_blocked);
   
   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host_self_blocked 	   = %d\n",    this_SHp->host_self_blocked);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       host_failed 		   = %d\n",    this_SHp->host_failed);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       io_port		   = %ld\n",    this_SHp->io_port);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       last_reset 		   = %ld\n",    this_SHp->last_reset);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_channel 		   = %d\n",    this_SHp->max_channel);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_cmd_len 		   = %d\n",    this_SHp->max_cmd_len);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_lun		   = %d\n",    this_SHp->max_lun);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_id			   = %d\n",    this_SHp->max_id);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       max_sectors 		   = %d\n",    this_SHp->max_sectors);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       sg_tablesize 		   = %d\n",    this_SHp->sg_tablesize);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       this_id 		   = %d\n",    this_SHp->this_id);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       unchecked_isa_dma 	   = %d\n",    this_SHp->unchecked_isa_dma);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       unique_id 		   = %d\n",    this_SHp->unique_id);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
        "       use_clustering              = %d\n",    this_SHp->use_clustering);

   VHBA_PROC_INFO_POS
   len = sprintf(&mppLnx_proc_info_buff[pos],
         "       dma_boundary		      = %ld\n\n",    this_SHp->dma_boundary);

   /*
   * go through all storage arrays and print rdac info
   */
   for(vTarget = 0; vTarget < mppLnx_Config_Parameters.mpp_MaxArrayModules; vTarget++)
   {
      if( (rdacInfo = mpp_ModuleArray[vTarget].RdacInfo) == NULL)
      {
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
            "no storage array at virtual target %d\n",vTarget));
         continue;
      }
      if( rdacInfo->ArrayType != arrayType )
      {
         /* only list the target information corresponding to this host */
         continue;
      }
      MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
         "virtula target %d array %s\n",
         rdacInfo->VirtualTargetId, (char *)rdacInfo->ModuleName));
     
      memset(nullWWN, 0, WWN_LENGTH);

	   /**
	   ** Global array information
	   **/
      VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos], 
         "MPP Information:\n");
	   
      VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
         "----------------\n");

	   VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
         "      ModuleName: %-32.32s         SingleController: %c\n",
		   (char *)rdacInfo->ModuleName, YES_NO(rdacInfo->SingleController));
	   
      VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
         " VirtualTargetID: 0x%03x                                       ScanTriggered: %c\n",
		   rdacInfo->VirtualTargetId, YES_NO(rdacInfo->ScanTriggered));
	   
      VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
         "     ObjectCount: 0x%03x                                          AVTEnabled: %c\n",
		   rdacInfo->ObjectCount, YES_NO(rdacInfo->AVTEnabled));
	   
     
      memset(asciiWWN,0,34);
      mpp_ConvertWWNtoAscii(&asciiWWN[0],rdacInfo->WWN);
      VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
         "             WWN:%s\n", asciiWWN);
	   
      
	   VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
         "                                                              RestoreCfg: %c\n", 
         YES_NO(rdacInfo->RestoreCfg));
	   

	   /**
	    ** Controller-specific information
	    **/
      //len = 0;
	   for(controllerCount = 0; controllerCount < 2; controllerCount++)
	   {
		   controllerInfo = rdacInfo->ControllerInfo[controllerCount];
		   if(!controllerInfo->ControllerPresent)
		   {
            //len = 0;
			   continue;
		   }

		   VHBA_PROC_INFO_POS
	      len = sprintf(&mppLnx_proc_info_buff[pos],
            "\nController '%c' Status:\n", controllerCount + 'A');

		   VHBA_PROC_INFO_POS
	      len = sprintf(&mppLnx_proc_info_buff[pos],
            "-----------------------\n");


		   VHBA_PROC_INFO_POS
	      len = sprintf(&mppLnx_proc_info_buff[pos],
            "ControllerPresent: %c\n",
		      YES_NO(controllerInfo->ControllerPresent));

		   if(controllerInfo->UTMLunExists)
		   {
			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "    UTMLunExists: %c (%-3.3d)                                            Failed: %c\n",
				   YES_NO(controllerInfo->UTMLunExists), controllerInfo->UTMLunNumber,
				   YES_NO(controllerInfo->Failed));
		   }
		   else
		   {
			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "    UTMLunExists: %c                                                  Failed: %c\n",
				   YES_NO(controllerInfo->UTMLunExists), YES_NO(controllerInfo->Failed));
		   }

		   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "   NumberOfPaths: %1d                                          FailoverInProg: %c\n",
			      controllerInfo->NumberOfPaths, YES_NO(controllerInfo->FailoverInProgress));

       
		   for(pathCount = 0; pathCount < controllerInfo->NumberOfPaths; pathCount++)
		   {
			   controllerPath = &controllerInfo->ControllerPath[pathCount];

			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "\n    Path #%1d\n", pathCount + 1);

			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "    ---------\n");

			   memset(strBuf,0,16);
            sprintf(strBuf,"h%dc%dt%d",
               controllerPath->Address.HostId,
               controllerPath->Address.ChannelId,
               controllerPath->Address.TargetId);
			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               " DirectoryVertex: %-10.10s                                        Present: %c\n", 
				   strBuf, YES_NO(controllerPath->Present));

			   VHBA_PROC_INFO_POS
             stateIndex = controllerPath->PathStateIndex;
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "                                                                     Failed: %x\n",
				   controllerPath->PathState[stateIndex]);
		   }
	   }

	   /**
	   ** Lun-specific information
	   **/

	   VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
            "\n\nLun Information\n");

	   VHBA_PROC_INFO_POS
	   len = sprintf(&mppLnx_proc_info_buff[pos],
           "---------------\n");
 
	   for(lunCount = 0; lunCount < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunCount++)
	   {
		   lunInfo = &rdacInfo->RdacLuns[lunCount];
		   if(!lunInfo->Present || lunInfo->PseudoLunObject == NULL)
         {
			   continue;
		   }

		   if(lunCount != 0)
		   {
			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "\n");
		   }

		   if(lunInfo->WWN[0] != 0)
		   {
            memset(asciiWWN,0,34);
            mpp_ConvertWWNtoAscii(&asciiWWN[0],lunInfo->WWN);
			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "    Lun #%d - WWN:%s\n ",lunCount,asciiWWN);
			
		   }
		   else
		   {
			   VHBA_PROC_INFO_POS
	         len = sprintf(&mppLnx_proc_info_buff[pos],
               "    Lun #%d - WWN:\n", lunCount);
		   }
         
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
            " Lun %d\n",lunCount));

		   VHBA_PROC_INFO_POS
	      len = sprintf(&mppLnx_proc_info_buff[pos],
               "    ----------------\n");
         
         memset(strBuf, 0, 16);

         
         if(NULL == lunInfo->PseudoLunObject)
         {
            sprintf(strBuf,"Null");
         }
         else
         {
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
               "PseudoLunObject 0x%lx\n",(unsigned long)lunInfo->PseudoLunObject));
            sprintf(strBuf,"h%dc%dt%dl%d",
               lunInfo->PseudoLunObject->host->host_no,
               lunInfo->PseudoLunObject->channel,
               lunInfo->PseudoLunObject->id,
               lunInfo->PseudoLunObject->lun);

         }

		   VHBA_PROC_INFO_POS
         len = sprintf(&mppLnx_proc_info_buff[pos],
            "       LunObject: %-10.10s                              CurrentOwningPath: %c\n",
            strBuf, A_OR_B(lunInfo->CurrentOwningPath));

		   VHBA_PROC_INFO_POS
         len = sprintf(&mppLnx_proc_info_buff[pos],
            "  RemoveEligible: %c                                          BootOwningPath: %c\n",
			   YES_NO(lunInfo->PseudoRemoveEligible), 
            A_OR_B(lunInfo->BootOwningPath));

		   VHBA_PROC_INFO_POS
         len = sprintf(&mppLnx_proc_info_buff[pos],
            "   NotConfigured: %c                                           PreferredPath: %c\n",
			   YES_NO(lunInfo->NotConfigured),
            A_OR_B(lunInfo->PreferredPath));

         VHBA_PROC_INFO_POS
         len = sprintf(&mppLnx_proc_info_buff[pos],
            "        DevState: %d                     ReportedPresent: %c\n",
			   lunInfo->PseudoDevState[lunInfo->PseudoDevStateIndex], YES_NO(lunInfo->ReportedPresent));


		   VHBA_PROC_INFO_POS
         len = sprintf(&mppLnx_proc_info_buff[pos],
            "                                         ReportedMissing: %c\n\n", YES_NO(lunInfo->ReportedMissing));

         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
            " Lun info is inserted %d\n",lunCount));

		   //len = 0;
         for(controllerCount = 0; controllerCount < 2; controllerCount++)
		   {
			   if(!lunInfo->LunControllers[controllerCount])
			   {
				   continue;
			   }
			   lunPath = lunInfo->LunControllers[controllerCount];
			   
            VHBA_PROC_INFO_POS
            len = sprintf(&mppLnx_proc_info_buff[pos],
               "    Controller '%c' Path\n", controllerCount + 'A');

			   VHBA_PROC_INFO_POS
            len = sprintf(&mppLnx_proc_info_buff[pos],
               "    --------------------\n");

			   VHBA_PROC_INFO_POS
            len = sprintf(&mppLnx_proc_info_buff[pos],
               "   NumLunObjects: %1d                                         RoundRobinIndex: %1d\n",
				   lunPath->NumberOfLunObjects, lunPath->RoundRobinPathIndex);

            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
               "   NumLunObjects: %1d                                         RoundRobinIndex: %1d\n",
				   lunPath->NumberOfLunObjects, lunPath->RoundRobinPathIndex));

            for(pathCount = 0; pathCount < lunPath->NumberOfLunObjects; pathCount++)
			   {
				   lunPathInfo = &lunPath->LunPaths[pathCount];
				   if(!lunPathInfo->LunPathDevice)
				   {
                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
                     "controller %d path %d is not available\n",
                     controllerCount,pathCount));

					   continue;
				   }

               MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
                  "h%dc%dt%dl%d",
                  lunPathInfo->LunPathDevice->host->host_no,
                  lunPathInfo->LunPathDevice->channel,
                  lunPathInfo->LunPathDevice->id,
                  lunPathInfo->LunPathDevice->lun));

               memset(strBuf, 0 , 16);
               sprintf(strBuf,"h%dc%dt%dl%d",
                  lunPathInfo->LunPathDevice->host->host_no,
                  lunPathInfo->LunPathDevice->channel,
                  lunPathInfo->LunPathDevice->id,
                  lunPathInfo->LunPathDevice->lun);
			      VHBA_PROC_INFO_POS
               len = sprintf(&mppLnx_proc_info_buff[pos],
                  "   PathRemoveState: 0x%x    PathStartState: 0x%x    PathPowerState: 0x%x\n", 
                  lunPathInfo->PathRemoveState, 
                  lunPathInfo->PathStartState, 
                  lunPathInfo->PathPowerState);

				   
               memset(strBuf2, 0 , 16);
               if(NULL == lunPathInfo->UtmLunDevice)
               {
                  sprintf(strBuf2,"Un-available");
               }
               else
               {
                  sprintf(strBuf2,"h%dc%dt%dl%d",
                        lunPathInfo->UtmLunDevice->host->host_no,
                        lunPathInfo->UtmLunDevice->channel,
                        lunPathInfo->UtmLunDevice->id,
                        lunPathInfo->UtmLunDevice->lun);
               }
                     
					VHBA_PROC_INFO_POS
               len = sprintf(&mppLnx_proc_info_buff[pos],
                     "         Path #%1d: LunPathDevice: %-12.12s           UTMLunDevice: %-10.10s\n",
						   pathCount + 1, strBuf, strBuf2);
            } /* for path Count */

			   VHBA_PROC_INFO_POS
            len = sprintf(&mppLnx_proc_info_buff[pos],"\n");

         } /*for controllerCount */

	   }/*for lunCount*/
   }/*for vTarget*/
   /*
   * In case the output length is greater than our pre-allocated buffer size,
   * terminate the processing.
   */
proc_info_output_stop:

   if(isTruncated)
   {
      pos = pos +len;
      len = sprintf(&mppLnx_proc_info_buff[pos],
         "\nOutput is too long. Truncated......\n");
      pos = pos + len;
   }else
   {
      pos = pos +len;
      len = sprintf(&mppLnx_proc_info_buff[pos],
         "\n---------End of the message-------------\n");
      pos = pos + len;
   }

   /*
   * the pos is the valid data length of our working buffer.
   */
	if (pos < offset) {
		len = 0;
		begin = pos;
      *start = NULL;
	}
   else
   {
      begin = offset;
      len = pos - offset;
      if(len >length)
      {
         len = length;
      }
      *start = buffer;
   }
  
   /*
   * copy our data from our working buffer to the buffer
   */
   if(len >0 )
   {
      memcpy(
         buffer,
         &mppLnx_proc_info_buff[begin],
         len);
   }
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_proc_info() buffer 0xlx offset %d begin %d pos %d len %d\n",
      (unsigned long)buffer,offset,begin,pos,len));
	return (len);

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_init_virtual_hba
* SUMMARY:  Initialize the virtual scsi host, scan the virtual host and 
* SCOPE:    public 
*
* DESCRIPTION:
   This function is invoked either by the virtual bus driver's probe() function or mppLnx_vhbaScanBus(). This function is responsible to
    1. allocate memory in common for all virtual hbas.
    2. Initialize queues.
    3. start kernel threads.

    1. create a virtual scsi host
    2. register the virtual host with the SCSI middle level
    3. scan virtual devices on the virtual host
    4. allocate necessary resources for the virtual HBA driver. The resources include memory and kernel threads
* RESTRICTIONS:
* 
* RETURNS:  0 for success. Otherwise failed.
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_init_virtual_hba(int fromVscan)
{
   struct scsi_host_template      * pshtp = NULL;
   
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,"Entering mppLnx_detect()\n"));
   
   /*
   * initialize certain attributes of this driver's template. They are dependent on the
   * physical HBA driver's attributes.
   */ 
   /*
   * check whether or not there are Engenio devices. 
   * 
   */
   if(NULL == mppLnxVHBAContext.phostList)
   {
      /*
      * We get here beause
      * 1. The host may not have a HBA adapter
      * 2. The HBA driver may not be loaded yet
      * 3. The HBAs may not connect to Engenio storage arrays
      * 4. The HBAs don't see any Engenio LUNs
      * we can not initialize ourself correctly if there is not physical HBA(s) that 
      * has discovered our devices. Sorry I have to leave. Bye.
      */
      pshtp = NULL;
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,901, 
         "mpp:vhba:mppLnx_detect: no manageable device is discovered. Leaving ...\n")); 
      
      printk("The mpp virtual HBA driver didn't discover any manageable devices. Leaving ...\n");
      goto no_device_errorout;

   }

   
   /*
   * register the virtual host with the middle level
   */
   /*
    * set up the proxy request memory
    */
    mppLnx_vhba_init_queues();
    if(mppLnx_vhba_mem_alloc(2*mppLnx_Config_Parameters.mpp_MaxPathsPerController) != 0)
    {
       goto no_device_errorout;
    }

    if(mppLnx_vhba_arMem_alloc(2*mppLnx_Config_Parameters.mpp_MaxPathsPerController) != 0)
    {
       goto no_arMem_errorout;
    }

   /*
   * initialize dpc queue
   */
   sema_init(&mppLnx_dpc_sem, 0);
   mppLnxDpcQueueContext.dpc_sem = &mppLnx_dpc_sem;
   sema_init(&mppLnx_detect_sem, 0);
   mppLnxDpcQueueContext.detect_sem = &mppLnx_detect_sem;
   //mppLnxDpcQueueContext.cmndQueueLock = &taskQueue_lock;
   mppLnxDpcQueueContext.isStopped = 0;
   atomic_set(&mppLnxDpcQueueContext.needSchedule, 1);
   
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Starting dpc thread.\n"));
	mppLnxDpcQueueContext.dpcQueue_pid = 
      kernel_thread(mppLnx_dpc_handler, NULL, 0); 

   if(mppLnxDpcQueueContext.dpcQueue_pid < 0)
   {
      goto dpc_errorout;
   }
	/*
	 * Now wait for the kernel dpc thread to initialize
	 * and go to sleep.
	 */   
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting sleeping. Waiting for dpc thread finishing init.\n"));
	down(&mppLnx_detect_sem);
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting wakes up.\n"));
  
   /*
   * start the failback scan thread
   */
   
   sema_init(&mppLnx_failback_sem, 0);
   mppLnxFailbackScanContext.failback_sem = &mppLnx_failback_sem;
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Starting failback scan thread.\n"));
   mppLnxFailbackScanContext.detect_sem = &mppLnx_detect_sem;
	mppLnxFailbackScanContext.failback_pid = 
      kernel_thread((int (*)(void *))mppLnx_failback_handler, NULL, 0);  
	
   if(mppLnxFailbackScanContext.failback_pid < 0)
   {
       goto failback_errorout;
   }
   /*
	 * Now wait for the kernel dpc thread to initialize
	 * and go to sleep.
	 */
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting sleeping. Waiting for failback thread finishing init.\n"));
	down(&mppLnx_detect_sem);
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting wakes up.\n"));
   
   /*
   * start the pathvalidate scan thread which performs PRRegistrations 
   */
   
   sema_init(&mppLnx_pathvalidate_sem, 0);
   mppLnxPathValidateContext.pathvalidate_sem = &mppLnx_pathvalidate_sem;
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Starting path validation  thread.\n"));
   mppLnxPathValidateContext.detect_sem = &mppLnx_detect_sem;
	mppLnxPathValidateContext.pathvalidate_pid = 
      kernel_thread((int (*)(void *))mppLnx_pathvalidate_handler, NULL, 0);  
	
   if(mppLnxPathValidateContext.pathvalidate_pid < 0)
   {
       goto pathvalidate_errorout;
   }
   /*
	 * Now wait for the kernel pathvalidate thread to initialize
	 * and go to sleep.
	 */
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting sleeping. Waiting for PathValidate thread finishing init.\n"));
	down(&mppLnx_detect_sem);
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting wakes up.\n"));
   
   /*
   * start the syncIO worker thread
   */
   sema_init(&mppLnx_syncIOWorker_sem, 0);
   mppLnxWorkerQueueContextContext.workerQueue_sem = &mppLnx_syncIOWorker_sem;
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Starting syncIO worker thread.\n"));
   mppLnxWorkerQueueContextContext.detect_sem = &mppLnx_detect_sem;
	mppLnxWorkerQueueContextContext.workerQueue_pid = 
      kernel_thread((int (*)(void *))mppLnx_worker_handler, NULL, 0);
   if(mppLnxWorkerQueueContextContext.workerQueue_pid < 0)
   {
      goto worker_errorout;
   }
	/*
	 * Now wait for the kernel dpc thread to initialize
	 * and go to sleep.
	 */   
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting sleeping. Waiting for syncIO thread finishing init.\n"));
	down(&mppLnx_detect_sem);
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Detecting wakes up.\n"));

      /* Timer initialization for checking path every interval */
      mppLnx_checkpath_timer_init();

      return 0;

   /*error out
   * we need to clean up all resources
   */
   /*
   *stop the sync IO worker thread
   */ 
   if(mppLnxWorkerQueueContextContext.workerQueue_pid >= 0)
   {
      mppLnxWorkerQueueContextContext.isStopped = 1;
      kill_proc(mppLnxWorkerQueueContextContext.workerQueue_pid, SIGHUP,1);
      if(mppLnxWorkerQueueContextContext.detect_sem)
      down(mppLnxWorkerQueueContextContext.detect_sem);
	   mppLnxWorkerQueueContextContext.detect_sem = NULL;

   }
   

worker_errorout:
   /*
   *stop the pathvalidate thread if it is still running.
   *(it may be killed by someone)
   */ 
   if(mppLnxPathValidateContext.pathvalidate_pid >= 0)
   {
      mppLnxPathValidateContext.isStopped = 1;
      kill_proc(mppLnxPathValidateContext.pathvalidate_pid, SIGHUP, 1);
      if(mppLnxPathValidateContext.detect_sem)
      down(mppLnxPathValidateContext.detect_sem);
      mppLnxPathValidateContext.detect_sem = NULL;
   }
   

pathvalidate_errorout:
   /*
   *stop the failback rescan thread if it is still running.
   *(it may be killed by someone)
   */ 
   if(mppLnxFailbackScanContext.failback_pid >= 0)
   {
      mppLnxFailbackScanContext.isStopped = 1;
      kill_proc(mppLnxFailbackScanContext.failback_pid, SIGHUP, 1);
      if(mppLnxFailbackScanContext.detect_sem)
      down(mppLnxFailbackScanContext.detect_sem);
      mppLnxFailbackScanContext.detect_sem = NULL;

   }
   
failback_errorout:
   /*
   *stop the dpc thread
   */ 
   if(mppLnxDpcQueueContext.dpcQueue_pid >= 0)
   {
      mppLnxDpcQueueContext.isStopped = 1;
      kill_proc(mppLnxDpcQueueContext.dpcQueue_pid, SIGHUP,1);
      if(mppLnxDpcQueueContext.detect_sem)
      down(mppLnxDpcQueueContext.detect_sem);
      mppLnxDpcQueueContext.detect_sem = NULL;
   }

dpc_errorout:
   mppLnx_vhba_arMem_free(mppLnx_vhba_arPool.size);

no_arMem_errorout:
    mppLnx_vhba_mem_free(mppLnx_vhba_pqPool.size);

no_device_errorout:
   /*
   * clean up the internal data model
   */
   mppLnx_cleanVHBAContext();

   /*
   * the virtual host doesn;t get initialized
   */
   mppLnx_is_virtual_hba_initialized = 0;
   return 1;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_setupHostTemplate
* SUMMARY:  Set up virtual host template's characteristics to fit the characteristics
             of the physical HBA template..
* SCOPE:    private
*
*
* DESCRIPTION: The values of certain fields in the struct scsi_host_template data struct 
         for the virtual HBA could be not be decided before hands. It must be matched 
         with the characteristics of the physical HBA. This function is invoked by 
         the virtual host initialization function to make the virtual HBA host 
         template matching the physical HBA's template
*
* RESTRICTIONS:
* 
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static void mppLnx_vhba_setupHostTemplate(struct scsi_host_template * pshtp,struct Scsi_Host *pshp)
{
   int numberOfArrays = 0;
   int i;
   MPPLNX_DUMP_SCSI_HOST_TEMPLATE(pshtp);
   for(i = 0; i<mppLnx_Config_Parameters.mpp_MaxArrayModules; i++)
   {
      if((mpp_ModuleArray +i)->RdacInfo != NULL)
      {
         numberOfArrays++;
      }
   }

   /*
   *QLA HBA driver for 2.6 has two modules. One is called qla2xxx that is
   *a common driver module for all HBA models. The other one is model specific
   *driver. The qla2xxx driver module initializes the struct scsi_host_template object.
   *The can_queue sets to 0. The model specific driver module will update the
   *value of can_queue at scsi_host level. Therefore, we must re-check our virtual
   *HBA host's can_queue again at virtual host registration time.
   */
    /* lpfc does not set it in host template it sets can_queue in scsi host */
   mppLnx_vhost_template.can_queue           = pshp->can_queue - 2*numberOfArrays;
   if(mppLnx_vhost_template.can_queue <= 0)
   {
      mppLnx_vhost_template.can_queue = 1;
   }

   mppLnx_vhost_template.sg_tablesize        = pshtp->sg_tablesize;
   mppLnx_vhost_template.cmd_per_lun         = pshtp->cmd_per_lun;
   mppLnx_vhost_template.unchecked_isa_dma   = pshtp->unchecked_isa_dma;
   mppLnx_vhost_template.use_clustering      = pshtp->use_clustering; 
   mppLnx_vhost_template.emulated            = pshtp->emulated;

   mppLnx_vhost_template.dma_boundary        = pshp->dma_boundary; 
   mppLnx_vhost_template.max_sectors         = pshp->max_sectors;

}

int mppLnx_min_can_queue(int vhost_can_queue, int phost_can_queue)
{
     int ret=0;

     int numberOfArrays = 0;
     int i;
     for(i = 0; i<mppLnx_Config_Parameters.mpp_MaxArrayModules; i++)
     {
        if((mpp_ModuleArray +i)->RdacInfo != NULL)
        {
           numberOfArrays++;
        }
     }

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_can_queue  vhost %d phost %d ret %d noofarrays %d\n",
               vhost_can_queue,phost_can_queue,ret,numberOfArrays));

     /*
     *QLA HBA driver for 2.6 has two modules. One is called qla2xxx that is
     *a common driver module for all HBA models. The other one is model specific
     *driver. The qla2xxx driver module initializes the struct scsi_host_template object.
     *The can_queue sets to 0. The model specific driver module will update the
     *value of can_queue at scsi_host level. Therefore, we must re-check our virtual
     *HBA host's can_queue again at virtual host registration time.
     */
     phost_can_queue = phost_can_queue - 2*numberOfArrays;
     if(phost_can_queue <= 0)
     {
        phost_can_queue = 1;
     }
     
     if( phost_can_queue < vhost_can_queue )
         ret = phost_can_queue;
     else
         ret = vhost_can_queue;
      
     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_can_queue  vhost %d phost %d ret %d\n",
               vhost_can_queue,phost_can_queue,ret));
     return ret; 
}

int mppLnx_min_max_cmd_len(int vhost_max_cmd_len, int phost_max_cmd_len)
{
     int ret;
     
     if( phost_max_cmd_len < vhost_max_cmd_len )
         ret = phost_max_cmd_len;
     else
         ret = vhost_max_cmd_len;

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_max_cmd_len  vhost %d phost %d ret %d\n",
               vhost_max_cmd_len,phost_max_cmd_len,ret));

     return ret;
}

int mppLnx_min_cmd_per_lun(int vhost_cmd_per_lun, int phost_template_cmd_per_lun)
{
     int ret;
     
     if( phost_template_cmd_per_lun < vhost_cmd_per_lun )
         ret = phost_template_cmd_per_lun;
     else
         ret = vhost_cmd_per_lun;

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_cmd_per_lun  vhost %d phost %d ret %d\n",
               vhost_cmd_per_lun,phost_template_cmd_per_lun,ret));

     return ret;
}

unsigned long  mppLnx_min_dma_boundary(unsigned long vhost_dma_boundary, unsigned long phost_dma_boundary)
{
     unsigned long ret;
     
     if( phost_dma_boundary < vhost_dma_boundary )
         ret = phost_dma_boundary;
     else
         ret = vhost_dma_boundary;

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_dma_boundary  vhost 0x%lx phost 0x%lx ret 0x%lx\n",
               vhost_dma_boundary,phost_dma_boundary,ret));

     return ret;
}

int mppLnx_min_use_clustering(int vhost_use_clustering, int phost_template_use_clustering)
{
     int ret;
      
     /* 0 - no clusering  :: 1 - use clustering */
     /* if two host adapters are connected to same target set the vhost clustering to lower value */
     if( phost_template_use_clustering < vhost_use_clustering )
         ret = phost_template_use_clustering;
     else
         ret = vhost_use_clustering;

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_use_clustering  vhost %d phost %d ret %d\n",
               vhost_use_clustering,phost_template_use_clustering,ret));

     return ret;
}

int mppLnx_min_sg_tablesize(int vhost_sg_tablesize, int phost_template_sg_tablesize)
{
     int ret;
      
     if( phost_template_sg_tablesize < vhost_sg_tablesize )
         ret = phost_template_sg_tablesize;
     else
         ret = vhost_sg_tablesize;

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_sg_tablesize  vhost %d phost %d ret %d\n",
               vhost_sg_tablesize,phost_template_sg_tablesize,ret));

     return ret;
}

int mppLnx_min_max_sectors(int vhost_max_sectors, int phost_max_sectors)
{
     int ret;
      
     if( phost_max_sectors < vhost_max_sectors )
         ret = phost_max_sectors;
     else
         ret = vhost_max_sectors;

     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppLnx_min_max_sectors  vhost %d phost %d ret %d\n",
               vhost_max_sectors,phost_max_sectors,ret));

     return ret;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_updateHostTemplate
* SUMMARY:  update virtual host template's characteristics to fit the characteristics
             of the physical HBA template..
* SCOPE:    private
*
*
* DESCRIPTION: The values of certain fields in the struct scsi_host_template data struct 
         for the virtual HBA could be not be decided before hands. It must be matched 
         with the characteristics of the physical HBA. This function is invoked by 
         the virtual host initialization function to make the virtual HBA host 
         template matching the physical HBA's template
	 This is called when multiple paths to the storage array are detected. we check if
	 any parameter needs to be updated. since we need to take the min of all the paths
	 connecting to storage array.
*
* RESTRICTIONS:
* 
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static void mppLnx_vhba_updateHostTemplate(struct Scsi_Host *vshp,struct scsi_host_template * pshtp, struct Scsi_Host *pshp)
{
   int ret;
   struct scsi_device          *vSDp=NULL;
   int queue_depth=0;
 
   /* first read in the values back from the virtual host */
   mppLnx_vhost_template.can_queue           = vshp->can_queue;
   mppLnx_vhost_template.cmd_per_lun         = vshp->cmd_per_lun;
   mppLnx_vhost_template.use_clustering      = vshp->use_clustering; 

   mppLnx_vhost_template.sg_tablesize        = vshp->sg_tablesize;
   mppLnx_vhost_template.max_sectors         = vshp->max_sectors;
   mppLnx_vhost_template.dma_boundary        = vshp->dma_boundary;

   mppLnx_vhost_template.unchecked_isa_dma   = vshp->unchecked_isa_dma;
   //mppLnx_vhost_template.emulated            = vshp->emulated;

   /* 
    * if the physical host template has different values than already registered to virtual hba
    * get the minimum and adjust the values 
    */
   if ( (ret=mppLnx_min_can_queue(mppLnx_vhost_template.can_queue, pshp->can_queue)) != mppLnx_vhost_template.can_queue )
   {
      /* update the min can queue value in virtual host and scsi template */
      vshp->can_queue                 = ret;
      mppLnx_vhost_template.can_queue = ret;     
   }

   /* 
    * if the physical host has different values than already registered to virtual hba
    * get the minimum and adjust the values 
    */
   if ( (ret=mppLnx_min_max_cmd_len(vshp->max_cmd_len, pshp->max_cmd_len)) != vshp->max_cmd_len )
   {
      /* update the min can queue value in virtual host  */
      vshp->max_cmd_len                 = ret;
   }

   if ( (ret=mppLnx_min_cmd_per_lun(mppLnx_vhost_template.cmd_per_lun, pshtp->cmd_per_lun)) != mppLnx_vhost_template.cmd_per_lun )
   {
      /* update the min cmd_per_lun value in virtual host and scsi template */
      vshp->cmd_per_lun                 = ret;
      mppLnx_vhost_template.cmd_per_lun = ret;     
   }

   if ( (ret=mppLnx_min_use_clustering(mppLnx_vhost_template.use_clustering, pshtp->use_clustering)) != mppLnx_vhost_template.use_clustering )
   {
      /* update the min can queue value in virtual host and scsi template */
      vshp->use_clustering                 = ret;
      mppLnx_vhost_template.use_clustering = ret;     

      if( vshp->use_clustering == DISABLE_CLUSTERING )
      {
         /* disable the clustering  for all the virtual luns already discovered by the virtual host */
         shost_for_each_device( vSDp, vshp)
         {
            clear_bit( QUEUE_FLAG_CLUSTER, &vSDp->request_queue->queue_flags);      
         }
      }
   }

   if ( (ret=mppLnx_min_sg_tablesize(mppLnx_vhost_template.sg_tablesize, pshtp->sg_tablesize)) != mppLnx_vhost_template.sg_tablesize )
   {
      /* update the min can queue value in virtual host and scsi template */
      vshp->sg_tablesize                 = ret;
      mppLnx_vhost_template.sg_tablesize = ret;     
      
      /* change the table size for all the virtual luns already discovered by the virtual host */
      shost_for_each_device( vSDp, vshp)
      {
         blk_queue_max_hw_segments( vSDp->request_queue, vshp->sg_tablesize);      
      }
   }

   /* software iscsi does not set the max_sectors in template hence retrive it from scsi host */
   if ( (ret=mppLnx_min_max_sectors(mppLnx_vhost_template.max_sectors, pshp->max_sectors)) != mppLnx_vhost_template.max_sectors )
   {
      /* update the min can queue value in virtual host and scsi template */
      vshp->max_sectors                 = ret;
      mppLnx_vhost_template.max_sectors = ret;     

      /* change the max_sectors for all the virtual luns already discovered by the virtual host */
      shost_for_each_device( vSDp, vshp)
      {
         blk_queue_max_sectors( vSDp->request_queue, vshp->max_sectors);      
      }
   }

   if ( (ret=mppLnx_min_dma_boundary(mppLnx_vhost_template.dma_boundary, pshp->dma_boundary)) != mppLnx_vhost_template.dma_boundary )
   {
      /* update the min can queue value in virtual host and scsi template */
      vshp->dma_boundary                 = ret;
      mppLnx_vhost_template.dma_boundary = ret;     
      
      /* change the table size for all the virtual luns already discovered by the virtual host */
      shost_for_each_device( vSDp, vshp)
      {
         blk_queue_segment_boundary( vSDp->request_queue, vshp->dma_boundary);      
      }
   }

 
   /* change the queue_depth for all the virtual luns already discovered by the virtual host */
   shost_for_each_device( vSDp, vshp)
   { 
      queue_depth =  mppLnx_get_min_queue_depth(vSDp);
      if( queue_depth > 0 )
         mppLnx_change_queue_depth(vSDp, queue_depth);
   }

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_regVirtualHost
* SUMMARY:  Register the virtual HBA SCSI host with the SCSI middle level and kernel
* SCOPE:    private
*
*
* DESCRIPTION: The function is invoked during the mpp virtual bus driver's probe() time.
* This function registers a virtual SCSI host with the scsi middle level. It also 
* initializes certain fields based on the physical HBA SCSI host.
*    1. create a virtual scsi host
*    2. register the virtual host with the SCSI middle level
*    3. scan virtual devices on the virtual host
*
* RESTRICTIONS:
* 
* RETURNS: 0 if successful. No-0 otherwise.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
static int mppLnx_vhba_regVirtualHost(struct Scsi_Host ** vshp,struct Scsi_Host * pshp,int hostnumber, int fromVscan)
{
   int error = 0;
   BOOL  needsfailover=TRUE;
   int   waittime=0; /* seconds */
   struct Scsi_Host *vhpnt;

   struct scsi_host_template      * pshtp = NULL;
   mppLnx_PhostEntry_t     * phostEntry=NULL;
   struct device           * pgendevice;
   struct list_head        *listPrt=NULL;

   
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                  "mppLnx_vhba_regVirtualHost  Setup host template fromVscan %d\n",fromVscan));

   for(listPrt = mppLnxVHBAContext.phostList->list.next; 
         listPrt != &(mppLnxVHBAContext.phostList->list); 
         listPrt =listPrt->next)
   {
      phostEntry = list_entry(listPrt, mppLnx_PhostEntry_t, list);

      if(phostEntry->hostNo == hostnumber) 
         break;
   }

   if ( phostEntry == NULL ) return -1;


   /*
   * We go the physical HBA host and host template
   * Set certain fields from the physical HBA's template to ours
   */
   pshtp = phostEntry->host->hostt;
   pshp = phostEntry->host;
   MPPLNX_DUMP_SCSI_HOST(pshp);

   /* if virtual hba is already iniitialized just check for lowest params of  physical hba and set the template*/
   if( *vshp != NULL )
   {
       MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                 "mppLnx_vhba_regVirtualHost vhost already exists for phost_no %d returning \n",hostnumber));

       mppLnx_vhba_updateHostTemplate(*vshp,pshtp,pshp);

       return error;
   }

   /*
   * setup the DMA mask for the virtual adapter based on the physical host's dma
   * mask.
   * get the physical host's genric device and then get its parent device's dma_mask.
   * the parent device of the physical host should be a pci adapter.
   */
   pgendevice = scsi_get_device(pshp);
   mppLnx_mppvadapter0.dma_mask = pgendevice->dma_mask;
   mppLnx_mppvadapter0.coherent_dma_mask = pgendevice->coherent_dma_mask;

   /*
   * make sure the virtual host's template matches the physical HBA's template
   */
   mppLnx_vhba_setupHostTemplate(pshtp,pshp);

   /*
   * setting proxy command time out value
   */
   mppLnx_set_command_timeout(pshtp);

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                  "mppLnx_vhba_regVirtualHost  Setup command timeout and call scsi_host_alloc\n"));


   vhpnt = scsi_host_alloc(&mppLnx_vhost_template, 0);
   
   if (unlikely(NULL == vhpnt)) 
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,907,
         "mppVhba: %s: scsi_host_alloc failed\n", __FUNCTION__));
                
      error = -ENODEV;
      *vshp = NULL;

      return error;
        
   }else
   {
      *vshp = vhpnt;
      MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2,
                      "mppLnx_vhba_regVirtualHost  scsi_host_alloc SUCCESS\n"));
   }
   vhpnt->max_channel = 0;
   vhpnt->max_id =  mppLnx_Config_Parameters.mpp_MaxArrayModules;
   vhpnt->max_lun = mppLnx_Config_Parameters.mpp_MaxLunsPerArray;
   vhpnt->can_queue           = mppLnx_vhost_template.can_queue;
   vhpnt->sg_tablesize        = mppLnx_vhost_template.sg_tablesize;
   vhpnt->max_sectors         = mppLnx_vhost_template.max_sectors;
   vhpnt->dma_boundary        = mppLnx_vhost_template.dma_boundary;
   vhpnt->unchecked_isa_dma   = mppLnx_vhost_template.unchecked_isa_dma;
   vhpnt->use_clustering      = mppLnx_vhost_template.use_clustering;
   vhpnt->cmd_per_lun         = mppLnx_vhost_template.cmd_per_lun;
   vhpnt->max_cmd_len         = pshp->max_cmd_len;

   error = scsi_add_host(vhpnt, &mppLnx_mppvadapter0);
   if (unlikely(error)) 
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,908,
         "%s: scsi_add_host failed\n", __FUNCTION__));

      printk(KERN_ERR "%s: scsi_add_host failed\n", __FUNCTION__);
      error = -ENODEV;
      scsi_host_put(vhpnt);
      *vshp = NULL;
   } else
   {
     /*
      *we perform virtual host scan immediately only if we don't need controller failover.
      *Otherwise, sleep for every 5 seconds and check again if max timeout is reached, 
      *deffer the host scan to the work queue thread.
      */
      while ( (needsfailover == TRUE) && ( waittime <  MPPLNX_BOOT_DISCOVERY_FAILOVER_WAIT))
      {
	if(fromVscan)
	{
		needsfailover = FALSE;
	}else
	{
		needsfailover = mppLnx_needsInitFailover();
	}
	if ( needsfailover == FALSE )
        {
		break;
        }
	else
	{
                if (( waittime % 30 ) == 0)
                {
                   MPP_ERRORLOGPRINT((MPP_ERR_PATH_FAILOVER,913,
                   "Waiting for %d seconds to discover LUNS on Current Owning Path\n", MPPLNX_BOOT_DISCOVERY_FAILOVER_WAIT - waittime));
                }

		msleep( 5 * 1000 ); /* sleep 5 seconds */
		waittime += 5;
	}

      } 

      if( needsfailover == TRUE )
      {
         /*
         *For some volumes, we see only one controller and the volume's current-owner is 
         *not the available controller. Because the circular dependency between the 
         *mppVhba and mppUpper driver modules, we can not do controller failover before 
         *the mppVhba driver module is completed its loading. Lets do the failover in 
         *a work queue thread.
         */
         mppLnx_delayHostScan(vhpnt);
      }else
      {
         /* Check and Set TAS bit for all volumes on all arrays */
         mppLnx_SetTAS_All_Arrays();

         scsi_scan_host(vhpnt);
      }
   }


   return error;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_slave_alloc
* SUMMARY:  Set up internal resource for a virtual device before the middle level scans this device
* SCOPE:    public
*
*
* DESCRIPTION: * Before the mid layer attempts to scan for a new device where none
	 * currently exists, it will call this entry in your driver.  Should
	 * your driver need to allocate any structs or perform any other init
	 * items in order to send commands to a currently unused target/lun
	 * combo, then this is where you can perform those allocations.  This
	 * is specifically so that drivers won't have to perform any kind of
	 * "is this a new device" checks in their queuecommand routine,
	 * thereby making the hot path a bit quicker.

* RESTRICTIONS:
* 
* RETURNS: 0 on success, non-0 on failure 
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
int  mppLnx_slave_alloc(struct scsi_device * sdev)
{
   /*
   * don't have any thing to do.
   */
   return 0;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_slave_configure
* SUMMARY:  Set up the virtual device's characteristics.
* SCOPE:    private
*
*
* DESCRIPTION: 
	 * The device has responded to an INQUIRY and we know the
	 * device is online. This function, must perform the task of setting the queue
	 * depth on the device based on its physical device's queue depth.
*
* RESTRICTIONS:
* 
* RETURNS: Return 0 on success, non-0 on error.  The device will be marked
	 *     as offline on error so that no access will occur.  If you return
	 *     non-0, your slave_detach routine will never get called for this
	 *     device, so don't leave any loose memory hanging around, clean
	 *     up after yourself before returning non-0
	 *
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/ 
int  mppLnx_slave_configure(struct scsi_device * sdev)
{
   RdacDeviceInformation_t    * rdacInfo = NULL;
   int                        queue_depth;
   int                        lunIndex;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_slave_configure scsi(%d:%d:%d:%d)\n",
      sdev->host->host_no, sdev->channel, sdev->id, sdev->lun));

   rdacInfo = mpp_ModuleArray[sdev->id].RdacInfo;
   queue_depth = mppLnx_get_min_queue_depth(sdev);
   
   if(unlikely( (queue_depth == 0) && (sdev->lun == 0) ) )
   {
      /*
      * this is the case of unconfigured lun 0 supoort
      * we don't have valid physical device for the lun 0.
      * we need to find a valid virtual device
      */
      for(lunIndex=1; lunIndex <mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
      {
         if( rdacInfo->RdacLuns[lunIndex].Present )
         {
            break;
         }
      }
      sdev->lun = lunIndex;
      queue_depth = mppLnx_get_min_queue_depth(sdev);
      sdev->lun = 0;
   }

   if(queue_depth > 3)
   {
      /*
      * we need 2 slots for path validation and failover commands to go through
      */
      queue_depth =queue_depth -2;
   }

         
   if(sdev->tagged_supported)
   {
      scsi_activate_tcq(sdev, queue_depth);
      printk(KERN_INFO
       "scsi(%d:%d:%d:%d): Enabled tagged queuing, queue depth %d.\n",
       sdev->host->host_no, sdev->channel, sdev->id, sdev->lun,
       sdev->queue_depth);
   }else
   {
      scsi_adjust_queue_depth(sdev, 0 /* TCQ off */,
		     sdev->host->hostt->cmd_per_lun);

   }


   /*
   * create the virtual device entry and virtual device proc file
   */
   
   rdacInfo->RdacLuns[sdev->lun].PseudoLunObject = sdev;

   mppLnx_insert_VdeviceEntry(sdev->lun, mpp_ModuleArray[sdev->id].RdacInfo);

   /*
   * allocate more proxy request memory
   */
   if(mppLnx_vhba_pqPool.size < sdev->host->can_queue + 2*2*mppLnx_Config_Parameters.mpp_MaxPathsPerController)
   {
      mppLnx_vhba_mem_alloc(queue_depth);
   }
   
   /*
   * DEBUG
   * check virtual and physical device's request_queue_t attributes
   */
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "virtual device bounce_pfn %ld bounce_gfp %d queue_flags %ld max_sectors %d max_phys_segments %d max_hw_segments %d hardsect_size %d max_segment_size %d seg_boundary_mask %ld dma_alignment %d\n",
      sdev->request_queue->bounce_pfn,
      sdev->request_queue->bounce_gfp,
      sdev->request_queue->queue_flags,
      sdev->request_queue->max_sectors,
      sdev->request_queue->max_phys_segments,
      sdev->request_queue->max_hw_segments,
      sdev->request_queue->hardsect_size,
      sdev->request_queue->max_segment_size,
      sdev->request_queue->seg_boundary_mask,
      sdev->request_queue->dma_alignment));


   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_slave_configure scsi(%d:%d:%d:%d)\n",
      sdev->host->host_no, sdev->channel, sdev->id, sdev->lun));
   return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_change_queue_depth
* SUMMARY:  Change the queue depth for virtual luns.
* SCOPE:    private
*
*
* DESCRIPTION:
     * This routine provides the mechanism for the user to set
     * the queue depth from the proc/sys file system on the fly.
     * this checks the value for 2 cases
     *   1. new qdepth value is still less than physical qdepth-2
     *   2. new qdepth is greater than 1
*
* RESTRICTIONS:
*
* RETURNS: 
     *   Returns the value of the queue depth.   
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int  mppLnx_change_queue_depth(struct scsi_device * sdev, int queue_depth)
{
   int   tagged=0;
   int   min_queue_depth=0;

   min_queue_depth = mppLnx_get_min_queue_depth(sdev);

   if( min_queue_depth == 0 )
      return 1;

   if(( queue_depth <= min_queue_depth - 2 ) && ( queue_depth >=1 ) )
    scsi_adjust_queue_depth(sdev, tagged, queue_depth);

   return (sdev->queue_depth);

}

static int mppLnx_get_min_queue_depth(struct scsi_device * vsdp)
{
   struct scsi_device            *psdp = NULL;
   RdacDeviceInformation_t       *rdacInfo;
   RdacLunInformation_t          *rdacLun;
   int                           path = 0;
   int                           controller = 0;
   unsigned int                  min_queue_depth=0xffffffff; /* initialize with huge number */
   
   rdacInfo = mpp_ModuleArray[vsdp->id].RdacInfo;
   if( rdacInfo == NULL )
   {
	  MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
            "mppLnx_get_min_queue_depth()- rdacInfo is NULL.\n" ));

	  return 0;
   }

   rdacLun  = & rdacInfo->RdacLuns[vsdp->lun];
   /* find the min acceptable queue depth from this lun on all paths*/
   for(controller = 0; controller < 2; controller++)
   {
      for(path = 0; path < mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
      {
         psdp = rdacLun->LunControllers[controller]->LunPaths[path].LunPathDevice;
         if(psdp == NULL)
         {
            continue;
         }else
         {
            if( psdp->queue_depth < min_queue_depth)
               min_queue_depth = psdp->queue_depth;
         }
      }
   }

   if( min_queue_depth == 0xffffffff)
      return 0;
   else
      return min_queue_depth;
}
void mppLnx_cleanup_requests_per_scsi_device( struct scsi_device *sdev)

{

   mppLnx_ProxyRequestEntry_t    *preq;
   unsigned long                 flags;

 printk("CAP called on %d:%d:%d\n",sdev->host->host_no,sdev->id,sdev->lun);
    /*
   * check the dispatch queue. The command in the physical device's queue must be the
   * proxy requests that are in the queued queue.
   */
   spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
   spin_lock(&mppLnx_DPProxyRequestQ.queueLock);
   spin_lock(&mppLnx_deferredProxyRequestQ.queueLock);
   list_for_each_entry(preq, &mppLnx_queuedProxyRequestQ.list, queued_list)
   {
      if(preq->vcmnd->device == sdev)
      {
         preq->isVcmdCompleted = TRUE;
         preq->vcmnd->result = DID_ABORT << 16;
         mppLnx_scsi_add_timer(preq->vcmnd ,15*HZ, preq->vcmd_timeoutfunc);
         preq->vcmnd->scsi_done(preq->vcmnd);
         printk("CAP QP %d:%d:%d not removed\n",
            preq->vcmnd->device->host->host_no,preq->vcmnd->device->id,preq->vcmnd->device->lun);
      }
   }
   /*
   * check the path validation command queue. This is the queue that has path validation
   * commands come back and wait for __mppLnx_pathvalidate_done
   */
   list_for_each_entry(preq, &mppLnx_deferredProxyRequestQ.list, pending_list)
   {
      if(preq->vcmnd->device == sdev)
      {
         preq->isVcmdCompleted = TRUE;
         preq->vcmnd->result = DID_ABORT << 16;
         mppLnx_scsi_add_timer(preq->vcmnd ,15*HZ, preq->vcmd_timeoutfunc);
         preq->vcmnd->scsi_done(preq->vcmnd);
         printk("CAP DQ %d:%d:%d not removed\n",
            preq->vcmnd->device->host->host_no,preq->vcmnd->device->id,preq->vcmnd->device->lun);
      }
   }

   /*
   * check the dispatch list
   */
   list_for_each_entry(preq, &mppLnx_DPProxyRequestQ.list, dispatch_list)
   {
      if( 
            //(preq->dispatchType != MPPLNX_VALIDATE_PATH_TYPE && 
             //preq->dispatchType != MPPLNX_DELAY_RESPONSE_TYPE) &&
            (preq->vcmnd->device == sdev)
        )
      {
          preq->isVcmdCompleted = TRUE;
          preq->vcmnd->result = DID_ABORT << 16;
          mppLnx_scsi_add_timer(preq->vcmnd ,15*HZ, preq->vcmd_timeoutfunc);
          preq->vcmnd->scsi_done(preq->vcmnd);
          printk("CAP DP %d:%d:%d not removed\n",
            preq->vcmnd->device->host->host_no,preq->vcmnd->device->id,preq->vcmnd->device->lun);
      }
   }

   spin_unlock(&mppLnx_deferredProxyRequestQ.queueLock);
   spin_unlock(&mppLnx_DPProxyRequestQ.queueLock);
   spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);
   printk("CAP return %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun);
   return;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_slave_destroy
* SUMMARY:  De-allocate resources for the virtual device that are allocated during mppLnx_salve_alloc() and mppLnx_slave_config.
* SCOPE:    public
*
*
* DESCRIPTION: 
  * Immediately prior to deallocating the device and after all activity
  * has ceased the mid layer calls this point so that the low level
  * driver may completely detach itself from the scsi device and vice
  * versa.  The low level driver is responsible for freeing any memory
  * it allocated in the slave_alloc or slave_configure calls. 
  *
* RESTRICTIONS:
* 
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
void mppLnx_slave_destroy(struct scsi_device * sdev)
{
   mppLnx_VDeviceEntry_t   *vDEp = NULL;
   RdacDeviceInformation_t    *RdacInfo = NULL;
   int                     Lun;
   LINT 	LockLevel;


   Lun= sdev->lun;
   RdacInfo = mpp_ModuleArray[(sdev->id)].RdacInfo;
   /*
   * during the slave_config phase (the middle level knows the device is there 
   * and the inquiry comes back without error), we create the mppLnx_VDeviceEntry_t
   * object for the virtual device. We need to delete it when the middle level tells us
   * the virtual device is ready to be removed.
   * 
   * Note: the middle level may call this function without calling slave_config().
   */
   //mppLnx_cleanup_requests_per_scsi_device(sdev);
   vDEp = mppLnx_get_VdeviceEntry(sdev);
   if(likely(vDEp))
   {
      mppLnx_delete_VdeviceEntry(vDEp);
   }
   
   if(( RdacInfo != NULL ) && (! RdacInfo->RdacLuns[Lun].PseudoRemoveEligible )) {
	return;
   }
   /*
   *If UTM LUN is the only lun mapped for the array, clear the memory of RdacLunInformation_t
   *will cause kernel panic when a user asks to rescan the virtual host. We clear the 
   *memory when all physical paths to the UTM lun are removed.
   */
   if((RdacInfo != NULL) && ( !RdacInfo->RdacLuns[Lun].UTMLun))
   {
	 MPP_LOCK(RdacInfo->Lock, LockLevel);
	/* Added the size of MPPCMN_LOAD_BALANCE_POLICY and 4 bytes added as reserved for allignment */
      bzero(&RdacInfo->RdacLuns[Lun],
				(sizeof(RdacLunInformation_t)-((2*sizeof(LunPathObjects_t *))+
				 16 + sizeof(MPPCMN_LOAD_BALANCE_POLICY)+sizeof(void *) + 4 )) );
	   bzero(&RdacInfo->RdacLuns[Lun].WWN[0], 16 );
	 MPP_UNLOCK(RdacInfo->Lock, LockLevel);
   }
   return;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_release_virtual_host
* SUMMARY:  release all resources associated with the virtual HBA host
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function is invoked from the virtual bus driver's driver remove function
   mppLnx_mppvbus_driver_remove() when the virtual HBA driver is un-loaded. This function 
   must release all system resources that are allocated during the virtual host 
   initialization. The release of resources must be performed in the following orders:
    1. unregister the virtual scsi host
    2. shutdown all mpp kernel thread
    3. release allocated memory

*  Notes: It may cause the rmmod process hang if the resources are not released orderly
* RESTRICTIONS:
* 
* RETURNS: 0 if all resources are successfully released. 1 otherwise
*
* ERRNO:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_release_virtual_host(void)
{
   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_release_virtual_host() \n"));

   /*
   *mark mppVhba driver module removing start.
   */
   mppLnx_startVhbaModRemove = 1;
   /* Timer cleanup for checking path every interval */
   mppLnx_checkpath_timer_exit();

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
         "Unregister the host with the middle level \n"));

   mppLnx_cleanup_requests(MPPLNX_CLEAN_FIRST_PHASE);

   /*
   * if we still have virtual device creation work task scheduled, flush them
   */
   if(mppLnx_workTaskRefCnt)
   {
      msleep(2*HZ);
      flush_scheduled_work();
   }
   if(mppLnx_vhba_virtual_host_fc)
       scsi_remove_host(mppLnx_vhba_virtual_host_fc);

   if(mppLnx_vhba_virtual_host_sas)
       scsi_remove_host(mppLnx_vhba_virtual_host_sas);

   if(mppLnx_vhba_virtual_host_iscsi)
       scsi_remove_host(mppLnx_vhba_virtual_host_iscsi);

   if(mppLnx_vhba_virtual_host_ib)
       scsi_remove_host(mppLnx_vhba_virtual_host_ib);

      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Done with Unregister the host with the middle level \n"));

   /* Flush the mpp_wq worker thread */
   if(mppLnx_wq != NULL)
      flush_workqueue(mppLnx_wq);
   
   /*
   *stop the failback rescan thread if it is still running.
   *(it may be killed by someone)
   */ 
   if(mppLnxFailbackScanContext.failback_pid >= 0)
   {
      mppLnxFailbackScanContext.isStopped = 1;
      kill_proc(mppLnxFailbackScanContext.failback_pid, SIGHUP, 1);
      if(mppLnxFailbackScanContext.detect_sem)
      down(mppLnxFailbackScanContext.detect_sem);
      mppLnxFailbackScanContext.detect_sem = NULL;

   }

   /*
   *stop the pathvalidate thread if it is still running.
   *(it may be killed by someone)
   */ 
   if(mppLnxPathValidateContext.pathvalidate_pid >= 0)
   {
      mppLnxPathValidateContext.isStopped = 1;
      kill_proc(mppLnxPathValidateContext.pathvalidate_pid, SIGHUP, 1);
      if(mppLnxPathValidateContext.detect_sem)
      down(mppLnxPathValidateContext.detect_sem);
      mppLnxPathValidateContext.detect_sem = NULL;

   }

   /*
   *stop the sync IO worker thread
   */ 
   if(mppLnxWorkerQueueContextContext.workerQueue_pid >= 0)
   {
      mppLnxWorkerQueueContextContext.isStopped = 1;
      kill_proc(mppLnxWorkerQueueContextContext.workerQueue_pid, SIGHUP,1);
      if(mppLnxWorkerQueueContextContext.detect_sem)
      down(mppLnxWorkerQueueContextContext.detect_sem);
      mppLnxWorkerQueueContextContext.detect_sem = NULL;
   }
   /*
   *stop the dpc thread
   */ 
   if(mppLnxDpcQueueContext.dpcQueue_pid >= 0)
   {
      mppLnxDpcQueueContext.isStopped = 1;
      kill_proc(mppLnxDpcQueueContext.dpcQueue_pid, SIGHUP,1);
      if(mppLnxDpcQueueContext.detect_sem)
      down(mppLnxDpcQueueContext.detect_sem);
      mppLnxDpcQueueContext.detect_sem = NULL;
   }

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Done with Unregister the host with the middle level \n"));
   
   mppLnx_cleanup_requests(MPPLNX_CLEAN_SECOND_PHASE);
   /*
   * free the slab memory
   */
   if(mppLnx_vhba_mem_free(mppLnx_vhba_pqPool.size))
   {
      printk("not all proxy request memory being freed\n");
   }

   if(mppLnx_vhba_arMem_free(mppLnx_vhba_arPool.size))
   {
      printk("not all async memory being freed\n");
   }
   /*
   * clean up internal data model
   */
   mppLnx_cleanVHBAContext();
   /*
   * tell the kernel that we are done with the virtual scsi host
   */
   if( mppLnx_vhba_virtual_host_fc)
       scsi_host_put(mppLnx_vhba_virtual_host_fc);

   if( mppLnx_vhba_virtual_host_sas)
       scsi_host_put(mppLnx_vhba_virtual_host_sas);

   if( mppLnx_vhba_virtual_host_iscsi)
       scsi_host_put(mppLnx_vhba_virtual_host_iscsi);

   if( mppLnx_vhba_virtual_host_ib)
       scsi_host_put(mppLnx_vhba_virtual_host_ib);

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_release_virtual_host()\n"));
   return 1;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_info
* SUMMARY:  supply information about given host: driver name plus data
 *          to distinguish given host
* SCOPE:    public 
*
* DESCRIPTION:
   This function is assigned to struct scsi_host_template data struct's "char (*info)" 
   function pointer. The information returned from this function will be used by
   the middle level in a number places. It will go to system boot message file.
   The returned string must be only one line.
   This function will return the following string:
	"mpp virtual host bus adapter: version: xx.xx.xx.xx, timestamp:xxxxxxxxx"

*
* RESTRICTIONS:
* 
* RETURNS: the info char string of this driver. 
*
* ERRNO:
*
* NOTES:It is assumed the returned information fits on one line 
*      (i.e. does not included embedded newlines).
*      The SCSI_IOCTL_PROBE_HOST ioctl yields the string returned by this
*      function.
*      In a similar manner, scsi_register_host() outputs to the console
*      each host's "info" (or name) for the driver it is registering.
*     .
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
const char * mppLnx_info(struct Scsi_Host * shp)
{
   static char infoBuf[1024];
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_info()\n"));
   memset(infoBuf,0,1024);
   sprintf(infoBuf,"%s:version:%s,%s:%s",
         MPPVHBASTRING,MPPLNX_VERSION,TIME_STAMP,MPPLNX_BUILDTIMESTAMP);
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_info()\n"));
   return infoBuf;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_queuecommand
* SUMMARY:  queue scsi command to the virtual HBA driver, invoke 'done' on completion
* SCOPE:    public 
*
* DESCRIPTION:
   This function is assigned to struct scsi_host_template data struct's "int (*queuecommand)" 
   function pointer. This function receives SCSI command from the Linux SCSI middle 
   level for virtual devices. The "void (*done)" function pointer is the command 
   completion function. This function must be returned before the "done" is called.

*  @vSCp: pointer to a scsi command object
*  @done: function pointer to be invoked on completion
*
*
* RESTRICTIONS:
* 
* RETURNS:1 if the virtual HBA adapter (host) is busy, else returns 0. 
*		One reason for an adapter to be busy is that the number
*		of outstanding queued commands is already equal to
*		struct Scsi_Host::can_queue. If the virtual HBA driver can not
*       allocate struct scsi_cmnd from all its available physical devices, returns 1.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_host_interface_type( struct Scsi_Host *vhost )
{

    if( vhost == NULL ) return -1;

    if(vhost == mppLnx_vhba_virtual_host_fc)
       return MPPCMN_TARGET_FC;

    if(vhost == mppLnx_vhba_virtual_host_iscsi)
       return MPPCMN_TARGET_ISCSI;

    if(vhost == mppLnx_vhba_virtual_host_sas)
       return MPPCMN_TARGET_SAS;

    if(vhost == mppLnx_vhba_virtual_host_ib)
       return MPPCMN_TARGET_IB;

    return -1; 
}

int mppLnx_queuecommand(struct scsi_cmnd * vSCp, void (*done)(struct scsi_cmnd *))
{
   int                  vchannel = vSCp->device->channel;
   int                  targetId = vSCp->device->id;
   int                  lun      = vSCp->device->lun;
   unsigned char        *cmd = (unsigned char *) vSCp->cmnd;
   void                 (*timeoutfunc) (struct scsi_cmnd *);
   SenseData_t          *senseData;
   int                  rejectIO = FALSE;
   int                  lunIndex;

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_queuecommand()\n"));
   
   /*
   * This is an IO request's entry point. Trace it
   */
   MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d LCP=EN\n",
      vSCp->serial_number,
      vSCp->cmnd[0],
      vSCp->device->host->host_no,
      vSCp->device->channel,
      vSCp->device->id,
      vSCp->device->lun));
   
   vSCp->scsi_done = done;
   /*
   * validate the command
   */
    if(  (targetId >= mppLnx_Config_Parameters.mpp_MaxArrayModules) ||
      (vchannel != 0) ||    /* no such channel */
      (mpp_ModuleArray[targetId].RdacInfo == NULL) || /*no such tid */
      /* no such LUN */
      (lun >= mppLnx_Config_Parameters.mpp_MaxLunsPerArray) ||
      (!mpp_ModuleArray[targetId].RdacInfo->RdacLuns[lun].Present) ||
      ((int)mpp_ModuleArray[targetId].RdacInfo->ArrayType != mppLnx_host_interface_type(vSCp->device->host)) 
     )
   {
      rejectIO = TRUE;
      /*schedule the tasklet to perform the response and return */
      /* the result in the vSCp will be DID_NO_CONNECT*/
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
         "mppLnx_queuecommand: No such device vtarget %d vlun %d\n",targetId, lun));

      if(targetId >= mppLnx_Config_Parameters.mpp_MaxArrayModules)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_queuecommand() return DID_NO_CONNECT because of id is out of range %d\n",
            targetId));
      }
      else if(mpp_ModuleArray[targetId].RdacInfo == NULL)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_queuecommand() return DID_NO_CONNECT because of rdacinfo == NULL %d\n",
            targetId));
      }
      else if(lun >= mppLnx_Config_Parameters.mpp_MaxLunsPerArray)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_queuecommand() return DID_NO_CONNECT because Lun is out off rang id%d lun%d\n",
            targetId,lun));
      }
      else if(!mpp_ModuleArray[targetId].RdacInfo->RdacLuns[lun].Present)
      {
         /*
         * We get here because the LUN is not present but the virtual target is valid
         * Check if the Lun is 0 and cdb is inquiry or report lun. We must let these two
         * commands through. Otherwise, no other LUNs will be discovered because the middle
         * level relies on the LUN0's present.
         * We allow only standard inquiry going through un-configured lun 0 by checking 
         * EVPD bit of the inquiry CDB
         */
         if(unlikely((lun==0) && ( ( (*cmd == INQUIRY) && ((cmd[1] & 0x01) == 0) ) 
            || *cmd == REPORT_LUNS)))
         {
            rejectIO = FALSE;
            /*
            * 1. find a virtual device that is present
            */
            for(lunIndex=1; lunIndex <mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
            {
               if( mpp_ModuleArray[targetId].RdacInfo->RdacLuns[lunIndex].Present)
               {
                  break;
               }
            }
            /*
            * the struct scsi_cmnd struct has a struct scsi_pointer for LLD to play with
            * The SCp's Statue is marked with a magic value. 
            * During the command completion time, if we see the magic value, we must
            * replace the device->lun back to 0
            */
            vSCp->SCp.Status = MPPLNX_LUN0_REPLACED_MAGIC;
            vSCp->device->lun = lunIndex;
            MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
               "mppLnx_queuecommand() Lun0 doesn't present. Send cmd %c to lun %d id%d lun%d\n",
               *cmd,lunIndex, targetId,lun));
         }else
         {
            MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
               "mppLnx_queuecommand() return DID_NO_CONNECT because Lun is not present id%d lun%d\n",
               targetId,lun));
         }
      } else if ( ((int)mpp_ModuleArray[targetId].RdacInfo->ArrayType != mppLnx_host_interface_type(vSCp->device->host))  ) 
      {
      /* Only accept commands from host type that has the same array type, this protects us during host_scan for scanning only related targets */
         MPP_ERRORLOGPRINT((MPP_ERR_ALL, 914,
         "mppLnx_queuecommand array type %d mismatch  with host type %d target %d lun %d cdb[0]=0x%02x\n",
          (int)mpp_ModuleArray[targetId].RdacInfo->ArrayType,mppLnx_host_interface_type(vSCp->device->host),
         targetId,lun,*cmd));
      }
      
      if(rejectIO == TRUE)
      {
         vSCp->result = DID_NO_CONNECT<<16;
         mppLnx_schedule_resp(vSCp);
         return 0;
      }
   }
   
   if(unlikely(done == NULL))
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 903,
         "mppLnx_queuecommand NULL done function target %d lun %d\n",
         targetId,lun));
      return 0;
   }
   if(unlikely(middle_level_scsi_done == NULL))
   {
      /**
      * save the middle level scsi done function
      */
      middle_level_scsi_done = done;
   }
   /*
   * if debug driver, dump read/write cdb
   */
   MPPLNX_DUMPREADWRITECDB(vSCp);
   /*
   *handling special IO 
   */
   switch (*cmd) 
   {
      case RESERVE:
      case RELEASE:
      case RESERVE_10:
      case RELEASE_10:
         /* 
         * what do we do if the command is targeted to a pre-Sonoran 4 array. FIX-ME
         * we need a decision.
         */
         if(!mpp_ModuleArray[targetId].RdacInfo->FWSonoran4OrLater)
         {
            /*
            * The array is a pre-Sonoran 4 array. It does not support SCSI-3
            * persistent reservation. We return an error back
            * status - check condition
            * sense key - illegal request
            * ASC/ASCQ 20/00 
            */
            MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 904,
               "Scsi-2 reserve/release command to a pre-05.40.xx.xx array is NOT SUPPORTED. Array:%s:Firmware Version:%02x.%02x.%02x.%02x\n",
               (char *)mpp_ModuleArray[targetId].ModuleName,
               mpp_ModuleArray[targetId].RdacInfo->FirmwareVersion[0],
               mpp_ModuleArray[targetId].RdacInfo->FirmwareVersion[1],
               mpp_ModuleArray[targetId].RdacInfo->FirmwareVersion[2],
               mpp_ModuleArray[targetId].RdacInfo->FirmwareVersion[3]));
            senseData = (SenseData_t *) vSCp->sense_buffer;
            senseData->Error_Code = 0x70; /* 
                                           *Error type: current, 
                                           *sense data format:fixed 
                                           */
            senseData->Sense_Key = SENSE_KEY_ILLEGAL_REQUEST;
            senseData->Additional_Len = 0x06; /*no addtional sense bytes*/
            senseData->ASC =0x20;    /* INVALID COMMAND OPERATION CODE*/
            senseData->ASCQ =0x00;
            vSCp->result = (COMMAND_COMPLETE << 8) |
				    (CHECK_CONDITION << 1);
            mppLnx_schedule_resp(vSCp);
            return 0;
            
         }
         mppLnx_Scsi2toScsi3_handler(vSCp);
	      return 0;
         break;
         /*
         *The following commands are supported by the Engenio storage array. Because the MPP 
         *driver has a scsi2-to-scsi3 emulation layer. The SCSI3 persistent reservation
         *with Exclusive-Access Registrants-only type allows those commands pass through.
         *But the Scsi2 spec says the target should reject those command with RESERVATION
         *CONFLICT status if the initiator is not the reservation owner. We must handle 
         *those command specially.
         */
      case PERSISTENT_RESERVE_OUT:
      case PERSISTENT_RESERVE_IN:
      {
         /* SCSI3 PRIN/PROUT command */
         mppLnx_Scsi2toScsi3_handler(vSCp);
         return 0;
      }

      case  SCSI_TEST_UNIT_READY:
      case  SCSI_SAFE_PASS_THROUGH:
      case  SCSI_LOG_SENSE:
      case  SCSI_READ_CAPACITY:
      case  SCSI_READ_CAPACITY_16:
      case  SCSI_START_STOP_UNIT:
      case  SCSI_LSI_QUIESCE:

         if(!mpp_ModuleArray[targetId].RdacInfo->FWSonoran4OrLater)
         {
           /* there is no scsi-2 emulation on pre-sonoran 4 array*/
            break;
         }
         else
         {

            mppLnx_Scsi2toScsi3_handler(vSCp);
	         return 0;
         }
         break;
      default:
	      break;
   }
   /*
   * delete the virtual command's timer
   */
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "VHBA Rcmd: Sn %ld, [%02x] vlun T%dL%d\n",
      vSCp->serial_number,
      vSCp->cmnd[0],
      vSCp->device->id,vSCp->device->lun));

   timeoutfunc = (void (*)(struct scsi_cmnd *))vSCp->eh_timeout.function;
   /*
   * remember the middle level's timer out function.
   * when we restore the timer in out virtual command completion function,
   * we need the middle level's timeout function
   */
   if(middle_level_timeoutfunc == NULL)
   {
      middle_level_timeoutfunc = timeoutfunc;
   }
   mppLnx_scsi_delete_timer(vSCp);
   
   mppLnx_schedule_queuecmd(vSCp);

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_queuecommand()\n"));
   return 0;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_eh_device_reset_handler(
* SUMMARY: issue SCSI bus reset
* SCOPE:    public 
*
* DESCRIPTION:
   This function is assigned to struct scsi_host_template data struct's 
   "int (*eh_device_reset_handler)" function pointer. It can be invoked by
   middle level SCSI error handler thread and/or a user space application
   that sends ioctl "SG_SCSI_RESET_DEVICE" command to a scsi pass-through
   device node.
   A "bus device reset" message causes a target device performing reset and
   clear all SCSI-2 reservations. The mpp virtual HBA driver must provide 
   this function and perform the following tasks:
      call common API to perform SCSI-2-to-SCSI-3 translation layer's 
      device_reset function
      select a physical path and invoke the physical HBA's eh_device_reset_handler.
*  @scp: SCSI target that contains this device should be reset
*
* RESTRICTIONS:
* The scsi middle level holds a spin-lock while invoking the eh_device_reset_handler.
* IO can not be issued in this function. This function returns immediately and schedules
* the scsi2-to-scsi3 bus device reset call to be run in the worker kernel thread.
* 
* RETURNS:SUCCESS if  else FAILED
*
* ERRNO:
*
* NOTES: Invoked from scsi_eh thread. No other commands will be
*      queued on current host during error handling.
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_eh_device_reset_handler(struct scsi_cmnd * scp)
{
   mppLnx_AsyncRequestEntry_t    *are; 
   struct scsi_cmnd                     *copyVscp = NULL;
   
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_eh_device_reset_handler Lun %d\n",scp->device->lun));
   
   /*
   * the command in the middle level is from stack. We must make a copy and give it
   * to the handler. The handler must free it after completing the operations
   */
   are = mppLnx_get_free_asyncRequest();
   
   if(!are)
   {
      /* error log*/
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 905,
         "Cannot get kernel memory for struct mppLnx_AsyncRequestEntry_t\n"));

      return FAILED;
   }
   else
   {
       copyVscp = are->cmnd;
       copyVscp->request = NULL;
       copyVscp->cmnd[0]=MPPLNX_BDR_CMND_OPCODE;
       memset(&copyVscp->eh_timeout, 0, sizeof(scp->eh_timeout));
       copyVscp->device                   = scp->device;
       copyVscp->use_sg                   = 0;
       copyVscp->underflow                = 0;
       copyVscp->transfersize             = 0;
       copyVscp->resid                    = 0;
       copyVscp->serial_number            = 0;
       copyVscp->host_scribble            = NULL;
       copyVscp->scsi_done                = scp->scsi_done;
       copyVscp->done                     = scp->done;
        
       copyVscp->request_buffer           = NULL;
       copyVscp->request_bufflen          = 0;

       copyVscp->sc_data_direction        = scp->sc_data_direction;

       /* sometimes the command can get back into the timer chain, so
        * use the pid as an identifier */
       copyVscp->pid                      = 0;
   }

   mppLnx_Scsi2toScsi3_handler(copyVscp);
   
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_eh_device_reset_handler- Lun %d\n",scp->device->lun));
   return SUCCESS;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_eh_bus_reset_handler
* SUMMARY:  Reset the virtual bus
* SCOPE:    public 
* DESCRIPTION:
   This function is assigned to struct scsi_host_template data struct's 
   "int (*eh_bus_reset_handler)" function pointer. This function is invoked by 
   the error handler thread as well as sg's ioctl operation. This function is 
   called to reset the SCSI bus.
   The implication of this function will have an empty function body and 
   return "SUCCESS" status. The consideration is that the virtual bus should not 
   reset physical buses since middle level error recovery processes already 
   performed necessary error recovery procedures for a proxy command.

*
*  @scp: SCSI bus that contains this device should be reset
* RESTRICTIONS:
* 
* RETURNS: SUCCESS
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_eh_bus_reset_handler(struct scsi_cmnd * scp)
{
   return SUCCESS;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_eh_host_reset_handler
* SUMMARY:  Resets the virtual HBA adaptor
* SCOPE:    public 
*
* DESCRIPTION:
   This function is assigned to struct scsi_host_template data struct's 
   "int (*eh_host_reset_handler)" function pointer. This function is invoked 
   by the error handler thread as well as sg's ioctl operation. The virtual HBA driver
   will not provide this function. This function asks a HBA driver to re-initialize the 
   HBA card.
   The implication of this function will have an empty function body and return "SUCCESS" status.
   The consideration is that the virtual bus should not reset physical buses since middle 
   level error recovery processes already performed necessary error recovery procedures 
   for a proxy command.

* @scp: SCSI host that contains this device should be reset
* RESTRICTIONS:
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_eh_host_reset_handler(struct scsi_cmnd * scp)
{
   return SUCCESS;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_bios_param
* SUMMARY:  fetch head, sector, cylinder info for a disk
* SCOPE:    public 
*
* DESCRIPTION:
   This function is assigned to struct scsi_host_template data struct's "int (*bios_param)" 
   function pointer. This function is invoked by a upper level driver to report head, 
   sector, cylinder info for a disk device.
   The implementation of this function will be a proxy function of a physical 
   HBA's same function. The algorithm will be:
      1.	call mppLnx_getAnyPathByVDev to get a physical device for this virtual device
      2.	get the physical device's Scsi_Host and then struct scsi_host_template.
      3.	call the bios_param function pointer of the struct scsi_host_template data struct
      4.	return to the caller.
*      @sdev: pointer to scsi device context (defined in sd.h)
*      @bdev: pointer to block device context (defined in fs.h)
*	   @capacity:  device size (in 512 byte sectors)
*      @params: three element array to place output:
*              params[0] number of heads (max 255)
*              params[1] number of sectors (max 63)
*              params[2] number of cylinders 
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
/*
 * This routine is no longer registered to the scsi host template.
 * Date April 27 2005 - CR 87871 
 */

int mppLnx_bios_param(struct scsi_device * dev, struct block_device * bdev,
			sector_t sectors, int *params)
{
   struct scsi_host_template   *phtp;
   struct scsi_device          * vSDp;
   struct scsi_device          * pSDp;
   RdacDeviceInformation_t    *rdacInfo = NULL;
   int                        lunIndex;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_bios_param()\n"));
   /** save the Disk's virtual struct scsi_device */
   vSDp = dev;
   rdacInfo = mpp_ModuleArray[vSDp->id].RdacInfo;
  
   /*
   * get a physical scsi device for the virtual
   * In this case, we are not selecting a path for IO,
   * we just want to get any path of this virtual scsi device
   **/
   pSDp = mppLnx_getAnyPathByVDev(vSDp);
   if(unlikely( (pSDp == NULL) && (vSDp->lun == 0) ) )
   {
      /*
      * this is the case of unconfigured lun 0 supoort
      * we don't have valid physical device for the lun 0.
      * we need to find a valid virtual device
      */
      for(lunIndex=1; lunIndex <mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
      {
         if( rdacInfo->RdacLuns[lunIndex].Present )
         {
            break;
         }
      }
      vSDp->lun = lunIndex;
      pSDp = mppLnx_getAnyPathByVDev(vSDp);
      vSDp->lun = 0;
   }
   /** get the physical device's struct scsi_host_template **/
   phtp = pSDp->host->hostt;
    
   if(NULL != phtp->bios_param)
   {
      /*replace the Disk's device with the physical struct scsi_device to make the
       * physical HBA happy */
      MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppLnx_bios_param() using hba's bios_param func\n"));
       
      phtp->bios_param(pSDp,bdev,sectors,params);
   }
   else
   {
      /* make up something meaningful
      */
      MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppLnx_bios_param() - Make up params\n"));
      //int scsicam_bios_param(struct block_device *bdev, sector_t capacity, int *ip)
      params[0] = 0x40;	/* 1 << 6 */
      params[1] = 0x20;	/* 1 << 5 */
      params[2] = sectors >> 11;
      scsicam_bios_param(
               bdev,	/* block device */
		         sectors,	/* capacity */
		         params);/* Heads, sectors, cylinders in that order */
   }
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_bios_param device %d:%d:%d:%d capacity 0x%x heads %d sectors %d cylinders %d \n",
         dev->host->host_no,
         dev->channel,
         dev->id,
         dev->lun,
         sectors,
         params[0] ,params[1] ,params[2] ));
   
   return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_newDeviceAdded
* SUMMARY:  Receives a notification of a new added struct scsi_device and create /proc 
*           node for the new device.
* SCOPE:    public 
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   The function is used for new-device-add event notification. During device 
   hot-add time, when a new device is added to the Engenio mpp driver common data model, 
   the Linux MPP upper level driver calls this function to notify the virtual 
   HBA driver of a newly added Scsi device.
   This function insert this new scsi device to its internal data model and creates 
   the device's /proc leaf node
*  @sdp a struct scsi_device object that is newly added to the MPP driver
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_newDeviceAdded(struct scsi_device * sdp)
{
   int                        isVirtualDevice = 0;
   mppLnx_PDeviceEntry_t      *pde;
   mppLnx_VDeviceEntry_t      *vde;

   int                        vTarget =0;
   int                        vLun = 0;
   int                        cntl = 0;
   int                        path = 0;
   int                        foundpSDp = 0;
   RdacDeviceInformation_t    *rdacInfo;
   RdacLunInformation_t       *rdaclun;
   LunPathObjects_t           * lunPathObject;
   struct scsi_device                *pSDp;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Entering mppLnx_newDeviceAdded() devies H%dC%dT%dL%d\n",
      sdp->host->host_no,
      sdp->channel,
      sdp->id,
      sdp->lun));
   /**
   * check if this device is a virtual or physical device
   */
   if((sdp->host->hostt->proc_name != NULL) &&
      strcmp(sdp->host->hostt->proc_name,MPPLNX_PROC_NAME) == 0)
   {
      isVirtualDevice = 1;
   }
   else
   {
      isVirtualDevice = 0;
   }
   /*
   *check if the device is already in the internal data model
   */
   if(isVirtualDevice)
   {
      vde = mppLnx_get_VdeviceEntry(sdp);
      if(vde != NULL)
      {
         return;
      }
   }
   else
   {
      pde = mppLnx_get_PdeviceEntry(sdp);
      if(pde != NULL)
      {
         return;
      }
   }

   if(!isVirtualDevice)
   {
      for(vTarget = 0; vTarget < mppLnx_Config_Parameters.mpp_MaxArrayModules; vTarget++)
      {
         if( (rdacInfo = mpp_ModuleArray[vTarget].RdacInfo) == NULL)
         {
            continue;
         }

         for(vLun = 0; vLun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; vLun++)
         {
            rdaclun = &(rdacInfo->RdacLuns[vLun]);
            if(!rdaclun->Present)
            {
               continue;
            }

            for(cntl = 0; cntl < 2; cntl++)
            {
               lunPathObject = rdaclun->LunControllers[cntl];
               for(path = 0; path < mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
               {
                  pSDp = lunPathObject->LunPaths[path].LunPathDevice;
                  if(sdp == pSDp)
                  {
                     /*
                     *We found the device from the RdacInfo.
                     *insert it to the internal data model
                     */
                     mppLnx_insert_PdeviceEntry(sdp, rdacInfo, cntl, path);
                     foundpSDp = 1;

                     if(!rdaclun->NotConfigured && !rdaclun->PseudoLunObject)
                     {
                           mppLnx_genVdAddEvent(rdacInfo, sdp,vTarget,vLun);
                     }

                     MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                           "Exiting mppLnx_newDeviceAdded()\n"));
                     return;
                  }
               }/* for path */


            }/* for cntl */

         } /* for vLun */

      } /* for vTarget */
   } /* else */

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_newDeviceAdded()\n"));
   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_genVdAddEvent
* SUMMARY:  Receives data path object creation event and generate virtual device creation 
*           work task conditionally.
* SCOPE:    local 
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function will generate a work task for virtual device creation conditionally. 
   If it is an initial device discovery condition, only one event will be generated. If it 
   is new device discovery after initial discovery, event storage array volume will 
   generate only one event even multiple data path objects are discovered
*  @ rdacInfo  - represents the storage array where the data path object is associated 
         with.

* @sdp  an I_T_L data path object which is just discovered and created.
* @tid  the storage arrays virtual target id
* @ lun  the logical unit number of the scsi device
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

static void mppLnx_genVdAddEvent(
   RdacDeviceInformation_t *rdacInfo, 
   struct scsi_device * sdp,
   int tid,
   int lun)
{
   static int              vhost_scan_event_fired_fc = 0;
   static int              vhost_scan_event_fired_ib = 0;
   static int              vhost_scan_event_fired_iscsi = 0;
   static int              vhost_scan_event_fired_sas = 0;
   static int              *vhost_scan_event_fired;
   mppLnx_vdAddEvent_t     *vdAddEvent;
   RdacLunInformation_t    *rdaclun;
   mppLnx_workqType_t      eventType;
   BOOL                    shouldFireEvent = FALSE;
   BYTE                    ctl;
   BYTE                    path;
   struct Scsi_Host        *vhost=NULL;
   
   
   rdaclun = &(rdacInfo->RdacLuns[lun]);                
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Entering mppLnx_genVdAddEvent H:%dC:%dT:%dL%d\n",
               sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));

   switch( rdacInfo->ArrayType )
   {
       case MPPCMN_TARGET_FC:
              vhost = mppLnx_vhba_virtual_host_fc;
              vhost_scan_event_fired = &vhost_scan_event_fired_fc;
              break;
       case MPPCMN_TARGET_IB:
              vhost = mppLnx_vhba_virtual_host_ib;
              vhost_scan_event_fired = &vhost_scan_event_fired_ib;
              break;
       case MPPCMN_TARGET_ISCSI:
              vhost = mppLnx_vhba_virtual_host_iscsi;
              vhost_scan_event_fired = &vhost_scan_event_fired_iscsi;
              break;
       case MPPCMN_TARGET_SAS:
              vhost = mppLnx_vhba_virtual_host_sas;
              vhost_scan_event_fired = &vhost_scan_event_fired_sas;
              break;
       default:
              break;

   }
   /*
   * check if we need to fire virtual device creation event
   */
   if(!rdaclun->PseudoLunObject)
   {
      if(!(*vhost_scan_event_fired) && !vhost)
      {
            /*
            * virtual host has not been created. This is the case where
            *vhba driver is loaded but doesn't discover any devices
            */
             MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                "mppLnx_genVdAddEvent gen event SCAN_VHOST H:%dC:%dT:%dL%d\n",
               sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));

            eventType = SCAN_VHOST;
            (*vhost_scan_event_fired)++;
            shouldFireEvent = TRUE;

            MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                "mppLnx_genVdAddEvent fc %d ib %d iscsi %d sas %d\n", 
                vhost_scan_event_fired_fc, vhost_scan_event_fired_ib, vhost_scan_event_fired_iscsi,vhost_scan_event_fired_sas));

       }else if(vhost)
       {
            /*
            *virtual host has not been created. New targets are plugged in or 
            * new devices are added
            */
             MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                "mppLnx_genVdAddEvent gen event ADD_VDEV H:%dC:%dT:%dL%d\n",
               sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
            eventType = ADD_VDEV;
            /*
            *we fire an event only if the first data path to a virtual LUN is added to the
            *Rdac data model. 
            *The 2.6.5-7.244-smp kernel has a bug that can not handle a big number of
            *works in a work_queue
            */
            for(ctl=0; ctl<2; ctl++)
            {
               for(path=0; path < mppLnx_Config_Parameters.mpp_MaxPathsPerController; ++path) 
               {
                  if( (rdacInfo->ControllerInfo[ctl]->ControllerPath[path].Present) )
                  {
                    if(lun == rdacInfo->ControllerInfo[ctl]->UTMLunNumber )
                    {
                       /*
                       * don't fire an event for UTM lun
                       */
                       break;
                    }
						if(rdacInfo->RdacLuns[lun].LunControllers[ctl]->LunPaths[path].LunPathDevice && 
							!rdacInfo->RdacLuns[lun].PseudoLunEventCreated)
                        {
							shouldFireEvent = TRUE;
							rdacInfo->RdacLuns[lun].PseudoLunEventCreated = 1;
                           break;
                        }
                     }
               } //for path
            }//for ctl
         } //else mppLnx_is_virtual_hba_initialized
   } //if !rdaclun->PseudoLunObject

   if(shouldFireEvent)
   {
	   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
				   "mppLnx_genVdAddEvent gen event %d Schedule event for  H:%dC:%dT:%dL%d\n",
				   eventType,sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));

	   if(eventType == SCAN_VHOST)
	   {
		   mppLnx_addWorkQueue(vhost, SCAN_VHOST);
	   }
	   else
	   {

		   vdAddEvent = MPP_INT_KMEM_ALLOC(sizeof(mppLnx_vdAddEvent_t));
		   if(!vdAddEvent)
		   {
			   //log an error
			   MPP_ERRORLOGPRINT((MPP_ERR_FATAL,910,
						   "Hot-Plug mppLnx_genVdAddEvent cannot get memory for mppLnx_vdAddEvent_t.\n"));
			   return;
		   }
		   vdAddEvent->rdacInfo   = rdacInfo;
		   vdAddEvent->lun        = lun;
		   vdAddEvent->tid        = tid;
		   vdAddEvent->sdp        = sdp;
		   vdAddEvent->startTime  = jiffies;
		   vdAddEvent->eventType  = eventType;

		   if(mppLnx_addWorkQueue(vdAddEvent, ADD_VDEV)!= 0)
		   {
			   MPP_FREE(vdAddEvent,sizeof(mppLnx_vdAddEvent_t));
		   }
	   }
   }
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_genVdAddEvent H:%dC:%dT:%dL%d\n",
               sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_set_command_timeout
* SUMMARY:  Tune-up proxy command timeout value based on physical HBA
* SCOPE:    public 
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   Qlogic HBA driver resets allowed command retries if the allowed retries is less
   than its default value. 
   The failover method in the common code would like to failover to alternative controller
   in the time window of mpp_ControllerIoWaitTime. We need to tune-up proxy command timeout
   value to meet the failover time requirement
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
#define  MPPLNX_CMD_TIMEOUT_MAX  200
#define  MPPLNX_CMD_TIMEOUT_MIN   80
static void mppLnx_set_command_timeout(struct scsi_host_template *pshtp)
{

   /*
   * check if it is qla HBA
   */
   if((pshtp->proc_name != NULL) &&
      memcmp(QLA_DRIVER_PROC_NAME_PREFIX,pshtp->proc_name,3) == 0)
   {
      /*
      * The QLA driver's default retry count is 20. With the following timeout per command
      * value, after the command comes back from the middle level, we still have a chance to
      * try the other controller.
      */
      mppLnx_proxy_cmnd_timeout = 10 + (mppLnx_Config_Parameters.mpp_ControllerIoWaitTime *HZ)/3;
   }
   else
   {
      
      if( mppLnx_Config_Parameters.mpp_ControllerIoWaitTime >= MPPLNX_CMD_TIMEOUT_MAX )
      		mppLnx_proxy_cmnd_timeout = MPPLNX_CMD_TIMEOUT_MAX * HZ;
      else
      		mppLnx_proxy_cmnd_timeout = mppLnx_Config_Parameters.mpp_ControllerIoWaitTime * HZ;
	
   }

   /*
   *The HBA port down timeout is usually 60 seconds. If the command timeout value is less 
   * than 60 seconds, during port down time, the middle level may kick off the error 
   * recovery because of the command timeout. The error recovery in the HBA driver is 
   * not a very well tested area. It could cause so many problems. We try to avoid the 
   * command timeout because of the target controller is pulled off or target port is down.
   */
   if(mppLnx_proxy_cmnd_timeout/HZ < MPPLNX_CMD_TIMEOUT_MIN)
   {
      mppLnx_proxy_cmnd_timeout = MPPLNX_CMD_TIMEOUT_MIN *HZ;

   }else if(mppLnx_proxy_cmnd_timeout/HZ > MPPLNX_CMD_TIMEOUT_MAX)
   {
      mppLnx_proxy_cmnd_timeout = MPPLNX_CMD_TIMEOUT_MAX*HZ;
   }
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_cleanup_requests
* SUMMARY:  cancel virtual command and clean up left over request
* SCOPE:    local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
* The clean up contains two phases. The first phase is invoked during the time 
of the virtual host is ready to be removed. In the first phase, this function 
goes over all internal queues to check whether or not some virtual  requests 
are still in the virtual HBA driver. If so, it will add the timer back to the 
virtual command. During the virtual host removing time, the scsi middle level 
will complete those virtual command with DID_ABORT error by checking a virtual 
command's timer.

The second phase of the clean up is invoked after the virtual host is being 
removed. At this phase, we check all queues other than the free request queue
 to see whether or not still have some requests left. If the request's proxy 
 request is already queued up to the physical HBA driver, we delete the physical
 command's timer so that the physical command will never come back to the virtual
 HBA driver. If a command's timer is deleted, the scsi middle level will not 
 process the command further after it is completed. The last thing in the second
 phase is to un-mark all "struct request"  object 's tag that is part of the proxy 
 request data struct.

* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static void mppLnx_cleanup_requests(int phase)
{
   mppLnx_ProxyRequestEntry_t    *preq;
   unsigned long                 flags;

   /*
   * Forget all deferred proxy request. The virtual commands are in this queue.
   * A virtual command in this queue is not associated with a physical request.
   */
   spin_lock_irqsave ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
   while (!list_empty(&(mppLnx_deferredProxyRequestQ.list))) 
   {
         preq = list_entry(mppLnx_deferredProxyRequestQ.list.next, 
            mppLnx_ProxyRequestEntry_t,pending_list);
         if(phase == MPPLNX_CLEAN_FIRST_PHASE)
         {
            /*
            * Add the timer back and
            * let the scsi_device_cancel forget all virtual command
            */
            mppLnx_scsi_add_timer(preq->vcmnd ,15*HZ, preq->vcmd_timeoutfunc);
         }
         else
         {
            /*
            * remove from the deferred list
            * and put it back to the free list
            */
            list_del(&preq->pending_list);
            /*
             * put it back to the free list
             */
            mppLnx_put_free_proxyRequest(preq);

         } 
   }
   spin_unlock_irqrestore ( &mppLnx_deferredProxyRequestQ.queueLock, flags);

   /*
   * work on the queued queue first. This will complete all proxy request in the
   */
   spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);

   while (!list_empty(&(mppLnx_queuedProxyRequestQ.list))) 
   {
         preq = list_entry(mppLnx_queuedProxyRequestQ.list.next, 
            mppLnx_ProxyRequestEntry_t,queued_list);
         if(phase == MPPLNX_CLEAN_FIRST_PHASE)
         {
            /*
            * Add the timer back to all IOs other than path validate and
            * let the scsi_device_cancel forget all virtual command
            */
		if(preq->dispatchType   == MPPLNX_VALIDATE_PATH_TYPE)
                {
                        continue;
                }

            mppLnx_scsi_add_timer(preq->vcmnd ,15*HZ, preq->vcmd_timeoutfunc);
         }else
         {
           /*
            * remove from the queued list
            * and put it back to the free list
            */
            list_del(&preq->queued_list);
            /*
             * put it back to the free list
             */
            mppLnx_put_free_proxyRequest(preq);
         }
   }
   spin_unlock_irqrestore ( &mppLnx_queuedProxyRequestQ.queueLock, flags);


   /*
   * working on the dispatch queue
   */
   
   spin_lock_irqsave ( &mppLnx_DPProxyRequestQ.queueLock, flags);

   while (!list_empty(&(mppLnx_DPProxyRequestQ.list))) 
   {
         preq = list_entry(mppLnx_DPProxyRequestQ.list.next, 
            mppLnx_ProxyRequestEntry_t,dispatch_list);
         if(phase == MPPLNX_CLEAN_FIRST_PHASE)
         {
            /*
            * Add the timer back  to all IOs other than path validate and
            * let the scsi_device_cancel forget all virtual command
            */
		if(preq->dispatchType   == MPPLNX_VALIDATE_PATH_TYPE)
                {
                        continue;
                }

            mppLnx_scsi_add_timer(preq->vcmnd ,15*HZ, preq->vcmd_timeoutfunc);
         }else
         {
            switch (preq->dispatchType)
            {
               case MPPLNX_QUEUED_CMND_TYPE:
               case MPPLNX_DELAY_RESPONSE_TYPE:
               case MPPLNX_COMPLETED_CMND_TYPE:
                  /*
                  * the request haven't be sent yet. Forget it
                  */
                  list_del(&preq->dispatch_list);
                  /*
                  * put it back to the free list
                  */
                  mppLnx_put_free_proxyRequest(preq);
                  break;
               default:
                  /*
                  *no default
                  */
                  break;

            }
         }
   }
   spin_unlock_irqrestore ( &mppLnx_DPProxyRequestQ.queueLock, flags);

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_SetTAS_All_Arrays
* SUMMARY:  Check and Set the TAS bit for luns in all arrays discovered.
*           
* SCOPE:    private
*
*
* DESCRIPTION: 
*      This function is invoked during the boot after physical devices
*   are discovered and just before mppVhba asks for scsi midlevel scan.
*   
* RESTRICTIONS:
* 
* RETURNS: None
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
static void mppLnx_SetTAS_All_Arrays()
{
   RdacDeviceInformation_t *RdacInfo;
   BYTE path,currctl;
   LWORD i, lun;
   BOOL AVTEnabled = TRUE;
   ArrayModuleEntry_t *ModPtr;
   BOOL found=FALSE;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_SetTAS_All_Arrays\n"));

   ModPtr = mpp_ModuleArray;
   for(i=0; i<mppLnx_Config_Parameters.mpp_MaxArrayModules; i++, ++ModPtr)
   {
      RdacInfo = ModPtr->RdacInfo;
      if( !RdacInfo )
         continue;

      if( RdacInfo->AVTEnabled == (LWORD) AVTEnabled  )
      {
         /*
         * AVT is enabled. We don't care about failover
         */
         continue;
      }

      for(lun=0; lun<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lun++)
      {
         if( RdacInfo->RdacLuns[lun].NotConfigured )
            continue;

         if( !RdacInfo->RdacLuns[lun].Present ) 
         {
            MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
            "mppLnx_SetTAS_All_Arrays:: physical dev is not discovered. Lun %d\n",lun));
            continue;
         }else
         {
            found = FALSE;
            currctl = RdacInfo->RdacLuns[lun].CurrentOwningPath;
            /* Try to set TAS bit on current owning controller */
            for( path=0; path<mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
            {
               if( RdacInfo->RdacLuns[lun].LunControllers[currctl]->LunPaths[path].LunPathDevice )
               {
                  mppCmn_CheckAndSetTASBit(RdacInfo->RdacLuns[lun].LunControllers[currctl]->LunPaths[path].LunPathDevice, RdacInfo, lun);
                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
                  "mppLnx_SetTAS_All_Arrays: lun %d path=%d NotConfigured %d Present %d owner %d\n",lun,path, 
                          RdacInfo->RdacLuns[lun].NotConfigured, RdacInfo->RdacLuns[lun].Present,currctl));
                  found = TRUE;
                  break;
               }
            }

            /* 
             * Try to set TAS bit on alternate of current owning controller 
             * This gets executed only during boot time when the owning controller is offline or
             * no paths to currentowning controller are found.
             */
            for( path=0; path<mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
               {
               if( RdacInfo->RdacLuns[lun].LunControllers[currctl^1]->LunPaths[path].LunPathDevice )
               {
                  mppCmn_CheckAndSetTASBit(RdacInfo->RdacLuns[lun].LunControllers[currctl^1]->LunPaths[path].LunPathDevice, RdacInfo, lun);
                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
                  "mppLnx_SetTAS_All_Arrays: lun %d path=%d NotConfigured %d Present %d owner %d\n",lun,path, 
                          RdacInfo->RdacLuns[lun].NotConfigured, RdacInfo->RdacLuns[lun].Present,currctl^1));
                  found = TRUE;
                  break;
               }
            }

            if( found == FALSE)
            {
                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
                  "mppLnx_SetTAS_All_Arrays::ERROR: lun %d path=%d no physicallun discovered on currentowningpath \n",lun,path));
            }
         }
       
      } /* for all luns */

   } /* for all array modules */

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_SetTAS_All_Arrays()\n"));

   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_needsInitFailover
* SUMMARY:  Check all devices in the configuration to see whether or not an 
*           initial failover is needed
* SCOPE:    private
*
*
* DESCRIPTION: 
* At the mpp viatual driver load time, if any volume is owned by an alternative
* controller but the alternative controller is not available to the host, we may 
* have to perform a failover. The failover operation will be called from the common 
* code through the mppUpper driver but the implementation of the failover is in the 
* mppVhba. To avoid the circular dependency betweem mppUpper and mppVba driver modules, 
* we need to complete the mppVhba driver load first and perform the host scan in another 
* thread. 
* This function checks if we need an initial failover.
* We dont need an initial failover if
*  a. Array is in AVT mode
*  b. See both controllers
*  c. The un-available controller doesnt own volumes 
* RESTRICTIONS:
* 
* RETURNS: 0 if dont need, non-0 if needs
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/

static BOOL mppLnx_needsInitFailover()
{
   BOOL ret = FALSE;
   RdacDeviceInformation_t *RdacInfo;
   BYTE path,currctl;
   LWORD i, lun;
   BOOL AVTEnabled = TRUE;
   ArrayModuleEntry_t *ModPtr;
   BOOL found=FALSE;

   MPP_DEBUGPRINT((MPP_AVTSCAN_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_needsInitFailover\n"));

   ModPtr = mpp_ModuleArray;
   for(i=0; i<mppLnx_Config_Parameters.mpp_MaxArrayModules; i++, ++ModPtr)
   {
      RdacInfo = ModPtr->RdacInfo;
      if( !RdacInfo )
         continue;

      if( RdacInfo->AVTEnabled == (LWORD) AVTEnabled  )
      {
         /*
         * AVT is enabled. We don't care about failover
         */
         continue;
      }

         /*
       * check for all luns to see if the physical device of currentowning path
       * is discovered.
         */
      for(lun=0; lun<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lun++)
         {
         if( RdacInfo->RdacLuns[lun].NotConfigured )
            continue;

         if( !RdacInfo->RdacLuns[lun].Present ) 
         {
            ret = TRUE;
            MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
               "mppLnx_needsInitFailover : lun %d configured but not present return TRUE\n",lun));
            break;
         }else
         {
            found = FALSE;
            for( path=0; path<mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
            {
               currctl = RdacInfo->RdacLuns[lun].CurrentOwningPath;
               if( RdacInfo->RdacLuns[lun].LunControllers[currctl]->LunPaths[path].LunPathDevice )
               {
                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
                          "mppLnx_needsInitFailover : lun %d path=%d NotConfigured %d Present %d owner %d\n",lun,path, 
                          RdacInfo->RdacLuns[lun].NotConfigured, RdacInfo->RdacLuns[lun].Present,currctl));
                  found = TRUE;
                  break;
      }
            } /* for all paths */
            if( found == FALSE)
      {
                  ret = TRUE;
                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_4,
                  "mppLnx_needsInitFailover : lun %d path=%d no physicallun discovred on currentowningpath return TRUE\n",lun,path));
         break;
      }
   }
      
         if( ret == TRUE )
            break;
       
      } /* for all luns */

      if( ret == TRUE )
         break;
   } /* for all array modules */

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_needsInitFailover() retval=%d\n",ret));

   return ret;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_delayHostScan
* SUMMARY:  A wrapper function which adds a work task to the work queue for virtual 
* host scan.
* SCOPE:    private
*
*
* DESCRIPTION: 
* @ vhost  virtual scsi host
* RESTRICTIONS:
* 
* RETURNS: 
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
static void mppLnx_delayHostScan(struct Scsi_Host *vhost)
{
   mppLnx_addWorkQueue(vhost, SCAN_VHOST);
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_delayHostScanWorkHandler
* SUMMARY:  Work queue handler for virtual host device scan task.
* SCOPE:    Global
*
*
* DESCRIPTION: 
* Performs virtual host device scan in the kernel work queue thread.
* RESTRICTIONS:
* @arg  the object of mppLnx_workQueueWrapper_t
* RETURNS: 
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
void mppLnx_delayHostScanWorkHandler(struct work_struct *ws)
#else
void mppLnx_delayHostScanWorkHandler(void * arg)
#endif
{

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
   mppLnx_workQueueWrapper_t *wqw = container_of(ws, mppLnx_workQueueWrapper_t, work);
   struct Scsi_Host *vhost = (struct Scsi_Host *)wqw->work_context;
#else
   struct Scsi_Host *vhost = (struct Scsi_Host *)(((mppLnx_workQueueWrapper_t *) arg)->work_context);
#endif

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "mppLnx_delayHostScanWorkHandler - Starting virtual host device scanning\n"));

   mppLnx_SetTAS_All_Arrays();

   /* 
    * This function is called when the vhba probe did not discover the volumes during probe.
    * Check if vhba host is NULL. If it is NULL then we need to re-scall all the Luns.
    */

   if(vhost)
      scsi_scan_host(vhost);
   else
      mppLnx_vhbaScanHost(1);

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "mppLnx_delayHostScanWorkHandler - Done with virtual host device scanning\n"));

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
   MPP_FREE(wqw, sizeof(mppLnx_workQueueWrapper_t));
#else
   MPP_FREE(arg, sizeof(mppLnx_workQueueWrapper_t));
#endif
   mppLnx_workTaskRefCnt--;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vdAddWorkHandler(
* SUMMARY:  Work queue handler for handling a virtual lun creation event
* SCOPE:    Global
*
* DESCRIPTION: 
* If the handler function determined that a virtual device creation action is ready to 
* take, it will take the action and clean-up memory. Otherwise, it will re-schedule the 
* work task again.
* RESTRICTIONS:
* @arg  the object of mppLnx_workQueueWrapper_t
* RETURNS: 
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
void mppLnx_vdAddWorkHandler(struct work_struct *ws)
#else
void mppLnx_vdAddWorkHandler(void * arg)
#endif
{

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
   mppLnx_workQueueWrapper_t * wqw        = container_of(ws, mppLnx_workQueueWrapper_t, dwork.work);
#else
   mppLnx_workQueueWrapper_t * wqw        = (mppLnx_workQueueWrapper_t *) arg;
#endif
   mppLnx_vdAddEvent_t       * vdAddEvent = (mppLnx_vdAddEvent_t *) wqw->work_context;
   RdacDeviceInformation_t    * rdacInfo  = vdAddEvent->rdacInfo;
   struct scsi_device         * sdp       = vdAddEvent->sdp;
   int                        lun         = vdAddEvent->lun;
   int                        tid         = vdAddEvent->tid;
   RdacLunInformation_t       *rdaclun    = &(rdacInfo->RdacLuns[lun]);
   //unsigned long              startTime   = vdAddEvent->startTime;
   mppLnx_workqType_t         eventType   = vdAddEvent->eventType;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "mppLnx_vdAddWorkHandler H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
   if( rdaclun->PseudoLunObject )
   {
      /*
      * the virutal device has been created. No-Op. Free memory.
      */
      MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
         "mppLnx_vdAddWorkHandler- virtual device is already created H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,0,tid,lun));
      MPP_FREE(vdAddEvent, sizeof(mppLnx_workQueueWrapper_t));
      MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
      mppLnx_workTaskRefCnt--;
      MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
         "mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
      return;
   }

   if(mppLnx_readyAddVd(vdAddEvent))
   {
      if(eventType == SCAN_VHOST)
      {
          MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
             "mppLnx_vdAddWorkHandler- Starting virtual device scanning - call mppLnx_vhbaScanHost H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));

         /*
         *This is the case where the mppVhba driver doesn't discover any devices at
         *its driver loading time. We need to initialize the driver and perform the discovery
         */
         mppLnx_vhbaScanHost(1);

         MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
             "mppLnx_vdAddWorkHandler- Done with virtual device scanning - call mppLnx_vhbaScanHost H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
         MPP_FREE(vdAddEvent, sizeof(mppLnx_workQueueWrapper_t));
         MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
         mppLnx_workTaskRefCnt--;
         MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
            "finished event SCAN_HOST mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
         return;
      }else
      {
          MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
             "mppLnx_vdAddWorkHandler- call scsi_add_device H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
         /*
          * We are ready to add the virtual device now
          * Check and set the TAS bit , this is discovery during hot_add or hot_plug
         */
         mppCmn_CheckAndSetTASBit(sdp, rdacInfo, lun);
         switch(rdacInfo->ArrayType)
         {
             case MPPCMN_TARGET_FC:
                    scsi_add_device(mppLnx_vhba_virtual_host_fc, 0, tid, lun);
                    break;
             case MPPCMN_TARGET_IB:
                    scsi_add_device(mppLnx_vhba_virtual_host_ib, 0, tid, lun);
                    break;
             case MPPCMN_TARGET_ISCSI:
                    scsi_add_device(mppLnx_vhba_virtual_host_iscsi, 0, tid, lun);
                    break;
             case MPPCMN_TARGET_SAS:
                    scsi_add_device(mppLnx_vhba_virtual_host_sas, 0, tid, lun);
                    break;
             default:
                    break;

         }
         MPP_FREE(vdAddEvent, sizeof(mppLnx_workQueueWrapper_t));
         MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
         mppLnx_workTaskRefCnt--;
         MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
            "finished event ADD_VDEV mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
         return;
      }
   }else
   {
      if(mppLnx_startVhbaModRemove)
      {
         MPP_FREE(vdAddEvent, sizeof(mppLnx_workQueueWrapper_t));
         MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
         mppLnx_workTaskRefCnt--;
         MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
            "mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
         return;
      }
      /*
      *It is not ready to take action yet. Re-schedule the work task.
       *delay 1 second
       */

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
      schedule_delayed_work(&wqw->dwork, HZ);
#else
      INIT_WORK(&wqw->work, mppLnx_vdAddWorkHandler, wqw);
      schedule_delayed_work(&wqw->work, HZ);
#endif

       MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
          "mppLnx_vdAddWorkHandler send back to re-schedule tid:%d lun:%d\n",tid,lun));
       MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
          "mppLnx_vdAddWorkHandler- schedule_delayed_work H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
   }

   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_hasOwningPath
* SUMMARY:  Check if a specific volume from a storage array has its owning controller 
* data path available
* SCOPE:    Local
*
* DESCRIPTION: 
* 	* RESTRICTIONS:
* @rdacInfo  represents a storage array
* @lun  the specified logical unit  in the storage array which is visible to this host
* RETURNS: TURE if the volumes owning controller data path available otherwise FALSE
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO
*/
static BOOL mppLnx_hasOwningPath(RdacDeviceInformation_t * rdacInfo, int lun)
{
   BYTE ctl;
   BYTE path;
   BOOL rtv = FALSE;

   ctl = rdacInfo->RdacLuns[lun].CurrentOwningPath;

   if( rdacInfo->ControllerInfo[ctl]->ControllerPresent &&
        !rdacInfo->ControllerInfo[ctl]->Failed )
   {
      for(path=0; path < mppLnx_Config_Parameters.mpp_MaxPathsPerController; ++path) 
      {
         if( (rdacInfo->ControllerInfo[ctl]->ControllerPath[path].Present) )
         {
            if(rdacInfo->RdacLuns[lun].LunControllers[ctl]->LunPaths[path].LunPathDevice)
            {
               rtv = TRUE;
               break;
            }
         }
      }
   }else
   {
      rtv = FALSE;
   }

   return rtv;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_isYoungerCreated
* SUMMARY:  Check if a luns immediate younger-configured lun has virtual device created
* SCOPE:    Local
*
* DESCRIPTION: 
* 	
* RESTRICTIONS:
* @rdacInfo  represents a storage array
* @lun  the specified logical unit  in the storage array which is visible to this host
* RETURNS: TURE if smaller luns virtual device is created otherwise FALSE
* ERRNO:
*
* NOTES: If the lun is 0, the function will always return TRUE. If a luns younger bother is
* UTM lun, this function will return true.
*
* EXAMPLES:
*
* SEE ALSO
*/
static BOOL mppLnx_isYoungerCreated(RdacDeviceInformation_t * rdacInfo, int lun)
{
   int                        lunIndex;
   BOOL                       rtv = TRUE;
   RdacLunInformation_t       *rdaclun;

   if(lun == 0)
   {
      return TRUE;
   }

   for(lunIndex = (lun-1); lunIndex >= 0; lunIndex--)
   {
      rdaclun = &(rdacInfo->RdacLuns[lunIndex]);
      if(!rdaclun->NotConfigured && !rdaclun->UTMLun)
      {
         if(!rdaclun->PseudoLunObject)
         {
            rtv = FALSE;
            break;
         }else 
         {
            rtv = TRUE;
            break;
         }
      }
   }
   /*
   * if all luns whose LU number are smaller than this lun is unconfigured, this function
   * will return true.
   */
   return rtv;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_isYoungerArrayCreated
* SUMMARY:  Check if an array that has a smaller virtual target id has all virtual device created
* SCOPE:    Local
*
* DESCRIPTION: 
* 	
* RESTRICTIONS:
* @tid  the virtual target id of this array
* RETURNS: TURE if smaller tids virtual device is created otherwise FALSE
* ERRNO:
*
* NOTES: If the tid is 0, the function will always return TRUE
*
* EXAMPLES:
*
* SEE ALSO
*/
static BOOL mppLnx_isYoungerArrayCreated(int tid)
{
   int                        tidIndex;
   int                        lunIndex;
   BOOL                       rtv = TRUE;
   BOOL                       testDone = FALSE;
   RdacDeviceInformation_t    * rdacInfo;
   RdacLunInformation_t       *rdaclun;
   ArrayModuleEntry_t         *ModPtr;

   if(tid == 0)
   {
      return TRUE;
   }

   for(tidIndex = (tid-1); tidIndex >= 0; tidIndex--)
   {
      ModPtr = &mpp_ModuleArray[tidIndex];
      rdacInfo = ModPtr->RdacInfo;
      if(!rdacInfo)
      {
         continue;
      }
      for(lunIndex=0; lunIndex < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
      {
         rdaclun = &(rdacInfo->RdacLuns[lunIndex]);
         if(!rdaclun->NotConfigured && !rdaclun->UTMLun)
         {
            if(!rdaclun->PseudoLunObject)
            {
               rtv = FALSE;
               testDone = TRUE;
               break;
            }
         }//if(!rdaclun->NotConfigured)
      } // for lun
      if(testDone)
      {
         break;
      }
   }
   /*
   * if all luns whose LU number are smaller than this lun is unconfigured, this function
   * will return true.
   */
   return rtv;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_readyAddVd
* SUMMARY:  Determine if it is ready to add virtual device or devices
* SCOPE:    Local
*
* DESCRIPTION: 
*  This function contains the core logic that makes decision if it is ready to create 
* a virtual device or virtual devices. For individual LUN mapping case, the function 
* will return true as long as:
*     A. Data path to the owning controller is available and
*     B. Immediate smaller LUNs virtual device is created
*  For virtual host scan case where the mppVhba driver module doesnt discover any devices
*  at driver module loading time. The function will return true as long as:
*     All volumes see owning controller data path
*  After MPPLNX_DEVICE_SCAN_WAITTIME seconds, if the above condition still can not be met, 
*  this function will return TRUE
* 	
* RESTRICTIONS:
* @ vdAddEvent  mppLnx_vdAddEvent_t object
* RETURNS: TURE if it is ready to add a virtual device or virtual devices otherwise FALSE
* ERRNO:
*
* NOTES: 
*
* EXAMPLES:
*
* SEE ALSO:
*/

static BOOL mppLnx_readyAddVd(mppLnx_vdAddEvent_t * vdAddEvent)
{
   RdacDeviceInformation_t    * rdacInfo  = vdAddEvent->rdacInfo;
#ifdef MPP_DEBUG
   struct scsi_device         * sdp       = vdAddEvent->sdp;
#endif
   int                        lun         = vdAddEvent->lun;
   RdacLunInformation_t       *rdaclun    = &(rdacInfo->RdacLuns[lun]);
   unsigned long              startTime   = vdAddEvent->startTime;
   mppLnx_workqType_t         eventType   = vdAddEvent->eventType;
   BOOL                       rtv = TRUE;
   ArrayModuleEntry_t         *ModPtr;
   int                        i;
   int                        lunIndex;
   BOOL                       doneCheck = TRUE;

    MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
       "Entering mppLnx_readyAddVd-  H:%dC:%dT:%dL%d\n",
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
   if( eventType == SCAN_VHOST)
   {
      if(jiffies > (startTime + MPPLNX_DEVICE_SCAN_WAITTIME*HZ))
      {
         /*
         * it is still not discovered all device yet. Give up
         */
         MPP_ERRORLOGPRINT((MPP_ERR_PATH_FAILOVER,911, 
               "%s:CTL:%d:Lun:%d mppLnx_readyAddVd: no volumes have owning data path after %d second. Force virtual host scan ...\n",
                  rdacInfo->ModuleName,
                  rdacInfo->RdacLuns[lun].CurrentOwningPath,
                  lun,
                  MPPLNX_DEVICE_SCAN_WAITTIME));
         return TRUE;
      }
      /*
      *This is the case where we don't have a virutal host. We need to wait all devices 
      *are discoved before starting virtual host scan.
      */
	   ModPtr = mpp_ModuleArray;
	   for(i=0; i<mppLnx_Config_Parameters.mpp_MaxArrayModules; ++i)
      {
		   if( (rdacInfo = ModPtr->RdacInfo) ) 
         {
            for(lunIndex=0; lunIndex < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lunIndex++)
            {
               rdaclun = &(rdacInfo->RdacLuns[lunIndex]);
               if(!rdaclun->NotConfigured && !rdaclun->Present)
               {
                  MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                      "mppLnx_readyAddVd not present lun:%d\n",lun));
                  doneCheck = TRUE;
                  rtv = FALSE;
                  break; /*break the for lunIndex */
               }else if( rdaclun->NotConfigured)
               {
                  continue;
               }else if(!mppLnx_hasOwningPath(rdacInfo, lunIndex))
               {
                  MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                     "mppLnx_readyAddVd not both path lun:%d\n",lun));
                  doneCheck = TRUE;
                  rtv = FALSE;
                  break; /*break the for lunIndex */
               }
            } //for lunIdex
    
            if(doneCheck)
            {
               break; /*break the for MaxArrayModules */
            }
			} //if rdacInfo
		   ++ModPtr;
      } // for mpp_MaxArrayModules

   }else
   {
      /*
      *This is the case where we dynamically add virtual devices after the virtual host
      *has been created. We return a true if we meet
      * 1. the owning controller path for this lun is present
      * 2. the virtual device for the immediate smaller LUN for this LUN is created
      * 3. the MPPLNX_DEVICE_SCAN_WAITTIME is reached
      */
      if(!mppLnx_isYoungerArrayCreated(vdAddEvent->tid))
      {
         rtv = FALSE;
      }else if(!mppLnx_hasOwningPath(rdacInfo, lun))
      {
         rtv = FALSE;
      }else if(!mppLnx_isYoungerCreated(rdacInfo, lun))
      {
         rtv = FALSE;
      }else 
      {
         rtv = TRUE;
      }

      if(rtv == FALSE && (jiffies > (startTime + MPPLNX_DEVICE_SCAN_WAITTIME*HZ)))
      {
         /*
         * it is still not discovered all device yet. start create virtual luns from the
         * lowest lun number
         */
         if(mppLnx_isYoungerCreated(rdacInfo, lun))
         {
            rtv = TRUE;
         }else if(jiffies > (startTime + 2*MPPLNX_DEVICE_SCAN_WAITTIME*HZ))
         {
            /*
            * in the case that a younger lun creation is failed for whatever reason,
            * we will go a head to create the virtual device for this lun.
            */
            MPP_ERRORLOGPRINT((MPP_ERR_PATH_FAILOVER,912, 
               "%s:CTL:%d:Lun:%d mppLnx_readyAddVd: virtual device has not been created after %d seconds. Force virtual device creation...\n",
                  rdacInfo->ModuleName,
                  rdacInfo->RdacLuns[lun].CurrentOwningPath,
                  lun,
                  2*MPPLNX_DEVICE_SCAN_WAITTIME));
            rtv = TRUE;
         }
      }
   }
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_readyAddVd- Ready?%d H:%dC:%dT:%dL%d\n",rtv,
            sdp->host->host_no,sdp->channel,sdp->id,sdp->lun));
   return rtv;
}
/*
* initialize the virtual HBA driver module
*/

    module_init(mppLnx_vhba_init);
    module_exit(mppLnx_vhba_exit);
MODULE_AUTHOR("IBM");
MODULE_DESCRIPTION("MPP Virtual HBA Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(MPPLNX_VERSION);
MODULE_INFO(supported,"yes");
