/*******************************************************************************
* Copyright 2004-2008 LSI Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppLnx26p_vhbaio.c
SUMMARY         %description%
VERSION         %version: 30 %
UPDATE DATE     %date_modified: Mon Aug 04 18:24:29 2008 %
PROGRAMMER      %created_by:    bmoger %


DESCRIPTION:This source file contains Linux MPP virtual HBA driver functions
   that are related to IO routing. The function mppLnx_queuecommand() in
   mppLnx_vhba.c receives Scsi command for all virtual Scsi devices from
   the Linux SCSI middle level. The functions in this file are responsible
   for routing those virtual Scsi commands to physical HBA drivers, taking
   care of path/controller failover, and completing those virtual commands.
   The functions in this file can be considered as "the Heart of the Linux
   MPP virtual HBA driver".

INCLUDE FILES:

NOTES: Error number range for this file is 480 to 530.

RESTRICTIONS:

SEE ALSO:

REFERENCE:

  349-1033500 Linux MPP Driver Feature Architecture Specification
  349-1034610 Linux MPP Virtual HBA Driver Feature Design Document
  349-1034750 Linux MPP Upper Level Driver Feature Design Document
  349-1034730 Linux MPP Utility Feature Design Document
  349-1034620 Linux MPP Installation Feature Design Document
  349-1043860 Linux MPP Driver for 2.6 Kernel
  349-1046490 On-the-fly Path Validation


IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/

#define __SRCmppLnx26p_vhbaio_c

/***  INCLUDES  ***/
/***  Linux kernel INCLUDES  ***/
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/genhd.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <scsi/scsi_host.h>

#include <asm/io.h>

#include <linux/blkdev.h>
#include <scsi/scsi.h>
#include <scsi/scsi_tcq.h>
#include <scsi/scsi_eh.h>

#include <linux/stat.h>

/***  MPP commomn API INCLUDES  ***/
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "MPP_Sysdep.h"
#include "mppLnx_version.h"
#include "mppCmn_s2tos3.h"
/***  Linux MPP Virtual HBA INCLUDES  ***/
#include "mppLnx26p_vhba.h"
#include "mppLnx26p_shared.h"

/***  CONSTANT DEFINITIONS  ***/


/***  MACRO DEFINITIONS  ***/

/***  TYPE DEFINITIONS  ***/

extern void (*middle_level_scsi_done)(struct scsi_cmnd *);
extern void (*middle_level_timeoutfunc)(struct scsi_cmnd *);

/*** EXTERNAL GLOBAL VARIABLES ***/

	// Following stuff is in mppLnx26_vhba.c
extern int mppLnx_proxy_cmnd_timeout;
extern mppLnx_pathvalidateContext_t mppLnxPathValidateContext;
extern mppLnx_dpcQueueContext_t   mppLnxDpcQueueContext;
extern mppLnx_failbackScanContext_t 	mppLnxFailbackScanContext;

/*** EXTERNAL FUNCTIONS ***/

	// Following stuff is in mppLnx26_upper.c
extern LWORD mppLnx_Translate_Linux_Status(int status);

	// Following stuff is in MPP_hba.c
extern BYTE mpp_IsFailoverPossible(RdacDeviceInformation_t *RdacInfo,
                                   BYTE Controller, BYTE Path, LWORD Lun, BYTE level);

/***  LOCAL FUNCTIONS  ***/

//static void mppLnx_scsi_init_proxy_cmnd(struct scsi_cmnd *Scpnt);
static void mppLnx_initProxyRequest(mppLnx_ProxyRequestEntry_t *preq, struct scsi_device *pSDp);
static void mppLnx_scsicmnd_virtual_to_proxy(mppLnx_ProxyRequestEntry_t *preq);
static void mppLnx_scsicmnd_proxy_to_virtual(struct scsi_cmnd *vscp, mppLnx_ProxyRequestEntry_t    *preq);
static void mppLnx_TakeCareOfWriteAndVerify(mppLnx_ProxyRequestEntry_t *preq, BYTE controller);


/***  PROCEDURES  ***/
#ifdef MPPLNX_SCSI_REQ_MAP_SG 

static int mppLnx_scsi_req_map_sg(struct request *rq, struct scatterlist *sgl,
               int nsegs, gfp_t gfp);
static int mppLnx_scsi_merge_bio(struct request *rq, struct bio *bio);

#else /* MPPLNX_SCSI_REQ_MAP_SG */

static void __mppLnx_bio_clone(struct bio *bio, struct bio *bio_src,request_queue_t *q);
static struct bio *mppLnx_bio_clone(struct bio *bio, gfp_t gfp_mask,request_queue_t *q);
static int mppLnx_clone_request_bios(struct request *clonerequest, struct request *vrequest, unsigned bufflen,gfp_t gfp, mppLnx_ProxyRequestEntry_t *preq);

#endif /* MPPLNX_SCSI_REQ_MAP_SG */

static int mppLnx_scsi_bi_endio(struct bio *bio, unsigned int bytes_done, int error);
int mppLnx_scsi_execute_async(struct scsi_device *sdev, const unsigned char *cmd,
                       int cmd_len, int data_direction, void *buffer, unsigned bufflen,
                       int use_sg, int timeout, int retries, void *privdata,
                       void (*done)(void *, char *, int, int), gfp_t gfp);
static void mppLnx_scsi_end_async(struct request *req, int uptodate);


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_do_queuecommand
* SUMMARY:  dispatch a virtual struct scsi_cmnd to physical HBA driver
* SCOPE:     public
*
*
* DESCRIPTION:
   This function receives a virtual struct scsi_cmnd and dispatches a proxy struct scsi_cmnd to
   a physical HBA driver.

*  @vscp The virtual struct scsi_cmnd needs to be routed.
*  @done The virutal struct scsi_cmnd's command completion function
*  @timeoutfunc   The middle level time out function
*  @ml_done       The middle level command completion function
*  @previous_controller If the virtual IO is re-sent to a new path, we need to know
                  which controller of the virtual IO was tried. For a virtual IO, which
                  is first time queued to the virtual HBA driver, the previous_controller
                  will be UNKNOWN_PREVIOUS_CONTROLLER.
*  @previous_path If the virtual IO is re-sent to a new path, we need to know
                  which path of the virtual IO was tried. For a virtual IO, which
                  is first time queued to the virtual HBA driver, the previous_controller
                  will be UNKNOWN_PREVIOUS_PATH.
*  @CommandTimeoutRetryCount  The time out retry count. If this function is invoked from
                              the mppLnx_queuecommand(), the value should be the
                              value from the field configurable parameter. If this is a
                              command retry, the value should be the value returned by the
                              common code
*  @RetryCount                Command retry counter. If this function is invoked from
                              the mppLnx_queuecommand(), the value should be the
                              value from the field configurable parameter. If this is a
                              command retry, the value should be the value returned by the
                              common code
*  @UaRetryCount              Unit Attention retry count.
*  @vcmndStartTime            The start up time of a virtual command being dispatched
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_do_queuecommand(mppLnx_ProxyRequestEntry_t *preq)
{
   struct scsi_device            * pSDp;
   struct scsi_cmnd              * vSCp = preq->vcmnd;
   RdacDeviceInformation_t       *rdacInfo;
   LWORD                         lun;
   LWORD                         lunindex;
   BYTE                          controller;
   BYTE                          path;
   int                           vtarget;
   LunPathInfo_t                 *lunpath;
   LINT                          LockLevel = 0;
   RdacControllerPath_t 	 *controllerPath;

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_do_queuecommand()\n"));

   if(unlikely(!preq))
   {
      MPPLNX_VHBA_BUG();
   }

   if(unlikely(!preq->vcmd_timeoutfunc))
   {
      MPPLNX_VHBA_BUG();
   }

   if(unlikely(!preq->done))
   {
      MPPLNX_VHBA_BUG();
   }

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "cmnd SN 0x%lx cmndtimeoutfunc 0x%ld ml_done 0x%lx \n",
      preq->vcmnd->serial_number,
      (unsigned long)preq->vcmd_timeoutfunc,
      (unsigned long)preq->done));


   vtarget = preq->vcmnd->device->id;
   lun = preq->vcmnd->device->lun;
   rdacInfo = mpp_ModuleArray[vtarget].RdacInfo;
   preq->rdacInfo = rdacInfo;

   if(unlikely(preq->isVcmdCompleted == TRUE))
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
         "doQ isVcnd-T %ld\n",preq->vcmnd->serial_number));
      /*
      *the virtual command is already completed.
      */
      mppLnx_put_free_proxyRequest(preq);
      return 0;
   }
   if( unlikely(rdacInfo == NULL))
   {
      /*return DID_NO_CONNECT. The IO will be failed*/
      //new error number. The array has been removed
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 513,
         "vtid %d:Lun %d RdacInfo is null.  vcmnd SN %ld\n",
          vtarget,
          lun,
          vSCp->serial_number));


      vSCp->result = DID_NO_CONNECT<<16;

      MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
             "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d LCP=LV EX=No-path CS=%08x\n",
             vSCp->serial_number,
             vSCp->cmnd[0],
             vSCp->device->host->host_no,
             vSCp->device->channel,
             vSCp->device->id,
             vSCp->device->lun,
             vSCp->result));
      mppLnx_scsi_add_timer(vSCp,5*HZ,preq->vcmd_timeoutfunc);
      /*
      * before we leave, we need to take care of the proxy request
       */
      mppLnx_put_free_proxyRequest(preq);
      vSCp->scsi_done(vSCp);
      return 0;

   }

   MPP_GETTIME(preq->startTime);

   lunpath = mppCmn_NewSelectPath(rdacInfo, lun, &controller, &path);
      /*
      * if there is not path available, (all of them failed)
      * we need let the middle level know
      */
   if( unlikely(lunpath == NULL))
   {
      /*return DID_NO_CONNECT. The IO will be failed*/
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 480,
         "%s:%d:%d:%d Selected path returns NULL.  vcmnd SN %ld\n",
          (char *)rdacInfo->ModuleName,
          controller,
          path,
          lun,
          vSCp->serial_number));


      vSCp->result = DID_NO_CONNECT<<16;

      MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
             "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d LCP=LV EX=No-path CS=%08x\n",
             vSCp->serial_number,
             vSCp->cmnd[0],
             vSCp->device->host->host_no,
             vSCp->device->channel,
             vSCp->device->id,
             vSCp->device->lun,
             vSCp->result));
      mppLnx_scsi_add_timer(vSCp,5*HZ,preq->vcmd_timeoutfunc);
      /*
      * before we leave, we need to take care of the proxy request
       */
      mppLnx_put_free_proxyRequest(preq);
      vSCp->scsi_done(vSCp);
      return 0;

    }

    /* Check the Device state */
    switch ( lunpath->DeviceState[lunpath->DeviceStateIndex] )
    {
    /* route the IO to the native device */
    case MPPCMN_OPTIMAL:
    case MPPCMN_OPTIMAL_NEED_CHECK:
    case MPPCMN_OPTIMAL_CHECKING:
        break;

    /* Defer IO request and schedule path validation command */
    case MPPCMN_FAILED_NEED_CHECK:
        MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "go to DFQ1 %ld\n",preq->vcmnd->serial_number));

        MPP_LOCK(rdacInfo->Lock, LockLevel);
        MPPCMN_SET_PATH_STATE(rdacInfo, controller, path, MPPCMN_FAILED_CHECKING);
        for(lunindex=0; lunindex<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lunindex)
        {
            if(rdacInfo->RdacLuns[lunindex].LunControllers[controller]->LunPaths[path].LunPathDevice)
            {
                MPPCMN_SET_DEVICE_PATH_STATE(rdacInfo, controller, path, lunindex, MPPCMN_FAILED_CHECKING);
            }
        }
        MPP_UNLOCK(rdacInfo->Lock, LockLevel);

        if(mppLnx_validatepath(rdacInfo, controller, path) != MPP_GOOD_STATUS)
        {
            // Validation request attempt failed.
            MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
                "mppLnx_do_queuecommand(): %s:%d:%d IssueValidation(Optimal) failed\n",
                rdacInfo->ModuleName, controller, path));
            MPP_LOCK(rdacInfo->Lock, LockLevel);
            MPPCMN_SET_PATH_STATE(rdacInfo, controller, path, MPPCMN_FAILED);
            //rdacInfo->ControllerInfo[controller]->ControllerPath[path].PathIdleTime = 0;
            rdacInfo->ControllerInfo[controller]->ControllerPath[path].PathFailedTime= 0;
            MPP_UNLOCK(rdacInfo->Lock, LockLevel);
        }

        /* Defer IO request */
        preq->controller = controller;
        preq->path = path;
        mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DEFERRED_LIST);
        MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_do_queuecommand(): %s:%d:%d Command added to DEFERRED List.\n",
            rdacInfo->ModuleName, controller, path));
        return 0;

        break;


    case MPPCMN_FAILED_CHECKING:
        /* Defer IO request */
        MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "go to DFQ2 %ld\n",preq->vcmnd->serial_number));
        preq->controller = controller;
        preq->path = path;

        mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DEFERRED_LIST);
        MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_do_queuecommand(): %s:%d:%d Command added to DEFERRED List.\n",
            rdacInfo->ModuleName, controller, path));
        return 0;

        break;

        /* Return IO Error Immediately */
    case MPPCMN_FAILED:
        /*return DID_NO_CONNECT. The IO will be failed*/
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 482,
            "%s:%d:%d:%d Selected path's Device Status is FAILED. vcmnd SN %ld\n",
            (char *)rdacInfo->ModuleName,
             controller,
             path,
             lun,
             vSCp->serial_number));

        vSCp->result = DID_NO_CONNECT<<16;

        MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
             "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d LCP=LV EX=No-path CS=%08x\n",
             vSCp->serial_number,
             vSCp->cmnd[0],
             vSCp->device->host->host_no,
             vSCp->device->channel,
             vSCp->device->id,
             vSCp->device->lun,
             vSCp->result));
        mppLnx_scsi_add_timer(vSCp,5*HZ,preq->vcmd_timeoutfunc);
        /*
        * before we leave, we need to take care of the proxy request
        */
        mppLnx_put_free_proxyRequest(preq);
        vSCp->scsi_done(vSCp);
        return 0;

         /* Linux should never get this state Return IO Error */
    case MPPCMN_SYSTEM_SPECIFIC:
    default:
        MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 483,
            "%s:%d:%d:%d mppLnx_do_queuecommand(): Invalid Device State %d.vcmnd SN %ld.\n",
            (char *)rdacInfo->ModuleName,
            controller,
            path,
            lun,
            lunpath->DeviceState[lunpath->DeviceStateIndex],
            vSCp->serial_number));

        /* Panic The system we should not get here */
        MPPLNX_VHBA_BUG();

        break;

    }

    pSDp = lunpath->LunPathDevice;
    MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
           "mppLnx_do_queuecommand: Selected Path H%dT%dL%d\n",pSDp->host->host_no, pSDp->id, pSDp->lun));

    mppLnx_TakeCareOfWriteAndVerify(preq, controller);

    /*
    * If the proxy command is routed to a different controller or path,
    * reset the retry count
    */
    if( ( (preq->path != UNKNOWN_PREVIOUS_PATH) &&
          (preq->controller != UNKNOWN_PREVIOUS_CONTROLLER) ) &&
       ( (preq->controller != controller) || (preq->path != path) ) )
    {
        preq->RetryCount = mppLnx_Config_Parameters.mpp_SyncRetryCount;
    }
    /*
    * Check if we need to reset wait time states and timers.
    * If the proxy command is routed to a different controller,
    * reset other controller based fields
    */
    if( (preq->controller != UNKNOWN_PREVIOUS_CONTROLLER) && (preq->controller != controller) )
    {
        // IO being resent on a new controller, reset counters and timers.
        MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_2,
              "mppLnx_do_queuecommand: IO retried on new controller, %s:%d:%d:%d\n",
              rdacInfo->ModuleName, controller, path,	lun));


        preq->UaRetryCount = 0;
        preq->CommandTimeoutRetryCount = 0;
        preq->SelectionTimeoutRetryCount = 0;
        rdacInfo->RdacLuns[lun].NotReady = FALSE;
        rdacInfo->RdacLuns[lun].Busy = FALSE;
        rdacInfo->RdacLuns[lun].Quiescent = FALSE;
        rdacInfo->RdacLuns[lun].QuiescienceStart = 0;
        rdacInfo->RdacLuns[lun].NotReadyStart = 0;
        rdacInfo->RdacLuns[lun].BusyStart = 0;
        MPP_GETTIME(preq->ControllerIoWaitTime);

    }
    /*
    *set up the scsi_request and request
    *copy data from virtual command to proxy scsi_request
    */
    mppLnx_initProxyRequest(preq, pSDp);
    /*
    * copy/set necessary fields from the virtual command to its proxy command.
    */
    preq->vtid = rdacInfo->VirtualTargetId;

    mppLnx_scsicmnd_virtual_to_proxy(preq);

    preq->errorRecovery = 0;

#ifdef MPP_DEBUG
    if(vSCp->cmnd[0] == SCSI_INQUIRY &&
      (vSCp->cmnd[1] & 0x01) == 0 )
    {
        printk("Dispatch Inquiry cmnd for virtual device %d:%d\n",
             vSCp->device->id,vSCp->device->lun);

    }
#endif

    preq->controller = controller;
    preq->path = path;

    /*
    * Check the device state before sending the IO , the device could be in remove state
    * or offline state. Mark the path offline so altenative optimal paths can be chosen
    * to route IO. If all the paths are failed then application throws IO error.
    */
    if(unlikely(lunpath->PathRemoveState || pSDp->sdev_state == SDEV_DEL ||
       pSDp->sdev_state == SDEV_CANCEL || (pSDp->sdev_state == SDEV_OFFLINE)))
    {
        MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
             "DEL send %d %d:%d\n",pSDp->host->host_no,pSDp->id,pSDp->lun));

        //this is the last change we can check the PathRemoveState
        MPP_ERRORLOGPRINT((MPP_ERR_FATAL,502,
                  "%s:%d:%d:%d Fail the path. Device in Remove or Offline State. PathRemoveState:%d dev_state:%d\n",
                  rdacInfo->ModuleName,
                  controller,
                  path,
                  pSDp->lun,
                  lunpath->PathRemoveState,
                  pSDp->sdev_state
                  ));

        /*
        * Mark the path and all devices on that path to failed state.
        * When the devices are recovred or become optimal path validation
        * thread will mark the path online accordingly.
        */
        MPP_LOCK(rdacInfo->Lock, LockLevel);
        MPPCMN_SET_PATH_STATE(rdacInfo, controller, path, MPPCMN_FAILED);
        rdacInfo->ControllerInfo[controller]->ControllerPath[path].PathFailedTime= 0;
        for(lunindex=0; lunindex<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lunindex)
        {
            if(rdacInfo->RdacLuns[lunindex].LunControllers[controller]->LunPaths[path].LunPathDevice)
            {
                MPPCMN_SET_DEVICE_PATH_STATE(rdacInfo, controller, path, lunindex, MPPCMN_FAILED);
            }
        }

        MPP_UNLOCK(rdacInfo->Lock, LockLevel);

        /* Retry the IO on other optimal paths if available */
        mppLnx_do_queuecommand(preq);
        return 0;
    }

    /*
    *insert the request to the queued queue and send the proxy request back to
    *the middle level.
    *Next time when we see the command, it will be from mppLnx_scsi_done()
    *function
    */
    mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_QUEUED_LIST);

    MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
        "mppLnx_do_queuecommand :: cs %d, bufflen %d use_sg %d \n ",COMMAND_SIZE(preq->vcmnd->cmnd[0]),preq->vcmnd->request_bufflen,preq->vcmnd->use_sg));
	
    /* Increment the Number of Outstanding commands count */
    controllerPath = &rdacInfo->ControllerInfo[controller]->ControllerPath[path];
    MPPSYS_ATOMIC_INC(controllerPath->RequestsInProgress);
    MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4, "mppLnx_do_queuecommand(%s) Lun:%d"
				" Sending the command on (Path %d controller %d) with pending %d requests\n",
				rdacInfo->ModuleName, lun, path, controller, MPPSYS_ATOMIC_GET(controllerPath->RequestsInProgress))); 

    if (mppLnx_scsi_execute_async(preq->pdev,preq->vcmnd->cmnd,COMMAND_SIZE(preq->vcmnd->cmnd[0]),
            preq->vcmnd->sc_data_direction, preq->vcmnd->request_buffer,
            preq->vcmnd->request_bufflen, preq->vcmnd->use_sg,mppLnx_proxy_cmnd_timeout,
            1, preq, mppLnx_scsi_done,GFP_ATOMIC))
    {

        MPP_ERRORLOGPRINT((MPP_ERR_FATAL,509,
              "mppLnx_do_queuecommand: mppLnx_scsi_execute_async failed.\n" ));

        /*
         * most likely out of mem, bio allocation or request allocation failed, retry again
         */
        mppLnx_scsi_done(preq,preq->sense_buffer,DID_ERROR<<16,preq->vcmnd->request_bufflen); 
        return 0;

    }
    /*
    **
    * call generic_unplug_device() to start the transfer
    *   Linux uses plugging to build bigger requests queues before letting
    *   the device have at them. If a queue is plugged, the I/O scheduler
    *   is still adding and merging requests on the queue. Once the queue
    *   gets unplugged, the request_fn defined for the queue is invoked and
    *   transfers started.
    **/
    generic_unplug_device(pSDp->request_queue);


    MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
        "Exiting mppLnx_do_queuecommand()\n"));
    return 0;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_RetrySamePath
* SUMMARY:  dispatch a virtual struct scsi_cmnd to a given struct scsi_device (path)
* SCOPE:     private
*
*
* DESCRIPTION:
   This function receives a virtual struct scsi_cmnd and dispatches a proxy struct scsi_cmnd to
   the specified struct scsi_device

*  @pSDp The physical struct scsi_device object to which the virtual struct scsi_cmnd will issue
*         a proxy command to
*  @vscp The virtual struct scsi_cmnd needs to be routed.
*  @done The virutal struct scsi_cmnd's command completion function
*  @timeoutfunc   The middle level time out function
*  @ml_done       The middle level command completion function
*  @CommandTimeoutRetryCount  The time out retry count. If this function is invoked from
                              the mppLnx_queuecommand(), the value should be the
                              value from the field configurable parameter. If this is a
                              command retry, the value should be the value returned by the
                              common code
*  @RetryCount                Command retry counter. If this function is invoked from
                              the mppLnx_queuecommand(), the value should be the
                              value from the field configurable parameter. If this is a
                              command retry, the value should be the value returned by the
                              common code
*  @UaRetryCount              Unit Attention retry count.
*  @vcmndStartTime            The start up time of a virtual command being dispatched
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_RetrySamePath(
           mppLnx_ProxyRequestEntry_t *preq)
{
   struct scsi_device            *pSDp;
   RdacDeviceInformation_t       *rdacInfo;
   BYTE                          controller;
   BYTE                          path;
   LWORD                         lunindex;
   LINT                          LockLevel = 0;
    RdacControllerPath_t          *controllerPath;


   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_RetrySamePath()\n"));

  MPP_GETTIME(preq->startTime);
  /*
   * copy/set necessary fields from the virtual command to its proxy command.
   */
   mppLnx_scsicmnd_virtual_to_proxy(preq);
   /*
   * save the device to avoid a race condition in generic_unplug_device
   */
   pSDp = preq->pdev;

   rdacInfo   = mpp_ModuleArray[preq->vtid].RdacInfo;
   controller = preq->controller;
   path       = preq->path;

   /*
    * Check the device state before sending the IO , the device could be in remove state
    * or offline state. Mark the path offline so altenative optimal paths can be chosen
    * to route IO. If all the paths are failed then application throws IO error.
    */
   if(unlikely( pSDp->sdev_state == SDEV_DEL ||
      pSDp->sdev_state == SDEV_CANCEL || (pSDp->sdev_state == SDEV_OFFLINE)))
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
         "send %d %d:%d\n",pSDp->host->host_no,pSDp->id,pSDp->lun));

      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,505,
                  "%s:%d:%d:%d Fail the path. Device in Remove or Offline State. sdev_state:%d\n",
                  rdacInfo->ModuleName,
                  controller,
                  path,
                  pSDp->lun,
                  pSDp->sdev_state
                  ));

      /*
       * Mark the path and all devices on that path to failed state.
       * When the devices are recovred or become optimal path validation
       * thread will mark the path online accordingly.
       */
      MPP_LOCK(rdacInfo->Lock, LockLevel);
      MPPCMN_SET_PATH_STATE(rdacInfo, controller, path, MPPCMN_FAILED);
      rdacInfo->ControllerInfo[controller]->ControllerPath[path].PathFailedTime= 0;
      for(lunindex=0; lunindex<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lunindex)
      {
	   if(rdacInfo->RdacLuns[lunindex].LunControllers[controller]->LunPaths[path].LunPathDevice)
	   {
		MPPCMN_SET_DEVICE_PATH_STATE(rdacInfo, controller, path, lunindex, MPPCMN_FAILED);
	   }
      }

      MPP_UNLOCK(rdacInfo->Lock, LockLevel);

      /* Retry the IO on other optimal paths if available */
      mppLnx_do_queuecommand(preq);
      return 0;
   }

   /*
   * the request is not in any queue now.
   * the mppLnx_scsi_done removed the request from the queued queue
   * we need to insert this request back to the queued queue
   */
   mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_QUEUED_LIST);

   /* Get the Controller Path */
   controllerPath = &rdacInfo->ControllerInfo[controller]->ControllerPath[path];
   MPPSYS_ATOMIC_INC(controllerPath->RequestsInProgress);
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4, "mppLnx_do_queuecommand(%s) Lun:%d"
                " Sending the command on (Path %d controller %d) with pending %d requests\n",
                rdacInfo->ModuleName, preq->vcmnd->device->lun, path, controller,
                MPPSYS_ATOMIC_GET(controllerPath->RequestsInProgress))); 

   if (mppLnx_scsi_execute_async(preq->pdev,preq->vcmnd->cmnd,COMMAND_SIZE(preq->vcmnd->cmnd[0]),
   			preq->vcmnd->sc_data_direction, preq->vcmnd->request_buffer,
   			preq->vcmnd->request_bufflen, preq->vcmnd->use_sg,mppLnx_proxy_cmnd_timeout,
   			1, preq, mppLnx_scsi_done,GFP_ATOMIC))
   {

        MPP_ERRORLOGPRINT((MPP_ERR_FATAL,510,
              "mppLnx_RetrySamePath: mppLnx_scsi_execute_async failed.\n" ));

        /*
         * most likely out of mem, bio allocation or request allocation failed, retry again
         */

        mppLnx_scsi_done(preq,preq->sense_buffer,DID_ERROR<<16,preq->vcmnd->request_bufflen); 
        return 0;
   }
   /*
   * unplug the device and let the request go
   */

   generic_unplug_device(pSDp->request_queue);

   return 0;
}

/****************************************************************************
* PROCEDURE
* NAME:    mppLnx_printsensebuffer
* SUMMARY: MPP System Dependent routine to print sense buffer
* SCOPE:   public
* DESCRIPTION: This function is an implementation of the system dependent function
* 		mppSys_PrintSenseBuffer( ), which is specified by the MPP driver command API
*               as a system dependent function.
*
*	     This routine is used for printing/dumping the sense buffer.
*
*
* RESTRICTIONS:
* RETURNS: 	VOID
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/
void
mppLnx_printsensebuffer (RdacDeviceInformation_t *RdacInfo, LWORD Controller,
			LWORD Path, void *SenseBuffer, ULONG SerialNumber)
{
    CHAR                mpp_PrintBuffer[PRINT_SENSE_BUFFER_LENGTH];
    SenseData_t         *SenseData;
    int                 SenseLength;
    int                 i;
    int                 len=0;
    int                 has_SenseData = 0;
    char                *buffer;

    /* perform this only when the config parameter is set to non zero value */
    if( mppLnx_Config_Parameters.mpp_PrintSenseBuffer == 0 )
       return;

    SenseData      = (SenseData_t *) SenseBuffer;
    SenseLength    = sizeof(SenseData_t);

    /* if sensedata buffer is not in format then do not print anything */
    has_SenseData = SenseData->Sense_Key != 0;
    if( has_SenseData == 0)
    {
	return;
    }


    buffer = ( char * ) SenseBuffer;


    memset(mpp_PrintBuffer, 0, PRINT_SENSE_BUFFER_LENGTH);

    len += sprintf( mpp_PrintBuffer+len, "SENSE_DATA_START SN %ld",SerialNumber);
    for(i=0;i<SenseLength;i++)
    {
        if( ( i % 16 ) == 0  )
        {
            len += sprintf( mpp_PrintBuffer+len, "\nSN %ld %04x " ,SerialNumber,i );

        }
        len += sprintf( mpp_PrintBuffer+len, "%02x " , (unsigned char)buffer[i]);
    }

    len += sprintf( mpp_PrintBuffer+len, "\nSENSE_DATA_END SN %ld\n",SerialNumber);

    MPP_ERRORLOGPRINT((MPP_ERR_PATH_FAILOVER, 485,
       "%s", mpp_PrintBuffer));

    return;

}

/****************************************************************************
* PROCEDURE
* NAME:    mppLnx_validatepath
* SUMMARY: MPP System Dependent routine to issue a Path Validation ( Inquiry or TUR ) command.
* SCOPE:   public
* DESCRIPTION: This function is an implementation of the system dependent function
* 		mppSys_ValidatePath( ), which is specified by the MPP driver command API
*               as a system dependent function.
*
*	     This routine is used for validating path and the command used can be
*            either Inquiry or TUR.
*
*
* RESTRICTIONS:
* RETURNS: 	MPP_GOOD_STATUS ( if the command is issued successfully ) or appropriate failure.
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/
LWORD
mppLnx_validatepath (RdacDeviceInformation_t *RdacInfo, LWORD Controller, LWORD Path)
{

	LINT                         lun;
	LunPathInfo_t                *lunPathInfo =NULL;
	struct scsi_device           *pSDp;
	struct scsi_cmnd             *validateVscp;
	mppLnx_ProxyRequestEntry_t   *validateRequestp;
        mppLnx_AsyncRequestEntry_t   *are;
        LINT                         foundDevice = 0;
   	LINT                         LockLevel = 0;
   	LWORD                        lunindex;
	RdacControllerPath_t         *controllerPath;


	if(unlikely(!RdacInfo))
	{
		MPPLNX_VHBA_BUG();
	}


	/* find a lun that is in "Present" state */
	for(lun = 0; lun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lun++)
	{
		lunPathInfo = &RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path];

		/* skip the lun if it is not present */
		if(!RdacInfo->RdacLuns[lun].Present || !lunPathInfo->LunPathDevice || lunPathInfo->PathRemoveState)
		{

				continue;
		}

      		foundDevice = 1;
		break;
	}

   	if(foundDevice == 0)
   	{
      		/*
      		 *if only UTM lun is mapped, RdacInfo->RdacLuns[lun].Present is false for UTM device
      		 * we end up has no physical device is selected from the above loop. Return from here
      		 */
      		return (MPP_HARDWARE_ERROR);
   	}

   	if(unlikely(!lunPathInfo))
   	{
      		MPPLNX_VHBA_BUG();
   	}

	pSDp = lunPathInfo->LunPathDevice;  /* Point to the SCSI Device */



	validateRequestp = mppLnx_get_free_proxyRequest();

   	if(unlikely(!validateRequestp))
   	{
      		MPPLNX_VHBA_BUG();
   	}

	/*
    	  * This command is generated locally and not received from middle level
   	  * so it does not have virtual command from the upper level driver associated
   	  * with it.  But the commmand would be completed asynchrounously, we need
   	  * to keep trackof it t by using a proxy request object.
   	  */
      are = mppLnx_get_free_asyncRequest();
      if(unlikely(!are))
   	{
         MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 481,
           "%s:%d:%d Error in allocating memory for path validation\n",
           (char *)RdacInfo->ModuleName,
           Controller,
           Path));

      /*
      		  * we didn't issue a command to the controller.
      		  * Return Error status.
      		  */
      		return (MPP_HARDWARE_ERROR);
   	}
   	validateVscp = are->cmnd;
      validateVscp->request = NULL;

	 /*
	   * Setup INQUIRY COMMAND
	   */
   	 memset(validateVscp, 0, sizeof(struct scsi_cmnd));
	 validateVscp->cmnd[0] = SCSI_INQUIRY;
	 validateVscp->cmnd[1] = 0;
	 validateVscp->cmnd[2] = 0;   /* Page Code */
	 validateVscp->cmnd[3] = 0;    /* Reserved   */
	 validateVscp->cmnd[4] = sizeof(InquiryData_t);
	 validateVscp->cmnd[5] = 0;    /* Control */

  	 /*
  	 * set up the path validation virtual command's related fields
  	 */

  	 validateVscp->cmd_len 		     = 6;
 	 validateVscp->sc_data_direction = DMA_FROM_DEVICE;
 	 validateVscp->underflow 	    = 0;
 	 validateVscp->use_sg 			= 0;
 	 validateVscp->sglist_len 		= 0;

	 validateVscp->request_buffer	= (char *)are->buffer;
	 validateVscp->request_bufflen	= sizeof( InquiryData_t );

	 MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
 	     "mppLnx_validatepath() setting up path validation scsi cmnd fields.\n"));

 	 /*
 	 * set up the request's related fields
 	 */
 	 validateRequestp->vcmd_timeoutfunc				= middle_level_timeoutfunc;
   	 validateRequestp->vcmnd						= validateVscp;
   	 validateRequestp->controller					= Controller;
  	 validateRequestp->path	    					= Path;
  	 validateRequestp->rdacInfo						= RdacInfo;
   	 validateRequestp->CommandTimeoutRetryCount	    = 0;
   	 validateRequestp->RetryCount					= mppLnx_Config_Parameters.mpp_SyncRetryCount;
   	 validateRequestp->UaRetryCount				    = 0;
   	 validateRequestp->SelectionTimeoutRetryCount	= 0;
	 validateRequestp->dispatchType 				= MPPLNX_VALIDATE_PATH_TYPE;
     validateRequestp->state 						= MPPLNX_REQUEST_DISPATCHED;

   	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      		"mppLnx_validatepath() setting up path validation Proxy Request fields.\n"));

	MPP_GETTIME(validateRequestp->startTime);
	MPP_GETTIME(validateRequestp->vcmndStartTime);

	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
		"mppLnx_validatepath() issuing path validation command\n"));

   	/*
   	* the command completion function mppLnx_pathvalidate_done() should remember
   	* to free the virtual command memory
   	*/

   	mppLnx_initProxyRequest(validateRequestp,pSDp);

   	/*
   	* send the validate command to the SCSI middle level
   	*/
   	mppLnx_insert_proxyRequest_to_list(validateRequestp,MPPLNX_QUEUE_QUEUED_LIST);
      validateRequestp->vtid = RdacInfo->VirtualTargetId;

   	mppLnx_scsicmnd_virtual_to_proxy(validateRequestp);

	/*
	 * Device is marked offline
         * Set the path offline and do not send the command down.
 	 */
       if(unlikely(scsi_host_in_recovery(pSDp->host)|| (pSDp->sdev_state == SDEV_BLOCK)))
        {
            MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,519,
                "%s:%d:%d:%d - pdev H%d:C%d:T%d:L%d physical scsi host is in recovery or sdev_state is %d.\n",
                RdacInfo->ModuleName,Controller, Path,lun, pSDp->host->host_no, 
                pSDp->channel,pSDp->id,lun, pSDp->sdev_state)); 

	    MPP_LOCK(RdacInfo->Lock, LockLevel);
	    MPPCMN_SET_PATH_STATE(RdacInfo, Controller, Path, MPPCMN_FAILED_NEED_CHECK);
	    for(lunindex=0; lunindex<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lunindex)
	    {
	        if(RdacInfo->RdacLuns[lunindex].LunControllers[Controller]->LunPaths[Path].LunPathDevice)
	        {
		        MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo, Controller, Path, lunindex, MPPCMN_FAILED_NEED_CHECK);
	        }
	    }
	    controllerPath = &RdacInfo->ControllerInfo[Controller]->ControllerPath[Path];
	    controllerPath->PathFailedTime = 0;
	    MPP_UNLOCK(RdacInfo->Lock, LockLevel);
	    mppLnx_remove_proxyRequest_from_list(validateRequestp,MPPLNX_QUEUE_QUEUED_LIST);
      	    mppLnx_put_free_asyncRequest(are);
   	    mppLnx_put_free_proxyRequest(validateRequestp);	

                return (MPP_GOOD_STATUS);
        }
        else if(  pSDp->sdev_state != SDEV_RUNNING ) {
		MPP_ERRORLOGPRINT((MPP_ERR_PATH_FAILOVER,506,
	         "Device T%dL%dCtrl%dPath%d is not in running state. sdev_state %d.\n",RdacInfo->VirtualTargetId, pSDp->lun, Controller,Path,pSDp->sdev_state));

		controllerPath = &RdacInfo->ControllerInfo[Controller]->ControllerPath[Path];

		MPP_LOCK(RdacInfo->Lock, LockLevel);
		MPPCMN_SET_PATH_STATE(RdacInfo, Controller, Path, MPPCMN_FAILED);
		for(lunindex=0; lunindex<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lunindex)
		{
			if(RdacInfo->RdacLuns[lunindex].LunControllers[Controller]->LunPaths[Path].LunPathDevice)
			{
				MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo, Controller, Path, lunindex, MPPCMN_FAILED);
			}
		}
		controllerPath->PathFailedTime = 0;
		MPP_UNLOCK(RdacInfo->Lock, LockLevel);

		mppLnx_remove_proxyRequest_from_list(validateRequestp,MPPLNX_QUEUE_QUEUED_LIST);
      		mppLnx_put_free_asyncRequest(are);
   		mppLnx_put_free_proxyRequest(validateRequestp);
		return (MPP_GOOD_STATUS);
	}

   /* Add a dummy serial number for debug purpose */
   validateRequestp->vcmnd->serial_number = 0x777;


   if (mppLnx_scsi_execute_async(validateRequestp->pdev,validateRequestp->vcmnd->cmnd,COMMAND_SIZE(validateRequestp->vcmnd->cmnd[0]),
   			validateRequestp->vcmnd->sc_data_direction, validateRequestp->vcmnd->request_buffer,
   			validateRequestp->vcmnd->request_bufflen, validateRequestp->vcmnd->use_sg,mppLnx_proxy_cmnd_timeout,
   			1, validateRequestp, mppLnx_validatepath_done,GFP_ATOMIC))
   {

        MPP_ERRORLOGPRINT((MPP_ERR_FATAL,511,
              "mppLnx_validatepath: mppLnx_scsi_execute_async failed.\n" ));

        /*
         * most likely out of mem, bio allocation or request allocation failed, retry again
         */
        mppLnx_validatepath_done(validateRequestp,validateRequestp->sense_buffer,DID_ERROR<<16,validateRequestp->vcmnd->request_bufflen); 
        return (MPP_GOOD_STATUS);
   }

   	generic_unplug_device(pSDp->request_queue);

   	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      		"Exitng mppLnx_validatePath() for T%dL%dCtrl%dPath%d\n",
      		RdacInfo->VirtualTargetId,
      		pSDp->lun,
      		Controller,Path));

	return (MPP_GOOD_STATUS);
}


/****************************************************************************
* PROCEDURE
* NAME:    mppLnx_validatepath_done
* SUMMARY: Completion handler for  Path Validation ( Inquiry or TUR ) command.
* SCOPE:   public
* DESCRIPTION:
*	Since this is called by scsi mid level in interrupt context queue the command
*	and return immediately. The processing of the completion of the command
*	can be done in a seperate thread.
*
*
* RESTRICTIONS:
* RETURNS: 	VOID
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/
void mppLnx_validatepath_done(void *data, char *sense, int result, int resid)
{
	mppLnx_ProxyRequestEntry_t  		*preq;
   	BYTE							Controller;
   	BYTE							Path;
   	LWORD							Lun;
	RdacDeviceInformation_t			*RdacInfo;

	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
		"Entering mppLnx_validatepath_done()\n"));

	preq			= (mppLnx_ProxyRequestEntry_t*)data;
  	Controller		= preq->controller;
   	Path			= preq->path;
   	Lun				= preq->lun;
	RdacInfo		= preq->rdacInfo;

       MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
			"mppLnx_pathvalidation_done():Validation Command Completed for %s:C %d:P %d: L %d \n",
			RdacInfo->ModuleName, Controller, Path, Lun));


	/*
	  * Since midlevel will call this completion handler in interrupt context we need to
	  * queue the command and complete it later.
	  */
	   if( unlikely(!preq))
	   {
	      /* Proxy request pointer is null */
	      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,484,
	         "mppLnx_pathvalidation_done():mppLnx_ProxyRequestEntry_t is NULL in the completed path validation scsi_cmnd.\n"));
	      /*
	      *
	      */
	      MPPLNX_VHBA_BUG();
	      return;
	   }
	   else
	   {
		preq->dispatchType = MPPLNX_VALIDATE_PATH_TYPE;
		preq->result		  = result;
		preq->resid		  = resid;

		mppLnx_remove_proxyRequest_from_list(preq,MPPLNX_QUEUE_QUEUED_LIST);
		mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DISPATCHED_LIST);

		if(atomic_read(&mppLnxDpcQueueContext.needSchedule))
		{
			up(mppLnxDpcQueueContext.dpc_sem);
		}
	   }

	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
		"Exiting mppLnx_validatepath_done()\n"));

}

/****************************************************************************
* PROCEDURE
* NAME:    __mppLnx_validatepath_done
* SUMMARY: performs the bottom half of pathvalidate_done function
* SCOPE:   public
* DESCRIPTION:
*		Failback handler is the thread that processes the completions
*		of path validation commands. The thread takes one completion
*		and calls this routine to perform the main task of completion.
* 		Path validation completion handler in non interrupt context .
*
*
* RESTRICTIONS:
* RETURNS: 	VOID
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/
void __mppLnx_validatepath_done( mppLnx_ProxyRequestEntry_t    *prequest)
{

   RdacDeviceInformation_t     *RdacInfo;
   BYTE                         Controller;
   BYTE                         Path;
   LWORD                        Lun;
   long                         StartTime;
   MPP_TIME                     currentTime;
   SenseData_t                 *SenseData;
   int                          SenseLength;
   long                         DataLengthXfered;
   long                         mpp_status = 0;
   long                         mppAction;
   RdacControllerPath_t        *controllerPath;
   LWORD                        lun,pathindex;
   LINT                         LockLevel;
   BOOL                         pathFailed = TRUE;
   BOOL                         flushDeferredQueue=FALSE;
   WORD                         host,target;
   LWORD                        pathStateIndex;
   MPPCMN_PATH_STATE            oldPathState;
   MPPCMN_PATH_STATE            pathstate;
   LWORD                        stateindex;
   BYTE                         anotherPathAvailable=FALSE;
   InquiryData_t               *inqdata;
   BYTE                         failcontroller=TRUE;
   BYTE                         wakeup_pathvalidate_thread=FALSE;
   mppLnx_PDeviceEntry_t       *pde;
   mppLnx_VDeviceEntry_t       *vde;

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering __mppLnx_validatepath_done()\n"));

   if(unlikely(prequest == NULL))
   {
      MPPLNX_VHBA_BUG();
   }

   RdacInfo			= prequest->rdacInfo;
   Controller		= prequest->controller;
   Path				= prequest->path;
   Lun				= prequest ->lun;
   SenseData		= (SenseData_t *)prequest->sense_buffer;
   SenseLength		= sizeof(prequest->sense_buffer);
   controllerPath	= &RdacInfo->ControllerInfo[Controller]->ControllerPath[Path];

	// Physical Host
	host 			= prequest->phost->host_no;

	// physical Target
	target 			= prequest->pdev->id;


   if ( prequest->req )
   {
         blk_put_request(prequest->req);
         prequest->req = NULL;
   }

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
					"__mppLnx_pathvalidation_done():Validation Command Completed for %s:%d:%d, Old path State %d \n",
					RdacInfo->ModuleName, Controller, Path,
					controllerPath->PathState[controllerPath->PathStateIndex]));

   if(prequest->sense_buffer[0] != 0)
   {
      MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
         "__mppLnx_validatepath_done():Sense Data Dump: Response Code 0x%02x Sense-Key/ASC/ASCQ 0x%02x/0x%02x/0x%02x \n",
         SenseData->Error_Code,
         SenseData->Sense_Key,
         SenseData->ASC,
         SenseData->ASCQ));
   }

   StartTime 		= prequest->startTime;
   DataLengthXfered 	= prequest->vcmnd->request_bufflen - prequest->resid;
   mpp_status 		= mppLnx_Translate_Linux_Status(prequest->result);

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "__mppLnx_validatepath_done() mppStatus %d\n",mpp_status));


	if( mpp_status != MPP_GOOD_STATUS )
	{
	   mppAction= mpp_AnalyseSyncError(mpp_status, SenseData, SenseLength, DataLengthXfered, StartTime,
						&prequest->SelectionTimeoutRetryCount,
						&prequest->CommandTimeoutRetryCount,
						&prequest->UaRetryCount);
	   switch(mppAction)
	  {
		case MPP_RETRY_DEC_COUNT:
			   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
			      "__mppLnx_validatepath_done() MPP_RETRY_DEC_COUNT\n"));
			      /* INQUIRY COMMAND SHOULD ALWAYS SUCCEED SHOULD NOT COME HERE */
				pathFailed = FALSE;  /* command is completed but with check condition */
			      	break;

		case MPP_RETRY_NO_DEC:
			   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
			      "__mppLnx_validatepath_done() MPP_RETRY_NO_DEC\n"));
			      /* INQUIRY COMMAND SHOULD ALWAYS SUCCEED SHOULD NOT COME HERE */
				pathFailed = FALSE;
				break;

		case MPP_NO_ERROR:
			   pathFailed = FALSE;

			   inqdata = (InquiryData_t *) prequest->vcmnd->request_buffer;
			   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
			      "__mppLnx_validatepath_done() MPP_NO_ERROR  Peripheral Qualifier=%d DeviceType =%d\n",
			      	inqdata->Periph_Qualifier,inqdata->Periph_Device_Type));
				break;

		case MPP_RETURN_ERROR:
		default:
			   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
			      "__mppLnx_validatepath_done() MPP_RETURN_ERROR\n"));
				// Don't retry these
				pathFailed = TRUE;
				break;
	   }
	}
	else
	{
			pathFailed = FALSE;

			 inqdata = (InquiryData_t *) prequest->vcmnd->request_buffer;
			 MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
			     "__mppLnx_validatepath_done() MPP_GOOD_STATUS  Peripheral Qualifier=%d DeviceType =%d\n",
			    	inqdata->Periph_Qualifier,inqdata->Periph_Device_Type));
	}


	if(pathFailed)
	{
		// The request needs to be re-attempted.  Since path validation occurs every second reset the PathState of this
		// path and let the timer run the inquiry request again.
		// Re-verify the controller is still present
		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1, "PathValidation: %s:%d:%d Failed\n",
			RdacInfo->ModuleName, Controller, Path));

		MPP_GETTIME(currentTime);
                MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 517, "%s:%d:%d - path validation failed. mpp_status %d mppAction %d scsi_status 0x%08x AGE=%ld\n",
                	RdacInfo->ModuleName, Controller,Path, mpp_status, mppAction,prequest->result,currentTime - prequest->vcmndStartTime));

		if(RdacInfo->ControllerInfo[Controller]->ControllerPresent && controllerPath->Present)
		{
			MPP_LOCK(RdacInfo->Lock, LockLevel);
			controllerPath->PathIdleTime	= 0;
			controllerPath->PathFailedTime	= 0;

         switch(controllerPath->PathState[controllerPath->PathStateIndex])
         {
            case MPPCMN_FAILED_CHECKING:
               MPPCMN_SET_PATH_STATE(RdacInfo, Controller, Path, MPPCMN_FAILED);

               for(lun=0; lun<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lun)
               {
                  if(RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice)
                  {
                     MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo, Controller, Path, lun, MPPCMN_FAILED);
                  }
               }

               MPP_UNLOCK(RdacInfo->Lock, LockLevel);

               /* Re dispatch the deferred commands */
               mppLnx_flushdeferredqueue(host, target);
               break;
            case MPPCMN_OPTIMAL_CHECKING:
               MPP_UNLOCK(RdacInfo->Lock, LockLevel);
               /* Check if path failover is possible if not mark controller as failed */
               anotherPathAvailable = mpp_FailPath(RdacInfo, Controller, Path);

               if ( anotherPathAvailable == TRUE )
               {
                  /* One more path still available to controller */
                  MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
                     "__mppLnx_validatepath_done() : One path set to FAILED. Controller %d Path %d."
                     "Another  path is available to controller.\n",
                     Controller, Path));
                 /* check the state of all other paths to the controller
                  * if all of them are either FAILED, FAILED_NEEDCHECK or FAILED_CHECKING
                  * then we can fail the controller .
                  */
                  for(pathindex=0;pathindex<mppLnx_Config_Parameters.mpp_MaxPathsPerController;pathindex++)
                  {
                     if(RdacInfo->ControllerInfo[Controller]->ControllerPath[pathindex].Present
                        && RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[pathindex].LunPathDevice)
                     {

                        stateindex = RdacInfo->ControllerInfo[Controller]->ControllerPath[pathindex].PathStateIndex;
                        pathstate  = RdacInfo->ControllerInfo[Controller]->ControllerPath[pathindex].PathState[stateindex];

                        if( (pathstate == MPPCMN_OPTIMAL) ||
                           (pathstate == MPPCMN_OPTIMAL_NEED_CHECK) ||
                           (pathstate == MPPCMN_OPTIMAL_CHECKING) )
                        {
                           failcontroller = FALSE;
                           break;
                        }
                     }

                  }

                  if( failcontroller == TRUE )
                  {

                     /* mark this controller as failed */
                     MPP_LOCK(RdacInfo->Lock, LockLevel);
                     RdacInfo->ControllerInfo[Controller]->Failed = TRUE;
                     MPP_UNLOCK(RdacInfo->Lock, LockLevel);
                  }
               } /*if ( anotherPathAvailable == TRUE )*/
               else
               {
                  /* mark this controller as failed */
                  MPP_LOCK(RdacInfo->Lock, LockLevel);
                  RdacInfo->ControllerInfo[Controller]->Failed = TRUE;
                  MPP_UNLOCK(RdacInfo->Lock, LockLevel);

                  MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
                     "__mppLnx_validatepath_done() : All paths  to Controller %d are FAILED. Marking controller as FAILED.\n",
                  Controller));
               }
               break;
            default:
               /*
               * all other states. The validation command expects the current path state is in
               * either MPPCMN_FAILED_CHECKING or MPPCMN_OPTIMAL_CHECKING. But the normal IO
               * could already move the path state to MPPCMN_FAILED
               */
               MPP_UNLOCK(RdacInfo->Lock, LockLevel);
               break;

         } /* end of switch */
      } /* end of if Present*/
	} /* if (pathfailed) */
	else
	{
		// Path is valid.  Do whatever processing is necessary to make the path available for I/O routing.
		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1, "PathValidation: %s:%d:%d Successful\n",
			RdacInfo->ModuleName, Controller, Path));
		MPP_LOCK(RdacInfo->Lock, LockLevel);

		//unfail the controller if the controller is in failed state earlier
		if(RdacInfo->ControllerInfo[Controller]->Failed == TRUE)
		{
			RdacInfo->ControllerInfo[Controller]->Failed = FALSE;
			MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_2, "Controller %s:%d Unfailed\n",
				RdacInfo->ModuleName, Controller));
			MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 486,
            "%s:%d:%d - Unfailed\n",
				RdacInfo->ModuleName, Controller,Path));
		}

		pathStateIndex 	= RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathStateIndex;
		oldPathState 	= RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathState[pathStateIndex];

		MPPCMN_SET_PATH_STATE(RdacInfo, Controller, Path, MPPCMN_OPTIMAL);
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED,504, "%s:%d:%d Path Unfailed\n", RdacInfo->ModuleName,
								Controller, Path));
		controllerPath->PathIdleTime = 0;
		controllerPath->PathFailedTime = 0;
		MPP_UNLOCK(RdacInfo->Lock, LockLevel);



		for(lun = 0; lun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lun)
		{
			if(!RdacInfo->RdacLuns[lun].Present
			|| !RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice
			||   RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].PathRemoveState)
			{
				continue;
			}

                        if( RdacInfo->RdacLuns[lun].UTMLun == TRUE )
                        {
                                MPP_LOCK(RdacInfo->Lock, LockLevel);
                                MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo,Controller,Path, lun, MPPCMN_OPTIMAL);
                                MPP_UNLOCK(RdacInfo->Lock, LockLevel);
                                continue;
                        }

                        vde =  RdacInfo->RdacLuns[lun].PlatformPrivateData;
                        pde = RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].PlatformPrivateData;
                        /* This is CR 132025 */
                   
                        if(vde == NULL)
                           continue;

                        if(pde == NULL)
                           continue;

                        if(RdacInfo == NULL)
                           continue; 
                        
                        /* Perform the PR Registration check only if the DEVICE was previously in MPPCMN_FAILED_CHECKING */
                        if( ((RdacInfo->RdacLuns[lun].NeedsReservationCheck) && (oldPathState == MPPCMN_FAILED_CHECKING)) || 
                            ((!pde->PRKeyRegistered) && (vde->PRKeyValid)))
			{
				wakeup_pathvalidate_thread = TRUE;
			}
			else
			{
					MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
						"DeviceValidation: %s:%d:%d:%d Successful, Check PR Registration not required.\n",
						RdacInfo->ModuleName, Controller, Path, lun));
					MPP_LOCK(RdacInfo->Lock, LockLevel);
					MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo,Controller,Path, lun, MPPCMN_OPTIMAL);
					MPP_UNLOCK(RdacInfo->Lock, LockLevel);

					/* We can restart the queued requests as the virtual device is optimal */
					flushDeferredQueue = TRUE;

			}
		} /* for */

		if( wakeup_pathvalidate_thread == TRUE )
		{
		   /* add proxy request to path validate thread */
		   mppLnx_insert_proxyRequest_to_list(prequest,MPPLNX_QUEUE_PATHVALIDATION_LIST);


		   /*we only wake up the path validate thread if it is sleep
		    */
		   if(atomic_read(&mppLnxPathValidateContext.scan_active) == 0)
		   {
		      MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
			 "__mppLnx_pathvalidate_done -- wake up pathvalidate thread\n"));
		      up(mppLnxPathValidateContext.pathvalidate_sem);
			}
			else
			{
		      MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
			 "__mppLnx_pathvalidate_done -- path validate scan is in progress\n"));
		   }

		}

			if( flushDeferredQueue == TRUE )
			{

				MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3, "State-0x%x\n",
					MPPCMN_OPTIMAL));

				// Flush the deferred commands
				mppLnx_flushdeferredqueue(host, target);

			}

	} /* pathfailed == false */

   if(unlikely(RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path].PathRemoveState == 1))
   {
      mppLnx_TestAndPutDevice(prequest->pdev,&RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path]);
   }
   /*
   * clean up
   */
   mppLnx_put_free_asyncRequest((mppLnx_AsyncRequestEntry_t *)
                  (((char *) prequest->vcmnd) - sizeof(mppLnx_AsyncRequestEntry_t)));



    /*
   * remove the request from queued queue
   * and put it back to the free queue
   * do this only if we do not need to check the
   * PR Registrations in the path validate thread.
   */
   if ( wakeup_pathvalidate_thread == FALSE )
   	mppLnx_put_free_proxyRequest(prequest);

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "exiting __mppLnx_validatepath_done() \n"));
}

/*
  * 	flush the deferred commads .
  *   Walk through the deferred command list and resubmit if any path becomes OPTIMAL.
  *
  *   FIX - Currently resubmitting all commands, we can probably check only the commands
  *   			for specific target whose path became OPTIMAL.
  */
void
mppLnx_flushdeferredqueue( WORD Host , BYTE Target )
{
	mppLnx_ProxyRequestEntry_t	*preq=NULL;
	unsigned long                 	iflages;
	struct list_head		*listPt,*temp;
	BYTE				controller,path;
	LWORD				lun;
	MPP_HANDLE			device;
	RdacDeviceInformation_t		*rdacInfo=NULL;
    	LWORD				devicestateindex;
    	MPPCMN_PATH_STATE		devicestate;


	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
							"EnteringmppLnx_flushdeferredqueue() \n"));

      /*
        * lock the queue and pick up one cmnd entry at a time. Remove the cmnd entry
        * from the list and unlock. In this way, we don't block any new entry being
        * added to the list.
        */

	list_for_each_safe( listPt, temp, &(mppLnx_deferredProxyRequestQ.list))
	{
		spin_lock_irqsave(&mppLnx_deferredProxyRequestQ.queueLock, iflages);
		preq = list_entry(listPt, mppLnx_ProxyRequestEntry_t, pending_list);
		spin_unlock_irqrestore(&mppLnx_deferredProxyRequestQ.queueLock, iflages);

		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
							"mppLnx_flushdefferedqueue() :  C %d P %d H %d T %d. \n",
							preq->controller,preq->path, Host, Target));

		/* flush only the commands from same host and  target */
		controller	= preq->controller;
		path 		= preq->path;
		lun		= preq->vcmnd->device->lun;
		device		= preq->rdacInfo->RdacLuns[lun].LunControllers[controller]->LunPaths[path].LunPathDevice;
                rdacInfo        = preq->rdacInfo;

		if((device->host->host_no == Host ) && ( device->id == Target ) )
		{
  
			devicestateindex  = rdacInfo->RdacLuns[lun].LunControllers[controller]->LunPaths[path].DeviceStateIndex;
			devicestate       = rdacInfo->RdacLuns[lun].LunControllers[controller]->LunPaths[path].DeviceState[devicestateindex];
  
			MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
				"mppLnx_flushdeferredqueue: IO retried on new controller, %s:%d:%d:%d:%ld devicestate %d\n",
				rdacInfo->ModuleName, controller, path, lun, preq->vcmnd->serial_number,devicestate));

			/* CR 142232 */
			if( devicestate == MPPCMN_FAILED_CHECKING )
			{
				continue;
			}

			/* Remove the command from the deferred list */
			mppLnx_remove_proxyRequest_from_list(preq, MPPLNX_QUEUE_DEFERRED_LIST);

			preq->controller	= UNKNOWN_PREVIOUS_CONTROLLER;
			preq->path 		= UNKNOWN_PREVIOUS_PATH;

                       /*
                        * If the proxy command is routed to a different controller,
                        * reset other controller based fields
                        */
                        // IO being resent on a new controller, reset counters and timers.

                       MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_2,
                        "mppLnx_flushdeferredqueue: IO retried on new controller, %s:%d:%d:%d:%ld\n",
                          rdacInfo->ModuleName, controller, path, lun, preq->vcmnd->serial_number));


                        preq->UaRetryCount = 0;
                        preq->CommandTimeoutRetryCount = 0;
                        preq->SelectionTimeoutRetryCount = 0;
                        rdacInfo->RdacLuns[lun].NotReady = FALSE;
                        rdacInfo->RdacLuns[lun].Busy = FALSE;
                        rdacInfo->RdacLuns[lun].Quiescent = FALSE;
                        rdacInfo->RdacLuns[lun].QuiescienceStart = 0;
                        rdacInfo->RdacLuns[lun].NotReadyStart = 0;
                        rdacInfo->RdacLuns[lun].BusyStart = 0;
                        MPP_GETTIME(preq->ControllerIoWaitTime);

		        /* resubmit the deferred command */
			mppLnx_do_queuecommand( preq );
		}
		else
		{
			MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
							"mppLnx_flushdefferedqueue() :  H %d T %d preq->Host %d preq->Target %d. \n",
							Host, Target,device->host->host_no,device->id));

			continue;
		}

	}

	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
			"Exiting mppLnx_flushdeferredqueue() \n"));

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_IssueFailover
* SUMMARY:  Issues an async Mode Select page-2c command to a selected physical device
* SCOPE:    public
*
* DESCRIPTION:;
   This function is an implementation of the system dependent function mppSys_IssueFailover(),
   which is specified by the MPP driver command API as a system dependent function.
   It issues a Model Select Page2c command to the alternative controller.
   This function performs the following operations:
      Asks the Linux Scsi middle level to allocate a struct scsi_cmnd object from
         the "FailovorHandle" struct scsi_device object
      Assigns the input data and CDB to the struct scsi_cmnd
      Assigns a special command completion function to this command. The special
         "(*done)" function will be mppLnx_failoverCmnd_done().
      Issues the command to the physical HBA driver
      The completion of this command will come to mppLnx_failoverCmnd_done() function.
*  @rdacInfo         the RdacDeviceInformation_t object which represents a storage array
*  @controller       Controller slot
*  @Path             The path index
*  @FailoverHandler  The physical struct scsi_device object to which the failover command will send to
*  @Page2C           The buffer which contains the Mode Select Page 2C
                     (Redundant Controller Page)
*  @cdb              The Command Descriptor Block of the Mode Select command
*  @ForceQuiescence  Whether force is needed
*  @noHoldInReset Whether to override global mpp_HoldAltInReset flag and leave alt. alive.
* RESTRICTIONS:
*
* RETURNS: Nothing
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_IssueFailover(RdacDeviceInformation_t *rdacInfo, BYTE controller,
		BYTE Path, MPP_HANDLE FailoverHandle, BYTE *Page2C, BYTE *cdb,
		MPPCMN_FAILOVER_METHOD FailoverMethod,
		BOOL ForceQuiescence, BOOL noHoldInReset)
{
   struct scsi_device            * pSDp;
   struct scsi_cmnd              * fakeVscp;
   mppLnx_ProxyRequestEntry_t    * fakeRequestp;
   LINT                          LockLevel = 0;
   mppLnx_AsyncRequestEntry_t    *are;
   RdacControllerPath_t		 *controllerPath;
   LWORD                         lunindex;
   LWORD                         lun;


   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_IssueFailover()\n"));

   /*
   *pSDp is the physical device where the failover command issused to
   */
   pSDp = FailoverHandle;

   if(unlikely(!pSDp))
   {
      MPPLNX_VHBA_BUG();
   }
   lun = pSDp->lun;

   if(unlikely(!rdacInfo))
   {
      MPPLNX_VHBA_BUG();
   }

   if(unlikely(!cdb))
   {
      MPPLNX_VHBA_BUG();
   }

   if(unlikely(!Page2C))
   {
      MPPLNX_VHBA_BUG();
   }

   /* Check to see if the device selected is offline , if it is we need to select a device that is online */
   while ((pSDp->sdev_state == SDEV_OFFLINE))
   {
       MPP_ERRORLOGPRINT((MPP_ERR_PATH_FAILOVER,507,
	      "%s:%d:%d:%d - pdev H%d:C%d:T%d:L%d is offline, pick up alternate path to send failover command.\n",
         rdacInfo->ModuleName,controller,
         Path,lun, pSDp->host->host_no,
         pSDp->channel,pSDp->id,lun));

       controllerPath = &rdacInfo->ControllerInfo[controller]->ControllerPath[Path];

       MPP_LOCK(rdacInfo->Lock, LockLevel);
       MPPCMN_SET_PATH_STATE(rdacInfo, controller, Path, MPPCMN_FAILED);
       for(lunindex=0; lunindex<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lunindex)
       {
           if(rdacInfo->RdacLuns[lunindex].LunControllers[controller]->LunPaths[Path].LunPathDevice)
           {
               MPPCMN_SET_DEVICE_PATH_STATE(rdacInfo, controller, Path, lunindex, MPPCMN_FAILED);
           }
       }
       controllerPath->PathFailedTime = 0;
       MPP_UNLOCK(rdacInfo->Lock, LockLevel);


       /* Select the alternate paths to same device if available */
       pSDp = mpp_SelectFailoverPath(
                     rdacInfo,
                     lun,
                     &controller,
                     &Path);

       if( pSDp == NULL )
       {

              MPP_ERRORLOGPRINT((MPP_ERR_FATAL,508,
                 "%s:%d:%d:%d No path to issue failover command.\n",
                 rdacInfo->ModuleName,
                 controller,
                 Path,
                 lun));

              /* Turn off the failover in progress flag */
              MPP_LOCK(rdacInfo->Lock, LockLevel);
              rdacInfo->ControllerInfo[controller]->FailoverInProgress = FALSE;
              MPP_UNLOCK(rdacInfo->Lock, LockLevel);
              return ;
       }
   }

   fakeRequestp = mppLnx_get_free_proxyRequest();

   if(unlikely(!fakeRequestp))
   {
      MPPLNX_VHBA_BUG();
   }

   /*
   * this is a manufacture IO command by the mpp driver. It does not have a
   * virtual command from the upper level driver associated with it.
   * But the commmand asynchly, we need to keep tracking it by using a proxy
   * request object
   */
   are = mppLnx_get_free_asyncRequest();
   if(unlikely(!are))
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 487,
         "%s:%d:%d:%d Error in allacating memory for fail-over command\n",
         (char *)rdacInfo->ModuleName,
          controller,
          Path,
          lun));
      /*
      * we didn't issue a command to the controller. What do I do.
      * the only thing I can do is to mark the failover-in-progress to false
      */
      MPP_LOCK(rdacInfo->Lock, LockLevel);
      rdacInfo->ControllerInfo[controller]->FailoverInProgress = FALSE;
	   MPP_UNLOCK(rdacInfo->Lock, LockLevel);

      return;
   }

   fakeVscp = are->cmnd;
   fakeVscp->request = NULL;
   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_IssueFailover() S4\n"));


   /*
   * set up the fake virtual command's related fields
   */
   memset(fakeVscp, 0, sizeof(struct scsi_cmnd));
   memcpy(fakeVscp->cmnd,cdb,10);
   fakeVscp->cmd_len = 10;
   fakeVscp->sc_data_direction = DMA_TO_DEVICE;
   fakeVscp->underflow = 0;
   fakeVscp->use_sg = 0;
   fakeVscp->sglist_len = 0;
   fakeVscp->request_buffer = Page2C;
   fakeVscp->request_bufflen = rdacInfo->Page2CSubPage1 ? sizeof(ModePage2CSubPage_t) : sizeof(ModePage2C_t);

   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_IssueFailover() S6\n"));
   /*
   * set up the request's related fields
   */
   fakeRequestp->vcmd_timeoutfunc            = middle_level_timeoutfunc;
   fakeRequestp->vcmnd                       = fakeVscp;
   fakeRequestp->controller                  = controller;
   fakeRequestp->path                        = Path;
   fakeRequestp->rdacInfo                    = rdacInfo;
   fakeRequestp->CommandTimeoutRetryCount    = 0;
   fakeRequestp->RetryCount                  = mppLnx_Config_Parameters.mpp_SyncRetryCount;
   fakeRequestp->UaRetryCount                = 0;
   fakeRequestp->SelectionTimeoutRetryCount  = 0;
   fakeRequestp->dispatchType                = MPPLNX_FAILOVER_CMND_TYPE;
   fakeRequestp->state                       = MPPLNX_REQUEST_DISPATCHED;
   fakeRequestp->FailoverMethod              = FailoverMethod;
   fakeRequestp->ForcedQuiescence            = ForceQuiescence;
   fakeRequestp->noHoldInReset               = noHoldInReset;


   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_IssueFailover() S7\n"));

   MPP_GETTIME(fakeRequestp->startTime);
   MPP_GETTIME(fakeRequestp->vcmndStartTime);
   MPP_GETTIME(fakeRequestp->ControllerIoWaitTime);
   MPP_GETTIME(fakeRequestp->ArrayIoWaitTime);

   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_IssueFailover() S8\n"));
   /**
   * the command completion function mppLnx_failoverCmd_done() should remember
   * to free the virtual command memory
   */


   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_IssueFailover() S10\n"));

   mppLnx_initProxyRequest(fakeRequestp,pSDp);
   /*
   * send the failover command to the SCSI middle level
   */
   mppLnx_insert_proxyRequest_to_list(fakeRequestp,MPPLNX_QUEUE_QUEUED_LIST);
   fakeRequestp->vtid = rdacInfo->VirtualTargetId;
   mppLnx_scsicmnd_virtual_to_proxy(fakeRequestp);


   if (mppLnx_scsi_execute_async(fakeRequestp->pdev,cdb,COMMAND_SIZE(cdb[0]),
   			fakeRequestp->vcmnd->sc_data_direction, Page2C,
   			fakeRequestp->vcmnd->request_bufflen, fakeRequestp->vcmnd->use_sg,
   			mppLnx_Config_Parameters.mpp_FailoverTimeout*HZ,
   			1, fakeRequestp, mppLnx_failoverCmd_done,GFP_ATOMIC))
   {

       MPP_ERRORLOGPRINT((MPP_ERR_FATAL,512,
              "mppLnx_IssueFailover: mppLnx_scsi_execute_async failed.\n" ));

       /*
        * most likely out of mem, bio allocation or request allocation failed, retry again
        */
        mppLnx_failoverCmd_done(fakeRequestp,fakeRequestp->sense_buffer,DID_ERROR<<16,fakeRequestp->vcmnd->request_bufflen); 
	return;
   }

   /*
   *
   */

   generic_unplug_device(pSDp->request_queue);

   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exitng mppLnx_IssueFailover() for T%dL%dCtrl%d\n",
      rdacInfo->VirtualTargetId,
      pSDp->lun,
      controller));
}




/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_failoverCmd_done
* SUMMARY:  Receives failover command's completion status
* SCOPE:    public
*
* DESCRIPTION:
   This is a special command completion function for a failover command.
   Unlike the normal command completion function, because the failover command
   is not requested by a host applications, this function will not cause a virtual I
   /O to be completed.

   This function will call mppCmn_AnalyseFailoverError() to determine whether or
   not the failover command needs to be retried.
   If the command is successful, it will make RdacControllerInformation_t object's
   FaiolverInProgress to false.

*  @scp The failover command which is completed by a physical HBA driver
* RESTRICTIONS:
*
* RETURNS: Nothing
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_failoverCmd_done(void *data, char *sense, int result, int resid)
{
	mppLnx_ProxyRequestEntry_t  	*preq;
   	BYTE							Controller;
   	BYTE							Path;
   	LWORD							Lun;
	RdacDeviceInformation_t			*RdacInfo;

	MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_3,
		"Entering mppLnx_failoverCmd_done()\n"));

	preq			= (mppLnx_ProxyRequestEntry_t*)data;
  	Controller		= preq->controller;
   	Path			= preq->path;
   	Lun				= preq->lun;
	RdacInfo		= preq->rdacInfo;

       MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_1,
			"mppLnx_failoverCmd_done(): Failover Command Completed for %s:C %d:P %d: L %d \n",
			RdacInfo->ModuleName, Controller, Path, Lun));


	/*
	  * Since midlevel will call this completion handler in interrupt context we need to
	  * queue the command and complete it later.
	  */
	   if( unlikely(!preq))
	   {
	      /* Proxy request pointer is null */
	      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,514,
	         "mppLnx_failoverCmd_done():mppLnx_ProxyRequestEntry_t is NULL in the completed failover command.\n"));
	      /*
	      *
	      */
	      MPPLNX_VHBA_BUG();
	      return;
	   }
	   else
	   {
		preq->dispatchType = MPPLNX_FAILOVER_CMND_TYPE;
		preq->result	   = result;
		preq->resid		   = resid;

		mppLnx_remove_proxyRequest_from_list(preq,MPPLNX_QUEUE_QUEUED_LIST);
		mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DISPATCHED_LIST);

		if(atomic_read(&mppLnxDpcQueueContext.needSchedule))
		{
			up(mppLnxDpcQueueContext.dpc_sem);
		}
	   }

	MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_3,
		"Exiting mppLnx_failoverCmd_done()\n"));

}

void __mppLnx_failoverCmd_done ( mppLnx_ProxyRequestEntry_t    *proxyRequest)
{
   long                          mpp_status = 0;
   MppCmn_FailoverCompletionContext_t failoverCompletionContext;
   LWORD                         failoverStatus;

   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering __mppLnx_failoverCmd_done()\n"));


   if(proxyRequest == NULL)
   {
      MPPLNX_VHBA_BUG();
   }

   failoverCompletionContext.FailoverContext.RdacInfo          = proxyRequest->rdacInfo;
   failoverCompletionContext.FailoverContext.ControllerIndex   = proxyRequest->controller;
   failoverCompletionContext.FailoverContext.PathIndex         = proxyRequest->path;
   failoverCompletionContext.FailoverContext.Lun               = proxyRequest->lun;
   failoverCompletionContext.FailoverContext.FailoverMethod    = proxyRequest->FailoverMethod ;
   failoverCompletionContext.ForcedQuiescenceAttempted         = proxyRequest->ForcedQuiescence;
   failoverCompletionContext.noHoldInReset                     = proxyRequest->noHoldInReset;
   failoverCompletionContext.SenseData                         = (SenseData_t *)proxyRequest->sense_buffer;
   failoverCompletionContext.SenseLength                       = sizeof(proxyRequest->sense_buffer);

   if ( proxyRequest->req )
   {
         blk_put_request(proxyRequest->req);
         proxyRequest->req = NULL;
   }


   if(proxyRequest->sense_buffer[0] != 0)
   {
      MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_3,
         "Sense Data Dump: Response Code 0x%02x Sense-Key/ASC/ASCQ 0x%02x/0x%02x/0x%02x \n",
         failoverCompletionContext.SenseData->Error_Code,
         failoverCompletionContext.SenseData->Sense_Key,
         failoverCompletionContext.SenseData->ASC,
         failoverCompletionContext.SenseData->ASCQ));
   }
   failoverCompletionContext.FailoverStartTime = proxyRequest->startTime;
   failoverCompletionContext.DataLengthTransferred = proxyRequest->vcmnd->request_bufflen - proxyRequest->resid;
   mpp_status = mppLnx_Translate_Linux_Status(proxyRequest->result);

   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_3,
      "__mppLnx_failoverCmd_done() mpp_status %d\n",mpp_status));

   failoverStatus = mppCmn_DoFailoverCompletion(mpp_status, &failoverCompletionContext);





   /*
   * clean up
   * 1. release the proxy command
   * 2. free proxy request's memory
   */
   mppLnx_put_free_asyncRequest((mppLnx_AsyncRequestEntry_t *)
                  (((char *) proxyRequest->vcmnd) - sizeof(mppLnx_AsyncRequestEntry_t)));



   /*
   * remove the request from queued queue
   * and put it back to the free queue
   */
   mppLnx_put_free_proxyRequest(proxyRequest);

   MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting __mppLnx_failoverCmd_done()\n"));
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_scsi_done
* SUMMARY:  The struct scsi_cmnd completion function of a proxy struct scsi_cmnd.
* SCOPE:    public
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function is the proxy command completion function. When a proxy command
   This function is the proxy command completion function. When a proxy command
   is completed by the HBA driver and further processed by the Scsi middle level,
   this function is called by the Scsi middle level to notify the proxy
   command's completion. This function is assigned to each proxy command's "(*done)" function.
   When USE_THREAD_FOR_RETRY is defined and sets to 1, this function will allocate a mppLnx_CmndEntry_t
   object and insert the completed proxy command to the command queue for completion or retry by mpp_DCR
   kernel thread.
*  @pscp The proxy struct scsi_cmnd that is completed by a physical HBA driver
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_scsi_done(void *data, char *sense, int result, int resid)
{

   mppLnx_ProxyRequestEntry_t  *preq;

   /*
   * get the proxy request from physical command's scsi_request object
   */
   preq = (mppLnx_ProxyRequestEntry_t *) data;

   /*
   * the pSCp is the completed command of the proxy scsi_request object
   */
   if( unlikely(!preq))
   {
      /* error.  */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,503,
         "cannot get proxy request\n"));
      /*
      *
      */
      MPPLNX_VHBA_BUG();
      return;
   }
   else
   {

      preq->dispatchType = MPPLNX_COMPLETED_CMND_TYPE;
      preq->result		  = result;
      preq->resid		  = resid;

      mppLnx_remove_proxyRequest_from_list(preq,MPPLNX_QUEUE_QUEUED_LIST);
      mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DISPATCHED_LIST);

      if(atomic_read(&mppLnxDpcQueueContext.needSchedule))
      {
         up(mppLnxDpcQueueContext.dpc_sem);
      }
   }

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_scsi_done\n"));
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     void __mppLnx_scsi_done( mppLnx_ProxyRequestEntry_t    *prequest)
* SUMMARY:  The struct scsi_cmnd completion function of a proxy struct scsi_cmnd.
* SCOPE:    public
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function is the proxy command completion function. When a proxy command
   is completed by the HBA driver and further processed by the Scsi middle level,
   this function is called by the Scsi middle level to notify the proxy
   command's completion. This function is assigned to each proxy command's "(*done)" function.
   This function is a core function of the Linux MPP virtual HBA driver.

*  @pscp The proxy struct scsi_cmnd that is completed by a physical HBA driver
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void __mppLnx_scsi_done( mppLnx_ProxyRequestEntry_t    *prequest)
{
   long                          mppAction;
   unsigned long                 roundTrip;
   RdacDeviceInformation_t       *RdacInfo;
   BYTE                          Controller;
   BYTE                          Path;
   LWORD                         Lun;
   LWORD                         mpp_status;
   LWORD                         ioErrorCondition=0;
   SenseData_t                   *SenseData;
   SINT                          SenseLength;
   LWORD                         DataLengthXfered;
   MPP_TIME                      StartTime;
   struct request                *req;
   struct scsi_cmnd              *vSCp;
   struct scsi_device            *pSDp = NULL;
   int                           ioErrorLevel;
   int                           isCompleteIo = 0;
   MPP_TIME                      currentTime;
   LINT                          LockLevel;
   int                           has_SenseData = 0;
   BOOL                          writeAndVerify, retryWithWrite;
   mppLnx_PDeviceEntry_t         * pde = NULL;
   mppLnx_VDeviceEntry_t         * vde = NULL;
   struct scatterlist            * sgel = NULL;
   unsigned char                 * inqBuffer = NULL;
   RdacControllerPath_t  	 *controllerPath;


   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering  __mppLnx_scsi_done()\n"));

   pSDp = prequest->pdev;


   if(unlikely(!prequest))
   {
      MPPLNX_VHBA_BUG();
   }

   vSCp = prequest->vcmnd;

   if(unlikely(NULL == vSCp))
   {
      MPPLNX_VHBA_BUG();
   }

   if(unlikely(NULL == pSDp))
   {
      MPPLNX_VHBA_BUG();
   }
   if(unlikely(NULL == vSCp->device))
   {
      MPPLNX_VHBA_BUG();
   }

   RdacInfo = mpp_ModuleArray[prequest->vtid].RdacInfo;

   if ( prequest->req )
   {
          blk_put_request(prequest->req);
          prequest->req = NULL;
   }

   if(unlikely(NULL == RdacInfo))
   {  //need new number
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
         "IO comes back for a already removed array v-tid %d psn %lx\n",prequest->vtid,
         vSCp->serial_number));
      //I don't know if the virtual device's memory still valid.

      mppLnx_do_queuecommand(
                 prequest);

      return;
   }

   Controller = prequest->controller;
   Path = prequest->path;
   Lun = prequest->lun;

    /* Get the Controller Path */
    controllerPath = &RdacInfo->ControllerInfo[Controller]->ControllerPath[Path];

   SenseData = (SenseData_t *)prequest->sense_buffer;

   if(unlikely(prequest->isVcmdCompleted == TRUE))
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
         "SD isVcnd-T %ld\n",prequest->vcmnd->serial_number));
      mppLnx_TestAndPutDevice(prequest->pdev, &RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path]);

      /* Decrement the count - NOT Sure */
      MPPSYS_ATOMIC_DEC(controllerPath->RequestsInProgress);

      mppLnx_put_free_proxyRequest(prequest);

      return;
   }

   if(unlikely(RdacInfo->RdacLuns[Lun].
            LunControllers[Controller]->LunPaths[Path].PathRemoveState == 1 ||
               pSDp->sdev_state == SDEV_DEL ||
               pSDp->sdev_state == SDEV_CANCEL))
   {

      if(prequest->result)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "BAD BAD IO comes back for a being removed device %d:%d:%d psn%lx\n",
                  Controller,Path,Lun,vSCp->serial_number));

         req = prequest->vcmnd->request;
         /*
          * The physical device may be ready to release. If the tag is not freed, the blk
          * will complain. For the normal case, we will do end_tag at the proxy request free time.
         */
         if (unlikely(blk_rq_tagged(req)))
         {
             unsigned long flags;
             struct request_queue *q = req->q;

             spin_lock_irqsave(q->queue_lock, flags);
             blk_queue_end_tag(q, req);
             spin_unlock_irqrestore(q->queue_lock, flags);
         }

         mppLnx_TestAndPutDevice(prequest->pdev, &RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path]);

         /* Decrement the count - NOT Sure */
         MPPSYS_ATOMIC_DEC(controllerPath->RequestsInProgress);

         mppLnx_do_queuecommand(
                 prequest);


         return;
      }else
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "Good IO come back for a being removed device %d:%d:%d psn%ld\n",
                  Controller,Path,Lun,vSCp->serial_number));
      }
   }


   SenseLength = sizeof(prequest->sense_buffer);
   /* Lessee what we've got */
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
       "__mppLnx_scsi_done: bufflen %d resid %d\n",
       prequest->vcmnd->request_bufflen, prequest->resid));
   DataLengthXfered = prequest->vcmnd->request_bufflen - prequest->resid;
   StartTime = prequest->startTime;


    /*
   * get the virtual device and physical device entry for updating its io statistics
   */
   pde = (mppLnx_PDeviceEntry_t *) RdacInfo->RdacLuns[Lun].
      LunControllers[Controller]->LunPaths[Path].PlatformPrivateData;

   if(unlikely(!pde))
   {
      /*
      * this is a new path being scaned after the virtual HBA driver is initialized.
      * Create is physical device's proc file
      */
      mppLnx_checkPdevEntry(RdacInfo);
      pde = (mppLnx_PDeviceEntry_t *) RdacInfo->RdacLuns[Lun].
            LunControllers[Controller]->LunPaths[Path].PlatformPrivateData;
   }

   vde = (mppLnx_VDeviceEntry_t *) RdacInfo->RdacLuns[Lun].PlatformPrivateData;


   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "__mppLnx_scsi_done() status %x v SN %ld on ctr%d:path%d:Lun%d\n",
      prequest->result,prequest->vcmnd->serial_number,
      Controller,Path,Lun));

   /*
   * get the current time. It is used later
   */
   MPP_GETTIME(currentTime);
   mpp_status = mppLnx_Translate_Linux_Status(prequest->result);

   /*
    * cr 134218 - lsi mptfc driver checks for underflow flag in scsi cmnd
    * DID_OK is returned even if all the data is not transfered.
    * set the flag equivalent to DID_SOFT_ERROR and retry the command.
    */
   if(unlikely(prequest->resid && prequest->vcmnd->underflow && !host_byte(prequest->result) && !status_byte(prequest->result) ))
   {
       mpp_status = MPP_HARDWARE_ERROR;
   }

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "%s-%d:%d:%d mppStatus %s\n",(char *)RdacInfo->ModuleName,
         Controller,Path,Lun,mppStatusStrings[mpp_status]));

#ifdef MPP_DEBUG
   if(vSCp->cmnd[0] == SCSI_INQUIRY &&
         (vSCp->cmnd[1] & 0x01) == 0 )
   {
      printk("the Inquiry cmnd for virtual device %d:%d come back from physical device %d:%d:%d:%d\n",
         vSCp->device->id,vSCp->device->lun,
         prequest->host_no,
         prequest->channel,
         prequest->id,
         prequest->lun);


      printk("physical command status 0x%04x\n",
         prequest->result);
   }
#endif


   /*
   * RAID5/RAID1 protection.
   */
   has_SenseData = SenseData->Sense_Key != 0;
   mpp_ChkIOForWandV(prequest->vcmnd->cmnd, has_SenseData,
            has_SenseData ? SenseData->Sense_Key : 0,
            has_SenseData ? SenseData->ASC : 0,
            has_SenseData ? SenseData->ASCQ : 0,
            &writeAndVerify, &retryWithWrite);

   if(mpp_status)
   {
      mppAction = mpp_AnalyseIoError(
         RdacInfo,
         Controller,
         Path,
         Lun,
         mpp_status,
         SenseData,
         SenseLength,
         DataLengthXfered,
         prequest->startTime,
         &prequest->SelectionTimeoutRetryCount,
         &prequest->CommandTimeoutRetryCount,
         &prequest->UaRetryCount,
         &prequest->RetryCount,
         prequest->ControllerIoWaitTime,
         prequest->ArrayIoWaitTime,
         &ioErrorCondition);

      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "%s-%d:%d:%d mppAction %s\n",(char *)RdacInfo->ModuleName,
         Controller,Path,Lun, mppActionStrings[mppAction]));

      switch( mppAction)
      {
         case MPP_NO_ERROR:
            /* no error. Complete the IO*/
            isCompleteIo = 1;
            roundTrip = currentTime - prequest->startTime;
            mppLnx_update_PdeviceStat(pde, MPPLNX_IO_SUCCESS, roundTrip);
            roundTrip = currentTime - prequest->vcmndStartTime;
            mppLnx_update_VdeviceStat(vde, MPPLNX_IO_SUCCESS, roundTrip);

            MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
               "__mppLnx_scsi_done() no error case\n"));
            MPP_LOCK(RdacInfo->Lock, LockLevel);
            RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathIdleTime = 0;
            RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathFailedTime = 0;
            MPP_UNLOCK(RdacInfo->Lock, LockLevel);

            break;
         case MPP_RETURN_ERROR:

            if(unlikely(retryWithWrite))
            {
               /*
               * RAID5/RAID1 protection
               * With this condition, retry the command if retryWithWrite is true. If the
               *    retryWithWrite is true, it indicates some special check-condition happends
               *    on the controller side. We need change the command back to "write"
               * 1. change the CDB back to write and then retry. The data_cmnd field has
               *     the original CDB
               * 2. reset the errorRecovery to 0
               * 3. retry the command
               */
               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "__mppLnx_scsi_done Error returned on W&V  %s-%d:%d:%d mppAction %s cdb=0x%x ErrorRecovery=%d b_WandV=%d retrying with write\n",
                  (char *)RdacInfo->ModuleName,
                  Controller,Path,Lun, mppActionStrings[mppAction],
                  prequest->vcmnd->cmnd[0],prequest->errorRecovery));


               prequest->errorRecovery = 0;
/*DVS CHECK THIS               memcpy(prequest->vcmnd->cmnd,prequest->vcmnd->data_cmnd,sizeof(prequest->vcmnd->cmnd));*/
               mppLnx_do_queuecommand(prequest);
               break;
            }

            /*
            * complete the IO with error. The middle level will receive an error
            * of the proxy command's
            * Shall we change the return status to DID_NO_CONNECT (selection timeout)?
            */
            MPP_ERRORLOGPRINT((MPP_ERR_FATAL,492,
               "%s:%d:%d:%d IO FAILURE. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
               RdacInfo->ModuleName,
               Controller,
               Path,
               Lun,
               prequest->vcmnd->serial_number,
               prequest->host_no,
               prequest->channel,
               prequest->id,
               prequest->lun,
               SenseData->Sense_Key,
               SenseData->ASC,
               SenseData->ASCQ,
               prequest->result,
               mpp_status));

            mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );

            MPP_LOCK(RdacInfo->Lock, LockLevel);
            RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathIdleTime = 0;
            RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathFailedTime = 0;
            MPP_UNLOCK(RdacInfo->Lock, LockLevel);


            isCompleteIo = 1;

            isCompleteIo = 1;
            roundTrip = currentTime - prequest->startTime;
            mppLnx_update_PdeviceStat(pde, MPPLNX_IO_FAILED, roundTrip);
            roundTrip = currentTime - prequest->vcmndStartTime;
            mppLnx_update_VdeviceStat(vde, MPPLNX_IO_FAILED, roundTrip);
            break;
         case MPP_RETRY_DEC_COUNT:
         case MPP_RETRY_NO_DEC:
            /* we need to perform a retry for this condition
            * MPP_RETRY_DEC_COUNT will perform a re-try and decrease the retry counter
            * MPP_RETRY_NO_DEC will perform a retry without changing the retry counter
            */
            /* don't complete the IO */
            isCompleteIo = 0;
            /*
            * don't release the physical command. We are going to re-send the
            * proxy command. As long as we don't release the proxy command, we
            * are still holding the command slot.
            */
            ioErrorLevel = MPPLNX_IO_FAILED;
            /* retry the command on the same path*/
            /*1. add the request back to list
             *2. copy data from virtual command to proxy command again
             *3. reset the proxy request's startTime
             *3. resend the command
             */

            MPP_GETTIME(prequest->startTime);

            roundTrip = currentTime - prequest->startTime;
            mppLnx_update_PdeviceStat(pde, MPPLNX_IO_FAILED, roundTrip);


            MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,493,
               "%s:%d:%d:%d Cmnd failed-retry the same path. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
               RdacInfo->ModuleName,
               Controller,
               Path,
               Lun,
               prequest->vcmnd->serial_number,
               prequest->host_no,
               prequest->channel,
               prequest->id,
               prequest->lun,
               SenseData->Sense_Key,
               SenseData->ASC,
               SenseData->ASCQ,
               prequest->result,
               mpp_status));

            mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );

            /*
            * RAID5/RAID1 protection
            * the mpp_SetWandVBit will set the errorRecovery to true if the CDB is writeAndVerify
            * and the command is not retryWithWrite
            */
            if(mppAction == MPP_RETRY_NO_DEC)
            {
               mpp_SetWandVBit((BOOL *)&prequest->errorRecovery, writeAndVerify, retryWithWrite);
            }
            MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=RT CS=%08x AGE=%ld\n",
                vSCp->serial_number,
                vSCp->cmnd[0],
                vSCp->device->host->host_no,
                vSCp->device->channel,
                vSCp->device->id,
                vSCp->device->lun,
                prequest->host_no,
                prequest->channel,
                prequest->id,
                prequest->lun,
                prequest->result,
                currentTime - prequest->vcmndStartTime));
            /*
            *04/14/2005 CR88221
            * If busy, not-ready or quiescence wait time exceeded, the first io request that
            * is trigged the time exceeded will start to go to the alternative controller
            * after a controller failover. Meanwhile, busy flag or the BusyStart timer
            * are cleared. But the already dispatched io requests that are rejected with
            * the same status will come here because analysis io error returns MPP_RETRY_NO_DEC.
            * In this case, we must check whether or not the current owning controller and
            * the proxy request's controller is the same. If not, we would like to re-select
            * the path.
            */
            if(prequest->controller == RdacInfo->RdacLuns[Lun].CurrentOwningPath)
            {
               mppLnx_RetrySamePath(prequest);
            }
            else
            {
               mppLnx_do_queuecommand(prequest);
            }

            MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
               "__mppLnx_scsi_done() retry on the \n"));
            break;

         case MPP_RETRY_ALTERNATE_CONTROLLER:
            /*
             * retry on alternative controller. Will select a new path and re-issue the
             * virtual IO
             * 1.don't complete the virtual IO
             * 2.release the physical command
             * 3.free proxy request's memory
             * 4.update physical device entry's statistics
             * 5.mark the ownership chanle of the volume
             * 6.issue the command to a new path
             */
              isCompleteIo = 0;

              roundTrip = currentTime - prequest->startTime;
              mppLnx_update_PdeviceStat(pde, MPPLNX_IO_FAILED, roundTrip);

              /*
              * Mark the change of the volume ownership so that the mpp_SelectPath
              * will pick up a path from the other controller
              */
              MPP_LOCK(RdacInfo->Lock, LockLevel);
              RdacInfo->RdacLuns[Lun].CurrentOwningPath = Controller^1;
              MPP_UNLOCK(RdacInfo->Lock, LockLevel);


              MPP_ERRORLOGPRINT((MPP_ERR_FATAL,494,
               "%s:%d:%d:%d Cmnd-failed try alt ctrl %d. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
               RdacInfo->ModuleName,
               Controller,
               Path,
               Lun,
               Controller^1,
               prequest->vcmnd->serial_number,
               prequest->host_no,
               prequest->channel,
               prequest->id,
               prequest->lun,
               SenseData->Sense_Key,
               SenseData->ASC,
               SenseData->ASCQ,
               prequest->result,
               mpp_status));

              mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );

              /*
              *try on a new controller. We need to reset the ControllerIoWaitTime
              */
              prequest->ControllerIoWaitTime = currentTime;
              /*
              * RAID5/RAID1 protection
              * the mpp_SetWandVBit will set the errorRecovery to true if the CDB is writeAndVerify
              * and the command is not retryWithWrite
              */
              mpp_SetWandVBit((BOOL *)&prequest->errorRecovery, writeAndVerify, retryWithWrite);
              /*
              * the command will be retried on different controller
              * the request currently now in any queues. The mpp_do_queuecommand()
              * will put the request back to the queued queue
              */
              MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                 "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=RT CS=%08x AGE=%ld\n",
                 vSCp->serial_number,
                 vSCp->cmnd[0],
                 vSCp->device->host->host_no,
                 vSCp->device->channel,
                 vSCp->device->id,
                 vSCp->device->lun,
                 prequest->host_no,
                 prequest->channel,
                 prequest->id,
                 prequest->lun,
                 prequest->result,
                 currentTime - prequest->vcmndStartTime));

              mppLnx_do_queuecommand(
                 prequest);

            //mppLnx_free_ProxyRequest(prequest);
              MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                 "__mppLnx_scsi_done() dispatched to alternative controller\n"));
              break;


         case MPP_FAILOVER:
            /*
            * fail the path
            */
            if( mpp_FailPath(RdacInfo, Controller,Path) )
            {
               /*
               *mpp_Failpath returns a value greater than 0 to indicate
               *that other paths to the controller are still OK
               *if returns 0, this case will fall to the controller fail-over
               *case.
               */
               /*
               * 1. don't complete the virtual IO
               * 2.release the physical command
               * 3.free proxy request's memory
               * 4.update physical device entry's statistics
               * 5.issue the command to a new path
               */
               isCompleteIo = 0;

               roundTrip = currentTime - prequest->startTime;
               mppLnx_update_PdeviceStat(pde, MPPLNX_IO_PATHFAILOVER, roundTrip);
               mppLnx_update_VdeviceStat(vde, MPPLNX_IO_PATHFAILOVER, roundTrip);

               MPP_ERRORLOGPRINT((MPP_ERR_FATAL,495,
                  "%s:%d:%d:%d Cmnd failed-retry on a new path. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
                  RdacInfo->ModuleName,
                  Controller,
                  Path,
                  Lun,
                  prequest->vcmnd->serial_number,
                  prequest->host_no,
                  prequest->channel,
                  prequest->id,
                  prequest->lun,
                  SenseData->Sense_Key,
                  SenseData->ASC,
                  SenseData->ASCQ,
                  prequest->result,
                  mpp_status));

               mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );

               /*
               * RAID5/RAID1 protection
               * the mpp_SetWandVBit will set the errorRecovery to true if the CDB is writeAndVerify
               * and the command is not retryWithWrite
               */
               mpp_SetWandVBit((BOOL *)&prequest->errorRecovery, writeAndVerify, retryWithWrite);

               MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                 "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=RT CS=%08x AGE=%ld\n",
                 vSCp->serial_number,
                 vSCp->cmnd[0],
                 vSCp->device->host->host_no,
                 vSCp->device->channel,
                 vSCp->device->id,
                 vSCp->device->lun,
                 prequest->host_no,
                 prequest->channel,
                 prequest->id,
                 prequest->lun,
                 prequest->result,
                 currentTime - prequest->vcmndStartTime));

               mppLnx_do_queuecommand(
                 prequest);

               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "__mppLnx_scsi_done() sent IO to the new path\n"));
               break;
            }
            else /*intentionally fall to the next case */
            {
                MPP_ERRORLOGPRINT((MPP_ERR_FATAL,496,
                  "%s:%d:%d:%d No new path: fall to failover controller case. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
                  RdacInfo->ModuleName,
                  Controller,
                  Path,
                  Lun,
                  prequest->vcmnd->serial_number,
                  prequest->host_no,
                  prequest->channel,
                  prequest->id,
                  prequest->lun,
                  SenseData->Sense_Key,
                  SenseData->ASC,
                  SenseData->ASCQ,
                  prequest->result,
                  mpp_status));

                mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );
                // and fall thru to the next case.
            }

         case MPP_FAILOVER_LUN:
            if (mppAction == MPP_FAILOVER_LUN ||
                RdacInfo->FailoverMethod == MPPCMN_FAILOVER_METHOD_LUN) // got here on drop-thru
            {
              if(RdacInfo->ControllerInfo[Controller^1]->ControllerPresent)
              {
                if (!RdacInfo->ControllerInfo[Controller^1]->Failed)
                {
                  if (mppCmn_Failover(RdacInfo, Controller, Lun, ioErrorCondition))
                  {
                    isCompleteIo = 0;

                    roundTrip = currentTime - prequest->startTime;
                    mppLnx_update_PdeviceStat(pde, MPPLNX_IO_PATHFAILOVER, roundTrip);

                    prequest->ControllerIoWaitTime = currentTime;

                    MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                    "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=RT CS=%08x AGE=%ld\n",
                    vSCp->serial_number,
                    vSCp->cmnd[0],
                    vSCp->device->host->host_no,
                    vSCp->device->channel,
                    vSCp->device->id,
                    vSCp->device->lun,
                    prequest->host_no,
                    prequest->channel,
                    prequest->id,
                    prequest->lun,
                    prequest->result,
                    currentTime - prequest->vcmndStartTime));

                    mppLnx_do_queuecommand(
                        prequest);

                    MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4,
                        "__mppLnx_scsi_done() cmd sent to the other controller after LUN failover\n"));
                  }
                  else
                  {
                    MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                        "__mppLnx_scsi_done() LUN failover failed\n"));
                  }
                }
                else
                {
                   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4, "__mppLnx_scsi_done LUN alt ctlr failed\n"));
                }
              }
              else
              {
                MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4, "__mppLnx_scsi_done LUN alt ctlr not present\n"));
              }

                break;
            }
            // else controller-level failure is selected; fall thru.
         case MPP_FAILOVER_CONTROLLER:
            if(RdacInfo->ControllerInfo[Controller^1]->ControllerPresent)
            {
             if(!RdacInfo->ControllerInfo[Controller^1]->Failed)
             {
              if(mppCmn_Failover(RdacInfo, Controller, Lun, ioErrorCondition))
              {
               /* the mppCmn_Failover() function will issue failover command
               */
               /*
               * 1. don't complete the virtual IO
               * 2.release the physical command
               * 3.free proxy request's memory
               * 4.update physical device entry's statistics
               * 5.issue the command to a new path
               */
               isCompleteIo = 0;

               roundTrip = currentTime - prequest->startTime;
               mppLnx_update_PdeviceStat(pde, MPPLNX_IO_PATHFAILOVER, roundTrip);
               mppLnx_update_VdeviceStat(vde, MPPLNX_IO_CTRLFAILOVER, roundTrip);


               MPP_ERRORLOGPRINT((MPP_ERR_FATAL,497,
                  "%s:%d:%d:%d Failed controller to %d. retry. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
                  RdacInfo->ModuleName,
                  Controller,
                  Path,
                  Lun,
                  Controller^1,
                  prequest->vcmnd->serial_number,
                  prequest->host_no,
                  prequest->channel,
                  prequest->id,
                  prequest->lun,
                  SenseData->Sense_Key,
                  SenseData->ASC,
                  SenseData->ASCQ,
                  prequest->result,
                  mpp_status));

                mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );


               /*
               * Mark the other controller as the owner of this lun
               */
               MPP_LOCK(RdacInfo->Lock, LockLevel);
               RdacInfo->RdacLuns[Lun].CurrentOwningPath = Controller^1;
               MPP_UNLOCK(RdacInfo->Lock, LockLevel);
               /*
               *try on a new controller. We need to reset the ControllerIoWaitTime
               */
               prequest->ControllerIoWaitTime = currentTime;
               /*
               * RAID5/RAID1 protection
               * the mpp_SetWandVBit will set the errorRecovery to true if the CDB is writeAndVerify
               * and the command is not retryWithWrite
               */
               mpp_SetWandVBit((BOOL *)&prequest->errorRecovery, writeAndVerify, retryWithWrite);

               MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                 "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=RT CS=%08x AGE=%ld\n",
                 vSCp->serial_number,
                 vSCp->cmnd[0],
                 vSCp->device->host->host_no,
                 vSCp->device->channel,
                 vSCp->device->id,
                 vSCp->device->lun,
                 prequest->host_no,
                 prequest->channel,
                 prequest->id,
                 prequest->lun,
                 prequest->result,
                 currentTime - prequest->vcmndStartTime));

               mppLnx_do_queuecommand(
                 prequest);

               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "__mppLnx_scsi_done() cmd sent to the other controller after ctl failover\n"));
              }
              else
              {
               /*
               * can not failover controller. We have to fail the virtual IO
               */
               /*
               * 1. complete the IO (failed)
               * 2. update statistics (both physical device entry and virtual
               *    device entry
               * 3. release the physical command
               * 4. free proxy request memory
               */
               isCompleteIo = 1;

               roundTrip = currentTime - prequest->startTime;
               mppLnx_update_PdeviceStat(pde, MPPLNX_IO_FAILED, roundTrip);
               roundTrip = currentTime - prequest->vcmndStartTime;
               mppLnx_update_VdeviceStat(vde, MPPLNX_IO_FAILED, roundTrip);


               MPP_ERRORLOGPRINT((MPP_ERR_FATAL,498,
                  "%s:%d:%d:%d Can not failctrl. IO failed. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
                  RdacInfo->ModuleName,
                  Controller,
                  Path,
                  Lun,
                  prequest->vcmnd->serial_number,
                  prequest->host_no,
                  prequest->channel,
                  prequest->id,
                  prequest->lun,
                  SenseData->Sense_Key,
                  SenseData->ASC,
                  SenseData->ASCQ,
                  prequest->result,
                  mpp_status));

                mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );


              }
             }
             else
             {
              MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4,
                 "__mppLnx_scsi_done FAILOVER_CONTROLLER alt ctlr failed\n"));
             }
            }
            else
            {
             MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4,
                 "__mppLnx_scsi_done FAILOVER_CONTROLLER alt ctlr missing\n"));
            }
            break;

         case MPP_FAILOVER_TO_CURRENT:
            /*
            * The command we sent came back with sense data that indicate the
            * controller does not own the LUN. But we think this controller should
            * own the LUN. We need to tell the other controller to failover to this
            * controller
            * The alternative controller must be marked as un-failed. Let the
            * mppCmn_Failover() change its state
            */
            MPP_LOCK(RdacInfo->Lock, LockLevel);
                RdacInfo->ControllerInfo[Controller^1]->Failed = FALSE;
                MPP_UNLOCK(RdacInfo->Lock, LockLevel);

            if(mppCmn_Failover(RdacInfo, Controller^1, Lun, ioErrorCondition))
            {

               /* the mppCmn_Failover() function will issue failover command
               */
               /*
               * 1. don't complete the virtual IO
               * 2.release the physical command
               * 3.free proxy request's memory
               * 4.update physical device entry's statistics
               * 5.issue the command to a new path
               */
               isCompleteIo = 0;


               //roundTrip = currentTime - prequest->startTime;
               //mppLnx_update_PdeviceStat(pde, MPPLNX_IO_FAILED, roundTrip);

               MPP_ERRORLOGPRINT((MPP_ERR_FATAL,500,
                  "%s:%d:%d:%d Fail to current ctrl %d.. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
                  RdacInfo->ModuleName,
                  Controller,
                  Path,
                  Lun,
                  Controller,
                  prequest->vcmnd->serial_number,
                  prequest->host_no,
                  prequest->channel,
                  prequest->id,
                  prequest->lun,
                  SenseData->Sense_Key,
                  SenseData->ASC,
                  SenseData->ASCQ,
                  prequest->result,
                  mpp_status));

                mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );

               /*
               * RAID5/RAID1 protection
               * the mpp_SetWandVBit will set the errorRecovery to true if the CDB is writeAndVerify
               * and the command is not retryWithWrite
               */
               mpp_SetWandVBit((BOOL *)&prequest->errorRecovery, writeAndVerify, retryWithWrite);

               MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                 "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=RT CS=%08x AGE=%ld\n",
                 vSCp->serial_number,
                 vSCp->cmnd[0],
                 vSCp->device->host->host_no,
                 vSCp->device->channel,
                 vSCp->device->id,
                 vSCp->device->lun,
                 prequest->host_no,
                 prequest->channel,
                 prequest->id,
                 prequest->lun,
                 prequest->result,
                 currentTime - prequest->vcmndStartTime));

               mppLnx_do_queuecommand(
                 prequest);

               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "__mppLnx_scsi_done() failover-to-current. sent the command\n"));
            }
            else
            {
               /*
               * can not failover controller. We have to fail the virtual IO
               * Why do we get here. We shouldn't get here at all.
               */
               /*
               * 1. complete the IO (failed)
               * 2. update statistics (both physical device entry and virtual
               *    device entry
               * 3. release the physical command
               * 4. free proxy request memory
               */
               isCompleteIo = 1;

               roundTrip = currentTime - prequest->startTime;
               mppLnx_update_PdeviceStat(pde, MPPLNX_IO_FAILED, roundTrip);
               roundTrip = currentTime - prequest->vcmndStartTime;
               mppLnx_update_VdeviceStat(vde, MPPLNX_IO_FAILED, roundTrip);

               MPP_ERRORLOGPRINT((MPP_ERR_FATAL,501,
                  "%s:%d:%d:%d Cannot fail to current ctrl %d.. vcmnd SN %ld pdev H%d:C%d:T%d:L%d 0x%02x/0x%02x/0x%02x 0x%08x mpp_status:%d\n",
                  RdacInfo->ModuleName,
                  Controller,
                  Path,
                  Lun,
                  Controller,
                  prequest->vcmnd->serial_number,
                  prequest->host_no,
                  prequest->channel,
                  prequest->id,
                  prequest->lun,
                  SenseData->Sense_Key,
                  SenseData->ASC,
                  SenseData->ASCQ,
                  prequest->result,
                  mpp_status));

               mppLnx_printsensebuffer ( RdacInfo, Controller, Path, prequest->sense_buffer, prequest->vcmnd->serial_number );

            }
      }/* switch */

   }/*if mpp_status */
   else
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "__mppLnx_scsi_done() no-error no-CC\n"));
      isCompleteIo = 1;

      /*
      *During controller SOD time, the read/write IO will return 02/04/01 (Not-Ready
      *check-condition) but the inquiry command will return good status. The same thing
      *is for 06/8b/20 condition. If an inquiry command is completed with good status,
      *we don't want to clear the timers.
      */
      MPP_LOCK(RdacInfo->Lock, LockLevel);
      if(vSCp->cmnd[0] != SCSI_INQUIRY)
      {
         RdacInfo->RdacLuns[Lun].QuiescienceStart =
            RdacInfo->RdacLuns[Lun].NotReadyStart =
            RdacInfo->RdacLuns[Lun].BusyStart      = 0;
         RdacInfo->RdacLuns[Lun].Busy = FALSE;
         RdacInfo->RdacLuns[Lun].NotReady = FALSE;
         RdacInfo->RdacLuns[Lun].Quiescent = FALSE;
      }

      RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathIdleTime	= 0;
      RdacInfo->ControllerInfo[Controller]->ControllerPath[Path].PathFailedTime	= 0;
      MPP_UNLOCK(RdacInfo->Lock, LockLevel);
      /*
      * free the memory of proxy request
      */
      roundTrip = currentTime - prequest->startTime;
      mppLnx_update_PdeviceStat(pde, MPPLNX_IO_SUCCESS, roundTrip);
      roundTrip = currentTime - prequest->vcmndStartTime;
      mppLnx_update_VdeviceStat(vde, MPPLNX_IO_SUCCESS, roundTrip);


   }

   MPPSYS_ATOMIC_DEC(controllerPath->RequestsInProgress);
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_4, "__mppLnx_scsi_done(%s) Lun:%d"
                " Completed the command on(Path %d controller %d) with pending %d requests\n",
                RdacInfo->ModuleName, Lun, Path, Controller, MPPSYS_ATOMIC_GET(controllerPath->RequestsInProgress)));

   if(isCompleteIo)
   {
      /*
      * copy the data back to the virtual command
      */
      mppLnx_scsicmnd_proxy_to_virtual(vSCp,prequest);
      /*
      * check the command to see whether or not it is a standard inquiry
      * if so, we need to change its vendor ID and product ID
      */
      if(vSCp->cmnd[0] == SCSI_INQUIRY &&
         (vSCp->cmnd[1] & 0x01) == 0 )
      {
         if(vSCp->use_sg)
         {
            /*
            * scatter-gather list is used for the inquiry command
            */
            sgel = (struct scatterlist *)vSCp->request_buffer;

            inqBuffer = page_address(sgel->page) + sgel->offset;

         }else
         {
            inqBuffer = vSCp->request_buffer;
         }

         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
               "__mppLnx_scsi_done- replace vendorId ProductId \n"));
         /*
         * Only replace the product ID with the value from the configuration file
         * The vendor ID will be the same as the array returned.
         */
         bcopy((caddr_t) mppLnx_Config_Parameters.mpp_ProductId, (caddr_t) (inqBuffer+ 16), 16);
         /*
         * For Egenio storage arrays, the scsi middle level will use REPORT LUN method
         * performs the device discovery. However, if the MPP driver returns "selection timeout"
         * for an inquiry command to the UTM virtual device, the SCSI middle level will
         * stop scanning the rest of LUNs since it considers the device is not responsible.
         * But if the inquiry data of the UTM virtual device's "Peripheral qualifier" bits
         * returns 011b, the SCSI middle level will not create a scsi device for this LUN
         * and still keeps scanning the rest of LUNs. Based on SPC-3, the value of 011b of
         * peripheral qualifier is describes as follows:
         *     The device server is not capable of supporting a physical device on
         *     this logical unit. For this peripheral qualifier the peripheral device
         *     type shall be set to 1Fh to provide compatibility with previous versions
         *     of SCSI. All other peripheral device type values are reserved for this peripheral
         *     qualifier.
         */
         if(RdacInfo->RdacLuns[Lun].UTMLun == TRUE &&
            vSCp->SCp.Status != MPPLNX_LUN0_REPLACED_MAGIC)
         {
           /*
            * see PR 92712. If the LUN 0 is a un-configured and the next smallest LUN is the
            * UTM LUN, we will use the UTM LUN to route the inquiry and report luns commands
            * to the virtual LUN 0. If the first byte of an inquiry data for the virtual LUN 0
            * is 0x3F, the middle level will consider the target device is SCSI-2 target and
            * only scan luns from 0 to 7.
            */
            ((unsigned char *)inqBuffer)[0] = ((unsigned char *)inqBuffer)[0]|0x7F;

         }
         if(unlikely(vSCp->SCp.Status == MPPLNX_LUN0_REPLACED_MAGIC))
         {
            ((unsigned char *)inqBuffer)[0] = ((unsigned char *)inqBuffer)[0]|0x7F;
         }
      }

      /*
      * For workaround the unconfigured LUN0 issue, during queue-command time,
      * we replaced the LUN 0's device->lun to a configured LUN for making the
      * inquiry and report lun through. Now, we must replace the lun value back
      * before the command completion.
      */
      if(unlikely(vSCp->SCp.Status == MPPLNX_LUN0_REPLACED_MAGIC))
      {
         //((unsigned char *)vSCp->buffer)[0] = ((unsigned char *)vSCp->buffer)[0]|0x7F;
         vSCp->device->lun = 0;
         vSCp->SCp.Status = 0;
      }
      mppLnx_scsi_add_timer(vSCp, vSCp->timeout_per_command, middle_level_timeoutfunc);
      /*
      * it is about to complete the command
      */
      if(unlikely(prequest->writeWithVerify))
      {
         /*
         * copy the original cdb back to the virtual command cdb
         */
/*DVS CHECK THIS          memcpy(vSCp->cmnd, vSCp->data_cmnd, sizeof(vSCp->cmnd)); */
      }

     /*
      * The IO request is ready to leave the mpp driver. Trace its physical device that is used
      * route the virtual IO request
      */
      MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
             "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d PD=H:%dC:%dT:%dL:%d LCP=LV CS=%08x AGE=%ld\n",
             vSCp->serial_number,
             vSCp->cmnd[0],
             vSCp->device->host->host_no,
             vSCp->device->channel,
             vSCp->device->id,
             vSCp->device->lun,
             prequest->host_no,
             prequest->channel,
             prequest->id,
             prequest->lun,
             prequest->result,
             currentTime - prequest->vcmndStartTime));

      mppLnx_put_free_proxyRequest(prequest);
      vSCp->scsi_done(vSCp);

   }

   if(unlikely(RdacInfo->RdacLuns[Lun].
            LunControllers[Controller]->LunPaths[Path].PathRemoveState == 1 ||
            pSDp->sdev_state == SDEV_DEL ||
            pSDp->sdev_state == SDEV_CANCEL))
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
         "TAP goodIO %d%d%d\n",prequest->host_no,
         prequest->id,prequest->lun));
            mppLnx_TestAndPutDevice(pSDp,&RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[Path]);
   }


   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting __mppLnx_scsi_done()\n"));
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_initProxyRequest
* SUMMARY:  Initialize the proxy request based on the scsi_device object
* SCOPE:    local
*
*
* DESCRIPTION:
 The virtual HBA driver pre-allocates a set of mppLnx_ProxyRequestEntry_t object
 for IO routing. The allocation of the mppLnx_ProxyRequestEntry_t also include
 the memory for scsi_request object and request object. This function is used to
 initialize the scsi_request object's scsi_device related fields.
*  @preq the mppLnx_ProxyRequestEntry_t object to be initialized
   @sdev the Scsi_device object where the proxy request will be routed to
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
#define SCSI_REQ_MAGIC		0x75F6D354
void mppLnx_initProxyRequest(mppLnx_ProxyRequestEntry_t *preq, struct scsi_device *sdev)
{

    preq->pdev   = sdev;
    preq->phost  = sdev->host;
    preq->lun   = sdev->lun;
    preq->host_no	= sdev->host->host_no;
    preq->channel	= sdev->channel;
    preq->id 		= sdev->id;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_scsicmnd_virtual_to_proxy
* SUMMARY:  Copy data from a virtual struct scsi_cmnd to a proxy struct scsi_cmnd
* SCOPE:    local
*
*
* DESCRIPTION:
   Before a proxy command is issued to a physical HBA driver, this function
   copies certain fields from the virtual struct scsi_cmnd object,  "*vcmnd", to
   the proxy struct scsi_cmnd object, "*pcmnd". Which fields in the virtual command
   need to be copied to the proxy command are specified in 349-1034610


*  @vscp the virtual struct scsi_cmnd
*  @pscp the Proxy struct scsi_cmnd
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static void mppLnx_scsicmnd_virtual_to_proxy(mppLnx_ProxyRequestEntry_t *preq)
{
   memset(preq->sense_buffer, 0 , SCSI_SENSE_BUFFERSIZE);
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_scsicmnd_proxy_to_virtual
* SUMMARY:  Copy data from a proxy struct scsi_cmnd to a virtual struct scsi_cmnd
* SCOPE:    local
*
* DESCRIPTION:
   After a proxy command is completed, this function copies certain fields from
   the proxy struct scsi_cmnd object,  "*pcmnd", to the virtual struct scsi_cmnd object, "*vcmnd".
   Which fields in the proxy command need to be copied to the virtual command are
   specified in 349-1034610.

*  @vscp the virtual struct scsi_cmnd (destination)
*  @pscp the proxt struct scsi_cmnd (source)
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static void mppLnx_scsicmnd_proxy_to_virtual(struct scsi_cmnd *vscp, mppLnx_ProxyRequestEntry_t    *preq)
{

   /*
   * set the retries to the max allowed. A virtual command does
   * not need to be retried
   */
   vscp->result = preq->result;
   vscp->retries = vscp->allowed;
   vscp->resid = preq->resid;
   memcpy(vscp->sense_buffer, preq->sense_buffer, SCSI_SENSE_BUFFERSIZE);

   return;
}

void mppLnx_TestAndGetDevice(struct scsi_device * sdev, LunPathInfo_t * lunPathInfo)
{

   mppLnx_ProxyRequestEntry_t    * preq;
   unsigned long                 flags;

   /*
   * check the dispatch queue. The command in the physical device's queue must be the
   * proxy requests that are in the queued queue.
   */
   spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_queuedProxyRequestQ.list, queued_list)
   {
      if(preq->pdev == sdev)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAG QP %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
          if(!lunPathInfo->holdKref)
         {
             get_device(&sdev->sdev_gendev);
             lunPathInfo->holdKref = TRUE;
         }
         spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAG QP %d%d%d return\n",sdev->host->host_no,sdev->id,sdev->lun));
         goto outsleep;
      }
   }
   spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);

   /*
   * check the path validation command queue. This is the queue that has path validation
   * commands come back and wait for __mppLnx_pathvalidate_done
   */
   spin_lock_irqsave (&mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_pathvalidationProxyRequestQ.list, pathvalidation_list)
   {
      if(preq->pdev == sdev)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAG PV %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
         if(!lunPathInfo->holdKref)
         {
             get_device(&sdev->sdev_gendev);
             lunPathInfo->holdKref = TRUE;
         }
         spin_unlock_irqrestore (&mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAG PV %d%d%d return\n",sdev->host->host_no,sdev->id,sdev->lun));
         goto outsleep;
      }
   }
   spin_unlock_irqrestore (&mppLnx_pathvalidationProxyRequestQ.queueLock, flags);

   /*
   * check the dispatch list
   */
   spin_lock_irqsave(&mppLnx_DPProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_DPProxyRequestQ.list, dispatch_list)
   {
      if(
         ((preq->dispatchType == MPPLNX_COMPLETED_CMND_TYPE) ||
            (preq->dispatchType == MPPLNX_VALIDATE_PATH_TYPE)) &&
            (preq->pdev == sdev)
    )
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAG DP %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
         if(!lunPathInfo->holdKref)
         {
             get_device(&sdev->sdev_gendev);
             lunPathInfo->holdKref = TRUE;
         }
         spin_unlock_irqrestore (&mppLnx_DPProxyRequestQ.queueLock, flags);
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAG DP %d%d%d return\n",sdev->host->host_no,sdev->id,sdev->lun));
         goto outsleep;
      }
   }
   spin_unlock_irqrestore (&mppLnx_DPProxyRequestQ.queueLock, flags);

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "TAG didn't call get_device %d%d%d return\n",sdev->host->host_no,sdev->id,sdev->lun));
   return;

outsleep:
   msleep(2*HZ);
   return;

}

void mppLnx_TestAndPutDevice(struct scsi_device * sdev, LunPathInfo_t * lunPathInfo)
{

   mppLnx_ProxyRequestEntry_t    * preq;
   unsigned long                 flags;

   /*
   * check the dispatch queue. The command in the physical device's queue must be the
   * proxy requests that are in the queued queue.
   */
   spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
   spin_lock(&mppLnx_pathvalidationProxyRequestQ.queueLock);
   spin_lock(&mppLnx_DPProxyRequestQ.queueLock);
   list_for_each_entry(preq, &mppLnx_queuedProxyRequestQ.list, queued_list)
   {
      if(preq->pdev == sdev)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAP QP %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
         goto out;
      }
   }
   /*
   * check the path validation command queue. This is the queue that has path validation
   * commands come back and wait for __mppLnx_pathvalidate_done
   */
   list_for_each_entry(preq, &mppLnx_pathvalidationProxyRequestQ.list, pathvalidation_list)
   {
      if(preq->pdev == sdev)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAP PV %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
         goto out;
      }
   }

   /*
   * check the dispatch list
   */
   list_for_each_entry(preq, &mppLnx_DPProxyRequestQ.list, dispatch_list)
   {
      if(
         ((preq->dispatchType == MPPLNX_COMPLETED_CMND_TYPE) ||
            (preq->dispatchType == MPPLNX_VALIDATE_PATH_TYPE)) &&
            (preq->pdev == sdev)
         )
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "TAP DP %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
         goto out;
      }
   }

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "TAP free %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
   spin_unlock(&mppLnx_DPProxyRequestQ.queueLock);
   spin_unlock(&mppLnx_pathvalidationProxyRequestQ.queueLock);
   spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);
   if(lunPathInfo->holdKref)
   {
       put_device(&sdev->sdev_gendev);
       lunPathInfo->holdKref=FALSE;
   }
   return;

out:
   spin_unlock(&mppLnx_DPProxyRequestQ.queueLock);
   spin_unlock(&mppLnx_pathvalidationProxyRequestQ.queueLock);
   spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
      "TAP return %d%d%d\n",sdev->host->host_no,sdev->id,sdev->lun));
   return;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_TakeCareOfWriteAndVerify
* SUMMARY:  Determine if Write And Verify needs to be done, and do it.
* SCOPE:    private
* take care of the raid1/raid5 write-and-verify
* if this is a new virtual IO, we don't have sense data
* if the logic of mpp_TestBuildWandVop returns ture, the virtual command will
* be modify to write-and-verify
*/
static void mppLnx_TakeCareOfWriteAndVerify(mppLnx_ProxyRequestEntry_t *preq, BYTE controller)
{
	BOOL   writeAndVerify, retryWithWrite;
	int    has_SenseData = (preq->controller != UNKNOWN_PREVIOUS_CONTROLLER);

	if (has_SenseData)
	{
		mpp_ChkIOForWandV(preq->vcmnd->cmnd, has_SenseData,
			((SenseData_t *)(preq->sense_buffer))->Sense_Key,
			((SenseData_t *)(preq->sense_buffer))->ASC,
			((SenseData_t *)(preq->sense_buffer))->ASCQ,
			&writeAndVerify, &retryWithWrite);
	}
	else
	{
		mpp_ChkIOForWandV(preq->vcmnd->cmnd, has_SenseData,
			0, 0, 0,
			&writeAndVerify, &retryWithWrite);
	}

	if(
		(preq->controller != UNKNOWN_PREVIOUS_CONTROLLER) &&
		(mpp_TestBuildWandVOp(preq->vcmnd->cmnd, preq->errorRecovery,
			writeAndVerify, retryWithWrite,
			preq->controller, controller))
  	  )
	{
		MPP_DEBUGPRINT((MPP_FAILOVER_IO_DEBUG+MPP_DEBUG_LEVEL_2,
			"mppLnx_do_queuecommand: Chg IO to WriteAndVerify, cdb 0x%x, ErrorRecovery=%d, writeAndVerify=%d\n",
			preq->vcmnd->cmnd, preq->errorRecovery, writeAndVerify));
		preq->writeWithVerify = 1;
		mpp_BuildWriteWithVerify(preq->vcmnd->cmnd); /* THIS IS NO LONGER ACCEPTABLE */
	}
}

static int mppLnx_scsi_bi_endio(struct bio *bio, unsigned int bytes_done, int error)
{

    if (bio->bi_size)
    {
    MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,518,
         "mppLnx_scsi_bi_endio:: residual data for bio %d bytes copied %d error %d\n",
          bio->bi_size, bytes_done, error));
        return 1;
    }

    bio_put(bio);
    return 0;
}

#ifdef MPPLNX_SCSI_REQ_MAP_SG
/**
 * scsi_req_map_sg - map a scatterlist into a request
 * @rq:     request to fill
 * @sg:     scatterlist
 * @nsegs:  number of elements
 * @bufflen:    len of buffer
 * @gfp:    memory allocation flags
 *
 * scsi_req_map_sg maps a scatterlist into a request so that the
 * request can be sent to the block layer. We do not trust the scatterlist
 * sent to use, as some ULDs use that struct to only organize the pages.
 */
static int mppLnx_scsi_req_map_sg(struct request *rq, struct scatterlist *sgl,
               int nsegs, gfp_t gfp)
{
    struct request_queue *q = rq->q;
    int nr_pages = 0;
    unsigned int data_len = 0, len, bytes, off;
    struct page *page;
    struct bio *bio = NULL;
    int i, err, nr_vecs = 0;

    for (i = 0; i < nsegs; i++) {
         nr_pages += (sgl[i].length + sgl[i].offset + PAGE_SIZE - 1) >> PAGE_SHIFT;
    }


    for (i = 0; i < nsegs; i++) {
        page = sgl[i].page;
        off = sgl[i].offset;
        len = sgl[i].length;
        data_len += len;

        while (len > 0) {
            bytes = min_t(unsigned int, len, PAGE_SIZE - off);

            if (!bio) {
                nr_vecs = min_t(int, BIO_MAX_PAGES, nr_pages);
                nr_pages -= nr_vecs;

                bio = bio_alloc(gfp, nr_vecs);
                if (!bio) {
                    err = -ENOMEM;
                    goto free_bios;
                }
                bio->bi_end_io = mppLnx_scsi_bi_endio;
            }

            if (bio_add_pc_page(q, bio, page, bytes, off) !=
                bytes) {
                bio_put(bio);
                err = -EINVAL;
                goto free_bios;
            }

            page++;
            len -= bytes;
            off = 0;
        }
    }

/** 
 * CR 124019 and 126741 - The following code was in the above for loop before. But, this lead 
 * to a case where scsi_cmnd->buffer_request was getting set to NULL before the command was 
 * submitted to HBA driver. So, to take care of it, it was moved out of the for loop but this
 * code does take one extra page than required to meet the scsi_cmnd request. There
 * may be possibility to run into memory_leak when LUNs are partitioned using
 * ext2/ext3 to do FileSystem IOs.
 */
    err = mppLnx_scsi_merge_bio(rq, bio);
    if (err) {
            bio_endio(bio, bio->bi_size, 0);
            goto free_bios;
    }
    bio = NULL;


    rq->buffer = rq->data = NULL;
    rq->data_len = data_len;
    return 0;

free_bios:
    while ((bio = rq->bio) != NULL) {
        rq->bio = bio->bi_next;
        /*
         * call endio instead of bio_put incase it was bounced
         */
        bio_endio(bio, bio->bi_size, 0);
    }

    return err;
}

static int mppLnx_scsi_merge_bio(struct request *rq, struct bio *bio)
{
    struct request_queue *q = rq->q;

    bio->bi_flags &= ~(1 << BIO_SEG_VALID);
    if (rq_data_dir(rq) == WRITE)
        bio->bi_rw |= (1 << BIO_RW);
    blk_queue_bounce(q, &bio);

    if (!rq->bio)
        blk_rq_bio_prep(q, rq, bio);
    else if (!q->back_merge_fn(q, rq, bio))
        return -EINVAL;
    else {
        rq->biotail->bi_next = bio;
        rq->biotail = bio;
        rq->hard_nr_sectors += bio_sectors(bio);
        rq->nr_sectors = rq->hard_nr_sectors;
    }

    return 0;
}
#else /* MPPLNX_SCSI_REQ_MAP_SG */

/**
 *  __mppLnx_bio_clone -   clone a bio
 *  @bio: destination bio
 *  @bio_src: bio to clone
 *
 *  Clone a &bio. Caller will own the returned bio, but not
 *  the actual data it points to. Reference count of returned
 *  bio will be one.
 */
static void __mppLnx_bio_clone(struct bio *bio, struct bio *bio_src,request_queue_t *q)
{
    memcpy(bio->bi_io_vec, bio_src->bi_io_vec,
        bio_src->bi_max_vecs * sizeof(struct bio_vec));

    bio->bi_sector = bio_src->bi_sector;
    bio->bi_bdev = bio_src->bi_bdev;
    bio->bi_flags |= 1 << BIO_CLONED;
    bio->bi_rw = bio_src->bi_rw;
    bio->bi_vcnt = bio_src->bi_vcnt;
    bio->bi_size = bio_src->bi_size;
    bio->bi_idx = bio_src->bi_idx;
    bio_phys_segments(q, bio);
    bio_hw_segments(q, bio);

}

/**
 *  mppLnx_bio_clone   -   clone a bio
 *  @bio: bio to clone
 *  @gfp_mask: allocation priority
 *
 *  Like __bio_clone, only also allocates the returned bio
 */
static struct bio *mppLnx_bio_clone(struct bio *bio, gfp_t gfp_mask,request_queue_t *q)
{
    struct bio *b = bio_alloc(gfp_mask, bio->bi_max_vecs);

    if (b) {
        __mppLnx_bio_clone(b, bio,q);
    }

    return b;
}

/*
 * clonereq - final request after cloning bios and request from the virtual request
 * vrequest - original virtual request that is attached to virtual scsi cmnd
 */
static int mppLnx_clone_request_bios(struct request *clonerequest, struct request *vrequest, unsigned bufflen,gfp_t gfp, mppLnx_ProxyRequestEntry_t *preq)
{

    struct bio *clonebio=NULL,*vbio=NULL,*tempbio=NULL;

    /* Clone Bio's from the virtual request into the clonerequest->bio */
    clonerequest->bio = NULL;
    rq_for_each_bio( vbio, vrequest )
    {
        clonebio = mppLnx_bio_clone( vbio, gfp, preq->pdev->request_queue );
        if ( !clonebio )
        {
                /* unable to allocate memory for bio
                 * cleanup all the older bio's 
                 * fail the io or retry handle the error here */
                 tempbio = clonerequest->bio;
                 vbio    = tempbio;
                 while( tempbio != NULL )
                 {
                     tempbio = vbio->bi_next;
                     bio_put(vbio);
                     vbio    = tempbio;
                 }
            return 1;
        }
        
        clonebio->bi_end_io = mppLnx_scsi_bi_endio;

        if( clonerequest->bio == NULL )
        {
            clonerequest->bio = clonebio;
            tempbio           = clonebio;
	}else
        {
            tempbio->bi_next  = clonebio;
            tempbio           = clonebio;
        }

    } /* for all bio's in the virtual request */
   
    /* link the last bio to the tail */
    clonerequest->biotail = clonebio;
    clonerequest->biotail->bi_next = NULL;


    /* Clone fields from the virtual request into the clonerequest 
       First three bits are identical in rq->cmd_flags and bio->bi_rw */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
    clonerequest->cmd_flags |= (clonerequest->bio->bi_rw & 3); 
#else
    clonerequest->flags |= (clonerequest->bio->bi_rw & 3); 
#endif
    clonerequest->data = NULL;
    clonerequest->sector = vrequest->sector;
    clonerequest->nr_sectors = vrequest->nr_sectors;
    clonerequest->current_nr_sectors = vrequest->current_nr_sectors;
    clonerequest->hard_sector = vrequest->hard_sector;
    clonerequest->hard_nr_sectors = vrequest->hard_nr_sectors;
    clonerequest->hard_cur_sectors = vrequest->hard_cur_sectors;
    clonerequest->nr_phys_segments = vrequest->nr_phys_segments;
    clonerequest->nr_hw_segments = vrequest->nr_hw_segments;
    clonerequest->buffer = vrequest->buffer;
    clonerequest->data_len = bufflen;

    return 0;
}
#endif /* MPPLNX_SCSI_REQ_MAP_SG */

/**
 * mppLnx_scsi_execute_async - insert request
 * @sdev:       scsi device
 * @cmd:        scsi command
 * @cmd_len:    length of scsi cdb
 * @data_direction: data direction
 * @buffer:     data buffer (this can be a kernel buffer or scatterlist)
 * @bufflen:    len of buffer
 * @use_sg:     if buffer is a scatterlist this is the number of elements
 * @timeout:    request timeout in seconds
 * @retries:    number of times to retry request
 * @flags:      or into request flags
 **/
int mppLnx_scsi_execute_async(struct scsi_device *sdev, const unsigned char *cmd,
                       int cmd_len, int data_direction, void *buffer, unsigned bufflen,
                       int use_sg, int timeout, int retries, void *privdata,
                       void (*done)(void *, char *, int, int), gfp_t gfp)
{
        struct request *req=NULL;
        mppLnx_ProxyRequestEntry_t *preq;
        int err = 0;
        int write = (data_direction == DMA_TO_DEVICE);

        if(unlikely(!privdata))
        {
            MPPLNX_VHBA_BUG();
        }

        if(unlikely(!done))
        {
            MPPLNX_VHBA_BUG();
        }

        if(unlikely(!sdev))
        {
            MPPLNX_VHBA_BUG();
        }

        preq = ((mppLnx_ProxyRequestEntry_t*)privdata);

        req = blk_get_request(sdev->request_queue, write, gfp);
        if (!req)
                goto free_sense;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
        req->cmd_type = REQ_TYPE_BLOCK_PC;
        req->cmd_flags |= REQ_QUIET;
#else
        req->flags |= REQ_BLOCK_PC | REQ_QUIET;
#endif

        if (use_sg)
        {
#ifdef  MPPLNX_SCSI_REQ_MAP_SG 
                err = mppLnx_scsi_req_map_sg(req, buffer, use_sg, gfp);
#else /* MPPLNX_SCSI_REQ_MAP_SG */
                err = mppLnx_clone_request_bios(req, preq->vcmnd->request,bufflen, gfp,preq);
#endif /* MPPLNX_SCSI_REQ_MAP_SG */
        }
        else if (bufflen)
                err = blk_rq_map_kern(req->q, req, buffer,bufflen, gfp);

        if (err)
                goto free_req;

        req->cmd_len = cmd_len;
        memcpy(req->cmd, cmd, req->cmd_len);
        req->sense = ((mppLnx_ProxyRequestEntry_t*)privdata)->sense_buffer;
        req->sense_len = 0;
        req->timeout = timeout;
        req->retries = retries;
        req->end_io_data = privdata;
        preq->mppLnx_done = done;
        preq->req         = req;

        MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_scsi_execute_async : io to lun %d\n", sdev->lun));

        blk_execute_rq_nowait(req->q, NULL, req, 1, mppLnx_scsi_end_async);
        return 0;

free_req:
        MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,515,
            "mppLnx_scsi_execute_async:: unable to allocate and clone bio's for lun  %d\n",sdev->lun));
        blk_put_request(req);
free_sense:
        MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,516,
            "mppLnx_scsi_execute_async:: allocation of block request failed. lun %d \n",sdev->lun));
        return DRIVER_ERROR << 24;
}

static void mppLnx_scsi_end_async(struct request *req, int uptodate)
{
        mppLnx_ProxyRequestEntry_t *preq;

        preq =  (mppLnx_ProxyRequestEntry_t*)req->end_io_data;

        MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_scsi_end_async : io cambeback for lun %d path %d ctl %d\n", preq->lun, preq->path,preq->controller));

        if (likely(preq->mppLnx_done))
            preq->mppLnx_done(req->end_io_data, preq->sense_buffer, req->errors, req->data_len);
        else
        {
            MPPLNX_VHBA_BUG();
        }


}


