/*******************************************************************************
*  Copyright 2004-2008 LSI Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            MPPutil.c
SUMMARY         %description%
VERSION         %version: 64 %
UPDATE DATE     %date_modified: Mon Apr 28 15:02:11 2008 %
PROGRAMMER      %created_by:    jwendel %

DESCRIPTION:
	Utility program for the MPP driver.

USAGE:
	mppUtil [-d debug_level] [-e error_level] [-g target_id] [-s "failback" | "avt" | "busscan" | "forcerebalance"]
	
	where:
		-d 	Set debug level based on defines found in MPP_Common.h
		-e 	Set error reporting level based on defines found in
			MPP_Common.h
		-g 	Get the RdacInfo data from the specified target ID.
		-s 	Initiate a scan for failed controllers, or AVT state change.

*******************************************************************************/


/***  INCLUDES  ***/  
#include	"mppUtil_Sysdep.h"
#include	"MPP_Sysdep.h"
#include	"MPP_Common.h"
#include	"MPP_RdacInfo.h"
#include	"mppUtil_Common.h"

/***  EXTERNAL REFERENCES  ***/

/***  PROCEDURES  ***/
	int	__cdecl main(LINT argc, char **argv);
static  VOID	utilPrintHeader(void);
static	BOOL	utilSetDebugLevel(char *level);
static	BOOL	utilSetErrorLevel(LINT level);
static	BOOL	utilGetRequest(LWORD target);
static	int	utilGetNextFeatureOption(char *args, char *keyword, LWORD *value, BOOL *valueProvided);
static	BOOL	utilScanRequest(char *req);
static	MppCmn_OptionControl_t *utilParseFeatureOptions(char *args, LWORD *bufferSize);
static	VOID	utilPrintRdacInfo(getRdacInfoIoctl_t *getInfo);
static	VOID	utilPrintFeatureOptions(MppCmn_OptionControlIoctl_t *ioctl);
static	BOOL	utilProcessFeatureOptions(char *args);
static  BOOL    utilUpdateMppConf(char * parameter, int value);
static  BOOL    utilProcessActionWord(MppCmn_OptionControl_t *optionControl);
static	VOID	utilHexDump(unsigned char *buffer, int bufferLength);
static  VOID    utilHexString(unsigned char *buf, unsigned char *buffer, int bufferLength);
static	BOOL	utilGetMaxLunsPerArray(void);
static	BOOL	utilGetMaxArrayModules(void);
static  BOOL    isString(MPP_STRING arg);
static  BOOL 	isHex(CHAR *arg); 
BOOL	isNum(MPP_STRING arg); 
static  BOOL	utilPrintTargetList(void);
static 	BOOL	utilGetTargetName(LINT target);
static 	BOOL	utilGetArrayInfo(MPP_STRING array_name);
static	BOOL	utilPrintLunPathCheckList(char *array_wwn, int controllerIndex);
static  void    mppUtilCmn_PrintArraySummary(mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxLuns, LWORD maxPaths, int PathState ) ;
static BOOL mppUtilCmn_ReadArraySummary(MPP_FILE_HANDLE fd, mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxPaths);
static  BOOL    mppUtilCmn_GetRealTimeStatus(void );
static  void mppUtilCmn_InitUtilStatus(mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxLuns, LWORD maxPaths);
static  BOOL mppUtilCmn_CompareUtilStatus(mpp_utilStatusInfo_t *utilStatusInfo, mpp_utilStatusInfo_t *utilStatusPtr, LWORD maxLuns, LWORD maxPaths ) ;
static BOOL mppUtilCmn_DeleteArrayWWNInfo(MPP_STRING array_name);
static BOOL mppUtilCmn_WriteArraySummary(MPP_FILE_HANDLE fd, mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxLuns, LWORD maxPaths );




/***  GLOBALS  ***/
	HANDLE   utilVirtualFd;
	LWORD    utilMaxPaths=-1;
	LWORD    utilMaxLuns=-1;
	LWORD	 utilMaxArrayModules=-1;
	BOOL     utilDebugEnabled;
	LINT    labelLength;
	CHAR	device[128];
	CHAR	hostName[HOST_NAME_LEN]={'\0'};
	CHAR	domainName[DOMAIN_NAME_LEN]={'\0'};
	CHAR	timeStr[TIME_STR_LEN]={'\0'};
	MPP_STRING progname;
	MPP_STRING  labelString;
	unsigned char		*module_name=NULL;  /* Array Module Name */
	unsigned char		*module_wwn=NULL; /* Array Module WWN */
	LWORD		        module_type=0; /* Array Module Type */
    char direct_string[MPP_MAX_PATH];
    char file_string[MPP_MAX_PATH];
    char *wwnDirPath;

    static char *ControllerStates[] = {
                "Missing",
                "Active",
                "Offline",
                "Service-Mode",
                "Reset",
                "Unknown"
    };
	
    /* Target Protocol Idendifier */
    static char *ArrayTypeString[] =  {
            "FC",          // Fibre Channel Array Type
            "invalid",     // No definition yet
            "invalid",     // No definition yet
            "invalid",     // No definition yet
            "IB",          // Infiniband Array Type
            "iSCSI",       // iSCSI Array Type
            "SAS"          // SAS Array Type
    };

	

int
__cdecl
main(LINT argc, char **argv)
{
	LINT	opt;
	BOOL	status = FALSE;
	char cmdLineArgs[MAX_CMDLINE_LENGTH];
	char	*array_wwn = NULL, *controller = NULL;
	LINT	controllerIndex;

	utilVirtualFd = (HANDLE) INVALID;
	utilMaxPaths = INVALID;
	utilMaxLuns = INVALID;
	utilDebugEnabled = FALSE;
	progname = argv[0];
	sprintf(cmdLineArgs, "%s%s", MPPCMN_UTIL_CMDLINE_ARGS,  MPPSYS_UTIL_CMDLINE_ARGS);


	if((opt = getopt(argc, argv, cmdLineArgs)) == EOF)
	{
		fprintf(stderr, "No Options supplied\n");
		utilUsage(progname);
	}
	while ( opt != EOF )
	{
		switch(opt)
		{
			case	'a':
				/* This option is overloaded.*/
				/* 'mpputil -a' prints the table of all arrays currently connected to this HOST */
				/* 'mppUtil -a "array_name"' prints the array info for the array "array_name". */
				optarg = argv[optind];
	

				/* print the array list if no options are passed */
				if ( ( optarg == NULL ) || ( *optarg == '-') )
				{
					utilPrintHeader();
					status = utilPrintTargetList();
				}
				else
					status = utilGetArrayInfo(optarg);
				break;

                        case    'c':
                                /* This option is overloaded.*/
                                /* 'mpputil -c' deletes wwn files of all arrays currently connected to this HOST */
                                /* 'mppUtil -c "array_name"' deletes the wwn info for the array "array_name". */
                                optarg = argv[optind];

                                /* print the array list if no options are passed */
                                if ( ( optarg == NULL ) || ( *optarg == '-') )
                                {
                                        utilPrintHeader();
					status = mppUtilSys_DeleteWWNList();
                                }
                                else
					status = mppUtilCmn_DeleteArrayWWNInfo(optarg);
                                break;


			case	'd':
				status = utilSetDebugLevel(optarg);
				break;

			case	'e':
				status = utilSetErrorLevel(atoi(optarg ));
				break;

			case	'g':
				
				if(!isNum(optarg))
				{
					fprintf(stderr, "This option accepts only integer.\n");
					break;
				}
				status = utilGetRequest(atoi(optarg));
				break;

			case	's':
				status = utilScanRequest(optarg);
				break;

                        case    'S':
                                status = mppUtilCmn_GetRealTimeStatus();

			case	't':
				utilDebugEnabled = TRUE;
				break;

			case	'o':
				optarg = argv[optind];
				status = utilProcessFeatureOptions(optarg);
				break;

			case	'm':
				status = utilGetMaxArrayModules();
				break;

			case	'l':
				status = utilGetMaxLunsPerArray();
				break;

			case	'w':
				optarg = argv[optind];

				if(optarg == NULL)
				{
					utilUsage(progname);
					break;
				}

				array_wwn = strtok(optarg, ",");

				// Multiply WWN_LENGTH by 2 since the string prints out a
				// hex value for each byte.
				if(strlen(array_wwn) != (WWN_LENGTH * 2))
				{
					fprintf(stderr, "Invalid WWN length\n");
					utilUsage(progname);
					break;
				}

				// Translate controller index character value to numeric if necessary.
				controller = strtok(NULL, ",");
				if(strlen(controller) != 1)
				{
					fprintf(stderr, "Invalid index value\n");
					utilUsage(progname);
					break;
				}

				if((controller[0] == 'a') || (controller[0] == 'A') ||
				   (controller[0] == '0'))
				{
					controllerIndex = 0;
				}
				else if((controller[0] == 'b') || (controller[0] == 'B') ||
				   (controller[0] == '1'))
				{
					controllerIndex = 1;
				}
				else
				{
					fprintf(stderr, "Invalid index value\n");
					utilUsage(progname);
					break;
				}

				utilDebug("array_wwn: '%s'\n", array_wwn);
				utilDebug("controllerIndex: '%d'\n", controllerIndex);
				status = utilPrintLunPathCheckList(array_wwn, controllerIndex);
				break;
				
			case	'h':
				utilUsage(progname);
				break;
				
			default:
				status = mppSys_ProcessCmdLineArgs(opt, optarg);
				break;
		}

		if ( status == INVALID ) 
			break;
		

		opt = getopt(argc, argv, cmdLineArgs);
	}

	return(status);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilUsage
* SUMMARY:	Display program usage.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	Display program usage.
*
* RETURNS:
*	N/A
*******************************************************************************/
VOID
utilUsage(char *prog)
{
	printf("Usage: %s %s %s\n", prog, MPPCMN_UTIL_USAGE, MPPSYS_UTIL_USAGE );
	return;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilPrintHeader
* SUMMARY:	Print Host specific information
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	Print Host specific information.
*
* RETURNS:
*	N/A
*******************************************************************************/
static VOID
utilPrintHeader(void)
{
	/* get host name */
	utilFormatHostName( hostName );

	/* get domian  name */
	utilFormatDomainName ( domainName );

	/* get current time */
	utilFormatCurrentGMTTime ( timeStr );
	
	/* print Host Name */
	if ( hostName[0] == '\0' )
		printf("Hostname    = N/A\n");
	else
		printf("Hostname    = %s\n",hostName);
	
	/* print domain Name */
	if ( domainName[0] == '\0' )
		printf("Domainname  = N/A\n");
	else
		printf("Domainname  = %s\n",domainName);

	/* print current time */
	if ( timeStr[0] == '\0' )
		printf("Time        = N/A\n"); 
	else
		printf("Time        = GMT %s \n", timeStr); 

	printf("\n");	/* extra new line */
	return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:		utilSetDebugLevel
* SUMMARY:	Set the current MPP debug level.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends an MPP_IOCTL_SETDEBUG request to the MPP
*	virtual bus.
*
* RETURNS:
*	Boolean indicating success or failure of the request.
*******************************************************************************/
static BOOL
utilSetDebugLevel(char *level)
{
	unsigned int	debugLevel = 0, idx = 0;

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	if(!isHex(optarg)) {
		fprintf(stderr, "Invalid option!\n");
		return(INVALID);
	}

	// See if the user began the debug level with '0x'
	if(!strncmp(level, "0x", 2) || !strncmp(level, "0X", 2) )
	{
		level += 2;
	}

	while(level && *level && isxdigit(*level))
	{
		if(islower(*level)) {
			idx = *level++ - ('a' - 'A');
			idx = idx - '0';
		}
		else
			idx = *level++ - '0';

		if(9 < idx)
			idx -= 7;
		debugLevel <<= 4;
		debugLevel |= (idx & 0xff);
	}
	
	utilDebug("Set debug level to 0x%x\n", debugLevel);
	if(ioctl(utilVirtualFd, MPP_IOCTL_SETDEBUG, &debugLevel) == INVALID)
	{
		fprintf(stderr, "Unable to send debug request - %s!\n", strerror(errno));
		return(FALSE);
	}
	return(TRUE);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilSetErrorLevel
* SUMMARY:	Set the current MPP error level.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends an MPP_IOCTL_SETERROR request to the MPP
*	virtual bus.
*
* RETURNS:
*	Boolean indicating success or failure of the request.
*******************************************************************************/
static BOOL
utilSetErrorLevel(LINT level)
{
	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	if(!isNum(optarg)) {
		fprintf(stderr, "Invalid option!\n");
		return(INVALID);
	}

	if(( level < 0 ) || ( level > 5 ))
	{
		fprintf(stderr, "The value %d for option ErrorLevel is out of range.  Valid range is 0-5 inclusive.\n",level);
                return(INVALID);
	}
       

	utilDebug("Set error level to %d\n", level);
	if(ioctl(utilVirtualFd, MPP_IOCTL_SETERROR, &level) == INVALID)
	{
		fprintf(stderr, "Unable to send error request - %s!\n", strerror(errno));
		return(FALSE);
	}
	return(TRUE);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilScanRequest
* SUMMARY:	Send a scan request to the MPP driver.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a scan request to the MPP virtual bus.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilScanRequest(char *req)
{
	LINT	ioctlReq;

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	if(strcmp(req, SCAN_FAILBACK) != 0 && strcmp(req, SCAN_AVT) != 0 &&
		strcmp(req, SCAN_BUSSCAN) != 0 && strcmp(req, SCAN_FORCE_REBALANCE) != 0)
	{
		fprintf(stderr, "Invalid option!\n");
		return(INVALID);
	}

	if(!strcmp(req, SCAN_FAILBACK))
		ioctlReq = MPP_IOCTL_FAILBACKSCAN;
	else if(!strcmp(req, SCAN_AVT))
		ioctlReq = MPP_IOCTL_AVTCHECKSCAN;
	else if(!strcmp(req, SCAN_BUSSCAN))
		ioctlReq = MPP_IOCTL_BUSSCAN;
   else if(!strcmp(req, SCAN_FORCE_REBALANCE))
      ioctlReq = MPP_IOCTL_FORCEREBALANCE;

	utilDebug("Send scan request 0x%x\n", ioctlReq);
	if(ioctl(utilVirtualFd, ioctlReq, NULL) == INVALID)
	{
		fprintf(stderr, "Unable to send ioctl request - %s!\n", strerror(errno));
		return(FALSE);
	}
	return(TRUE);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilInitModuleArray
* SUMMARY:	Allocate memory for handling module name and wwn
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a ioctl to find the max supported array modules,
*	allocates the memory for module_name and module_wwn array's.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilInitModuleArray(void)
{

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	/* Get Max Array Modules supported */
	if ( utilMaxArrayModules == -1 ) 
	{
		utilDebug("sending ioctl for maxArrayModules \n");
		if((ioctl(utilVirtualFd, MPP_IOCTL_GETMODULES, &utilMaxArrayModules)) == INVALID)
		{
			fprintf(stderr, "Unable to get maxmodules request - %s!\n", strerror(errno));
			return(FALSE);
		}
		utilDebug("MaxArrayModules = %d\n", utilMaxArrayModules);
	}

	/* allocate memory for module_name array */
	if(( module_name = (unsigned char *)malloc( utilMaxArrayModules * MAX_ARRAY_STR_LEN )) == NULL)
	{
		fprintf(stderr, "Unable to malloc memory for module_name array - %s!\n", strerror(errno));
		return(FALSE);
	}

	/* allocate memory for module_wwn array */
	if(( module_wwn = (unsigned char *)malloc( utilMaxArrayModules * WWN_LENGTH )) == NULL)
	{
		fprintf(stderr, "Unable to malloc memory for module_wwn array - %s!\n", strerror(errno));
		return(FALSE);
	}
	
	return (TRUE);
}
/*******************************************************************************
* PROCEDURE
*
* NAME:		utilFreeModuleArray
* SUMMARY:	Free memory for handling module name and wwn
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a ioctl to find the max supported array modules,
*	Frees the memory for module_name and module_wwn array's.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilFreeModuleArray(void)
{

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	/* Get Max Array Modules supported */
	if ( utilMaxArrayModules == -1 ) 
	{
		utilDebug("sending ioctl for maxArrayModules \n");
		if((ioctl(utilVirtualFd, MPP_IOCTL_GETMODULES, &utilMaxArrayModules)) == INVALID)
		{
			fprintf(stderr, "Unable to get maxmodules request - %s!\n", strerror(errno));
			return(FALSE);
		}
		utilDebug("MaxArrayModules = %d\n", utilMaxArrayModules);
	}

	/* free memory for module_name array */
	if( module_name )
	{
		free(module_name);
	}

	/* free memory for module_wwn array */
	if( module_wwn ) 
	{
		free(module_wwn);
	}
	
	return (TRUE);
}
/*******************************************************************************
* PROCEDURE
*
* NAME:		utilPrintTargetList
* SUMMARY:	Print the info of all Array's detected by this Host.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a get request to the MPP virtual bus for each
*	target and gets the array info.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilPrintTargetList(void)
{
	LWORD 	i;
	BOOL	ret;

	if(( module_name == NULL) || (module_wwn == NULL))
	{
		ret= utilInitModuleArray();
		if( ( ret == INVALID ) || ( ret == FALSE ) )
		{
			utilDebug("failure in initialization of Module Array.\n");
			return (ret);
		}
	}

	fprintf(stdout,"---------------------------------------------------------------\n");
	fprintf(stdout,"Info of Array Module's seen by this Host. \n");
	fprintf(stdout,"---------------------------------------------------------------\n");
	fprintf(stdout,"ID		WWN		   	 Type     Name         \n");
	fprintf(stdout,"---------------------------------------------------------------\n");
	
	for(i=0; i<utilMaxArrayModules; i++)
	{	
		//fprintf(stdout,"Array Module Number %d\n",i);

		ret = utilGetTargetName(i);
		if (ret == FALSE )
			continue;

    // 03/15/07 CR 118927 - changed all module_wwn[i] and module_name[i] references to module_wwn[0],  module_name[0]
		fprintf(stdout,"%2d	",i);
		utilHexDump(&module_wwn[0], WWN_LENGTH);
		fprintf(stdout," %s", ArrayTypeString[module_type]);
		fprintf(stdout,"	%s     ",&module_name[0]);
		fprintf(stdout,"\n");
	}	

	fprintf(stdout,"---------------------------------------------------------------\n");

	utilFreeModuleArray(); /* Free the global Memory */
	return (TRUE);

}

/*******************************************************************************
* PROCEDURE
*
* NAME:		utilGetTargetName
* SUMMARY:	Get the target name from rdacinfo at index(target).
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a get request to the MPP virtual bus to get the
*	array name at given target index.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilGetTargetName(LINT target)
{
	RdacDeviceInformation_t *reqInfo = NULL;
	unsigned char		reqName[MAX_ARRAY_STR_LEN];
	getRdacInfoIoctl_t	rdacInfo;
	LWORD			Lun;
	BYTE			*reqPtr = NULL;

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	/* Get maxluns supported */
	if ( utilMaxLuns == -1 ) 
	{
		if( ioctl(utilVirtualFd, MPP_IOCTL_GETLUNS, &utilMaxLuns) == INVALID)
		{
			utilDebug("Unable to get maxluns request - %s!\n", strerror(errno));
			return(FALSE);
		}
	}

	/* Get maxpaths supported */
	if ( utilMaxPaths == -1 )
	{
		if( ioctl(utilVirtualFd, MPP_IOCTL_GETPATHS, &utilMaxPaths) == INVALID)
		{
			utilDebug("Unable to get maxpaths request - %s!\n", strerror(errno));
			return(FALSE);
		}
	}

	utilDebug("utilGetTargetName() : utilMaxLuns =%d  utilMaxPaths=%d.\n",utilMaxLuns,utilMaxPaths);

	if((reqInfo = (RdacDeviceInformation_t *)malloc(MPP_RDACINFOSIZE(utilMaxLuns, utilMaxPaths))) == NULL)
	{
		fprintf(stderr, "Unable to malloc memory for RdacInfo - %s!\n", strerror(errno));
		return(FALSE);
	}

	rdacInfo.target = target;
	rdacInfo.BufferSize = MPP_RDACINFOSIZE(utilMaxLuns, utilMaxPaths);
	rdacInfo.RdacInfoAdrs = reqInfo;
	rdacInfo.RdacName = &reqName;

	utilDebug("Send RdacInfo request to target %d. \n",target);
	if(ioctl(utilVirtualFd, MPP_IOCTL_GETRDACINFO, &rdacInfo) == INVALID)
	{
		utilDebug("Unable to send ioctl request - %s!\n", strerror(errno));
		return(FALSE);
	}

	labelString = (MPP_STRING) 0;
	labelLength = 0;

  // 03/15/07 CR 118927 - changed all module_wwn[i] and module_name[i] references to module_wwn[0],  module_name[0]
	strcpy(&module_name[0],rdacInfo.RdacName);
	memcpy(&module_wwn[0],(unsigned char *)reqInfo->WWN,WWN_LENGTH);
    module_type = (LWORD) reqInfo->ArrayType;

	utilDebug("Freeing the memory allocated for rdac info.\n");
	/* free the memory allocated to req info */	
	free(reqInfo);

	return(TRUE);
}

/*******************************************************************************
* PROCEDURE
*
* NAME:		mppUtilCmn_DeleteArrayWWNInfo
* SUMMARY:	Delete Array WWN file corresponding to an array wwn name
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function deletes the wwn file for a particular array wwn name.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/

static BOOL
mppUtilCmn_DeleteArrayWWNInfo(MPP_STRING array_name)
{
        LWORD                   targetNum;
        char temp_string[MPP_MAX_PATH];
        char file_name[MPP_MAX_PATH];
        BYTE                    i;
        BOOL                    ret;

        if(( module_name == NULL) || (module_wwn == NULL))
        {
                ret= utilInitModuleArray();
                if( ( ret == FALSE ) || ( ret == INVALID ) )
                {
                        utilDebug("failure in initialization of Module Array.\n");
                        return (ret);
                }
        }

        if ( !isString(array_name )) {

                fprintf (stderr," Please enter an array name as input\n");
                return FALSE;
        }

        utilDebug("string passed = %s\n",array_name);
        targetNum = -1;

        for(i=0; i<utilMaxArrayModules; i++)
        {
                /* fill the ModuleList array with module names */
                ret = utilGetTargetName(i);
                if( ret == FALSE )
                        continue;
                        
                // 03/15/07 CR 118927 - changed all module_wwn[i] and module_name[i] references to module_wwn[0],  module_name[0]
                if(strcmp(array_name,&module_name[0])==0)
                {
                        utilDebug("found the matching module %d\n", i);
                        targetNum = i;
                        break;
                }
        }
        if (targetNum == -1 )
        {
                fprintf(stderr,"Array Name '%s' not found.\n",array_name);
                return (ret);
        }
        else
        {

                wwnDirPath = mppUtilSys_setWwnDirPath();
                if (wwnDirPath == NULL)
                {
                    return(FALSE);
                }
                strcpy(temp_string, wwnDirPath);
                utilHexString(&file_name[0], &module_wwn[0], WWN_LENGTH);
                strcat(file_name, ".wwn");
                utilDebug("Deleting file %s in %s\n", file_name, MPPSYS_UTIL_WWN_DIR_NAME);
                strcat(temp_string, file_name);
                mppUtilSys_DeleteFile(temp_string);

                utilFreeModuleArray(); /* Free the memory */
        }
        return TRUE;

}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilGetArrayInfo
* SUMMARY:	Get the RDAC information for a given target.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a get request to the MPP virtual bus.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilGetArrayInfo(MPP_STRING array_name)
{
	LWORD			targetNum;
	BYTE			i;
	BOOL			ret;

	if(( module_name == NULL) || (module_wwn == NULL))
	{
		ret= utilInitModuleArray();
		if( ( ret == FALSE ) || ( ret == INVALID ) )
		{
			utilDebug("failure in initialization of Module Array.\n");
			return (ret);
		}
	}

        if ( !isString(array_name )) {

                fprintf (stderr," Please enter an array name as input\n");
                return FALSE;
        }

	utilDebug("string passed = %s\n",array_name);
	targetNum = -1;
	
	for(i=0; i<utilMaxArrayModules; i++)
	{
		/* fill the ModuleList array with module names */
		ret = utilGetTargetName(i);	
		if( ret == FALSE )
			continue;

    // 03/15/07 CR 118927 - changed all module_wwn[i] and module_name[i] references to module_wwn[0],  module_name[0]
		if(strcmp(array_name,&module_name[0])==0)
		{
			utilDebug("found the matching module %d\n", i);
			targetNum = i;
			break;
		}
	}

	if (targetNum == -1 )
	{
		fprintf(stderr,"Array Name '%s' not found.\n",array_name);
		ret = utilPrintTargetList();
		return (ret);
	}
	else
	{
		ret = utilGetRequest(targetNum);	
		utilFreeModuleArray(); /* Free the memory */
		return (ret);
	}
		
}

/*******************************************************************************
* PROCEDURE
*
* NAME:         mppUtilCmn_GetRealTimeStatus
* SUMMARY:      Get Real Time controller and path status from the array.
* SCOPE:        LOCAL
*
* DESCRIPTION:
*       This function sends an MPP_IOCTL_UTIL_STATUS request to the MPP
*       virtual bus for each Array Virtual Target Id.
*
* RETURNS:
*       Boolean indicating success or failure of the request.
*******************************************************************************/
static BOOL
mppUtilCmn_GetRealTimeStatus(void )
{
        LWORD maxLunsPerArray=0, maxPathsPerController=0, maxArrayModules=0;
        mpp_utilGetStatusIoctl_t utilStatusIoctl;
        LWORD utilStatusSize = 0, i =0;
        BYTE *ptr = NULL ;
        mpp_utilStatusInfo_t *utilStatusPtr = NULL, *utilStatusPointer=NULL;
        LWORD status = 0;
        MPP_FILE_HANDLE fd;
        BOOL ret= FALSE;
        PWWN_FILES wwnFileData;
        int j = 0;
	CHAR buf[UTIL_STATUS_BUF_LENGTH];
	CHAR arrayName[DEVNM_LEN+1];
	int n;
	BOOL missingArray;


        if(!utilDetermineBusFd())
        {
                fprintf(stderr, "Unable to determine virtual bus!!\n");
                return(INVALID);
        }

        utilDebug("sending ioctl for maxLunsPerArray\n");
        if(ioctl(utilVirtualFd, MPP_IOCTL_GETLUNS, &maxLunsPerArray) == INVALID)
        {
                fprintf(stderr, "Unable to get maxluns request - %s!\n", strerror(errno));
                return(FALSE);
        }

        utilDebug("sending ioctl for maxPathsPerController\n");
        if(ioctl(utilVirtualFd, MPP_IOCTL_GETPATHS, &maxPathsPerController) == INVALID)
        {
                fprintf(stderr, "Unable to get maxpaths request - %s!\n", strerror(errno));
                return(FALSE);
        }

        utilDebug("sending ioctl for maxArrayModules\n");
        if(ioctl(utilVirtualFd, MPP_IOCTL_GETMODULES, &maxArrayModules) == INVALID)
        {
                fprintf(stderr, "Unable to get getmodules request - %s!\n", strerror(errno));
                return(FALSE);
        }

        if (( utilStatusPtr = (mpp_utilStatusInfo_t *)malloc(MPP_UTILSTATUSSIZE( maxLunsPerArray, maxPathsPerController))) == NULL )
        {
                fprintf(stderr, "Unable to malloc memory for UtilStatusInfo - %s!\n", strerror(errno));
                return(FALSE);
        }


        if(( module_name == NULL) || (module_wwn == NULL))
        {
                ret= utilInitModuleArray();
                if( ( ret == INVALID ) || ( ret == FALSE ) )
                {
                        utilDebug("Failure in allocation of Module Array.\n");
                        return (ret);
                }
        }

	
        if(mppUtilSys_PrintWWNList(&wwnFileData) == TRUE) {
                utilDebug("mppUtilSys_PrintWWNList wwnFileData numWwnFiles %d\n", wwnFileData->numWwnFiles);
        }
        else {
                fprintf(stderr, "mppUtilSys_PrintWWNList returned FALSE\n");
        }
        for(j=0; j< wwnFileData->numWwnFiles; j++) {
                 wwnFileData->arrayDataList[j].active = FALSE;
        }


        for(i=0; i < maxArrayModules; i++)
        {

                ret = utilGetTargetName(i);
                if (ret == FALSE || ret == INVALID ) {
                        utilDebug("No Array found !\n");
                        continue;
                }

                utilHexString(&file_string[0], &module_wwn[0], WWN_LENGTH);
                strcat(file_string, ".wwn");

                wwnDirPath = mppUtilSys_setWwnDirPath();
                if (wwnDirPath  == NULL )
                {
                        fprintf(stderr, "No WWN DIR PATH defined !\n");
                }
                strcpy(direct_string, wwnDirPath);

                memset(utilStatusPtr, '\0', MPP_UTILSTATUSSIZE( maxLunsPerArray, maxPathsPerController));
                utilStatusIoctl.mpp_Target =  i;
                utilStatusIoctl.mpp_BufferSize = MPP_UTILSTATUSSIZE( maxLunsPerArray, maxPathsPerController);
                utilStatusIoctl.mpp_StatusAddress = utilStatusPtr;
                utilDebug("sending utilstatus ioctl for target # %d\n", utilStatusIoctl.mpp_Target);
                status = ioctl(utilVirtualFd, MPP_IOCTL_UTIL_STATUS, &utilStatusIoctl);
                if( status == INVALID)
                {
                     utilDebug("Unable to get utilstatus request - %s!\n", strerror(errno));
                     continue;
                }

                for(j=0; j< wwnFileData->numWwnFiles; j++) {
                        if(strcmp(&wwnFileData->arrayDataList[j].fileName[0], file_string) == 0) {
                                wwnFileData->arrayDataList[j].active = TRUE;
                        }
                }


                ret = mppUtilSys_DoesWwnDirPathFileExist(file_string);
                if (ret == TRUE) {


                   if (( utilStatusPointer = (mpp_utilStatusInfo_t *)malloc(MPP_UTILSTATUSSIZE( maxLunsPerArray, maxPathsPerController))) == NULL )
                   {
                        fprintf(stderr, "Unable to malloc memory for UtilStatusInfo - %s!\n", strerror(errno));
                        return(FALSE);
                   }

                   mppUtilCmn_InitUtilStatus(utilStatusPointer, maxLunsPerArray,  maxPathsPerController);

                   strcat(direct_string, file_string);
                   fd = mppUtilSys_OpenFile(direct_string, O_RDONLY, 0);
		   if ( fd == MPP_OPEN_FILE_ERROR )
			continue;		
                   ret = mppUtilCmn_ReadArraySummary(fd, utilStatusPointer, maxPathsPerController);
                   if( ret == FALSE) {
                        fprintf(stderr, "Unable to read Array File- %s!\n", strerror(errno));
                   	ret = mppUtilSys_CloseFile(fd);
			continue;
		   }

                   ret = mppUtilSys_CloseFile(fd);

                   ret = mppUtilCmn_CompareUtilStatus(utilStatusPtr, utilStatusPointer, maxLunsPerArray,  maxPathsPerController );
                   mppUtilCmn_PrintArraySummary(utilStatusPointer, maxLunsPerArray, maxPathsPerController, PATH_UP);
                   fd = mppUtilSys_OpenFile(direct_string, O_CREAT|O_RDWR, 0600);
		   if ( fd == MPP_OPEN_FILE_ERROR )
			continue;		
                   ret = mppUtilCmn_WriteArraySummary(fd, utilStatusPointer, maxLunsPerArray, maxPathsPerController);
                   if( ret == FALSE)
                        fprintf(stderr, "Unable to write Array File- %s!\n", strerror(errno));
                   ret = mppUtilSys_CloseFile(fd);
                   free(utilStatusPointer);
                }
                else {
                   strcat(direct_string, file_string);
                   fd = mppUtilSys_OpenFile(direct_string, O_CREAT|O_RDWR, 0600);
		   if ( fd == MPP_OPEN_FILE_ERROR )
			continue;		
                   ret = mppUtilCmn_WriteArraySummary(fd, utilStatusPtr, maxLunsPerArray, maxPathsPerController);
                   if( ret == FALSE) {
                   	ret = mppUtilSys_CloseFile(fd);
                        continue;
		   }
                   ret = mppUtilSys_CloseFile(fd);
                   mppUtilCmn_PrintArraySummary(utilStatusPtr, maxLunsPerArray, maxPathsPerController, PATH_UP);
                }
        }
	missingArray = FALSE;
	printf("\n   Missing Arrays \n");
        for(j=0; j< wwnFileData->numWwnFiles; j++) {
                if(wwnFileData->arrayDataList[j].active != TRUE)  {

                   wwnDirPath = mppUtilSys_setWwnDirPath();
                   if (wwnDirPath  == NULL )
                   {
                        fprintf(stderr, "No WWN DIR PATH defined !\n");
                   }
                   strcpy(direct_string, wwnDirPath);
                   strcat(direct_string, &wwnFileData->arrayDataList[j].fileName[0]);
                   fd = mppUtilSys_OpenFile(direct_string, O_RDONLY, 0);
                   if ( fd == MPP_OPEN_FILE_ERROR )
                        continue;

		   memset(buf, '\0', UTIL_STATUS_BUF_LENGTH);
   		   if((n = mppUtilSys_ReadFile(fd, buf, VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH + DEVNM_LEN)) > 0) { 
			if(n < (VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH + DEVNM_LEN) )
			{
				ret = mppUtilSys_CloseFile(fd);
				continue;
			}
			else {
				missingArray = TRUE;
				memset(arrayName, '\0', DEVNM_LEN+1);
				memcpy(&arrayName[0], &buf[VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH], DEVNM_LEN-1);
				printf("  %s\n", &arrayName[0]);
			}
		    }
                    ret = mppUtilSys_CloseFile(fd);

		 }

        }
	if( missingArray == FALSE )
		printf(" There are no missing arrays\n");

        free(wwnFileData);

        utilFreeModuleArray(); /* Free the global Memory */

        free(utilStatusPtr);

        return(TRUE);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:         mppUtilCmn_WriteArraySummary
* SUMMARY:      Write the Array Information into a File
* SCOPE:        LOCAL
*
* DESCRIPTION:
*       This function writes the Array Information into a .wwn file
*
* RETURNS:
*       Boolean indicating success or failure of the action.
*******************************************************************************/

static BOOL
mppUtilCmn_WriteArraySummary( MPP_FILE_HANDLE fd, mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxLuns, LWORD maxPaths )
{
        LWORD Lun=0, Path =0;
        CHAR    virtual_target[STR_LENGTH], physical_target[STR_LENGTH], controller_state[2][STR_LENGTH];
        CHAR    tmpBuf[UTIL_STATUS_BUF_LENGTH];
        LWORD   width=0;
        INT curPos=0;
        BYTE    *ptr = NULL,  *other_ptr= NULL;
        mpp_utilLunInfo_t   *lunInfo = NULL;
        mpp_utilPathInfo_t  *pathInfo = NULL, *other_pathInfo= NULL;
        BOOL first=TRUE;
        char temp_string[RECORD_LENGTH];



        if( utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualTargetId != -1 ) {

                mppUtilSys_FormatVirtualTarget(&utilStatusInfo->mpp_PlatformStatusInfo, virtual_target);

                memset(tmpBuf, ' ', UTIL_STATUS_BUF_LENGTH);
                width = strlen(virtual_target);
                if(width > VIRTUAL_TARGET_WIDTH)
                        width = VIRTUAL_TARGET_WIDTH;
                memcpy(&tmpBuf[VIRTUAL_TARGET_STATUS_COLUMN], virtual_target, width);
                tmpBuf[VIRTUAL_TARGET_WIDTH-1] = ':';
                if (mppUtilSys_WriteFile(fd, &tmpBuf[0] , VIRTUAL_TARGET_WIDTH) <= 0 )
                        return FALSE;
                strcpy (controller_state[0], ControllerStates[utilStatusInfo->mpp_ControllerState[0] - 0x0A] );

                width = strlen(controller_state[0]);
                if(width > CONTROLLER_STATUS_WIDTH)
                        width = CONTROLLER_STATUS_WIDTH;
                memcpy(&tmpBuf[VIRTUAL_TARGET_WIDTH], controller_state[0], width);
                tmpBuf[VIRTUAL_TARGET_WIDTH + CONTROLLER_STATUS_WIDTH-1] = ':';
                if (mppUtilSys_WriteFile(fd, &tmpBuf[VIRTUAL_TARGET_WIDTH], CONTROLLER_STATUS_WIDTH) <= 0)
                        return FALSE;
                strcpy (controller_state[1], ControllerStates[utilStatusInfo->mpp_ControllerState[1] - 0x0A] );

                width = strlen(controller_state[1]);
                if(width > CONTROLLER_STATUS_WIDTH)
                        width = CONTROLLER_STATUS_WIDTH;
                memcpy(&tmpBuf[VIRTUAL_TARGET_WIDTH + CONTROLLER_STATUS_WIDTH], controller_state[1], width);
                tmpBuf[VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH-1] = ':';
                if ( mppUtilSys_WriteFile(fd, &tmpBuf[VIRTUAL_TARGET_WIDTH + CONTROLLER_STATUS_WIDTH], CONTROLLER_STATUS_WIDTH) <= 0)
                        return FALSE;

                memcpy(&tmpBuf[VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH], utilStatusInfo->mpp_SAName, strlen(utilStatusInfo->mpp_SAName));
                tmpBuf[VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH+DEVNM_LEN -1] = '\n';
                if (mppUtilSys_WriteFile(fd, &tmpBuf[VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH], DEVNM_LEN) <= 0)
                        return FALSE;


                ptr =  (BYTE *)utilStatusInfo + sizeof(mpp_utilStatusInfo_t) ;
                lunInfo = (mpp_utilLunInfo_t *) ptr;
                ptr= ptr + sizeof(mpp_utilLunInfo_t)* maxLuns;
                pathInfo = (mpp_utilPathInfo_t *) ptr;

                sprintf(temp_string, "maxLun%3d:", maxLuns);
                mppUtilSys_WriteFile(fd, &temp_string[0], RECORD_LENGTH);

                for(Path=0; Path < maxPaths; Path++) {
                        sprintf(temp_string, "CAP%2d    :",Path);
                        if (mppUtilSys_WriteFile(fd, &temp_string[0], RECORD_LENGTH ) <= 0)
                                return FALSE;

                }

                for(Path=0; Path < maxPaths; Path++) {
                sprintf(temp_string, "CBP%2d    :",Path);
                if (mppUtilSys_WriteFile(fd, &temp_string[0], RECORD_LENGTH) <= 0)
                        return FALSE;
                }

                mppUtilSys_WriteFile(fd, "\n", 1);

                for(Lun=0; Lun < maxLuns; ++Lun) {

                        if(lunInfo->mpp_LunNumber == NO_LUN) {

                                ptr = (BYTE *)pathInfo;
                                ptr += (sizeof(mpp_utilPathInfo_t ) * 2*maxPaths) ;
                                pathInfo = (mpp_utilPathInfo_t *)ptr;
                                goto lun_step;
                        }

                        sprintf(tmpBuf, "L%03d     :", lunInfo->mpp_LunNumber );
                        if (mppUtilSys_WriteFile(fd, &tmpBuf[0], RECORD_LENGTH) <= 0)
                                return FALSE;

                        for(Path=0; Path < maxPaths; Path++) {

                                ptr = (BYTE *)pathInfo;
                                other_ptr = ptr + sizeof(mpp_utilPathInfo_t ) * maxPaths ;
                                other_pathInfo = (mpp_utilPathInfo_t *) other_ptr;

                                if( pathInfo->mpp_PathState != NO_PATH ) {

                                        memset(physical_target, '\0', STR_LENGTH );
                                        mppUtilSys_FormatPhysicalTarget(&pathInfo->mpp_CtlAddr, physical_target);
                                        sprintf(tmpBuf, "%sL%03d  %s", physical_target, lunInfo->mpp_LunNumber, pathInfo->mpp_PathState == 1 ? UP : DOWN );
                                        width = strlen(tmpBuf);
                                        if(width > CONTROLLER_STATUS_WIDTH)
                                                width = CONTROLLER_STATUS_WIDTH;

                                        sprintf(tmpBuf, "%s:", physical_target);
                                        if (mppUtilSys_WriteFile(fd, &tmpBuf[0], RECORD_LENGTH) <= 0)
                                                return FALSE;
                                        if((curPos=mppUtilSys_SetFilePointer(fd, 0, SEEK_CUR)) < 0)
                                                return FALSE;

                                }
                                else {

                                        sprintf(tmpBuf, "---------:");
                                        mppUtilSys_WriteFile(fd, &tmpBuf[0], RECORD_LENGTH);
                                        if ((curPos=mppUtilSys_SetFilePointer(fd, 0, SEEK_CUR)) < 0)
                                                return FALSE;
                                }

                                if( other_pathInfo->mpp_PathState != NO_PATH ) {

                                        memset(physical_target, '\0', STR_LENGTH );
                                        mppUtilSys_FormatPhysicalTarget(&other_pathInfo->mpp_CtlAddr, physical_target);
                                        sprintf(tmpBuf, "%sL%03d  %s", physical_target, lunInfo->mpp_LunNumber, other_pathInfo->mpp_PathState == 1 ? UP : DOWN );
                                        width = strlen(tmpBuf);
                                        if(width > CONTROLLER_STATUS_WIDTH)
                                                width = CONTROLLER_STATUS_WIDTH;

                                        sprintf(tmpBuf, "%s:", physical_target);
                                        if ( mppUtilSys_SetFilePointer(fd, (maxPaths-1)*RECORD_LENGTH, SEEK_CUR) < 0)
                                                return FALSE;
                                        if (mppUtilSys_WriteFile(fd, &tmpBuf[0], RECORD_LENGTH) <= 0 )
                                                return FALSE;

                                }
                                else {
                                        sprintf(tmpBuf, "---------:");
                                        if ( mppUtilSys_SetFilePointer(fd, (maxPaths-1)*RECORD_LENGTH, SEEK_CUR) < 0 )
                                                return FALSE;

                                        if (mppUtilSys_WriteFile(fd, &tmpBuf[0], RECORD_LENGTH) <= 0 )
                                                return FALSE;
                                }


                                if (mppUtilSys_SetFilePointer(fd, curPos, SEEK_SET) < 0)
                                        return FALSE;
                                ptr = (BYTE *)pathInfo;
                                ptr += sizeof(mpp_utilPathInfo_t);
                                pathInfo = (mpp_utilPathInfo_t *)ptr;

                        }
                        if ((curPos=mppUtilSys_SetFilePointer(fd, 0, SEEK_CUR)) < 0)
                                return FALSE;
                        if ( mppUtilSys_SetFilePointer(fd, (maxPaths)*RECORD_LENGTH, SEEK_CUR) < 0)
                                return FALSE;
                        if ( mppUtilSys_WriteFile(fd, "\n", 1) <= 0)
                                return FALSE;
                        ptr = (BYTE *)pathInfo;
                        ptr += (sizeof(mpp_utilPathInfo_t ) * maxPaths);
                        pathInfo = (mpp_utilPathInfo_t *)ptr;

              lun_step: ptr = (BYTE *)lunInfo;
                        ptr += sizeof(mpp_utilLunInfo_t);
                        lunInfo = (mpp_utilLunInfo_t *)ptr;
                }

        }
        return TRUE;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:         mppUtilCmn_PrintArraySummary
* SUMMARY:      Print Real Time Controller and Path Summary for each Array.
* SCOPE:        LOCAL
*
* DESCRIPTION:
*               This function prints out the real time status of the controller and paths
*               on the controller.
*
* RETURNS:
*               void
*******************************************************************************/
void
mppUtilCmn_PrintArraySummary(mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxLuns, LWORD maxPaths, int PathState )
{
        LWORD Lun=0, Path =0;
        CHAR    virtual_target[STR_LENGTH], physical_target[STR_LENGTH], controller_state[2][STR_LENGTH];
        CHAR    buf[UTIL_STATUS_BUF_LENGTH], tmpBuf[UTIL_STATUS_BUF_LENGTH];
        LWORD   width=0 ;
        BYTE    printRow = FALSE;
        BYTE    *ptr = NULL, *lunPtr = NULL, *other_ptr= NULL;
        mpp_utilLunInfo_t   *lunInfo = NULL;
        mpp_utilPathInfo_t  *pathInfo = NULL, *other_pathInfo= NULL;



        if( utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualTargetId != -1 ) {

                mppUtilSys_FormatVirtualTarget(&utilStatusInfo->mpp_PlatformStatusInfo, virtual_target);

                memset(buf, ' ', UTIL_STATUS_BUF_LENGTH);
                memset(tmpBuf, ' ', UTIL_STATUS_BUF_LENGTH);
                width = strlen(virtual_target);
                if(width > VIRTUAL_TARGET_WIDTH)
                        width = VIRTUAL_TARGET_WIDTH;
                memcpy(&buf[VIRTUAL_TARGET_STATUS_COLUMN], virtual_target, width);

                strcpy (controller_state[0], ControllerStates[utilStatusInfo->mpp_ControllerState[0] - 0x0A] );
                width = strlen(controller_state[0]);
                if(width > CONTROLLER_STATUS_WIDTH)
                        width = CONTROLLER_STATUS_WIDTH;
                memcpy(&buf[A_CONTROLLER_STATUS_COLUMN], controller_state[0], width);
                if(PathState == PATH_MISSING) {
                        width = strlen(ControllerStates[0]);
                        memcpy(&buf[A_CONTROLLER_STATUS_COLUMN], ControllerStates[0], width);
                }

                strcpy (controller_state[1], ControllerStates[utilStatusInfo->mpp_ControllerState[1] - 0x0A] );
                width = strlen(controller_state[1]);
                if(width > CONTROLLER_STATUS_WIDTH)
                        width = CONTROLLER_STATUS_WIDTH;
                memcpy(&buf[B_CONTROLLER_STATUS_COLUMN], controller_state[1], width);
                if(PathState == PATH_MISSING) {
                        width = strlen(ControllerStates[0]);
                        memcpy(&buf[B_CONTROLLER_STATUS_COLUMN], ControllerStates[0], width);
                }


                memcpy(&buf[SANAME_COLUMN], utilStatusInfo->mpp_SAName, strlen(utilStatusInfo->mpp_SAName));

                buf[UTIL_STATUS_BUF_LENGTH-1] = '\0';
                printf("%s\n", buf);

                ptr =  (BYTE *)utilStatusInfo + sizeof(mpp_utilStatusInfo_t) ;
                lunInfo = (mpp_utilLunInfo_t *) ptr;
                ptr= ptr + sizeof(mpp_utilLunInfo_t)* maxLuns;
                pathInfo = (mpp_utilPathInfo_t *) ptr;


                for(Lun=0; Lun < maxLuns; ++Lun) {

                        if(lunInfo->mpp_LunNumber == NO_LUN) {

                                ptr = (BYTE *)pathInfo;
                                ptr += (sizeof(mpp_utilPathInfo_t ) * 2*maxPaths) ;
                                pathInfo = (mpp_utilPathInfo_t *)ptr;
                                goto lun_step;
                        }


                        /**
                         **  There are two columns printed, one for each controller.  If information for that controller is available it
                         **  should be printed else that column is left blank.  Printing "Unknown" may cause the user some confusion
                         **  since it might imply a path exists but its status is undetermined.
                         **/
                        for(Path=0; Path < maxPaths; Path++) {

                                ptr = (BYTE *)pathInfo;
                                other_ptr = ptr + sizeof(mpp_utilPathInfo_t ) * maxPaths ;
                                other_pathInfo = (mpp_utilPathInfo_t *) other_ptr;
                                memset(buf, ' ', UTIL_STATUS_BUF_LENGTH);
                                printRow = FALSE;

                                if( pathInfo->mpp_PathState != NO_PATH ) {

                                        memset(physical_target, '\0', STR_LENGTH );
                                        mppUtilSys_FormatPhysicalTarget(&pathInfo->mpp_CtlAddr, physical_target);
                                        sprintf(tmpBuf, "%sL%03d  %s", physical_target, lunInfo->mpp_LunNumber, pathInfo->mpp_PathState == 1 ? UP : DOWN );
                                        if(PathState == PATH_MISSING)
                                                sprintf(tmpBuf, "%sL%03d  %s", physical_target, lunInfo->mpp_LunNumber, ControllerStates[0] );
                                        width = strlen(tmpBuf);
                                        if(width > CONTROLLER_STATUS_WIDTH)
                                                width = CONTROLLER_STATUS_WIDTH;

                                        memcpy(&buf[A_CONTROLLER_STATUS_COLUMN], tmpBuf, width);

                                        printRow = TRUE;
                                }


                                if( other_pathInfo->mpp_PathState != NO_PATH ) {

                                        memset(physical_target, '\0', STR_LENGTH );
                                        mppUtilSys_FormatPhysicalTarget(&other_pathInfo->mpp_CtlAddr, physical_target);
                                        sprintf(tmpBuf, "%sL%03d  %s", physical_target, lunInfo->mpp_LunNumber, other_pathInfo->mpp_PathState == 1 ? UP : DOWN );
                                        if(PathState == PATH_MISSING)
                                                sprintf(tmpBuf, "%sL%03d  %s", physical_target, lunInfo->mpp_LunNumber, ControllerStates[0] );
                                        width = strlen(tmpBuf);
                                        if(width > CONTROLLER_STATUS_WIDTH)
                                                width = CONTROLLER_STATUS_WIDTH;
                                        memcpy(&buf[B_CONTROLLER_STATUS_COLUMN], tmpBuf, width);


                                        printRow = TRUE;
                                }

                                if(printRow)
                                {
                                        buf[UTIL_STATUS_BUF_LENGTH-1] = '\0';
                                        printf("%s\n", buf);
                                }

                                ptr = (BYTE *)pathInfo;
                                ptr += sizeof(mpp_utilPathInfo_t);
                                pathInfo = (mpp_utilPathInfo_t *)ptr;

                        }
                        ptr = (BYTE *)pathInfo;
                        ptr += (sizeof(mpp_utilPathInfo_t ) * maxPaths);
                        pathInfo = (mpp_utilPathInfo_t *)ptr;

              lun_step: ptr = (BYTE *)lunInfo;
                        ptr += sizeof(mpp_utilLunInfo_t);
                        lunInfo = (mpp_utilLunInfo_t *)ptr;
                }

        }

}

/*******************************************************************************
* PROCEDURE
*
* NAME:         mppUtilCmn_ReadArraySummary
* SUMMARY:      Read the Array Information from a File
* SCOPE:        LOCAL
*
* DESCRIPTION:
*       This function reads the Array Information from a .wwn file
*
* RETURNS:
*       Boolean indicating success or failure of the action.
*******************************************************************************/
static BOOL
mppUtilCmn_ReadArraySummary(MPP_FILE_HANDLE fd, mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxPaths )
{
	LWORD  Path =0, maxLuns=0;
	CHAR    virtual_target[STR_LENGTH], physical_target[STR_LENGTH], controller_state[2][STR_LENGTH];
	CHAR    buf[UTIL_STATUS_BUF_LENGTH], tmpBuf[UTIL_STATUS_BUF_LENGTH], temp;
	LWORD   width;
	BYTE    *ptr = NULL, *other_ptr= NULL;
	mpp_utilLunInfo_t   *lunInfo = NULL;
	mpp_utilPathInfo_t  *pathInfo = NULL, *other_pathInfo= NULL;
	INT i=0, LunNo=0, LastLun= 0;
	INT curPos = 0, Lun = 0;
	BOOL first = 1;
	int n;



	if((n = mppUtilSys_ReadFile(fd, buf,VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH + DEVNM_LEN)) > 0) { 

		buf[VIRTUAL_TARGET_WIDTH + 2*CONTROLLER_STATUS_WIDTH + DEVNM_LEN] = '\0';

		memcpy(tmpBuf, &buf[0], VIRTUAL_TARGET_WIDTH);
		tmpBuf[VIRTUAL_TARGET_WIDTH] = '\0';
		mppUtilSys_ReadVirtualTarget(&utilStatusInfo->mpp_PlatformStatusInfo, tmpBuf);

		memcpy(tmpBuf, &buf[VIRTUAL_TARGET_WIDTH], CONTROLLER_STATUS_WIDTH );
		tmpBuf[CONTROLLER_STATUS_WIDTH] = '\0';
		i = 0 ;
		while ( ControllerStates[i] != NULL ) {
			if(strstr(tmpBuf, ControllerStates[i]) != NULL ) {
				utilStatusInfo->mpp_ControllerState[0] = i + 0x0A;
				break;
			}
			i++;
		}

		memcpy(tmpBuf, &buf[VIRTUAL_TARGET_WIDTH + CONTROLLER_STATUS_WIDTH] , CONTROLLER_STATUS_WIDTH );
		tmpBuf[CONTROLLER_STATUS_WIDTH] = '\0';
		i = 0;
		while ( ControllerStates[i] != NULL ) {
			if(strstr(tmpBuf, ControllerStates[i] ) != NULL ) {
				utilStatusInfo->mpp_ControllerState[1] = i + 0x0A;
				break;
			}
			i++;
		}

		memcpy(utilStatusInfo->mpp_SAName, &buf[VIRTUAL_TARGET_WIDTH+1+2*CONTROLLER_STATUS_WIDTH-1], DEVNM_LEN-1);
	}
	else
		return FALSE;


	if (mppUtilSys_ReadFile(fd, buf, RECORD_LENGTH) < 0 )
		return FALSE;
	memcpy(tmpBuf, &buf[0], RECORD_LENGTH);
	tmpBuf[RECORD_LENGTH] = '\0';
	sscanf(tmpBuf, "maxLun%3d", &maxLuns );

	i = 0;
	for(Path=0; Path < maxPaths; Path++) {

		if (mppUtilSys_ReadFile(fd, buf, RECORD_LENGTH) < 0 )
			return FALSE;
		buf[RECORD_LENGTH] = '\0';
		if(strstr(buf, "CAP") != NULL )
			i++;

	}

	if ( i != maxPaths )
		fprintf (stderr,"No of maxPaths has changed \n");

	i = 0;
	for(Path=0; Path < maxPaths; Path++) {

		if (mppUtilSys_ReadFile(fd, buf, RECORD_LENGTH) < 0 )
			return FALSE;
		buf[RECORD_LENGTH] = '\0';
		if(strstr(buf, "CBP") != NULL )
			i++;

	}

	if ( i != maxPaths )
		fprintf (stderr,"No of maxPaths has changed \n");

	if (mppUtilSys_ReadFile(fd, buf, 1) < 0 )
		return FALSE;
	memcpy(tmpBuf, &buf[0], 1);
	tmpBuf[1] = '\0';


	ptr =  (BYTE *)utilStatusInfo + sizeof(mpp_utilStatusInfo_t) ;
	lunInfo = (mpp_utilLunInfo_t *) ptr;
	ptr= ptr + sizeof(mpp_utilLunInfo_t)* maxLuns;
	pathInfo = (mpp_utilPathInfo_t *) ptr;

	LastLun = 0;
	first = 1;

	while(mppUtilSys_ReadFile(fd, buf, RECORD_LENGTH) > 0 ) {

		memcpy(tmpBuf, &buf[0], RECORD_LENGTH);
		tmpBuf[RECORD_LENGTH] = '\0';
		sscanf(tmpBuf, "L%3d     :",&LunNo );

		if( first != 1 ) {
			LastLun = LastLun + 1;
		}
		if( first  == 1 ) {
			first = 0;
		}

		for(Lun=LastLun; Lun <= LunNo ; ++Lun) {

			if( LunNo != Lun ) {
				lunInfo->mpp_LunNumber = NO_LUN;
				ptr = (BYTE *)pathInfo;
				ptr += (sizeof(mpp_utilPathInfo_t ) * 2*maxPaths) ;
				pathInfo = (mpp_utilPathInfo_t *)ptr;
				goto lun_step;
			}
			lunInfo->mpp_LunNumber = LunNo;
			for(Path=0; Path < maxPaths; Path++) {

				ptr = (BYTE *)pathInfo;
				other_ptr = ptr + sizeof(mpp_utilPathInfo_t ) * maxPaths ;
				other_pathInfo = (mpp_utilPathInfo_t *) other_ptr;

				if (mppUtilSys_ReadFile(fd, buf, RECORD_LENGTH) < 0 )
					return FALSE;
				memcpy(tmpBuf, &buf[0], RECORD_LENGTH);
				tmpBuf[RECORD_LENGTH] = '\0';

				if( strcmp(&tmpBuf[0], "---------:") != 0)  {

					mppUtilSys_ReadPhysicalTarget(&pathInfo->mpp_CtlAddr, tmpBuf) ;
					pathInfo->mpp_PathState = PATH_UP;
				}
				else {
					pathInfo->mpp_PathState = NO_PATH;
				}

				if ((curPos = mppUtilSys_SetFilePointer(fd, 0, SEEK_CUR)) < 0 )
					return FALSE;
				if(mppUtilSys_SetFilePointer(fd, (maxPaths-1)*RECORD_LENGTH, SEEK_CUR) < 0)
					return FALSE;
				if (mppUtilSys_ReadFile(fd, buf, RECORD_LENGTH) < 0 )
					return FALSE;
				memcpy(tmpBuf, &buf[0], RECORD_LENGTH);
				tmpBuf[RECORD_LENGTH] = '\0';

				if( strcmp(&tmpBuf[0], "---------:") != 0)  {

					mppUtilSys_ReadPhysicalTarget(&other_pathInfo->mpp_CtlAddr, tmpBuf) ;
					other_pathInfo->mpp_PathState = PATH_UP;
				}
				else {
					other_pathInfo->mpp_PathState = NO_PATH;
				}

				ptr = (BYTE *)pathInfo;
				ptr += sizeof(mpp_utilPathInfo_t);
				pathInfo = (mpp_utilPathInfo_t *)ptr;

				if( mppUtilSys_SetFilePointer(fd, curPos, SEEK_SET) < 0 )
					return FALSE;

			}
			ptr = (BYTE *)pathInfo;
			ptr += (sizeof(mpp_utilPathInfo_t ) * maxPaths);
			pathInfo = (mpp_utilPathInfo_t *)ptr;

			if (mppUtilSys_SetFilePointer(fd, curPos+maxPaths*RECORD_LENGTH, SEEK_SET) < 0)
				return FALSE;
			if (mppUtilSys_ReadFile(fd, buf, 1) < 0 )
				return FALSE;
			memcpy(tmpBuf, &buf[0], 1);
			tmpBuf[1] = '\0';


lun_step:
			ptr = (BYTE *)lunInfo;
			ptr += sizeof(mpp_utilLunInfo_t);
			lunInfo = (mpp_utilLunInfo_t *)ptr;
		}

		LastLun = LunNo;
	}
	return TRUE;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:         mppUtilCmn_InitUtilStatus
* SUMMARY:      Initialize the Array Information data structure
* SCOPE:        LOCAL
*
* DESCRIPTION:
*	This function initializes the data structure for an array to default values
* RETURNS:
*	void
*******************************************************************************/
void
mppUtilCmn_InitUtilStatus(mpp_utilStatusInfo_t *utilStatusInfo, LWORD maxLuns, LWORD maxPaths )
{
        BYTE*   ptr;
        BYTE                    Path;
        LWORD                   Lun;
        ControllerAddress_t DummyCtlAddr = { -1, -1, -1 };
        mpp_utilLunInfo_t   *lunInfo = NULL;
        mpp_utilPathInfo_t  *pathInfo = NULL;

        memset(utilStatusInfo->mpp_SAName, 0, sizeof(DevName_t));
        utilStatusInfo->mpp_ControllerState[0] = MPPCMN_CTLR_UNKNOWN_STATE;
        utilStatusInfo->mpp_ControllerState[1] = MPPCMN_CTLR_UNKNOWN_STATE;

        ptr = (BYTE *)utilStatusInfo + sizeof(mpp_utilStatusInfo_t);
        utilStatusInfo->mpp_utilLunInfoAddress = (mpp_utilLunInfo_t *)ptr;
        lunInfo = (mpp_utilLunInfo_t *) utilStatusInfo->mpp_utilLunInfoAddress;

        ptr= ptr + sizeof(mpp_utilLunInfo_t)* maxLuns;
        pathInfo = (mpp_utilPathInfo_t *) ptr;
        lunInfo->mpp_utilPathInfoAddress = pathInfo;

        utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualTargetId = -1;
        utilStatusInfo->mpp_PlatformStatusInfo.mpp_VirtualHostId = -1;


        for(Lun=0; Lun<maxLuns; ++Lun) {
                lunInfo->mpp_LunNumber = NO_LUN;
                lunInfo->mpp_utilPathInfoAddress = pathInfo;

                for(Path=0; Path < 2 * maxPaths; Path++) {

                        pathInfo->mpp_PathState = NO_PATH;
                        bcopy(&DummyCtlAddr, &pathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                        ptr = (BYTE *)pathInfo;
                        ptr += sizeof(mpp_utilPathInfo_t);
                        pathInfo = (mpp_utilPathInfo_t *)ptr;
                }
                ptr = (BYTE *)lunInfo;
                ptr += sizeof(mpp_utilLunInfo_t);
                lunInfo = (mpp_utilLunInfo_t *)ptr;
        }
}


/*******************************************************************************
* PROCEDURE
*
* NAME:         mppUtilCmn_CompareUtilStatus
* SUMMARY:      Compare two Array Information data structures
* SCOPE:        LOCAL
*
* DESCRIPTION:
*		This function Compares the Util Data Structures read from file and data got back from the driver
* RETURNS:
*       Boolean indicating success or failure of the action.
*******************************************************************************/
BOOL
mppUtilCmn_CompareUtilStatus(mpp_utilStatusInfo_t *utilStatusInfo, mpp_utilStatusInfo_t *utilStatusPtr, LWORD maxLuns, LWORD maxPaths ) {
        BYTE    *ptr = NULL, *other_ptr= NULL, *utilPtr = NULL, *other_utilPtr = NULL;
        mpp_utilLunInfo_t   *lunInfo = NULL, *utilLunInfo = NULL;
        mpp_utilPathInfo_t  *pathInfo = NULL, *utilPathInfo = NULL, *other_utilPathInfo = NULL, *other_pathInfo= NULL;
        LWORD Lun = 0;
        BYTE Path = 0;

           strcpy(utilStatusPtr->mpp_SAName, utilStatusInfo->mpp_SAName);
           utilStatusPtr->mpp_ControllerState[0] = utilStatusInfo->mpp_ControllerState[0];
           utilStatusPtr->mpp_ControllerState[1] = utilStatusInfo->mpp_ControllerState[1];
           bcopy( &utilStatusInfo->mpp_PlatformStatusInfo, &utilStatusPtr->mpp_PlatformStatusInfo, sizeof(mpp_utilSysStatusInfo_t));

           ptr =  (BYTE *)utilStatusInfo + sizeof(mpp_utilStatusInfo_t) ;
           lunInfo = (mpp_utilLunInfo_t *) ptr;
           ptr= ptr + sizeof(mpp_utilLunInfo_t)* maxLuns;
           pathInfo = (mpp_utilPathInfo_t *) ptr;

           utilPtr =  (BYTE *)utilStatusPtr + sizeof(mpp_utilStatusInfo_t) ;
           utilLunInfo = (mpp_utilLunInfo_t *) utilPtr;
           utilPtr= utilPtr + sizeof(mpp_utilLunInfo_t)* maxLuns;
           utilPathInfo = (mpp_utilPathInfo_t *) utilPtr;

           for(Lun= 0; Lun < maxLuns ; ++Lun) {

                if( lunInfo->mpp_LunNumber == utilLunInfo->mpp_LunNumber && lunInfo->mpp_LunNumber == NO_LUN && utilLunInfo->mpp_LunNumber == NO_LUN ) {

                     ptr = (BYTE *)pathInfo;
                     ptr += (sizeof(mpp_utilPathInfo_t ) * 2*maxPaths) ;
                     pathInfo = (mpp_utilPathInfo_t *)ptr;

                     utilPtr = (BYTE *)utilPathInfo;
                     utilPtr += (sizeof(mpp_utilPathInfo_t ) * 2*maxPaths) ;
                     utilPathInfo = (mpp_utilPathInfo_t *)utilPtr;
                     goto lun_step;
                }
                 for(Path=0; Path < maxPaths; Path++) {

                          ptr = (BYTE *)pathInfo;
                          other_ptr = ptr + sizeof(mpp_utilPathInfo_t ) * maxPaths ;
                          other_pathInfo = (mpp_utilPathInfo_t *) other_ptr;

                          utilPtr = (BYTE *)utilPathInfo;
                          other_utilPtr = utilPtr + sizeof(mpp_utilPathInfo_t ) * maxPaths ;
                          other_utilPathInfo = (mpp_utilPathInfo_t *) other_utilPtr;


                          if ( lunInfo->mpp_LunNumber == NO_LUN && utilLunInfo->mpp_LunNumber != NO_LUN ) {

                                if ( utilPathInfo->mpp_PathState != NO_PATH ) {
                                        utilPathInfo->mpp_PathState = PATH_DOWN;
                                }
                                if ( other_utilPathInfo->mpp_PathState != NO_PATH ) {
                                        other_utilPathInfo->mpp_PathState = PATH_DOWN;
                                }
                                goto path_step;

                          }

                          if ( lunInfo->mpp_LunNumber != NO_LUN && utilLunInfo->mpp_LunNumber == NO_LUN ) {

                                utilLunInfo->mpp_LunNumber = lunInfo->mpp_LunNumber;

                                if ( pathInfo->mpp_PathState != NO_PATH ) {

                                        bcopy(&pathInfo->mpp_CtlAddr, &utilPathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                                        utilPathInfo->mpp_PathState = pathInfo->mpp_PathState;

                                }

                                if ( other_pathInfo->mpp_PathState != NO_PATH ) {
                                        bcopy(&other_pathInfo->mpp_CtlAddr, &other_utilPathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                                        other_utilPathInfo->mpp_PathState = other_pathInfo->mpp_PathState;

                                }
                                goto path_step;

                          }


                          if (  lunInfo->mpp_LunNumber != NO_LUN && utilLunInfo->mpp_LunNumber != NO_LUN ) {
                                if ( pathInfo->mpp_PathState == NO_PATH  && utilPathInfo->mpp_PathState != NO_PATH) {
                                        utilPathInfo->mpp_PathState = PATH_DOWN;

                                }
                                if ( other_pathInfo->mpp_PathState == NO_PATH  && other_utilPathInfo->mpp_PathState != NO_PATH) {
                                        other_utilPathInfo->mpp_PathState = PATH_DOWN;

                                }

                                if ( pathInfo->mpp_PathState != NO_PATH  && utilPathInfo->mpp_PathState == NO_PATH) {

                                        bcopy(&pathInfo->mpp_CtlAddr, &utilPathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                                        utilPathInfo->mpp_PathState = pathInfo->mpp_PathState;

                                }
                                if ( other_pathInfo->mpp_PathState != NO_PATH  && other_utilPathInfo->mpp_PathState == NO_PATH) {
                                        bcopy(&other_pathInfo->mpp_CtlAddr, &other_utilPathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                                        other_utilPathInfo->mpp_PathState = other_pathInfo->mpp_PathState;

                                }
                                if ( mppUtilSys_CompareCtlAddr(&utilPathInfo->mpp_CtlAddr, &pathInfo->mpp_CtlAddr) == TRUE ) {
                                        utilPathInfo->mpp_PathState = pathInfo->mpp_PathState;
                                }
                                if ( mppUtilSys_CompareCtlAddr(&other_utilPathInfo->mpp_CtlAddr, &other_pathInfo->mpp_CtlAddr) == TRUE ) {
                                        other_utilPathInfo->mpp_PathState = other_pathInfo->mpp_PathState;
                                }

                                if ( mppUtilSys_CompareCtlAddr(&utilPathInfo->mpp_CtlAddr, &pathInfo->mpp_CtlAddr) == FALSE ) {
                                   if ( pathInfo->mpp_PathState != NO_PATH ) {

                                        bcopy(&pathInfo->mpp_CtlAddr, &utilPathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                                        utilPathInfo->mpp_PathState = PATH_UP;


                                   }

                                }

                                if ( mppUtilSys_CompareCtlAddr(&other_utilPathInfo->mpp_CtlAddr, &other_pathInfo->mpp_CtlAddr) == FALSE ) {
                                  if ( other_pathInfo->mpp_PathState != NO_PATH ) {
                                        bcopy(&other_pathInfo->mpp_CtlAddr, &other_utilPathInfo->mpp_CtlAddr, sizeof(ControllerAddress_t));
                                        other_utilPathInfo->mpp_PathState = PATH_UP;

                                   }

                                }
                           }
  	path_step:
                  	   ptr = (BYTE *)pathInfo;
                  	   ptr += sizeof(mpp_utilPathInfo_t);
                  	   pathInfo = (mpp_utilPathInfo_t *)ptr;

                  	   utilPtr = (BYTE *)utilPathInfo;
	                   utilPtr += sizeof(mpp_utilPathInfo_t);
         	           utilPathInfo = (mpp_utilPathInfo_t *)utilPtr;
            	   }

            	   ptr = (BYTE *)pathInfo;
            	   ptr += sizeof(mpp_utilPathInfo_t) * maxPaths;
	           pathInfo = (mpp_utilPathInfo_t *)ptr;

         	   utilPtr = (BYTE *)utilPathInfo;
            	   utilPtr += sizeof(mpp_utilPathInfo_t) * maxPaths;
            	   utilPathInfo = (mpp_utilPathInfo_t *)utilPtr;

  	lun_step:  ptr = (BYTE *)lunInfo;
            	   ptr += sizeof(mpp_utilLunInfo_t ) ;
            	   lunInfo = (mpp_utilLunInfo_t *)ptr;

            	   utilPtr = (BYTE *)utilLunInfo;
            	   utilPtr += sizeof(mpp_utilLunInfo_t );
            	   utilLunInfo = (mpp_utilLunInfo_t *)utilPtr;

        }
        return TRUE;

}



/*******************************************************************************
* PROCEDURE
*
* NAME:		utilGetRequest
* SUMMARY:	Get the RDAC information for a given target.
* SCOPE:	LOCAL
*
* DESCRIPTION:
*	This function sends a get request to the MPP virtual bus.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilGetRequest(LWORD target)
{
	RdacDeviceInformation_t *reqInfo = NULL;
	unsigned char			reqName[MAX_ARRAY_STR_LEN];
	getRdacInfoIoctl_t		rdacInfo;
	LWORD					Lun;
	BYTE					*reqPtr = NULL;

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	
	utilPrintHeader();

	/* Get maxluns supported */
	if ( utilMaxLuns == -1 ) 
	{
        	if( ioctl(utilVirtualFd, MPP_IOCTL_GETLUNS, &utilMaxLuns) == INVALID)
        	{
                	fprintf(stderr, "Unable to get maxluns request - %s!\n", strerror(errno));
                	return(FALSE);
        	}
	}

	/* Get maxpaths supported */
	if ( utilMaxPaths == -1 )
	{
        	if( ioctl(utilVirtualFd, MPP_IOCTL_GETPATHS, &utilMaxPaths) == INVALID)
        	{
                	fprintf(stderr, "Unable to get maxpaths request - %s!\n", strerror(errno));
                	return(FALSE);
        	}
	}

	utilDebug("target number = %d\n",target);

	if((reqInfo = (RdacDeviceInformation_t *)malloc(MPP_RDACINFOSIZE(utilMaxLuns, utilMaxPaths))) == NULL)
	{
		fprintf(stderr, "Unable to malloc memory for RdacInfo - %s!\n", strerror(errno));
		return(FALSE);
	}

	rdacInfo.target = target;
	rdacInfo.BufferSize = MPP_RDACINFOSIZE(utilMaxLuns, utilMaxPaths);
	rdacInfo.RdacInfoAdrs = reqInfo;
	rdacInfo.RdacName = &reqName;

	utilDebug("Send RdacInfo request\n");
	if(ioctl(utilVirtualFd, MPP_IOCTL_GETRDACINFO, &rdacInfo) == INVALID)
	{
		fprintf(stderr, "Unable to get rdacinfo request - %s!\n", strerror(errno));
		return(FALSE);
	}

	reqPtr = (BYTE *)reqInfo;
	reqPtr = reqPtr + sizeof(RdacDeviceInformation_t) +
		(sizeof(RdacLunInformation_t) * (utilMaxLuns - 1));
	reqInfo->ControllerInfo[0] = (RdacControllerInformation_t *)reqPtr;

	reqPtr = (reqPtr + sizeof(RdacControllerInformation_t) +
		(sizeof(RdacControllerPath_t) * (utilMaxPaths - 1)));
	reqInfo->ControllerInfo[1] = (RdacControllerInformation_t *)reqPtr;

	reqPtr = reqPtr + sizeof(RdacControllerInformation_t) +
		(sizeof(RdacControllerPath_t) * (utilMaxPaths - 1));
	for(Lun = 0; Lun < utilMaxLuns; ++Lun)
	{
		reqInfo->RdacLuns[Lun].LunControllers[0] = (LunPathObjects_t *)reqPtr;

		reqPtr = reqPtr + sizeof(LunPathObjects_t) +
			(sizeof(LunPathInfo_t) * (utilMaxPaths - 1));
		reqInfo->RdacLuns[Lun].LunControllers[1] = (LunPathObjects_t *)reqPtr;

		reqPtr = reqPtr + sizeof(LunPathObjects_t) +
			(sizeof(LunPathInfo_t) * (utilMaxPaths - 1));
	}

	labelString = (MPP_STRING) 0;
	labelLength = 0;
	utilPrintRdacInfo(&rdacInfo);

	/* free the memory allocated to req info */	
	free(reqInfo);

	return(TRUE);
}

static VOID
utilPrintRdacInfo(getRdacInfoIoctl_t *getInfo)
{
	int				controllerCount;
	LWORD				lunCount, pathCount;
	unsigned char			nullWWN[WWN_LENGTH];
	RdacDeviceInformation_t 	*rdacInfo;
	RdacControllerInformation_t	*controllerInfo;
	RdacControllerPath_t		*controllerPath;
	RdacLunInformation_t		*lunInfo;
	LunPathObjects_t		*lunPath;
	LunPathInfo_t			*lunPathInfo;
	char				strBuf[STR_LENGTH], strBuf2[STR_LENGTH];
	char				timeBuf[TIME_BUFFER_LENGTH];
	char				ctlAddressStrBuf[50] = "";

	rdacInfo = getInfo->RdacInfoAdrs;

	memset(nullWWN, '\0', WWN_LENGTH);

	/**
	 ** Global array information
	 **/
	printf("MPP Information:\n");
	printf("----------------\n");

	printf("      ModuleName: %-32.32s         SingleController: %c\n",
		getInfo->RdacName, YES_NO(rdacInfo->SingleController));
	printf(" VirtualTargetID: 0x%03x                                       ScanTriggered: %c\n",
		rdacInfo->VirtualTargetId, YES_NO(rdacInfo->ScanTriggered));
	printf("     ObjectCount: 0x%03x                                          AVTEnabled: %c\n",
		rdacInfo->ObjectCount, YES_NO(rdacInfo->AVTEnabled));
	printf("             WWN: ");
	utilHexDump(rdacInfo->WWN, WWN_LENGTH);
	printf("               RestoreCfg: %c\n", YES_NO(rdacInfo->RestoreCfg));
	utilFormatHandle(rdacInfo->ModuleHandle, strBuf);
	printf("    ModuleHandle: %-18.18s                          Page2CSubPage: %c\n",
		strBuf, YES_NO(rdacInfo->Page2CSubPage1));
	memset(strBuf, '\0', STR_LENGTH);

        /**
	 **CR~117078 As VPD page only giving 3 BYTE  of info Displaying 4th byte as "xx"
         **/
        sprintf(strBuf, "%x.%x.%x.%s", rdacInfo->FirmwareVersion[0], rdacInfo->FirmwareVersion[1], rdacInfo->FirmwareVersion[2],"xx");
	printf(" FirmwareVersion: %-18.18s                          \n",strBuf);
	printf("   ScanTaskState: 0x%08x\n", rdacInfo->ScanTaskState);
	switch(rdacInfo->LoadBalancePolicy) //77 util
	{
		case MPPCMN_LB_ROUND_ROBIN_SUBSET:
			printf("        LBPolicy: %s\n", MPP_LBP_ROUND_ROBIN_WITH_SUBSET); //77 util
			break;
		case MPPCMN_LB_LEAST_QUEUE_DEPTH:
			printf("        LBPolicy: %s\n", MPP_LBP_LEAST_QUEUE_DEPTH); //77 util	    
			break;
		case MPPCMN_LB_PATH_WEIGHT:
			printf("        LBPolicy: %s\n", MPP_LBP_LEAST_PATH_WEIGHT); //77 util	    
			break;
		default:
			printf("        LBPolicy: UnKnown\n"); //77 util	    
			break;	      	    	    
	}
	
    printf("\n");

	/**
	 ** Controller-specific information
	 **/
	for(controllerCount = 0; controllerCount < 2; controllerCount++)
	{
		controllerInfo = rdacInfo->ControllerInfo[controllerCount];
		if(!controllerInfo->ControllerPresent)
		{
			continue;
		}

		printf("\nController '%c' Status:\n", controllerCount + 'A');
		printf("-----------------------\n");
		utilFormatHandle(controllerInfo->ControllerSlotHandle, strBuf);
		printf("ControllerHandle: %-18.18s                      ControllerPresent: %c\n",
			strBuf, YES_NO(controllerInfo->ControllerPresent));
		if(controllerInfo->UTMLunExists)
		{
			printf("    UTMLunExists: %c (%-3.3d)                                            Failed: %c\n",
				YES_NO(controllerInfo->UTMLunExists), controllerInfo->UTMLunNumber,
				YES_NO(controllerInfo->Failed));
		}
		else
		{
			printf("    UTMLunExists: %c                                                  Failed: %c\n",
				YES_NO(controllerInfo->UTMLunExists), YES_NO(controllerInfo->Failed));
		}
		printf("   NumberOfPaths: %1d                                          FailoverInProg: %c\n",
			controllerInfo->NumberOfPaths, YES_NO(controllerInfo->FailoverInProgress));
		printf("                                                                ServiceMode: %c\n",
			YES_NO(controllerInfo->ServiceMode));

		// for(pathCount = 0; pathCount < controllerInfo->NumberOfPaths; pathCount++)
		// 08/23/04 - Changed NumberOfLunObjects to utilMaxPaths.  NumberOfLunObjects reflects the current number
		// of lun paths seen on a given controller.  Under Windows PnP a given controller may not have the same number of paths
		// as another controller on the same array.  This is similar to the change made on 08/07/03 for lun paths.
		for(pathCount = 0; pathCount < utilMaxPaths; pathCount++)
		{
			controllerPath = &controllerInfo->ControllerPath[pathCount];
			if(controllerPath->Present == 0)
			{
				continue;
			}

			printf("\n    Path #%1d\n", pathCount + 1);
			printf("    ---------\n");
			utilFormatVertex(controllerPath->DirectoryVertex, strBuf);
			printf(" DirectoryVertex: %-18.18s                                Present: %c\n", 
				strBuf, YES_NO(controllerPath->Present));
			printf("       PathState: %-21.21s\n", utilFormatDevState(controllerPath->PathState[controllerPath->PathStateIndex]));
			utilFormatControllerAddress(controllerPath->Address, ctlAddressStrBuf, controllerPath->PathId);
			printf("          PathId: %s\n", ctlAddressStrBuf);
		}

		printf("\n");
	}

	/**
	 ** Lun-specific information
	 **/

	printf("\n\nLun Information\n");
	printf("---------------\n");
	for(lunCount = 0; lunCount < utilMaxLuns; lunCount++)
	{
		lunInfo = &rdacInfo->RdacLuns[lunCount];

		if(!memcmp(nullWWN, lunInfo->WWN, WWN_LENGTH))
		{
			continue;
		}

		if(lunCount != 0)
		{
			printf("\n");
		}

		if(lunInfo->WWN[0] != 0)
		{
			printf("    Lun #%d - WWN: ", lunCount);
			utilHexDump(lunInfo->WWN, WWN_LENGTH);
			printf("\n");
		}
		else
		{
			printf("    Lun #%d - WWN:\n", lunCount);
		}

		printf("    ----------------\n");
		utilFormatVertex(lunInfo->PseudoLunObject, strBuf),
		printf("       LunObject: %-18.18s                      CurrentOwningPath: %c\n", strBuf,
			A_OR_B(lunInfo->CurrentOwningPath));
		printf("  RemoveEligible: %c                                          BootOwningPath: %c\n",
			YES_NO(lunInfo->PseudoRemoveEligible), A_OR_B(lunInfo->BootOwningPath));
		printf("   NotConfigured: %c                                           PreferredPath: %c\n",
			YES_NO(lunInfo->NotConfigured), A_OR_B(lunInfo->PreferredPath));
		printf("        DevState: %-21.21s                     ReportedPresent: %c\n",
			utilFormatDevState(lunInfo->PseudoDevState[lunInfo->PseudoDevStateIndex]), YES_NO(lunInfo->ReportedPresent));
		printf("                                                            ReportedMissing: %c\n", YES_NO(lunInfo->ReportedMissing));
		printf("                                                      NeedsReservationCheck: %c\n",
			YES_NO(lunInfo->NeedsReservationCheck));
		printf("                                                                  TASBitSet: %c\n",
			YES_NO(lunInfo->TASBitSet));

		if(lunInfo->NotReadyStart)
		{
			utilFormatTime(lunInfo->NotReadyStart, timeBuf);
			printf("  NotReady Start: %-19.19s                               NotReady: %c\n",
				timeBuf, YES_NO(lunInfo->NotReady));
		}
		else
		{
			printf("                                                                   NotReady: %c\n",
				YES_NO(lunInfo->NotReady));
		}

		if(lunInfo->BusyStart)
		{
			utilFormatTime(lunInfo->BusyStart, timeBuf);
			printf("      Busy Start: %-19.19s                                     Busy: %c\n",
				timeBuf, YES_NO(lunInfo->Busy));
		}
		else
		{
			printf("                                                                       Busy: %c\n",
				YES_NO(lunInfo->Busy));
		}

		if(lunInfo->QuiescienceStart)
		{
			utilFormatTime(lunInfo->QuiescienceStart, timeBuf);
			printf("Quiescience Start: %-19.19s                            Quiescent: %c\n\n",
				timeBuf, YES_NO(lunInfo->Quiescent));
		}
		else
		{
			printf("                                                                  Quiescent: %c\n\n", YES_NO(lunInfo->Quiescent));
		}

		for(controllerCount = 0; controllerCount < 2; controllerCount++)
		{
			if(!lunInfo->LunControllers[controllerCount])
			{
				continue;
			}

			lunPath = lunInfo->LunControllers[controllerCount];
			printf("    Controller '%c' Path\n", controllerCount + 'A');
			printf("    --------------------\n");
			printf("   NumLunObjects: %1d                                         RoundRobinIndex: %1d\n",
				lunPath->NumberOfLunObjects, lunPath->RoundRobinPathIndex);

			//for(pathCount = 0; pathCount < lunPath->NumberOfLunObjects; pathCount++)
			// 08/07/03 - Changed NumberOfLunObjects to utilMaxPaths.  NumberOfLunObjects reflects the current number
			// of lun paths seen for a given lun.  Under Windows PnP a given lun may not have the same number of lun
			// paths as another lun.  For example, Lun 0 may have 4 paths to each controller but Lun 1 may only have
			// 3 paths (due to some problem with that lun on some path).  As a result, Lun 3 may have paths 1,2, and 4
			// defined.  If NumberOfLunObjects is 3 it will only print the first 3 slots of the LunPaths[] structure
			// (ie. 1,2, and 3).  Since path 4 doesn't print the user is not aware another path exists.  Changing the
			// logic to use MaxPathsPerController, and checking for UtmLunDevice or LunPathDevice is a better alternative.
			for(pathCount = 0; pathCount < utilMaxPaths; pathCount++)
			{
				lunPathInfo = &lunPath->LunPaths[pathCount];
				if(!(lunPathInfo->UtmLunDevice || lunPathInfo->LunPathDevice))
				{
					continue;
				}

				utilFormatVertex(lunPathInfo->LunPathDevice, strBuf);

                if(lunPathInfo->UtmLunDevice != 0)
				{
					utilFormatVertex(lunPathInfo->UtmLunDevice, strBuf2),
					printf("         Path #%1d: LunPathDevice: %-18.18s            UTMLunDevice: %-18.18s\n",
                           pathCount + 1, strBuf, strBuf2);
				}
				else
				{
					printf("         Path #%1d: LunPathDevice: %-18.18s\n",
						pathCount + 1, strBuf);
				}

				printf("                       DevState: %-21.21s\n",
					utilFormatDevState(lunPathInfo->DeviceState[lunPathInfo->DeviceStateIndex]));
				printf("                    RemoveState: 0x%x  StartState: 0x%x  PowerState: 0x%x\n",
					lunPathInfo->PathRemoveState, lunPathInfo->PathStartState, lunPathInfo->PathPowerState);
			}
			printf("\n");
		}
	}
}

static BOOL
utilPrintLunPathCheckList(char *array_wwn, int controllerIndex)
{
	RdacDeviceInformation_t	*reqInfo = NULL;
	RdacLunInformation_t	*lunInfo;
	LunPathObjects_t	*lunPath;
	LunPathInfo_t		*lunPathInfo;
	MPPCMN_PATH_STATE	pathState;
	unsigned char		reqName[MAX_ARRAY_STR_LEN];
	unsigned char		nullWWN[WWN_LENGTH];
	getRdacInfoIoctl_t	rdacInfo;
	LWORD			lunCount, pathCount;
	LWORD			totalPathCount;
	BOOL			found = FALSE;
	LINT			retVal;
	BYTE			*reqPtr = NULL;
	char			wwn[(WWN_LENGTH * 2) + 1];
	LWORD			i, j, firstTime;

	firstTime = 1;
	for(;;)
	{
		if(firstTime)
		{
			retVal = utilGetBusFd(FIRST_DRIVER);
			firstTime = 0;
		}
		else
		{
			retVal = utilGetBusFd(NEXT_DRIVER);
		}
		
		if(retVal == DONE)
		{
			// No more multipath drivers found.
			return(TRUE);
		}
		else if(retVal == FALSE)
		{
			// A problem was found getting information for this driver.  Try the next one.
			continue;
		}

		if(utilMaxLuns == -1)
		{
			if((ioctl(utilVirtualFd, MPP_IOCTL_GETLUNS, &utilMaxLuns)) == INVALID)
			{
				fprintf(stderr, "Unable to get maxluns request - %s!\n", strerror(errno));
				continue;
			}
		}

		if(utilMaxPaths == -1)
		{
			if((ioctl(utilVirtualFd, MPP_IOCTL_GETPATHS, &utilMaxPaths)) == INVALID)
			{
				fprintf(stderr, "Unable to get maxpaths request - %s!\n", strerror(errno));
				continue;
			}
		}

		if ( utilMaxArrayModules == -1 ) 
		{
			if((ioctl(utilVirtualFd, MPP_IOCTL_GETMODULES, &utilMaxArrayModules)) == INVALID)
			{
				fprintf(stderr, "Unable to get maxmodules request - %s!\n", strerror(errno));
				continue;
			}
		}

		if((reqInfo = (RdacDeviceInformation_t *)malloc(MPP_RDACINFOSIZE(utilMaxLuns, utilMaxPaths))) == NULL)
		{
			fprintf(stderr, "Unable to malloc memory for RdacInfo - %s!\n", strerror(errno));
			continue;
		}

		memset(nullWWN, '\0', WWN_LENGTH);
		for(i=0; i<utilMaxArrayModules; i++)
		{
			rdacInfo.target = i;
			rdacInfo.BufferSize = MPP_RDACINFOSIZE(utilMaxLuns, utilMaxPaths);
			rdacInfo.RdacInfoAdrs = reqInfo;
			rdacInfo.RdacName = &reqName[0];
	
			utilDebug("Send RdacInfo request to target %d.\n", i);
			if(ioctl(utilVirtualFd, MPP_IOCTL_GETRDACINFO, &rdacInfo) == INVALID)
			{
				utilDebug("Unable to send ioctl request - %s!\n", strerror(errno));
				continue;
			}
	
			// Convert the binary WWN value to a string for comparison.
			memset(wwn, '\0', (WWN_LENGTH * 2) + 1);
			for(j=0; j < WWN_LENGTH; j++)
				(VOID)sprintf(&wwn[j*2], "%02x", reqInfo->WWN[j]);
	
			if(!memcmp(array_wwn, wwn, WWN_LENGTH))
			{
				found = TRUE;
				break; // utilMaxArrayModules loop
			}
		}

		if(found == TRUE)
		{
			break; // GetBusFd() loop
		}

		// Reset variables before starting the loop again.  These may have different values in another
		// multipath driver.
		utilMaxLuns = -1;
		utilMaxPaths = -1;
		utilMaxArrayModules = -1;
	}

	if(found == TRUE)
	{
		// Setup up rdacInfo pointers before parsing the results.
		reqPtr = (BYTE *)reqInfo;
		reqPtr = reqPtr + sizeof(RdacDeviceInformation_t) +
			(sizeof(RdacLunInformation_t) * (utilMaxLuns - 1));
		reqInfo->ControllerInfo[0] = (RdacControllerInformation_t *)reqPtr;

		reqPtr = (reqPtr + sizeof(RdacControllerInformation_t) +
			(sizeof(RdacControllerPath_t) * (utilMaxPaths - 1)));
		reqInfo->ControllerInfo[1] = (RdacControllerInformation_t *)reqPtr;

		reqPtr = reqPtr + sizeof(RdacControllerInformation_t) +
			(sizeof(RdacControllerPath_t) * (utilMaxPaths - 1));
		for(lunCount = 0; lunCount < utilMaxLuns; ++lunCount)
		{
			reqInfo->RdacLuns[lunCount].LunControllers[0] = (LunPathObjects_t *)reqPtr;

			reqPtr = reqPtr + sizeof(LunPathObjects_t) +
				(sizeof(LunPathInfo_t) * (utilMaxPaths - 1));
			reqInfo->RdacLuns[lunCount].LunControllers[1] = (LunPathObjects_t *)reqPtr;

			reqPtr = reqPtr + sizeof(LunPathObjects_t) +
				(sizeof(LunPathInfo_t) * (utilMaxPaths - 1));
		}

		// Now print any results.
		for(lunCount = 0; lunCount < utilMaxLuns; ++lunCount)
		{
			lunInfo = &reqInfo->RdacLuns[lunCount];
			if(!memcmp(nullWWN, lunInfo->WWN, WWN_LENGTH))
			{
				continue;
			}

			if(!lunInfo->LunControllers[controllerIndex])
			{
				continue;
			}

			lunPath = lunInfo->LunControllers[controllerIndex];

			totalPathCount = 0;
			for(pathCount = 0; pathCount < utilMaxPaths; pathCount++)
			{
				lunPathInfo = &lunPath->LunPaths[pathCount];
				if(!(lunPathInfo->UtmLunDevice || lunPathInfo->LunPathDevice))
				{
					continue;
				}

				pathState = lunPathInfo->DeviceState[lunPathInfo->DeviceStateIndex];
				if((pathState == MPPCMN_OPTIMAL) || (pathState == MPPCMN_OPTIMAL_NEED_CHECK) ||
				   (pathState == MPPCMN_OPTIMAL_CHECKING))
				{
					totalPathCount++;
				}
			}

			if(totalPathCount != 0)
			{
				utilHexDump(lunInfo->WWN, WWN_LENGTH);
				printf(",%d\n", totalPathCount);
			}
		}
	}

	return(TRUE);
}

static BOOL
utilGetMaxArrayModules( ) 
{
	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	utilDebug("sending ioctl for maxArrayModules \n");
	if(ioctl(utilVirtualFd, MPP_IOCTL_GETMODULES, &utilMaxArrayModules) == INVALID)
	{
		fprintf(stderr, "Unable to get max Array modules request - %s!\n", strerror(errno));
		return(FALSE);
	}
	fprintf(stdout,"MaxArrayModules = %d\n", utilMaxArrayModules);
	return(TRUE);

}

static BOOL
utilGetMaxLunsPerArray( ) 
{
	LWORD maxLunsPerArray = 0;
	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	utilDebug("sending ioctl for maxLunsPerArray\n");
	if(ioctl(utilVirtualFd, MPP_IOCTL_GETLUNS, &maxLunsPerArray) == INVALID)
	{
		fprintf(stderr, "Unable to get maxluns request - %s!\n", strerror(errno));
		return(FALSE);
	}
	fprintf(stdout,"MaxLunsPerArray = %d\n", maxLunsPerArray);
	return(TRUE);

}



VOID
utilFormatTime(MPP_TIME rawTime, MPP_STRING buffer)
{
	tm_t	*tmPtr;

	memset(buffer, '\0', TIME_BUFFER_LENGTH);

	tmPtr = gmtime((time_t *) &rawTime);

	if(tmPtr)
	{
		sprintf(buffer, "%02d/%02d/%04d %02d:%02d:%02d",
			tmPtr->tm_mon+1, tmPtr->tm_mday, tmPtr->tm_year + 1900, 
			tmPtr->tm_hour, tmPtr->tm_min, tmPtr->tm_sec);
	}
	else
		sprintf(buffer, "Format error");

	return;
}


MPP_STRING
utilFormatDevState(MPPCMN_PATH_STATE state)
{
	switch(state)
	{
		case	MPPCMN_OPTIMAL:
			return("OPTIMAL");
			break;

		case	MPPCMN_FAILED:
			return("FAILED");
			break;

		case	MPPCMN_OPTIMAL_NEED_CHECK:
			return("OPTIMAL_NEED_CHECK");
			break;

		case	MPPCMN_OPTIMAL_CHECKING:
			return("OPTIMAL_CHECKING");
			break;

		case	MPPCMN_FAILED_NEED_CHECK:
			return("FAILED_NEED_CHECK");
			break;

		case	MPPCMN_FAILED_CHECKING:
			return("FAILED_CHECKING");
			break;

		case	MPPCMN_SYSTEM_SPECIFIC:
			return("SYSTEM_SPECIFIC");
			break;

		default:
			return("UNKNOWN");
	}
}


static VOID
utilHexDump(unsigned char *buffer, int bufferLength)
{
	int		cnt;

	for(cnt = 0; cnt < bufferLength; cnt++)
		printf("%02x", buffer[cnt]);
}

static VOID
utilHexString(unsigned char *buf, unsigned char *buffer, int bufferLength)
{
        int             cnt, inc=0;

        for(cnt = 0; cnt < bufferLength; cnt++) {
                sprintf(&buf[inc], "%02x", buffer[cnt]);
                inc = inc +2;
        }
}



TINY	debugBuffer[DEBUG_BUFFER_LENGTH];

/*******************************************************************************
* PROCEDURE
*
* NAME:		utilDebug
* SUMMARY:	Display debug message.
* SCOPE:	GLOBAL
*
* DESCRIPTION:
*	Display debug message.
*
* RETURNS:
*	N/A
*******************************************************************************/
VOID
utilDebug(TINY *debugMessage, ...)
{
	va_list	ap;

	if(utilDebugEnabled)
	{
		va_start(ap, debugMessage);
		vsprintf(debugBuffer, debugMessage, ap);
		va_end(ap);
		fprintf(stderr, debugBuffer);
	}
	return;
}

static BOOL
isHex(CHAR *arg) 
{
	BOOL status = FALSE;

	if( ! arg)
		return status;

	// See if the user began the debug level with '0x'
	if(!strncmp(arg, "0x", 2) || !strncmp(arg, "0X", 2) )
	{
		arg += 2;
	}
	else
	{
		fprintf(stderr, "Did not begin value with 0x -should be a hexadecimal value \n");
	}
	while(arg && *arg && isxdigit(*arg))
	{
		arg++;
	}
	
	if ( *arg == '\0' ) {
		status = TRUE;	
		return status;
	}

	if ( !isxdigit(*arg))
		status = FALSE;

	return status;
}

BOOL
isNum(MPP_STRING arg) 
{
	BOOL status = FALSE;

	if( ! arg)
		return status;

	while(arg && *arg && isdigit(*arg))
	{
		arg++;
	}
	
	if (*arg == '\0' ) {
		status = TRUE;	
		return status;
	}

	if ( !isdigit(*arg))
		status = FALSE;

	return status;
}

BOOL 
isString(MPP_STRING arg)
{
        BOOL status  = FALSE;
        
        if ( !arg)
                return FALSE;

        if ( !isNum(arg) )
                return TRUE;

        return status;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:		utilProcessFeatureOptions
* SUMMARY:	Mainline function for Feature Option processing.
* SCOPE:	LOCAL
*
* DESCRIPTION:
* 	This is the "mainline" function for processing the Feature Option
* 	argument ('-o').  It builds an MppCmn_OptionControlIoctl_t structure
* 	and calls support functions to build an array of MppCmn_OptionControl_t
* 	structures if necessary.  The function then sends the request to the
* 	MPP driver and analyzes the results, printing status information if
* 	necessary.
*
* RETURNS:
*	Boolean indicating success or failure.
*******************************************************************************/
static BOOL
utilProcessFeatureOptions(char *args)
{
	MppCmn_OptionControlIoctl_t	optionControlIoctl;
	MppCmn_OptionControl_t		*optionControl;
	LWORD				bufferSize;
	LWORD				numOptions;
	LWORD				optionCount;

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(FALSE);
	}

	if(args == NULL)
	{
		utilDebug("No input arguments\n");
		memset(&optionControlIoctl, '\0', sizeof(MppCmn_OptionControlIoctl_t));
		optionControlIoctl.Magic = MPPCMN_FEATURE_OPTION_MAGIC;
		optionControlIoctl.RequestType = MPPCMN_OPTION_REQUEST_GET;

		// other parameters are set to zero or NULL, so there's no need to explicitly
		// set them here

		optionControl = NULL;
		while(TRUE)
		{
			utilDebug("Send FeatureOption request\n");
			if(ioctl(utilVirtualFd, MPP_IOCTL_OPTIONCONTROL, &optionControlIoctl) == INVALID)
			{
				utilDebug("Request failed\n");
				// The request failed.
				return(FALSE);

			}
			else
			{
				// Request successful.  Check the TotalBytesNeeded, and if not zero
				// allocate the requested size and retry.
				utilDebug("Magic: 0x%x\n", optionControlIoctl.Magic);
				utilDebug("RequestType: 0x%x\n", optionControlIoctl.RequestType);
				utilDebug("BufferSize: 0x%x\n", optionControlIoctl.BufferSize);
				utilDebug("BytesTransferred: 0x%x\n", optionControlIoctl.BytesTransferred);
				utilDebug("TotalBytesNeeded: 0x%x\n", optionControlIoctl.TotalBytesNeeded);

				if(optionControlIoctl.TotalBytesNeeded == 0)
				{
					utilDebug("TotalBytesNeeded zero.  Break loop\n");
					break;
				}

				if(optionControl)
				{
					// just in case memory was allocated.
					free(optionControl);
				}

				optionControl = (MppCmn_OptionControl_t *)malloc(optionControlIoctl.TotalBytesNeeded);
				if(optionControl == NULL)
				{
					utilDebug("Unable to allocate memory\n");
					return(FALSE);
				}

				optionControlIoctl.OptionBuffer = optionControl;
				optionControlIoctl.BufferSize = optionControlIoctl.TotalBytesNeeded;
				optionControlIoctl.TotalBytesNeeded = 0;

				// Loop back and try the request again.
				utilDebug("Retry request\n");
			}
		}

		utilPrintFeatureOptions(&optionControlIoctl);

		if(optionControl)
		{
			free(optionControl);
		}
	}
	else
	{
		utilDebug("Input arguments found\n");
		memset(&optionControlIoctl, '\0', sizeof(MppCmn_OptionControlIoctl_t));
		optionControlIoctl.Magic = MPPCMN_FEATURE_OPTION_MAGIC;
		optionControlIoctl.RequestType = MPPCMN_OPTION_REQUEST_SET;
		optionControlIoctl.OptionBuffer = utilParseFeatureOptions(args, &bufferSize);
		if(optionControlIoctl.OptionBuffer == NULL)
		{
			// The request failed.
			utilDebug("Request failed\n");
			return(FALSE);
		}

		optionControlIoctl.BufferSize = bufferSize;

		// other parameters are set to zero or NULL, so there's no need to explicitly
		// set them here
		utilDebug("Send FeatureOption request\n");
		if(ioctl(utilVirtualFd, MPP_IOCTL_OPTIONCONTROL, &optionControlIoctl) == INVALID)
		{
			// The request failed.
			fprintf(stderr, "Request failed - %s!\n", strerror(errno));
			free(optionControlIoctl.OptionBuffer);
			return(FALSE);
		}

		numOptions = bufferSize / sizeof(MppCmn_OptionControl_t);
		optionControl = optionControlIoctl.OptionBuffer;
		for(optionCount = 0; optionCount < numOptions; optionCount++)
		{
			utilDebug("Option '%s'.  Status 0x%x\n", optionControl[optionCount].Keyword, optionControl[optionCount].Status);
			switch(optionControl[optionCount].Status)
			{
				case	MPPCMN_OPTION_VALUE_OUT_OF_RANGE:
					printf("The value ('0x%x') for option '%s' is out of range.  Valid range is '0x%x' to '0x%x'\n",
						optionControl[optionCount].Value,
						optionControl[optionCount].Keyword,
						optionControl[optionCount].MinValue,
						optionControl[optionCount].MaxValue);
					break;

				case	MPPCMN_OPTION_KEYWORD_UNKNOWN:
					printf("Option '%s' is unknown\n", optionControl[optionCount].Keyword);
					break;

				case	MPPCMN_OPTION_INVALID_PARAMETER:
					printf("Option '%s' has an invalid parameter\n", optionControl[optionCount].Keyword);
					break;

				case	MPPCMN_OPTION_STATUS_UNKNOWN:
					printf("Option '%s' returned an unknown status\n", optionControl[optionCount].Keyword);
					break;

				default:
					// Do nothing
					break;
			}
		}

		free(optionControlIoctl.OptionBuffer);
	}

	return(TRUE);
}

/*******************************************************************************
* PROCEDURE
*
* NAME:		utilUpdateMppConf
* SUMMARY:	This function will write the parameter to the file /etc/mpp.conf file.
*               This function will use temperary file for intermediate processing.
* RETURNS:
*   TRUE   - The parameter is updated and written to /etc/mpp.conf
*   FALSE  - The file is not present
*******************************************************************************/
static BOOL
utilUpdateMppConf(char * parameter, int value)
{
	char buffer[120]; /* Max characters in a line is 120 */ 
	FILE * mpp_conf_file, * temp_file;
	char temp_file_name[20]; 
	int found=0;

	utilDebug("Entering function utilUpdateMppConf:\n");

	/* Open the file */
	mpp_conf_file = fopen("/etc/mpp.conf", "r" );
	if (mpp_conf_file == NULL){
		fprintf(stderr, "File /etc/mpp.conf open request failed - %s!\n", strerror(errno));
		return FALSE;
	}

	/* Open/create the temporary file */
	sprintf(temp_file_name, "/tmp/mpp.conf.%d", getpid());
	temp_file = fopen(temp_file_name, "w+" );
	if (temp_file == NULL){
		fprintf(stderr, "Temporary file %s open request failed - %s!\n", temp_file_name, strerror(errno));
		return FALSE;
	}

	/* Loop until you find the specified line, or EOF is reached */
	while (fgets(buffer, 120, mpp_conf_file ) ) {
		if(strstr(buffer, parameter)){
			utilDebug("%s found in the mpp.conf file:\n", parameter);
			found =1;
			sprintf(buffer, "%s=%d\n", parameter, value);
			fputs(buffer, temp_file);
		}else {
			fputs(buffer, temp_file);
		}
	}

	/* If the parameter is not found then write at the end of the file */
	if (!found){
		utilDebug("%s not found in the mpp.conf file, writing at the end:\n",parameter);
		sprintf(buffer, "%s=%d\n", parameter, value);
		fputs(buffer, temp_file);
	}

	// Close files
	fclose(mpp_conf_file);
	fclose(temp_file);
	rename(temp_file_name, "/etc/mpp.conf");

	utilDebug("Leaving function utilUpdateMppConf:\n");

	return TRUE;
}
 
/*******************************************************************************
* PROCEDURE
*
* NAME:		utilProcessActionWord
* SUMMARY:	This function will check if the parameter passed support the
*               persistance. If yes it will call the function to update the conf file.
*
* RETURNS:
*   TRUE   - the action word is recognized and valid for the option
*   FALSE  - The file is not present
*******************************************************************************/
static BOOL
utilProcessActionWord(MppCmn_OptionControl_t *optionControl)
{
	int retVal = FALSE;

	utilDebug("Entering function utilProcessActionWord:\n");

	if (!strcmp(MPP_DEBUG_LEVEL, (char *)&optionControl->Keyword[0]))
	{
		/* SaveSettings is passed. We have to update the /etc/mpp.conf file */
		utilDebug("Calling utilUpdateMppConf to update mpp.conf file:\n");
		retVal = utilUpdateMppConf(MPP_DEBUG_LEVEL,  optionControl->Value);
	}
	else if (!strcmp(MPP_ERRORLEVEL, (char *)&optionControl->Keyword[0]))
	{
		/* SaveSettings is passed. We have to update the /etc/mpp.conf file */
		utilDebug("Calling utilUpdateMppConf to update mpp.conf file:\n");
		retVal = utilUpdateMppConf(MPP_ERRORLEVEL,  optionControl->Value);
	}
	else if (!strcmp(MPP_DISABLELUNREBALANCE, (char *)&optionControl->Keyword[0]))
	{
		/* SaveSettings is passed. We have to update the /etc/mpp.conf file */
		utilDebug("Calling utilUpdateMppConf to update mpp.conf file:\n");
		retVal = utilUpdateMppConf(MPP_DISABLELUNREBALANCE,  optionControl->Value);
	}
	else if (!strcmp(MPP_LOAD_BALANCE_POLICY, (char *)&optionControl->Keyword[0]))
	{
		/* SaveSettings is passed. We have to update the /etc/mpp.conf file */
		utilDebug("Calling utilUpdateMppConf to update mpp.conf file:\n");
		retVal = utilUpdateMppConf(MPP_LOAD_BALANCE_POLICY,  optionControl->Value);
	}
	else
	{
		fprintf(stderr, "SaveSettings action is not allowed on %s \n", optionControl->Keyword);
	}

	utilDebug("Leaving function utilProcessActionWord:\n");

	return retVal;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:		utilParseFeatureOptions
* SUMMARY:	Parse the command-line argument for Feature Options
* SCOPE:	LOCAL
*
* DESCRIPTION:
* 	This function parses the '-o' command-line argument and builds an
* 	array of MppCmn_OptionControl_t structures that is passed to the
* 	MPP driver for processing.
*
* RETURNS:
* 	Pointer to a block of allocated MppCmn_OptionControl_t structures,
* 	else NULL if an error has occurred.
*******************************************************************************/
static MppCmn_OptionControl_t *
utilParseFeatureOptions(char *args, LWORD *bufferSize)
{
	MppCmn_OptionControl_t		*optionControl = NULL;
	char				keyword[MPPCMN_KEYWORD_OPTION_LENGTH + 1];
	LWORD				value;
	BOOL				firstTime = TRUE;
	BOOL				valueProvided;
	int				optionReturn;
	LWORD				numOptions = 0;
	LWORD				retVal;

	utilDebug("ParseFeatureOptions: '%s'\n", args);
	while(TRUE)
	{
		if(firstTime)
		{
			// utilGetNextFeatureOption is similar to strtok() in that you pass in args to get started,
			// then subsequent calls parse the arg string.
			optionReturn = utilGetNextFeatureOption(args, keyword, &value, &valueProvided);
			firstTime = FALSE;
		}
		else
		{
			optionReturn = utilGetNextFeatureOption(NULL, keyword, &value, &valueProvided);
		}

		if(optionReturn == FALSE)
		{
			// No more options to process.  Its okay to do this here for both firstTime
			// and !firstTime.  If firstTime numOptions is zero so bufferSize will be
			// zero and optionControl is NULL.
			break;
		}
		else if(optionReturn == INVALID)
		{
			// A parsing error occurred.  Free any allocated memory and return.
			if(optionControl)
			{
				fprintf(stderr, "Error parsing options\n");
				free(optionControl);
				return(NULL);
			}
		}
	
		/* We dont want to allow Least Path weight in Linux for LoadBalancePolicy */
		if(!strcmp(keyword, "LoadBalancePolicy") && (value == 2))
		{
			fprintf(stderr, "This LoadBalancePolicy(Least Path Weight) is not supported\n");
			if(optionControl)
				free(optionControl);
			return(NULL);
		}

		if(optionControl == NULL)
		{
			optionControl = (MppCmn_OptionControl_t *)malloc(sizeof(MppCmn_OptionControl_t));
			if(optionControl == NULL)
			{
				fprintf(stderr, "Unable to allocate memory - %s!\n", strerror(errno));
				return(NULL);
			}
		}
		else
		{
			// Check for action word associated with previous option. If action word entered,
			// SaveSettings then call the function update the conf file.
			if(!strcmp(keyword, "SaveSettings")){
				retVal = utilProcessActionWord(&optionControl[numOptions-1]);
				if (retVal == TRUE)  
				{
					continue;
				}
				else if (retVal == FALSE)
				{ // The action word is not supported with the option
					free(optionControl);
					return(NULL);			    
				}

			}

			optionControl = (MppCmn_OptionControl_t *)realloc(optionControl, sizeof(MppCmn_OptionControl_t) * (numOptions + 1));
			if(optionControl == NULL)
			{
				fprintf(stderr, "Unable to reallocate memory - %s!\n", strerror(errno));
				return(NULL);
			}
		}

		if(valueProvided == TRUE)
		{
			optionControl[numOptions].Status = MPPCMN_OPTION_VARIABLE_SUCCESS;
		}
		else
		{
			optionControl[numOptions].Status = MPPCMN_OPTION_ACTION_SUCCESS;
		}
		strncpy(optionControl[numOptions].Keyword, keyword, MPPCMN_KEYWORD_OPTION_LENGTH);
		optionControl[numOptions].Keyword[MPPCMN_KEYWORD_OPTION_LENGTH] = '\0';
		optionControl[numOptions].Value = value;
		optionControl[numOptions].DefaultValue = 0;
		optionControl[numOptions].MinValue = 0;
		optionControl[numOptions].MaxValue = 0;

		numOptions++;
	}

	*bufferSize = numOptions * sizeof(MppCmn_OptionControl_t);
	return(optionControl);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilGetNextFeatureOption
* SUMMARY:	Retrieve the next Feature Option from the command-line argument.
* SCOPE:	LOCAL
*
* DESCRIPTION:
* 	This function returns the next Feature Option.
* 	It works similar to strtok(), where you pass in a string to initialize the
* 	function, then subsequent calls parse that string.
*
* 	A Feature Option has the following format:
* 		[keyword[=value][,keyword[=value]]]...
*
*	Given this the only delimeters of concern are:
*		'=' : indicates a value is being set for a keyword
*		',' or EOL : indicates the end of a keyword (action) or keyword/value definition
*
* 	The input arguments 'keyword' and 'value' will contain the parsed values.  Since
* 	'value' is unsigned the argument 'valueProvided' is used to indicate whether
* 	a value was provided.  This can be used to distinquish an action from
* 	a keyword=value condition.
*
* RETURNS:
* 	TRUE - a successful parse occurred
* 	FALSE - no addition parsing is necessary
* 	INVALID - an error during parsing occurred
*******************************************************************************/
static int
utilGetNextFeatureOption(char *args, char *keyword, LWORD *value, BOOL *valueProvided)
{
	static char			*savedArgs;
	char				*curPtr;
	char				lastDelimeter;
	BOOL				needValue;
	LWORD				newValue = 0, idx;

	if(args)
	{
		// Initialize a new parse operation
		utilDebug("GetNextFeatureOption: Reset parsing\n");
		savedArgs = args;
	}

	utilDebug("savedArgs: '%s'\n", savedArgs);
	// Check other input arguments before going on
	if(keyword == NULL || value == 0)
	{
		utilDebug("GetNextFeatureOption: No keyword or value buffer provided\n");
		return(INVALID);
	}

	if(savedArgs == NULL)
	{
		utilDebug("GetNextFeatureOption: No more args\n");
		return(FALSE);
	}

	curPtr = savedArgs;
	needValue = FALSE;
	while(TRUE)
	{
		utilDebug("savedArgs(Loop): '%s'; '%c'\n", savedArgs, *curPtr);
		if(*curPtr == MPPCMN_DEL_EQUAL)
		{
			// A key/value pair has been discovered
			*curPtr++ = '\0';
			strncpy(keyword, savedArgs, MPPCMN_KEYWORD_OPTION_LENGTH);
			savedArgs = curPtr;
			needValue = TRUE;
		}
		else if(*curPtr == '\0' || *curPtr == MPPCMN_DEL_COMMA)
		{
			// Either an arglist has been completely processed or the end of a key
			// or key/value definition has been discovered.  Either way were done
			// processing for this iteration.
			lastDelimeter = *curPtr;
			*curPtr = '\0';
			if(needValue == TRUE)
			{
				utilDebug("GetNextFeatureOption: Convert '%s'\n", savedArgs);
				// Convert this portion of the savedArg string to a decimal value.

				// See if the user began with '0x'
				if(!isHex(savedArgs))
				{
					return(INVALID);
				}

				// See if the user began the debug level with '0x'
				if(!strncmp(savedArgs, "0x", 2) || !strncmp(savedArgs, "0X", 2) )
				{
					savedArgs += 2;
				}

				while(savedArgs && *savedArgs && isxdigit(*savedArgs))
				{
					if(islower(*savedArgs)) {
						idx = *savedArgs++ - ('a' - 'A');
						idx = idx - '0';
					}
					else
						idx = *savedArgs++ - '0';

					if(9 < idx)
						idx -= 7;
					newValue <<= 4;
					newValue |= (idx & 0xff);
				}
	
				*value = newValue;
				*valueProvided = TRUE;
			}
			else
			{
				// Must be an action which is copied into the keyword.  Set the value
				// parameter to zero.
				strncpy(keyword, savedArgs, MPPCMN_KEYWORD_OPTION_LENGTH);
				*value = 0;
				*valueProvided = FALSE;
			}

			if(lastDelimeter == MPPCMN_DEL_COMMA)
			{
				*curPtr++;
				savedArgs = curPtr;
			}
			else
			{
				savedArgs = NULL;
			}

			utilDebug("GetNextFeatureOption: Keyword '%s', Value 0x%x, ValueProvided 0x%x\n", keyword, *value, *valueProvided);
			return(TRUE);
		}
		else
		{
			*curPtr++;
		}
	}

	utilDebug("GetNextFeatureOption: Bad parse\n");
	return(INVALID);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:		utilPrintFeatureOptions
* SUMMARY:	Print a list of supported Feature Options
* SCOPE:	LOCAL
*
* DESCRIPTION:
* 	This function prints a list of supported Feature Options based on information
* 	passed back from the MPP driver.
*
* RETURNS:
* 	None.
*******************************************************************************/
static	VOID
utilPrintFeatureOptions(MppCmn_OptionControlIoctl_t *ioctl)
{
	MppCmn_OptionControl_t		*optionControl;
	LWORD				numOptions, optionCount, numPrinted;
	char				strValue[11];

	if(ioctl == NULL)
	{
		return;
	}

	// Print variable options first...
	printf("                                          Current     Default     Minimum     Maximum\n");
	printf("Variable Options                          Value       Value       Value       Value\n");
	printf("----------------------------------------  ----------  ----------  ----------  ----------\n");

	optionControl = ioctl->OptionBuffer;
	numOptions = ioctl->BytesTransferred / sizeof(MppCmn_OptionControl_t);
	numPrinted = 0;
	for(optionCount = 0; optionCount < numOptions; optionCount++)
	{
		if(optionControl[optionCount].Status == MPPCMN_OPTION_VARIABLE_SUCCESS)
		{
			// This code aligns the values on a column boundry by first printing the value as
			// hex to a temporary buffer.  Then the buffer is printed with a formatting string.
			// This makes the output cleaner than using a "0x%10x" format which would pad the
			// hex number with zeroes.   (ie. 0x0000000000).
			printf("%-40.40s", optionControl[optionCount].Keyword);
			sprintf(strValue, "0x%x", optionControl[optionCount].Value);
			printf("  %-10.10s", strValue);
			sprintf(strValue, "0x%x", optionControl[optionCount].DefaultValue);
			printf("  %-10.10s", strValue);
			sprintf(strValue, "0x%x", optionControl[optionCount].MinValue);
			printf("  %-10.10s", strValue);
			sprintf(strValue, "0x%x", optionControl[optionCount].MaxValue);
			printf("  %-10.10s\n", strValue);
			numPrinted++;
		}
	}
	
	if(numPrinted == 0)
	{
		printf("No variables defined\n\n");
	}
	else
	{
		printf("\n\n");
	}

	// ...followed by actions
	printf("Action Options\n");
	printf("---------------------------------------\n");

	optionControl = ioctl->OptionBuffer;
	numPrinted = 0;
	for(optionCount = 0; optionCount < numOptions; optionCount++)
	{
		if(optionControl[optionCount].Status == MPPCMN_OPTION_ACTION_SUCCESS)
		{
			printf("%-40.40s\n", optionControl[optionCount].Keyword);
			numPrinted++;
		}
	}

	if(numPrinted == 0)
	{
		printf("No actions defined\n");
	}

	return;
}


