/*******************************************************************************
*  Copyright 2004-2008 LSI Logic Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppLnx26p_upper.c
SUMMARY         %description%
VERSION         %version: 18.1.1 %
UPDATE DATE     %date_modified: Thu Jun 26 14:07:34 2008 %
PROGRAMMER      %created_by:    bmoger %


DESCRIPTION: This is the main file that contains all the functionality of the linux Upper
Level driver.

INCLUDE FILES:

NOTES: MPP upper level driver reserves error number 700-760.

RESTRICTIONS:

SEE ALSO:

REFERENCE:

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/


/***  MACRO DEFINITIONS  ***/
#define		WWN_LENGTH 16

#define __SRCMPP_Target
#define MPP_VBUSNODE "mppUpper"
#define DISKDRIVERNAME "sd"


/***  INCLUDES  ***/


#include "MPP_Sysdep.h"
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "copyright.h"
#include "mppLnx_version.h"
#include "mppLnx26p_shared.h"
#include "mppLnx26p_prototypes.h"
#include "mppCmn_SysInterface.h"
#include <linux/list.h>
#include <linux/err.h>
#include <scsi/scsi_eh.h>

/*** LOCALS  ***/
int mppDriverInterceptCallback(struct device_driver *, void *);
int mppDriverRestoreCallback(struct device_driver *, void *);
int mppLnx_scsi_execute(struct scsi_device *sdev, const unsigned char *cmd,
         int data_direction, void *buffer, unsigned bufflen,
         unsigned char *sense, int timeout, int retries, int flags);
int mppLnx_scsi_execute_req(struct scsi_device *sdev, const unsigned char *cmd,
             int data_direction, void *buffer, unsigned bufflen,
             struct scsi_sense_hdr *sshdr, int timeout, int retries);

/* The following function pointers stores the old function pointers for sd/sg driver */
int (*sg_old_add) (struct class_device *,struct class_interface *);
void (*sg_old_remove) (struct class_device *, struct class_interface *);
static struct class *mppUpper_class;
int (*sd_old_probe) (struct device * dev);

static void mppLnx_process_dummy_lun0(struct scsi_device *sdp);
static void mppLnx_CheckRemoveModule(RdacDeviceInformation_t *RdacInfo, MPP_HANDLE DeviceVertex);
/*
* The /proc/mpp/array/controller/path directory needs the physical HBA driver's identification.
* But at the time of building mpp_CreateControllerVertex, the physical device is not in
* RdacDeviceInformation struct yet. The following two variables are used to set
* mppLnx_host_procName that are used in the MPP_Sysdep.c function.
*/
extern char             * mppLnx_host_procName;

/* function declaration for MPP virtual bus mode creation/destruction */
int mppLnx_ioctl_init(void);
int mppLnx_ioctl_exit(void);

MPP_MUTEX_T	 mppLnx_rdac_mutex;
struct list_head mpp_hot_plug_list;
spinlock_t mpp_hot_plug_spin_lock = SPIN_LOCK_UNLOCKED;

/* MPP configuration parameter structure declaration */
struct mpp_Config_Parameters mppLnx_Config_Parameters;
/* MPP proc entry declaration */
extern struct proc_dir_entry *proc_mpp;
/* file operation structure used for module initialization */
static struct file_operations mppLnx_fops = {
    .owner = THIS_MODULE,
    .read = NULL,
    .write = NULL,
    .open = mppLnx_open,
    .ioctl = mppLnx_ioctl,
    .release = mppLnx_release
};

static struct scsi_driver mpp_template = {
        .owner = THIS_MODULE,
        .gendrv = {
                .name = "mpp",
                .probe = mppLnx_driver_probe,
                .remove = mppLnx_driver_remove,
        },
        .init_command = mppLnx_init_command,
};

static struct class_interface mpp_interface = {
	.add = mppLnx_class_add,
	.remove = mppLnx_class_remove,
};

/* mpp_SysInterface data structure used for Sys Interface function registration */
static mppCmn_SysInterface_t mppLnx_sysinterface = {

	/* REQUIRED */
	.mpp_SysdepSynchronousIo	= mpp_SysdepSynchronousIo,
	.mpp_IssueFailover		= mpp_IssueFailover,
	.mpp_makeVirtualLun		= mpp_makeVirtualLun,
	.mppSys_GetUtilSysdepData       = mppSys_GetUtilSysdepData,
	.mppSys_ValidatePath		= mppSys_ValidatePath,
	.mppSys_CopyRdacInfoToUser	= mppSys_CopyRdacInfoToUser,
	.mppSys_QueueRequest            = mppSys_QueueRequest,

	/* OPTIONAL */
	.mpp_IsOkayToClearRdacLuns	= mppSys_IsOkayToClearRdacLuns,  /* return false */
	.mpp_RemoveControllerDirectory	= mpp_RemoveControllerDirectory,
	.mpp_ThreadDelay		= mpp_ThreadDelay,
	.mppSys_AreControllerPathsFailed = mppSys_AreControllerPathsFailed,
	.mpp_CreateUtmDevice		= NULL, /* USE DEFAULT */
	.mpp_RemoveUTM			= NULL, /* USE DEFAULT */
	.mpp_RemoveLunLink		= mpp_RemoveLunLink,
	.mpp_RemoveVirtualLun		= mpp_RemoveVirtualLun,
	.mpp_CheckPathGood		= NULL, /* USE DEFAULT */
	.mpp_makeVirtualTarget		= mpp_makeVirtualTarget,
	.mpp_CreateControllerVertex	= mpp_CreateControllerVertex,
	.mpp_CreateLunLink		= mpp_CreateLunLink,
	.mpp_RemovePathDirectory	= mpp_RemovePathDirectory,
	.mpp_RemoveModule		= mpp_RemoveModule,
	.mppSys_DeviceAlreadyDiscovered	= mppSys_DeviceAlreadyDiscovered,
	.mppSys_PrintSenseBuffer	= mppSys_PrintSenseBuffer,
	.mppSys_AcquireReferenceCount	= NULL, /* USE DEFAULT */
	.mppSys_ReleaseReferenceCount	= NULL,  /* USE DEFAULT */
	.mppSys_IsConfiguredDevice = mppSys_IsConfiguredDevice,
	.mppSys_PreFailoverProcessing	= NULL,  /* USE DEFAULT */
	.mppSys_PostFailoverProcessing	= NULL  /* USE DEFAULT */
};

static spinlock_t unconfiguredDevice_lock = SPIN_LOCK_UNLOCKED;
static mppLnx_UnconfiguredDeviceList_t  unconfiguredDeviceList;
static int mppLnx_major;

BOOL sgFound=FALSE;
BOOL sdFound=FALSE;


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_init
* SUMMARY:  MPP upper level driver initialization routine
* SCOPE:    local
*
* DESCRIPTION: This function is called when loading MPP upper level driver. This function will:
*              1. register as a device driver with scsi bus.
*              2. Register as a class interface with scsi_device class.
*              3. register with Linux kernelas a character device using dynamically allocated major number
*              4. read configurable parameter file, persistent binding file and translation key.
*              5. replace sd/sg driver function pointers
*              6. create virtual bus node
*              7. allocate mpp_ModuleArray memory resources
*
* RESTRICTIONS:
*
* RETURNS: 0 if sucess, 1 if error
*
* ERRNO:
*
*
*/

static int __init mppLnx_init (void)
{
	static int rc;
	struct class_interface *class_intf;
	int ret=0;
	struct list_head *listp;
	mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
	unsigned long iflages;
    int intercept=0;
   

	INIT_LIST_HEAD(&mpp_hot_plug_list);
	/* assign default values to MPP driver configurable parameters */
	mppLnx_Config_Parameters.mpp_Debug =       mpp_Debug;
	mppLnx_Config_Parameters.mpp_NotReadyWaitTime = mpp_NotReadyWaitTime;
	mppLnx_Config_Parameters.mpp_BusyWaitTime =     mpp_BusyWaitTime;
	mppLnx_Config_Parameters.mpp_QuiescenceWaitTime =       mpp_QuiescenceWaitTime;
	mppLnx_Config_Parameters.mpp_InquiryWaitTime =  mpp_InquiryWaitTime;
	mppLnx_Config_Parameters.mpp_MaxLunsPerArray =  mpp_MaxLunsPerArray;
	mppLnx_Config_Parameters.mpp_MaxPathsPerController =    mpp_MaxPathsPerController = 4;
	mppLnx_Config_Parameters.mpp_ScanInterval =     mpp_ScanInterval;
	mppLnx_Config_Parameters.mpp_InquiryInterval =  mpp_InquiryInterval;
	mppLnx_Config_Parameters.mpp_MaxArrayModules =  mpp_MaxArrayModules;
	mppLnx_Config_Parameters.mpp_ErrorLevel =       mpp_ErrorLevel;
	mppLnx_Config_Parameters.mpp_SelectionTimeoutRetryCount =       mpp_SelectionTimeoutRetryCount;
	mppLnx_Config_Parameters.mpp_CommandTimeoutRetryCount =         mpp_CommandTimeoutRetryCount;
	mppLnx_Config_Parameters.mpp_UaRetryCount =     mpp_UaRetryCount;
	mppLnx_Config_Parameters.mpp_SyncRetryCount =   mpp_SyncRetryCount;
	mppLnx_Config_Parameters.mpp_SynchTimeout =     mpp_SynchTimeout = 60;
	mppLnx_Config_Parameters.mpp_FailOverQuiescenceTime =   mpp_FailOverQuiescenceTime;
	mppLnx_Config_Parameters.mpp_FailoverTimeout =  mpp_FailoverTimeout;
	mppLnx_Config_Parameters.mpp_FailBackToCurrentAllowed =         mpp_FailBackToCurrentAllowed;
	mppLnx_Config_Parameters.mpp_HoldAltInReset =   mpp_HoldAltInReset;
	mppLnx_Config_Parameters.mpp_DoUARetry =        mpp_DoUARetry = 1;
	mppLnx_Config_Parameters.mpp_ControllerIoWaitTime = mpp_ControllerIoWaitTime;
	mppLnx_Config_Parameters.mpp_ArrayIoWaitTime = mpp_ArrayIoWaitTime;
	mppLnx_Config_Parameters.mpp_DisableLUNRebalance = mpp_DisableLUNRebalance;

	// reset mpp_S2ToS3Key memory
	memset (mpp_S2ToS3Key, 0, 8);
	memset (mppLnx_Config_Parameters.mpp_S2ToS3Key, 0, 8);

	mppLnx_Config_Parameters.mpp_MaxArrayFailoverLength = mpp_MaxArrayFailoverLength;
	mppLnx_Config_Parameters.mpp_IdlePathCheckingInterval = mppCmn_IdlePathCheckingInterval;
	mppLnx_Config_Parameters.mpp_RecheckFailedPathWaitTime = mppCmn_RecheckFailedPathWaitTime;
	mppLnx_Config_Parameters.mpp_FailedPathCheckingInterval = mppCmn_FailedPathCheckingInterval;
	mppLnx_Config_Parameters.mpp_PrintSenseBuffer = mppCmn_PrintSenseBuffer;
	mppLnx_Config_Parameters.mpp_ClassicModeFailover = mppCmn_ClassicModeFailover;
	mppLnx_Config_Parameters.mpp_ClassicModeFailover = mppCmn_AVTModeFailover;
	mppLnx_Config_Parameters.mpp_LunFailoverDelay = mppCmn_LunFailoverDelay;
	mppLnx_Config_Parameters.mpp_LoadBalancePolicy = mppCmn_LoadBalancePolicy;

	// assigned default values to parameters of type char array
        strcpy (mppLnx_Config_Parameters.mpp_VendorId, "LSI     \0");
        strcpy (mppLnx_Config_Parameters.mpp_ProductId,"VirtualDisk     \0");
        strcpy (mpp_VendorId, "LSI     \0");
        strcpy (mpp_ProductId,"VirtualDisk     \0");
	strcpy (mppLnx_Config_Parameters.mpp_Version,RAID_MANAGER_VERSION);


	MPP_MUTEX_INIT(mppLnx_rdac_mutex);
	/*
	* Register SysInterface to common code
	*/
	if( mppCmn_RegisterSysInterface( &mppLnx_sysinterface  ) == FALSE )
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL,700,
			"mpp:upper:mppLnx_init: Failed at registering System Interface fuctions.\n"));
		ret = -EAGAIN;
		goto errorOut0;
	}
	else
	{
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2, "mpp:upper:mppLnx_init: PASSED at registering System Interface fuctions.\n"));
	}

	/*
	*The Linux MPP driver currently doesn't have system specific feature options. When it does,
	*it should provide the options and number of options here to register it with the command
	*code.
	*/

	if(mppCmn_InitializeFeatureOptions((MppCmn_FeatureOptionControl_t *)NULL, 0) != TRUE)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 701, "Failed to initialize Feature Options\n"));
	}


	// read configurable parameter file
	mppLnx_readConfigData();

	// allocate and reset mpp_ModuleArray[] data structure
	mpp_ModuleArray = (ArrayModuleEntry_t*) MPP_KMEM_ALLOC(sizeof(ArrayModuleEntry_t)*mpp_MaxArrayModules);
	if (mpp_ModuleArray == NULL)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 702,
			"Failed Allocating mpp_ModuleArray[]\n"));
		ret = -ENOMEM;
		goto errorOut0;
	}

	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2, "mpp_ModuleArray allocated starting at address 0x%p\n", mpp_ModuleArray));

	// read persistent binding data
	mppLnx_readBindingData();

	/*
	* initialize the unconfigured device list and its lock
	*/

	unconfiguredDeviceList.unconfiguredDeviceLock = &unconfiguredDevice_lock;
	unconfiguredDeviceList.deviceEntryList =
		(mppLnx_UnconfiguredDeviceEntry_t *) MPP_INT_KMEM_ALLOC(sizeof(mppLnx_UnconfiguredDeviceEntry_t));
	if(!unconfiguredDeviceList.deviceEntryList)
	{
		/* log error*/
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 709,
			"Cannot allocate memory for unconfigured device entry\n"));
		goto errorOut1;
	}

	INIT_LIST_HEAD(&(unconfiguredDeviceList.deviceEntryList->list));

	// register MPP as a device driver with scsi bus
	rc = scsi_register_driver(&mpp_template.gendrv);
	if (rc != 0)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL,703,
			"mpp:upper:mppLnx_init: Failed at registering driver with scsi bus.\n"));
		ret = rc;
		goto errorOut2;
	}
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2, "mpp: upper:mppLnx_init:passed driver interface registration\n"));

	intercept = bus_for_each_drv(mpp_template.gendrv.bus,
                        (struct device_driver *)NULL,
                        (void *)NULL,
                        mppDriverInterceptCallback);

	if (!sdFound)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL,704,
			"mpp:upper:mppLnx_init: cannot find sd driver.\n"));
		ret = -EAGAIN;
		goto errorOut3;
	}

	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp: upper:mppLnx_init: intercepted sd function pointer\n"));

	// register MPP as a class interface with scsi_device class
	rc = scsi_register_interface(&mpp_interface);
	if (rc != 0)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL,705,
			"mpp:upper:mppLnx_init: Failed at registering interface with scsi_device class.\n"));
		ret = rc;
		goto errorOut4;
	}
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_2, "mpp: upper:mppLnx_init:passed class interface registration\n"));

	list_for_each_entry(class_intf, &mpp_interface.class->interfaces, node)
	{
		if (class_intf != &mpp_interface)
		// Okay, we have found sg class interface
		{
			sgFound = TRUE;
			sg_old_add = class_intf->add;
			sg_old_remove = class_intf->remove;
			class_intf->add = mppLnx_sg_add;
			class_intf->remove = mppLnx_sg_remove;
		}
	}

	if (!sgFound)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL,706,
			"mpp:upper:mppLnx_init: cannot find sg class interface.\n"));
		ret = -EAGAIN;
		goto errorOut5;
	}
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: intercepted sg function pointer\n"));

	// Now register with the kernel as a char device using dynamically allocated major number.
	if ( (rc = mppLnx_ioctl_init()) != 0)
	/* char device registration or virtual bus node creation fails */
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 707,
			"Failed registering mppLnx, returned %d\n", mppLnx_major));
		goto errorOut6;
	}

	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4, "mpp:upper:mppLnx_init: device successfully   registered, Major No. = %d\n", mppLnx_major));


	/* create the /proc/mpp directory */
	proc_mpp = proc_mkdir(MPPLNX_PROC_NAME, 0);
	if (NULL == proc_mpp)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 708,
			"Failed to create MPP driver proc entry %s\n",MPPLNX_PROC_NAME));
		ret = -EAGAIN;
		goto errorOut7;
	}
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: created MPP /proc entry\n"));

	return 0;


errorOut7:
	/* unregister with the kernel */
	mppLnx_ioctl_exit();
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: unregister with kernel.\n"));

errorOut6:
	/* restore function pointers */
	list_for_each_entry(class_intf, &mpp_interface.class->interfaces, node)
	{
		if (class_intf != &mpp_interface)
		// Okay, we have found sg class interface
		{
			class_intf->add = sg_old_add;
			class_intf->remove = sg_old_remove;
		}
	}
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: restore sg function pointer.\n"));

errorOut5:
	/* unregister class interface */
	scsi_unregister_interface(&mpp_interface);
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: unregister class interface.\n"));

errorOut4:
	intercept = bus_for_each_drv(mpp_template.gendrv.bus,
                        (struct device_driver *)NULL,
                        (void *)NULL,
                        mppDriverRestoreCallback);

errorOut3:
	/* unregister device driver interface */
	scsi_unregister_driver(&mpp_template.gendrv);
	MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: unregister driver interface.\n"));

	/*
	* clean-up the un-configured device list
	*/
	for(listp = unconfiguredDeviceList.deviceEntryList->list.next;
	listp != &(unconfiguredDeviceList.deviceEntryList->list);
	listp = listp->next)
	{
		deviceEntry = list_entry(listp, mppLnx_UnconfiguredDeviceEntry_t, list);
		if (deviceEntry) {
			MPP_ERRORLOGPRINT((MPP_ERR_FATAL,738, "mpp:upper:mppLnx_exit: Unconfigured Device found in list. Should not happen.\n"));
			spin_lock_irqsave(unconfiguredDeviceList.unconfiguredDeviceLock, iflages);
			list_del(&(deviceEntry->list));
			spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);
			MPP_FREE(deviceEntry,sizeof(mppLnx_UnconfiguredDeviceEntry_t));
		}
	}

errorOut2:
	/* free unconfiguredDeviceList head*/
	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: freeing unconfigured device list head.\n"));
	MPP_FREE(unconfiguredDeviceList.deviceEntryList,sizeof(mppLnx_UnconfiguredDeviceEntry_t));

errorOut1:
	/* free mpp_ModuleArray[] data structure */
	if (mpp_ModuleArray != NULL)
	{
		/* Free the allocated memory for mpp_ModuleArray[] data structure */
		MPP_FREE(mpp_ModuleArray, sizeof(ArrayModuleEntry_t)*mppLnx_Config_Parameters.mpp_MaxArrayModules);
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_init: free mpp_ModuleArray[].\n"));
	}

errorOut0:
	return ret;

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppDriverInterceptCallback
* SUMMARY:  sd driver interception function
* SCOPE:    local
*
*
* DESCRIPTION: This function checks the driver name. If it is sd driver, store its probe function
*		and replace it with mppLnx_sd_probe function.
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES:
*
*/
int mppDriverInterceptCallback (struct device_driver *drv, void * data)
{
	if (strcmp(drv->name,DISKDRIVERNAME) == 0)
	/* we have found sd driver */
	{
		sdFound=TRUE;
		sd_old_probe = drv->probe;
		drv->probe = mppLnx_sd_probe;
	}
	return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppDriverRestoreCallback
* SUMMARY:  read configurable parameter file
* SCOPE:    local
*
*
* DESCRIPTION: This function checks the driver name. If it is sd driver, restore its probe function
*		to its original routine.
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES:
*
*/
int mppDriverRestoreCallback (struct device_driver *drv, void * data)
{
	if (sdFound)
	{
		if (strcmp(drv->name,DISKDRIVERNAME) == 0)
		/* we have found sd driver */
		{
			drv->probe = sd_old_probe;
		}
		MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppDriverRestoreCallback: restore sd function pointer.\n"));
	}
	return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_exit
* SUMMARY:  MPP upper level driver clean up routine
* SCOPE:    local
*
* DESCRIPTION: This function is called when unloading MPP upper level driver. This function will:
*              1. unregister with scsi bus and scsi_device class.
*              2. Restore sd/sg driver function pointers
*              3. Remove virtual bus node
*              4. Free mpp_ModuleArray memory resources
*
* RESTRICTIONS:
*
* RETURNS: none
*
* ERRNO:
*
*
*/
static void __exit mppLnx_exit (void)
{
	struct class_interface *class_intf;
	struct list_head *listp;
	mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
	unsigned long iflages;
    int intercept=0;


	/* roll back the resources that we allocated if any error occurs */
	if (mpp_ModuleArray != NULL)
	{
		/* Free the allocated memory for mpp_ModuleArray[] data structure */
		MPP_FREE(mpp_ModuleArray, sizeof(ArrayModuleEntry_t)*mppLnx_Config_Parameters.mpp_MaxArrayModules);
	}
	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: free mpp_ModuleArray[].\n"));

	mppLnx_ioctl_exit();
	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: remove MPP character device.\n"));

	if (proc_mpp != NULL)
	{
		/* remove MPP proc entry */
		remove_proc_entry(MPPLNX_PROC_NAME, 0);
	}
	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: remove MPP /proc entry.\n"));

	/* restore function pointers */
	list_for_each_entry(class_intf, &mpp_interface.class->interfaces, node)
	{
		if (class_intf != &mpp_interface)
		// Okay, we have found sg class interface
		{
			class_intf->add = sg_old_add;
			class_intf->remove = sg_old_remove;
		}
	}
	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: restore sg function pointer.\n"));

	intercept = bus_for_each_drv(mpp_template.gendrv.bus,
                        (struct device_driver *)NULL,
                        (void *)NULL,
                        mppDriverRestoreCallback);

	/* unregister device driver interface and class interface */
	scsi_unregister_driver(&mpp_template.gendrv);
	scsi_unregister_interface(&mpp_interface);
	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: unregister MPP driver/class interface.\n"));


	/*
	* clean-up the un-configured device list
	*/
	for(listp = unconfiguredDeviceList.deviceEntryList->list.next;
	listp != &(unconfiguredDeviceList.deviceEntryList->list);
	listp = listp->next)
	{
		deviceEntry = list_entry(listp, mppLnx_UnconfiguredDeviceEntry_t, list);
		if (deviceEntry) {
			MPP_ERRORLOGPRINT((MPP_ERR_FATAL,740, "mpp:upper:mppLnx_exit: Unconfigured Device found in list. Should not happen.\n"));
			spin_lock_irqsave(unconfiguredDeviceList.unconfiguredDeviceLock, iflages);
			list_del(&(deviceEntry->list));
			spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);
			MPP_FREE(deviceEntry,sizeof(mppLnx_UnconfiguredDeviceEntry_t));
		}
	}


	MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mpp:upper:mppLnx_exit: freeing unconfigured device list head.\n"));
	MPP_FREE(unconfiguredDeviceList.deviceEntryList,sizeof(mppLnx_UnconfiguredDeviceEntry_t));

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_ioctl_init
* SUMMARY:  MPP virtual bus node creation function
* SCOPE:    local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION: This function creates MPP viutual bus node /dev/mppUpper
*
* RESTRICTIONS:
*
* RETURNS: 0 if succeeds, non-0 if fails
*
* ERRNO:
*
* NOTES:
*
*/
int mppLnx_ioctl_init(void)
{
	struct class_device *class;
	int err;

        mppUpper_class = class_create(THIS_MODULE, MPP_VBUSNODE);
        if (IS_ERR(mppUpper_class)) {
                mppUpper_class = NULL;
                return 1;

        }

        mppLnx_major = register_chrdev(0, MPP_VBUSNODE, &mppLnx_fops);
        if (mppLnx_major < 0) {
                class_destroy(mppUpper_class);
                mppUpper_class = NULL;

                return mppLnx_major;
        }
        class = class_device_create(mppUpper_class, NULL, MKDEV(mppLnx_major, 0), NULL,
           MPP_VBUSNODE);

	if (IS_ERR(class))
	{
		err = PTR_ERR(class);
		return err;
	}

        return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_ioctl_exit
* SUMMARY:  MPP virtual bus node removal function
* SCOPE:    local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION: This function removes MPP viutual bus node /dev/mppUpper
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES:
*
*/
int mppLnx_ioctl_exit(void)
{
        if (!mppUpper_class)
                return 1;
        class_device_destroy(mppUpper_class, MKDEV(mppLnx_major, 0));

        unregister_chrdev(mppLnx_major, MPP_VBUSNODE);

        class_destroy(mppUpper_class);

        mppUpper_class = NULL;

        return 0;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_readConfigData
* SUMMARY:  read configurable parameter file
* SCOPE:    local
*
*
* DESCRIPTION: This function is called in MPP upper level driver initialization routine to read configurable parameter values.
*              The file has the following format:
*              <ParameterName1>=<ParameterValue1>'\n'
*              <ParameterName1>=<ParameterValue1>'\n'
*              ...
*              <ParameterName1>=<ParameterValue1>'\n'
*	       Leading or trailing spaces are allowed around parameter name and parameter value for more flexibility.
*	       User can insert blank lines.
*	       User can have comment lines as long as they do not have '='
*
* RESTRICTIONS:
*
* RETURNS: 0 if suceeds, 1 if error
*
* ERRNO: 1
*
* NOTES: if error occurs, the parameter values are set back to their default values.
*
*/
static int mppLnx_readConfigData(void)
{
	struct file* filp;
	char* buf = NULL, *bufBeginPtr = NULL;
	char* tag = NULL;
	char* value = NULL;
	LWORD bytesRead = 0;
	LWORD fileSize = 0;

	/*** Now open the file /etc/mpp.conf to read ***/
	filp = filp_open(CONFIG_FILE_NAME,O_RDONLY,0);
	if(!filp || IS_ERR(filp)) {
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 710,
			"mppLnx_readConfigData: Error opening /etc/mpp.conf\n"));
		return 1;
	}
	fileSize = filp->f_dentry->d_inode->i_size;
	/*** allocate the buffer to hold file contents ***/
	buf = MPP_KMEM_ALLOC(fileSize);
	if(!buf) {
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 711,
			"mppLnx_readConfigData: Error allocating buffer of size %d\n",fileSize));
		filp_close(filp,0);
		return 1;
	}
	bufBeginPtr = buf;

	/*** read the whole contents of the file into the buffer ***/
	bytesRead = kernel_read(filp,0,buf,fileSize);
	if(bytesRead != fileSize) {
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 712,
			"mppLnx_readConfigData: Only get %d bytes out of %d bytes.\n",
			bytesRead, fileSize));
		filp_close(filp,0);
		MPP_FREE(buf,fileSize);
		return 1;
	}
	/*** close file handler ***/
	filp_close(filp,0);

 	/*** Now parse the buffer ***/
	while (buf < bufBeginPtr+fileSize && buf != NULL && *buf != 0)
	{
		mppLnx_parseLine(&buf, "=", &tag, &value);
		mppLnx_modifyConfigParameter(tag, value);
		tag = value = NULL;
	}
	/*** Now free the buffer and other memory resources that we allocated ***/
	MPP_FREE(bufBeginPtr,fileSize);
	return 0;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_modifyConfigParameter
* SUMMARY:  Modify the value of MPP driver configurable parameters according to /etc/mpp.conf
* SCOPE:    local
*
*
* DESCRIPTION: Modify the value of MPP driver configurable parameters according to /etc/mpp.conf
*	       @1: string containing parameter name
*	       @2: string containing parameter value
*
* RESTRICTIONS:
*
* RETURNS: void
*
* NOTES: if error occurs, the parameter values are set back to their default values.
*
*/
static void mppLnx_modifyConfigParameter(char* tag, char* value)
{

	LWORD index = 0;
	BOOL S2ToS3KeyInvalid = FALSE;
	char* endOfString;
	unsigned long parseValue = 0;
	int size;
	if (tag == NULL || value == NULL || *tag == 0 || *value == 0)
		return;
	/*** now parse the parameter value and assign it to proper parameters ***/
	if(strncmp(tag,MPP_DEBUG_LEVEL,strlen(MPP_DEBUG_LEVEL))== 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_Debug = mppLnx_Config_Parameters.mpp_Debug = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_DEBUG_LEVEL,
			mpp_Debug));
		}
	}
	else if(strncmp(tag,MPP_NOTREADYWAITTIME, strlen(MPP_NOTREADYWAITTIME))== 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_NotReadyWaitTime = mppLnx_Config_Parameters.mpp_NotReadyWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_NOTREADYWAITTIME,
			mpp_NotReadyWaitTime));
		}
	}
	else if(strncmp(tag,MPP_BUSYWAITTIME, strlen(MPP_BUSYWAITTIME)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_BusyWaitTime = mppLnx_Config_Parameters.mpp_BusyWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_BUSYWAITTIME,
				mpp_BusyWaitTime));
		}
	}
	else if(strncmp(tag,MPP_QUIESCENCEWAITTIME,strlen(MPP_QUIESCENCEWAITTIME)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_QuiescenceWaitTime = mppLnx_Config_Parameters.mpp_QuiescenceWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_QUIESCENCEWAITTIME,
				mpp_QuiescenceWaitTime));
		}
	}
	else if (strncmp(tag,MPP_INQUIRYWAITTIME,strlen(MPP_INQUIRYWAITTIME)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_InquiryWaitTime = mppLnx_Config_Parameters.mpp_InquiryWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_INQUIRYWAITTIME,
				mpp_InquiryWaitTime));
		}
	}
	else if (strncmp(tag,MPP_MAXLUNSPERARRAY,strlen(MPP_MAXLUNSPERARRAY)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_MaxLunsPerArray = mppLnx_Config_Parameters.mpp_MaxLunsPerArray =  simple_strtoul(value,&endOfString,0);
			if(	(mpp_MaxLunsPerArray > 256)	|| (mpp_MaxLunsPerArray	== 0)) {
				MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 713,
					"The mpp driver	configuration parameter	%s(%d) is incorrect\n",
				MPP_MAXLUNSPERARRAY, mpp_MaxLunsPerArray));
				mpp_MaxLunsPerArray = mppLnx_Config_Parameters.mpp_MaxLunsPerArray = 32;
			}
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_MAXLUNSPERARRAY,
				mpp_MaxLunsPerArray));
		}
	}
	else if (strncmp(tag,MPP_MAXPATHSPERCONTROLLER,strlen(MPP_MAXPATHSPERCONTROLLER)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_MaxPathsPerController = mppLnx_Config_Parameters.mpp_MaxPathsPerController = parseValue;
			if(	(mpp_MaxPathsPerController > 255) || (mpp_MaxPathsPerController	== 0) )	{
				MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 714,
					"The mpp driver	configuration parameter	%s(%d) is incorrect\n",
					MPP_MAXPATHSPERCONTROLLER, mpp_MaxPathsPerController));
				mpp_MaxPathsPerController = mppLnx_Config_Parameters.mpp_MaxPathsPerController = 4;
			}
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n",
				MPP_MAXPATHSPERCONTROLLER, mpp_MaxPathsPerController));
		}
	}
	else if (strncmp(tag,MPP_SCANINTERVAL,strlen(MPP_SCANINTERVAL)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_ScanInterval = mppLnx_Config_Parameters.mpp_ScanInterval = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_SCANINTERVAL,
				mpp_ScanInterval));
		}
	}
	else if (strncmp(tag,MPP_INQUIRYINTERVAL,strlen(MPP_INQUIRYINTERVAL)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_InquiryInterval = mppLnx_Config_Parameters.mpp_InquiryInterval = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_INQUIRYINTERVAL,
				mpp_InquiryInterval));
		}
	}
	else if (strncmp(tag,MPP_MAXARRAYMODULES,strlen(MPP_MAXARRAYMODULES)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_MaxArrayModules = mppLnx_Config_Parameters.mpp_MaxArrayModules = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", "MaxArrayModules",
				mpp_MaxArrayModules));
		}
	}
	else if (strncmp(tag,MPP_ERRORLEVEL,strlen(MPP_ERRORLEVEL)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_ErrorLevel = mppLnx_Config_Parameters.mpp_ErrorLevel = parseValue;
			if(	mpp_ErrorLevel <= 5	) {
				MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_ERRORLEVEL,
					mpp_ErrorLevel));
			} else {
				mpp_ErrorLevel = mppLnx_Config_Parameters.mpp_ErrorLevel = 0;
				MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 715,
				"The mpp driver	configuration parameter	mpp_ErrorLevel is invalid\n	"));
			}
		}
	}
	else if (strncmp(tag,MPP_SELECTIONTIMEOUTRETRYCOUNT,strlen(MPP_SELECTIONTIMEOUTRETRYCOUNT)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_SelectionTimeoutRetryCount = mppLnx_Config_Parameters.mpp_SelectionTimeoutRetryCount = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_SELECTIONTIMEOUTRETRYCOUNT,	mpp_SelectionTimeoutRetryCount));
		}
	}
	else if (strncmp(tag,MPP_COMMANDTIMEOUTRETRYCOUNT,strlen(MPP_COMMANDTIMEOUTRETRYCOUNT)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_CommandTimeoutRetryCount = mppLnx_Config_Parameters.mpp_CommandTimeoutRetryCount = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_COMMANDTIMEOUTRETRYCOUNT, mpp_CommandTimeoutRetryCount));
		}
	}
	else if (strncmp(tag,MPP_UARETRYCOUNT,strlen(MPP_UARETRYCOUNT)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_UaRetryCount = mppLnx_Config_Parameters.mpp_UaRetryCount = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n", MPP_UARETRYCOUNT,
				mpp_UaRetryCount));
		}
	}
	else if (strncmp(tag,MPP_RETRYCOUNT,strlen(MPP_RETRYCOUNT)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_SyncRetryCount = mppLnx_Config_Parameters.mpp_SyncRetryCount = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n", MPP_RETRYCOUNT,
				mpp_SyncRetryCount));
		}
	}
	else if (strncmp(tag,MPP_SYNCHTIMEOUT,strlen(MPP_SYNCHTIMEOUT)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_SynchTimeout = mppLnx_Config_Parameters.mpp_SynchTimeout= parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n", MPP_SYNCHTIMEOUT,
				mpp_SynchTimeout));
		}
	}
	else if (strncmp(tag,MPP_FAILOVERQUIESCETIME,strlen(MPP_FAILOVERQUIESCETIME)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_FailOverQuiescenceTime = mppLnx_Config_Parameters.mpp_FailOverQuiescenceTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_FAILOVERQUIESCETIME, mpp_FailOverQuiescenceTime));
		}
	}
	else if (strncmp(tag,MPP_FAILOVERTIMEOUT,strlen(MPP_FAILOVERTIMEOUT)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_FailoverTimeout = mppLnx_Config_Parameters.mpp_FailoverTimeout = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_FAILOVERTIMEOUT, mpp_FailoverTimeout));
		}
	}
	else if (strncmp(tag,MPPCMN_RECHECKFAILEDPATHWAITTIME,strlen(MPPCMN_RECHECKFAILEDPATHWAITTIME)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mppCmn_RecheckFailedPathWaitTime = mppLnx_Config_Parameters.mpp_RecheckFailedPathWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPPCMN_RECHECKFAILEDPATHWAITTIME, mppCmn_RecheckFailedPathWaitTime));
		}
	}
	else if (strncmp(tag,MPPCMN_FAILEDPATHCHECKINGINTERVAL,strlen(MPPCMN_FAILEDPATHCHECKINGINTERVAL)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mppCmn_FailedPathCheckingInterval = mppLnx_Config_Parameters.mpp_FailedPathCheckingInterval = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPPCMN_FAILEDPATHCHECKINGINTERVAL, mppCmn_FailedPathCheckingInterval));
		}
	}
	else if (strncmp(tag,MPPCMN_IDLEPATHCHECKINGINTERVAL,strlen(MPPCMN_IDLEPATHCHECKINGINTERVAL)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mppCmn_IdlePathCheckingInterval = mppLnx_Config_Parameters.mpp_IdlePathCheckingInterval= parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPPCMN_IDLEPATHCHECKINGINTERVAL, mppCmn_IdlePathCheckingInterval));
		}
	}
	else if (strncmp(tag,MPPCMN_PRINTSENSEBUFFER,strlen(MPPCMN_PRINTSENSEBUFFER)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mppCmn_PrintSenseBuffer = mppLnx_Config_Parameters.mpp_PrintSenseBuffer= parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPPCMN_PRINTSENSEBUFFER, mppCmn_PrintSenseBuffer));
		}
	}
	else if (strncmp(tag,MPP_MAXARRAYFAILOVERLENGTH,strlen(MPP_MAXARRAYFAILOVERLENGTH)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_MaxArrayFailoverLength = mppLnx_Config_Parameters.mpp_MaxArrayFailoverLength  = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_MAXARRAYFAILOVERLENGTH , mpp_MaxArrayFailoverLength ));
		}
	}
	else if (strncmp(tag,MPP_FAILBACKTOCURRENTALLOWED,strlen(MPP_FAILBACKTOCURRENTALLOWED)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_FailBackToCurrentAllowed = mppLnx_Config_Parameters.mpp_FailBackToCurrentAllowed = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_FAILBACKTOCURRENTALLOWED,mpp_FailBackToCurrentAllowed));
		}
	}
	else if (strncmp(tag,MPP_HOLDALTINRESET,strlen(MPP_HOLDALTINRESET)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_HoldAltInReset = mppLnx_Config_Parameters.mpp_HoldAltInReset = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
			MPP_HOLDALTINRESET,mpp_HoldAltInReset));
		}
	}
	else if (strncmp(tag,MPP_VENDORID,strlen(MPP_VENDORID)) == 0)
	{
		size = strlen(value);

		/* Max Supported Vendor ID Str Length is 8  */
		if (size > (sizeof(mpp_VendorId)-1) )
			size = sizeof(mpp_VendorId)-1;

		/* Copy the string */
		strncpy(mppLnx_Config_Parameters.mpp_VendorId,value,size);
		strncpy(mpp_VendorId,value,size);

		/* According to SCSI Spec fill the rest of spaces with 0x20 (space) charecters */
		memset ((mppLnx_Config_Parameters.mpp_VendorId+size),0x20,(sizeof(mpp_VendorId)-size));
		memset ((mpp_VendorId+size),0x20,(sizeof(mpp_VendorId)-size));

		mppLnx_Config_Parameters.mpp_VendorId[sizeof(mpp_VendorId)-1] = '\0';
		mpp_VendorId[sizeof(mpp_VendorId)-1] = '\0';
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s is %s\n",
			MPP_VENDORID,	mpp_VendorId));
	}
	else if (strncmp(tag, MPP_PRODUCTID,strlen( MPP_PRODUCTID)) == 0)
	{
		size = strlen(value);

		/* Max Supported Product ID Str Length is 16  */
		if (size > (sizeof(mpp_ProductId)-1) )
			size = sizeof(mpp_ProductId)-1;

		/* Copy the string */
		strncpy(mppLnx_Config_Parameters.mpp_ProductId,value,size);
		strncpy(mpp_ProductId,value,size);

		/* According to SCSI Spec fill the rest of spaces with 0x20 (space) charecters */
		memset ((mppLnx_Config_Parameters.mpp_ProductId+size),0x20,(sizeof(mpp_ProductId)-size));
		memset ((mpp_ProductId+size),0x20,(sizeof(mpp_ProductId)-size));

		mppLnx_Config_Parameters.mpp_ProductId[sizeof(mpp_ProductId)-1] = '\0';
		mpp_ProductId[sizeof(mpp_ProductId)-1] = '\0';
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s is %s\n",
			MPP_PRODUCTID,	mpp_ProductId));
	}
	else if (strncmp(tag, MPP_S2TOS3KEY,strlen( MPP_S2TOS3KEY)) == 0)
	{
		/*** translate character into digit number ***/
		for (index = 0; index < 16; index++)
		{
			if (value[index] >= '0' && value[index] <= '9')
			{
				value[index] = value[index] + 0 - '0';
			}
			else if (value[index] >= 'a' && value[index] <= 'f')
			{
				value[index] = value[index] + 10 - 'a';
			}
			else if (value[index] >= 'A' && value[index] <= 'F')
			{
				value[index] = value[index] + 10 - 'A';
			}
			else
			{
				MPP_ERRORLOGPRINT ((MPP_ERR_RECOVERED,716,
				"mpp_ReadConfigData: unrecognized char %c in parsing mpp_S2ToS3Key\n",
					value[index]));
				S2ToS3KeyInvalid = TRUE;
				break;
			}
		}
		/*** if the key is invalid, discard the key ***/
		if (S2ToS3KeyInvalid)
			return;
		/*** combine the adjacent two characters into one BYTE ***/
		for (index = 0; index < 8; index++)
		{
			mpp_S2ToS3Key[index] = mppLnx_Config_Parameters.mpp_S2ToS3Key[index] =
			(((BYTE)(value[2*index])) << 4) | ((BYTE)(value[2*index+1]));
		}
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3, "mpp_readConfigData: %s is 0x", MPP_S2TOS3KEY));
		for (index = 0; index < 8; index++)
		{
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,"%02x", mpp_S2ToS3Key[index]));
		}
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,"\n"));
	}

	else if (strstr(tag,MPP_CONTROLLERIOWAITTIME)!= NULL)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_ControllerIoWaitTime = mppLnx_Config_Parameters.mpp_ControllerIoWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_CONTROLLERIOWAITTIME,mpp_ControllerIoWaitTime));
		}
	}
	else if (strncmp(tag,MPP_ARRAYIOWAITTIME,strlen(MPP_ARRAYIOWAITTIME)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_ArrayIoWaitTime = mppLnx_Config_Parameters.mpp_ArrayIoWaitTime = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_ARRAYIOWAITTIME,mpp_ArrayIoWaitTime));
		}
	}
	else if (strncmp(tag, MPP_DISABLELUNREBALANCE, strlen(MPP_DISABLELUNREBALANCE)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mpp_DisableLUNRebalance = mppLnx_Config_Parameters.mpp_DisableLUNRebalance = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPP_DISABLELUNREBALANCE,mpp_DisableLUNRebalance));
		}
	}
	else if (strncmp(tag, MPPCMN_LUNFAILOVERDELAY, strlen(MPPCMN_LUNFAILOVERDELAY)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			if (parseValue > 60 || parseValue < 0)
			{
				parseValue = 3;
			}
			mppCmn_LunFailoverDelay = mppLnx_Config_Parameters.mpp_LunFailoverDelay = parseValue;
			MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
				MPPCMN_LUNFAILOVERDELAY, mppCmn_LunFailoverDelay));
		}
	}
	else if (strncmp(tag, MPPCMN_CLASSICMODEFAILOVER, strlen(MPPCMN_CLASSICMODEFAILOVER)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue) && parseValue == MPPCMN_FAILOVER_METHOD_LUN )
			mppCmn_ClassicModeFailover = mppLnx_Config_Parameters.mpp_ClassicModeFailover = MPPCMN_FAILOVER_METHOD_LUN;
		else
			mppCmn_ClassicModeFailover = mppLnx_Config_Parameters.mpp_ClassicModeFailover = MPPCMN_FAILOVER_METHOD_CONTROLLER;
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
			MPPCMN_CLASSICMODEFAILOVER, mppCmn_ClassicModeFailover));
	}
	else if (strncmp(tag, MPPCMN_AVTMODEFAILOVER, strlen(MPPCMN_AVTMODEFAILOVER)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue) && parseValue == MPPCMN_FAILOVER_METHOD_LUN )
			mppCmn_AVTModeFailover = mppLnx_Config_Parameters.mpp_AVTModeFailover = MPPCMN_FAILOVER_METHOD_LUN;
		else
			mppCmn_AVTModeFailover = mppLnx_Config_Parameters.mpp_AVTModeFailover = MPPCMN_FAILOVER_METHOD_CONTROLLER;
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,	"mpp_readConfigData: Configuration Parameter %s =	%d\n",
			MPPCMN_AVTMODEFAILOVER, mppCmn_AVTModeFailover));
	}
	else if (strncmp(tag, MPP_LOAD_BALANCE_POLICY, strlen(MPP_LOAD_BALANCE_POLICY)) == 0)
	{
		if (mppLnx_simple_strtoul(value,&endOfString,0,&parseValue))
		{
			mppCmn_LoadBalancePolicy = mppLnx_Config_Parameters.mpp_LoadBalancePolicy = parseValue;

			/* Right now we support only RoundRobin(0) and LeastQueueDepth(1)-default */
			/* Check if the value is either one of this */
			if(( mppCmn_LoadBalancePolicy < 0 ) || ( mppCmn_LoadBalancePolicy > 1 )) {
				mppCmn_LoadBalancePolicy = mppLnx_Config_Parameters.mpp_LoadBalancePolicy = 1;
				MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 745,
				"The mpp driver	configuration parameter	mppCmn_LoadBalancePolicy is invalid\n	"));
			}else {
				MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
					"mpp_readConfigData: Configuration Parameter %s =	0x%x\n", MPP_LOAD_BALANCE_POLICY,
					mppCmn_LoadBalancePolicy));
			}
		}
	}

	else {
		// unrecognized parameter
		MPP_ERRORLOGPRINT ((MPP_ERR_RECOVERED,717, "mpp_readConfigData: Unrecognized parameter tag %s\n", tag));
	}

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_simple_strtoul
* SUMMARY:  convert a string to an unsigned long integer
* SCOPE:    local
*
*
* DESCRIPTION: This function convert a string to an unsigned long integer. It does sanity checking for the string, also.
* RESTRICTIONS:
*
*
* ERRNO: 1
*
*/
BOOL mppLnx_simple_strtoul(const char* cp, char** endp, unsigned int base, unsigned long *parseValue)
{
	*parseValue = simple_strtoul(cp, endp, base);
	if (**endp !='\0')
	{
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 718,
			"The mpp driver	configuration parameter	%d is incorrect\n",
			*parseValue));
		**endp = '\0';
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_readBindingData
* SUMMARY:  read persistent binding file
* SCOPE:    local
*
*
* DESCRIPTION: This function is called in MPP upper level driver initialization routine to read persistent binding information.
*              This information is used to fill in the array name information in the proper slot in mpp_ModuleArray
*              according to the virtual target ID of the array.
*              The file has the following format:
*              <VirtualId1>:<arrayName1>'\n'
*              <VirtualId2>:<arrayName2>'\n'
*              ...
*              <VirtualIdN>:<arrayNameN>'\n'
*	       leading or trailing spaces are allowed.
*              Entries that exceed the maximum number of array modules that MPP driver can support will be discarded.
*              Comment lines are allowed as long as they do not have '='
*
* RESTRICTIONS:
*
* RETURNS: 0 if suceeds, 1 if error (file open error, kernel read error, etc)
*
* ERRNO: 1
*
* NOTES: if error occurs, no persistent binding information is used.
*
*/
static int mppLnx_readBindingData(void)
{
	struct file* filp = NULL;
	char* buf = NULL;
	char* bufBeginPtr = NULL;
	char* virtualId = NULL;
	char* arrayName = NULL;
	int bytesRead = 0;
	int numberOfArrays = 0;
	LWORD fileSize = 0;
	/*** open parameter file /var/mpp/devicemapping ***/
	filp = filp_open(BINDING_FILE_NAME,O_RDONLY,0);
	if(!filp || IS_ERR(filp)) {
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 719,
			"mppLnx_readBindingData: Error opening /var/mpp/devicemapping\n"));
		return 1;
	}
	fileSize = filp->f_dentry->d_inode->i_size;
	buf = MPP_KMEM_ALLOC(fileSize);
	if(!buf) {
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 720,
			"mppLnx_readBindingData: Error allocating buffer of size %d\n", fileSize ));
		filp_close(filp,0);
		return 1;
	}
	bufBeginPtr = buf;

	/*** Now read the persistent binding file into the buffer ***/
	bytesRead = kernel_read(filp,0,buf,fileSize);

	/*** If read is incomplete, return error status without parsing the buffer ***/
	if(bytesRead != fileSize) {
		MPP_ERRORLOGPRINT ((MPP_ERR_RECOVERED,721,"mpp_ReadBindingData: need to read %d bytes. Only get %d bytes.\n",
			filp->f_dentry->d_inode->i_size,bytesRead));
		filp_close(filp,0);
		MPP_FREE(buf,fileSize);
		return 1;
	}
	filp_close(filp,0);

 	/*** Now parse the buffer ***/
	while (buf < bufBeginPtr+fileSize && numberOfArrays < mpp_MaxArrayModules && buf != NULL && *buf != 0)
	{
		mppLnx_parseLine(&buf, ":",&virtualId, &arrayName);
		mppLnx_FillInPersistentArray(virtualId, arrayName);
		numberOfArrays++;
		virtualId = arrayName = NULL;
	}
	/*** Now free the buffer and other memory resources that we allocated ***/
	MPP_FREE(bufBeginPtr, fileSize);
	return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_FillInPersistentArray
* SUMMARY:  Fill in module name information for persistent binding arrays
* SCOPE:    local
*
*
* DESCRIPTION: Fill in module name information for persistent binding arrays
*              @1: The virtual target id of the array
*              @2: The array name
*
* RESTRICTIONS:
*
* RETURNS: void
*
* NOTES: 1. discard any array information with virtual target id exceeding the number that we support
*        2. discard any array information if there is conflicting array name in the slot
*
*/
static void mppLnx_FillInPersistentArray(char* virtualId, char* arrayName)
{
	struct ArrayModuleEntry* pArray = NULL;
	LWORD arrayVirtualId = 0;
	if (virtualId == NULL || arrayName == NULL || *virtualId == 0 || *arrayName == 0)
		return;
	arrayVirtualId = simple_strtoul(virtualId,NULL,0);

	if(arrayVirtualId >= mpp_MaxArrayModules)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 722,
			"mppLnx_readBindingData: The mpp driver	virtual ID %d is incorrect\n", arrayVirtualId));
	}
	else
	{
		/*** Now assign the array name to the mpp_ModuleArray slot
		with proper virtual target ID.***/
		pArray = mpp_ModuleArray + arrayVirtualId;
		if (pArray->ModuleNameInConfig)
		{
			MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 723,
				"mppLnx_FillInPersistentArray: Array slot %d has duplicate information\n", arrayVirtualId));
		}

		strncpy(pArray->ModuleName, arrayName, DEVNM_LEN-1);
		pArray->ModuleName[DEVNM_LEN-1] = '\0';
		pArray->ModuleNameInConfig = TRUE;
		MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
			"Assign array name %s to virtual target ID %d\n",
			arrayName, arrayVirtualId));
	}

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_parseLine
* SUMMARY:  Line parsing function to parse a line into two fields
* SCOPE:    local
*
* DESCRIPTION: This function is called to parse a buffer into two fields delimited by
*              delimiter and strip off all spaces around the field.
*              Lines without '=' (delimiter) are treated as comment lines and are thus ignored.
*              @1: Address of the buffer to be parsed
*              @2: delimiter character
*              @3: address of the string buffer that is to hold the first part
*              @4: address of the secoond buffer that is to hold the first part
*
* RESTRICTIONS:
*
* RETURNS: void
*
* ERRNO:
*
* NOTES:
*
*/
static void mppLnx_parseLine(char** buf, char* delimiter, char** part1, char** part2)
{
	/* In the following cases:
	   1. The buf contains NULL content
	   2. The buffer pointer is invalid (NULL)
	   return immediately */
	if (*buf == NULL || **buf == 0)
	{
		return ;
	}
	/* devide the buffer into two fields seperated with delimiter */
	*part2 = strsep(buf,"\n");
	*part1 = strsep(part2, delimiter);
	if (part2 == NULL || *part2 == 0)
		return ;
	/* strip off all spaces around the fields */
	*part1 = mppLnx_strstrip(*part1, " \t");
	*part2 = mppLnx_strstrip(*part2, " \t");
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_strstrip
* SUMMARY:  strip leading and trailing characters in string s off string str
* SCOPE:    local
*
*
* DESCRIPTION:  strip leading and trailing characters in string s off string str
*		@1: string to be processed
*		@2: the characters to be stripped off
*
* RESTRICTIONS:
*
* RETURNS: char *
*/
static char* mppLnx_strstrip (char *str, char* s)
{
        char *eos = str + strlen(str);          /* -> end of string */

        while (*str && strchr(s, *str))
                ++str                           /* strip leading spaces */
        ;
        while ((eos > str) && strchr(s, *(eos - 1)))
                --eos                           /* strip trailing spaces */
        ;
        *eos = '\0';
        return str;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_sd_probe
* SUMMARY:  interception routine for sd device driver
* SCOPE:    local
*
*
* DESCRIPTION: sd driver claimes Engenio virtual devices and non-Engenio devices only.
*				It does not claim Engenio physical devices.
*
* RESTRICTIONS:
*
* RETURNS: 0 for Engenio virtual devices and non-Engenio devices,
*	   -ENODEV for Engenio physical devices.
*
* ERRNO:
*
* NOTES:
*
*/
static int mppLnx_sd_probe(struct device *dev)
{
	struct scsi_device *sdp = to_scsi_device(dev);
	ControllerAddress_t              ctlAddress;

	RdacDeviceInformation_t	*pRdacDevice;
	RdacLunInformation_t	*pRdacLun;
	LINT LockLevel;

	ctlAddress.ChannelId = sdp->channel;
	ctlAddress.TargetId  = sdp->id;
	ctlAddress.HostId    = sdp->host->host_no;

	MPP_DEBUGPRINT ((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_sd_probe invoked for H%dT%dL%d\n",
			sdp->host->host_no, sdp->id, sdp->lun));

	/* check if the struct scsi_device is a virtual device */
	/* virtual device: sd attaches to it */
	if ((sdp->host->hostt->proc_name != NULL) &&
      strcmp(sdp->host->hostt->proc_name, MPPLNX_PROC_NAME) == 0)
	{
    	//make sure you point to the right RdacInfo while locking and unlocking
		pRdacDevice = mpp_ModuleArray[(sdp->id)].RdacInfo;
		pRdacLun = &pRdacDevice->RdacLuns[sdp->lun];


      if(unlikely(!pRdacLun->Present && sdp->lun == 0))
      {
         /*
         * This is the dummy lun 0. We don't want give it to the sd driver.
         * We would like to put it into the un-configured device list
         */
         mppLnx_process_dummy_lun0(sdp);
         return -ENODEV;
      }
		if ( !pRdacLun->PseudoLunObject ) {
			MPP_LOCK(pRdacDevice->Lock, LockLevel);
			pRdacLun->PseudoLunObject = sdp;
			pRdacLun->ReportedMissing = 0;
			MPP_UNLOCK(pRdacDevice->Lock, LockLevel);
		}

		return sd_old_probe(dev);
	}

        /*
        * workaround for setting mppLnx_host_procName
        * For SUSE 10 SP1 and RHEL 5 sdp->host->hostt->module->name is "<NULL>"
        * and that is displayed in /proc/mpp so replacing that by proc_name
        */
        if ( (sdp->host->hostt->proc_name != NULL)&& strncmp(sdp->host->hostt->proc_name,"iscsi",strlen("iscsi")) == 0 )
        {
              mppLnx_host_procName = sdp->host->hostt->proc_name;
        }
        else
        {
              mppLnx_host_procName = sdp->host->hostt->module->name;
        }

	/* insert the struct scsi_device into mpp_ModuleArray */
	if(mppLnx_BuildRdacinfo((MPP_HANDLE)sdp, &ctlAddress, sdp->lun))
    	{
		return -ENODEV;

	}
	else  /* the struct scsi_device is a non-Engenio device */
	{
		return sd_old_probe(dev);
	}

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_driver_probe
* SUMMARY:  MPP device driver probe interface
* SCOPE:    local
*
*
* DESCRIPTION: MPP driver claims Engenio physical devices.
*
* RESTRICTIONS:
*
* RETURNS: 0 for Engenio physical devices,
*	   -ENODEV for Engenio virtual devices and non-Engenio devices.
*
* ERRNO:
*
* NOTES: This function always returns sccess
*
*/
static int mppLnx_driver_probe(struct device *dev)
{
	RdacDeviceInformation_t	*pRdacDevice;
	RdacLunInformation_t	*pRdacLun;
	LINT LockLevel;
	struct scsi_device *sdp = to_scsi_device(dev);
	ControllerAddress_t              ctlAddress;
	void (*mppLnx_newDeviceAdded)(struct scsi_device * sdp);

	mppLnx_UnconfiguredDeviceEntry_t * unConfiguredDeviceEntry = NULL;
	BOOL unconfiguredDevNeedsClaim = FALSE;
	LunPathInfo_t *lunPathInfo = NULL;
	LWORD device_type;

	ctlAddress.ChannelId = sdp->channel;
	ctlAddress.TargetId  = sdp->id;
	ctlAddress.HostId    = sdp->host->host_no;


	MPP_DEBUGPRINT ((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_driver_probe invoked for H%dT%dL%d\n", sdp->host->host_no, sdp->id, sdp->lun));

	/* check if the struct scsi_device is a virtual device */
	/* virtual device: sd attaches to it */
	if ( (sdp->host->hostt->proc_name != NULL) &&
        strcmp(sdp->host->hostt->proc_name, MPPLNX_PROC_NAME) == 0)
	{
    	//make sure you point to the right RdacInfo while locking and unlocking
		pRdacDevice = mpp_ModuleArray[(sdp->id)].RdacInfo;
		pRdacLun = &pRdacDevice->RdacLuns[sdp->lun];
		if ( !pRdacLun->PseudoLunObject ) {
			MPP_LOCK(pRdacDevice->Lock, LockLevel);
			pRdacLun->PseudoLunObject = sdp;
			pRdacLun->ReportedMissing = 0;
			MPP_UNLOCK(pRdacDevice->Lock, LockLevel);
		}

		return -ENODEV;
	}

	/*
	* workaround for setting mppLnx_host_procName
	*/
	mppLnx_host_procName = sdp->host->hostt->module->name;

	/* insert the struct scsi_device into mpp_ModuleArray */
	if(mppLnx_BuildRdacinfo((MPP_HANDLE)sdp, &ctlAddress, sdp->lun))
    	{

		unConfiguredDeviceEntry = mppLnx_getUnconfiguredDevice(sdp);

		if (unConfiguredDeviceEntry != NULL)
		{
      			if (unConfiguredDeviceEntry->isUnconfigured == TRUE &&
         		unConfiguredDeviceEntry->isDetected == FALSE )
      			{
         			unConfiguredDeviceEntry->isDetected = TRUE;
         			unConfiguredDeviceEntry->isAttached = TRUE;
				unconfiguredDevNeedsClaim = TRUE;
      			}
			if (!unconfiguredDevNeedsClaim)
				return -ENODEV;
		}

		/* increase device reference counter in rdacinfo data structure */
		if (( device_type = mppLnx_DeviceType(sdp, &lunPathInfo, &pRdacDevice)) ) {
			if (lunPathInfo != NULL)
			{
				lunPathInfo->refCount++;
			}
		}

                mppLnx_newDeviceAdded = (void *)symbol_get(mppLnx_newDeviceAdded);

                if(mppLnx_newDeviceAdded) {
                     (*mppLnx_newDeviceAdded) (sdp);
                     symbol_put(mppLnx_newDeviceAdded);
                }
                else {
                        MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 729,
                                 "mppLnx_driver_probe: symbol_get did not return a function pointer of mppLnx_newDeviceAdded\n"));
                }

		return 0;

	}
	else  /* the struct scsi_device is a non-Engenio device */
	{
		return -ENODEV;
	}

	return -ENODEV;

}

/****************************************************************************
* PROCEDURE
* NAME:    mppLnx_failbackrescan
* SUMMARY: This function starts the failbackscan thread
* SCOPE:   public
* DESCRIPTION: This function just calls the failbackrescan function provided by the virtual HBA driver.
* This function is called in response to an ioctl call to the Upper Level driver.
* RESTRICTIONS:
* RETURNS: void
* ERRNO:
* NOTES:
* EXAMPLES:
* SEE ALSO:
****************************************************************************/

static void
mppLnx_failBackRescan(void )
{
  /*
   * The function pointer mppLnx_failbackRescan will be retrieved from the mppVhba driver module through symbol_get() kernel API.
   * the name of the function pointer must be same as that of the exported symbol name of the mppVhba driver module.
   * The exported symbols of the mppVhba driver module is defined in mppLnx26p_vhbamisc.c
   */
   void (*mppLnx_failbackRescan) (void );

	mppLnx_failbackRescan = (void *) symbol_get(mppLnx_failbackRescan);

	if(mppLnx_failbackRescan) {
		(*mppLnx_failbackRescan) ();
		symbol_put(mppLnx_failbackRescan);
	}
	else
   {
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 724,
			"mppLnx_failBackRescan: symbol_get did not return a function pointer\n"));
   }

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_driver_remove
* SUMMARY:  MPP device driver interface remove routine
* SCOPE:    local
*
* DESCRIPTION: This function is invoked when an Engenio physical device is going to be removed.
*               It tears down associated data structure in rdacInfo for this physical device.
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES:
*
*/
static int mppLnx_driver_remove(struct device *dev)
{
	struct scsi_device *sdp = to_scsi_device(dev);

	MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
	"mppLnx_driver_remove mppLnx_removeLun invoked for h%dt%dl%d, device vertex 0x%p\n",
	sdp->host->host_no, sdp->id, sdp->lun, sdp));

	/* remove the scsi_device from RdacInfo data structure if we are the last one to be invoked */
	mppLnx_removeLun(sdp);
	return 0;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_removeLun
* SUMMARY:  Linux MPP fucntion to remove a scsi_device from RdacInfo
* SCOPE:    local
*
* DESCRIPTION: This function removes a scsi_device from RdacInfo
*               It tears down associated data structure in rdacInfo for this physical device.
*
* RESTRICTIONS:
*
* RETURNS: none
*
* ERRNO:
*
* NOTES:
*
*/
static void mppLnx_removeLun(struct scsi_device *sdp)
{
	ControllerAddress_t ctlAddress;
	ArrayModuleEntry_t *ModuleArray;
	RdacDeviceInformation_t* RdacInfo;
	RdacLunInformation_t *pRdacLun;
	LunPathObjects_t *pCtlPath;
	LunPathInfo_t *pLunPathInfo;
	BOOL pathAlreadyFound = FALSE;
	BYTE ctlIndex;
	BYTE pathIndex;
	LWORD count;
	LINT LockLevel;

	MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3,
	"mppLnx_removeLun invoked for h%dt%dl%d, device vertex 0x%p\n",
	sdp->host->host_no, sdp->id, sdp->lun, sdp));


	ctlAddress.ChannelId = sdp->channel;
	ctlAddress.TargetId  = sdp->id;
	ctlAddress.HostId    = sdp->host->host_no;

	/* Found the device information in RdacInfo. If it is in RdacInfo data structure,
       remove the asssociated structure from RdacInfo
	   and decrement the attach counter for that device if it is a physical device.*/
	for(count=0; count<mpp_MaxArrayModules; ++count)
	{
		ModuleArray = mpp_ModuleArray+count;
		if( (RdacInfo = ModuleArray->RdacInfo) )
		{
			if(RdacInfo->RdacLuns[sdp->lun].Present || (RdacInfo->ControllerInfo[0]->UTMLunNumber == sdp->lun) || (RdacInfo->ControllerInfo[1]->UTMLunNumber == sdp->lun)) {
				pRdacLun = &RdacInfo->RdacLuns[sdp->lun];
				for (ctlIndex = 0; ctlIndex < 2; ctlIndex++)
				{
					pCtlPath = pRdacLun->LunControllers[ctlIndex];
					/* We need to loop through 0..MaxLunsPerArray to real with the following scenarios:
					   We have Lun 1,2,3, the user removes a single lun(e.g., Lun0 and leave Lun1, Lun2,
					   if we loop through 0.. NumberofLunObjects(2), we can not find Lun2
					   in RdacInfo data structure. */
					for (pathIndex = 0; pathIndex < mpp_MaxPathsPerController; pathIndex++)
					{
						pLunPathInfo = &pCtlPath->LunPaths[pathIndex];
						if (!pLunPathInfo->LunPathDevice)
						{
							continue;
						}
						else
						{
							if ((struct scsi_device*)pLunPathInfo->LunPathDevice == sdp)
							{
                        /*
                        * set the devic path to failed state unconditionally
                        */
			/* CR142652 - Reomved the earlier code to call mpp_FailPath */
                                                                MPP_LOCK(RdacInfo->Lock, LockLevel);
                                                                MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo, ctlIndex, pathIndex , sdp->lun, MPPCMN_FAILED);
                                                                pLunPathInfo->refCount--;
                                                                MPP_UNLOCK(RdacInfo->Lock, LockLevel);
								if ( pLunPathInfo->refCount == 0 )
								{
									mpp_RemoveLun((MPP_HANDLE)sdp, RdacInfo, ctlIndex, pathIndex, sdp->lun);
            								pRdacLun->ReportedMissing = FALSE;
									mppLnx_CheckRemoveModule(RdacInfo,sdp);
									MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3,"mppLnx_removeLun: Removed physical dev 0x%p, T%d L%d Array %s, Ctl %c, Lun %d, Path %d\n",
									    sdp, sdp->id,sdp->lun, ModuleArray->ModuleName, ctlIndex+'A'-0, sdp->lun, pathIndex));
								}
								pathAlreadyFound = TRUE;
								break;
							}
							else
							{
								continue;
							}
						}
					}
					if (pathAlreadyFound)
						break;
				}
				if(pathAlreadyFound)
					break;
			}
		}
		if(pathAlreadyFound)
			break;
	}
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_class_add
* SUMMARY:  MPP class interface add routine
* SCOPE:    local
*DESCRIPTION: no-op
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES:
*
*/
int mppLnx_class_add(struct class_device *cl_dev,struct class_interface *cl_intf)
{
#ifdef MPP_DEBUG
	struct scsi_device *sdp = to_scsi_device(cl_dev->dev);
	MPP_DEBUGPRINT ((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_2, "mppLnx_class_add: add scsi device H%dT%dL%d\n", sdp->host->host_no, sdp->id, sdp->lun));
#endif
	return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_class_remove
* SUMMARY:  MPP class interface remove routine
* SCOPE:    local
*
* DESCRIPTION: no-op
*
* RESTRICTIONS:
*
* RETURNS: none
*
* ERRNO:
*
* NOTES:
*
*/
void mppLnx_class_remove(struct class_device *cl_dev,struct class_interface *cl_intf)
{
#ifdef MPP_DEBUG
	struct scsi_device *sdp = to_scsi_device(cl_dev->dev);
	MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_2, "mppLnx_class_remove: remove scsi device H%dT%dL%d\n", sdp->host->host_no, sdp->id, sdp->lun));
#endif
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_init_command
* SUMMARY:  MPP upper level driver command initialization routine
* SCOPE:    local
*
* DESCRIPTION: This function is a dummy function. It should never be invoked.
*
* RESTRICTIONS:
*
* RETURNS: 1
*
* ERRNO:
*
* NOTES: This function log an error message in MPP error log
*
*/
static int mppLnx_init_command(struct scsi_cmnd *sdp) {
	MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 725, "mpp_target_init_command: invoked"));
        return 1;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_open
* SUMMARY:  MPP upper level driver open function
* SCOPE:    local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION: This function is called by the kernel whenever some program tries to open the
* upper level character driver.
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES: This function always returns success
*
*/
static int mppLnx_open (struct inode *inode, struct file *file)
{
	return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_release
* SUMMARY:  MPP upper level driver release function
* SCOPE:    local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION: This function is called by the kernel whenever some program tries to close the
* upper level character driver.
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES: This function always returns success
*
*/
static int mppLnx_release (struct inode *inode, struct file *file)
{
	return 0;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_ioctl
* SUMMARY:  MPP upper level driver ioctl function
* SCOPE:    local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION: This function is called by the kernel whenever some program tries to use the ioctl
* interface of the upper level character driver.
*
* RESTRICTIONS:
*
* RETURNS: 0
*
* ERRNO:
*
* NOTES: This function returns success or EFAULT or ENXIO depending on the outcome of the
* copy-to-user and copy-from-user operations
*
*/

static int mppLnx_ioctl(struct inode *inode, struct file *file, unsigned int cmd_in, unsigned long arg)
{
	getRdacInfoIoctl_t UserArg;
	mpp_utilGetStatusIoctl_t UtilArg;
	ArrayModuleEntry_t *ModPtr;
	RdacDeviceInformation_t *RdacInfo;
	mpp_utilStatusInfo_t *utilStatusPtr;
	BOOL bSilent = FALSE;
	int ioctl_status = 0;
	struct host_target_t host_target_arg;
	BYTE arrayIndex = 0;
	BYTE controllerIndex = 0;
	BYTE pathIndex = 0;
	LWORD lunIndex = 0;
	MPP_HANDLE deviceObject = NULL;
	LINT LockLevel;
	LWORD UTMLun;
	BOOL UTMEnabled;
	BOOL devFound = FALSE;
	LWORD  StateIndex;
   /*
   * The function pointer mppLnx_vhbaScanHost will be retrieved from the mppVhba driver module through symbol_get() kernel API.
   * the name of the function pointer must be same as that of the exported symbol name of the mppVhba driver module.
   * The exported symbols of the mppVhba driver module is defined in mppLnx26p_vhbamisc.c
   */
   void (*mppLnx_vhbaScanHost) (int );


	MPP_DEBUGPRINT((MPP_IOCTL_DEBUG+MPP_DEBUG_LEVEL_4, "mpp_ioctl: called for cmd = 0x%lx, arg = 0x%lx\n", cmd_in, arg));


	switch (cmd_in) {
		/* Return the MaxPathsPerController parameter */
	       	case MPP_IOCTL_GETPATHS:
			ioctl_status = copy_to_user((void *)arg, &mpp_MaxPathsPerController, sizeof(LWORD));
			if( ioctl_status )
				ioctl_status = -EFAULT;
			break;
		/* Return the MaxLunsPerArray parameter */
       		case MPP_IOCTL_GETLUNS:
			ioctl_status = copy_to_user((void *)arg, &mpp_MaxLunsPerArray,  sizeof(LWORD));
			if( ioctl_status )
				ioctl_status = -EFAULT;
			break;
		/* Return the MaxArrayModules parameter */
       		case MPP_IOCTL_GETMODULES:
			ioctl_status = copy_to_user((void *)arg, &mpp_MaxArrayModules,  sizeof(LWORD));
			if( ioctl_status )
				ioctl_status = -EFAULT;
			break;
		/* Return the MaxTargetPerPhysicalHost parameter for the physical host*/
       		case MPP_IOCTL_GET_PHYSICALHOST_MAX_TARGET:
			ioctl_status = copy_from_user(&host_target_arg, (void *)arg, sizeof(struct host_target_t));
			/* find the first available device object */
			for (arrayIndex = 0; arrayIndex < mpp_MaxArrayModules; arrayIndex++)
			{
				RdacInfo = mpp_ModuleArray[arrayIndex].RdacInfo;
				if( !RdacInfo )
					continue;
				for (lunIndex=0; lunIndex < mpp_MaxLunsPerArray; ++lunIndex)
				{
					if (!RdacInfo->RdacLuns[lunIndex].Present)
						continue;
					for (controllerIndex = 0; controllerIndex < 2; ++controllerIndex)
					{
						if(!RdacInfo->ControllerInfo[controllerIndex]->ControllerPresent)
							continue;
						for(pathIndex = 0; pathIndex < mpp_MaxPathsPerController; ++pathIndex)
						{
							if(!RdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].Present ||
							!RdacInfo->RdacLuns[lunIndex].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice)
								continue;
							deviceObject = RdacInfo->RdacLuns[lunIndex].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice;
							devFound = TRUE;
							break;
						}
						if (devFound)
							break;
					}
					if (devFound)
						break;
				}
				if (devFound)
					break;
			}
			/* get the host pointer from scsi_device object.
			   If there is no object available (no array is attached or no LUN is mapped),
			   a value of -1 is returned */
			if (deviceObject != NULL)
				host_target_arg.max_target_id = deviceObject->host->max_id;
			else
				host_target_arg.max_target_id = -1;

			ioctl_status = copy_to_user((void *)arg, &host_target_arg, sizeof(struct host_target_t));
			if( ioctl_status )
				ioctl_status = -EFAULT;

			break;

		/* Return the UtilStatusInfo corresponding to the virtual target ID supplied */
                case MPP_IOCTL_UTIL_STATUS:
                        ioctl_status = copy_from_user(&UtilArg, (void *)arg, sizeof(mpp_utilGetStatusIoctl_t));
                        if( ioctl_status ) {
                                ioctl_status = -EFAULT;
                                break;
                        }

                        ModPtr = mpp_ModuleArray;
                        if( UtilArg.mpp_Target > (mpp_MaxArrayModules-1) ) {
                                ioctl_status = -ENXIO;
                                break;
                        }

                        ModPtr += UtilArg.mpp_Target;
                        RdacInfo = ModPtr->RdacInfo;
                        if( !RdacInfo ) {
                                ioctl_status = -ENXIO;
                                /*
                                * the caller goes through all virtual target ids from 0 to
                                * MaxArrayModules. It is not an error if the virtual target slot
                                * doesn't have its RdacInfo. Don't log error message
                                */
                                bSilent = TRUE;
                                break;
                        }


                        if( UtilArg.mpp_BufferSize < MPP_UTILSTATUSSIZE(mpp_MaxLunsPerArray, mpp_MaxPathsPerController) ) {
                                MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 726, "mppLnx_ioctl: ioctl failed on device input Buffer size %d, UtilInfo size %d\n", UtilArg.mpp_BufferSize, MPP_UTILSTATUSSIZE(mpp_MaxLunsPerArray, mpp_MaxPathsPerController)));
                                ioctl_status = -ENOMEM;
                                break;
                        }


                        if (( utilStatusPtr = (mpp_utilStatusInfo_t *)MPP_KMEM_ALLOC(MPP_UTILSTATUSSIZE( mpp_MaxLunsPerArray , mpp_MaxPathsPerController))) == NULL )
                        {
                                MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 727, "mppLnx_ioctl: Failed Allocating mpp_UtilStatusInfo\n"));
                                ioctl_status = -ENOMEM;
                                break;
                        }

                        mppCmn_GetArrayStatus(RdacInfo, utilStatusPtr);

                        ioctl_status = copy_to_user(UtilArg.mpp_StatusAddress, utilStatusPtr, MPP_UTILSTATUSSIZE(mpp_MaxLunsPerArray, mpp_MaxPathsPerController));

                        if( ioctl_status ) {
                                ioctl_status = -EFAULT;
                        }

                        MPP_FREE(utilStatusPtr, sizeof(MPP_UTILSTATUSSIZE(mpp_MaxLunsPerArray, mpp_MaxPathsPerController)));
                        break;

		/* Return the RdacInfo corresponding to the virtual target ID supplied */
       		case MPP_IOCTL_GETRDACINFO:
		{
			ioctl_status = copy_from_user(&UserArg, (void *)arg, sizeof(getRdacInfoIoctl_t));
			if( ioctl_status ) {
				ioctl_status = -EFAULT;
				break;
			}

			ModPtr = mpp_ModuleArray;
			if( UserArg.target > (mpp_MaxArrayModules-1) ) {
				ioctl_status = -ENXIO;
				break;
			}
			ModPtr += UserArg.target;
			RdacInfo = ModPtr->RdacInfo;
			if( !RdacInfo ) {
            /*
             * the caller (mppUtil -a) goes through all virtual target ids from 0 to
             * MaxArrayModules. It is not an error if the virtual target slot
             * doesn't have its RdacInfo. Don't log error message
             */
            bSilent = TRUE;
				ioctl_status = -ENXIO;
				break;
			}
			if( UserArg.BufferSize < MPP_RDACINFOSIZE(mpp_MaxLunsPerArray, mpp_MaxPathsPerController) ) {
				MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 728, "mppLnx_ioctl: ioctl failed on device input Buffer size %d, RdacInfo size %d\n", UserArg.BufferSize, MPP_RDACINFOSIZE(mpp_MaxLunsPerArray, mpp_MaxPathsPerController)));
				ioctl_status = -ENOMEM;
				break;
			}

			ioctl_status = (int) mpp_SysInterface->mppSys_CopyRdacInfoToUser( UserArg.RdacInfoAdrs, RdacInfo);
			if( ioctl_status ) {
				break;
			}

			ioctl_status = copy_to_user( UserArg.RdacName, RdacInfo->ModuleName, sizeof(DevName_t) );
			if( ioctl_status ) {
				ioctl_status = -EFAULT;
				break;
			}
			break;
		}
       		case MPP_IOCTL_FAILBACKSCAN:
			/* Call the FailBackScan function */
			mppLnx_failBackRescan();
			break;
       		case MPP_IOCTL_AVTCHECKSCAN:
			/* Call the CheckAVTState function */
			mpp_CheckAVTState();
			break;

        	case MPP_IOCTL_BUSSCAN:
    			mppLnx_RescanUnconfiguredDevices();
           /*
            *Check whether or not the virtual host has been created in case at the virtual
            * HBA driver initialization time, no Engenio's device was discovered.
            */
	         mppLnx_vhbaScanHost = (void *)symbol_get(mppLnx_vhbaScanHost);

	         if(mppLnx_vhbaScanHost)
            {
		         (*mppLnx_vhbaScanHost) (0);
		         symbol_put(mppLnx_vhbaScanHost);
	         }
	         else
            {
		         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 743, 
			         "mppLnx_ioctl: symbol_get did not return a function pointer of mppLnx_vhbaScahHost\n"));
            }
      			break;
		case MPP_IOCTL_SETDEBUG:
			/* Set the debug value to the value supplied by the user */
			ioctl_status = copy_from_user( &mpp_Debug, (void *)arg, sizeof(LWORD));
			mppLnx_Config_Parameters.mpp_Debug = mpp_Debug;
			if( ioctl_status )
				ioctl_status = -EFAULT;
			break;

		case MPP_IOCTL_SETERROR:
			/* Set the error level to the value supplied by the user */
			ioctl_status = copy_from_user( &mpp_ErrorLevel, (void *)arg, sizeof(LWORD));
			mppLnx_Config_Parameters.mpp_ErrorLevel = mpp_ErrorLevel;
			if( ioctl_status )
				ioctl_status = -EFAULT;
			break;

		case MPP_IOCTL_GET_UTMLUN_INFO:
			/* Refresh UTM Lun information for all discovered arrays */
			deviceObject = NULL;
			for(arrayIndex=0; arrayIndex<mpp_MaxArrayModules; ++arrayIndex)
			{
				if (!mpp_ModuleArray[arrayIndex].RdacInfo)
					continue;
				RdacInfo = mpp_ModuleArray[arrayIndex].RdacInfo;
				for(controllerIndex = 0; controllerIndex < 2; ++controllerIndex)
				{

					for(pathIndex=0; pathIndex < mpp_MaxPathsPerController && !deviceObject; ++pathIndex)
					{
						StateIndex = RdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].PathStateIndex;
						if(!RdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].Present ||
			    			RdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].PathState[StateIndex] == MPPCMN_FAILED)
						{
							continue;
						}

						for(lunIndex=0; lunIndex < mpp_MaxLunsPerArray; ++lunIndex)
						{
							if(!RdacInfo->RdacLuns[lunIndex].Present ||
				   			!RdacInfo->RdacLuns[lunIndex].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice ||
				    			RdacInfo->RdacLuns[lunIndex].LunControllers[controllerIndex]->LunPaths[pathIndex].PathRemoveState)
							{
								continue;
							}
							else
							{
								deviceObject = RdacInfo->RdacLuns[lunIndex].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice;
								break;
							}
						}
						if (deviceObject != NULL)
							break;
					}
					if (deviceObject != NULL)
					{
						ioctl_status = mpp_GetUTMData(deviceObject, &UTMLun, &UTMEnabled);
						if (!ioctl_status)
						{
							MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 730,
								"mppLnx_ioctl: mpp_GetUTMData returned an error on Array %s, controller %d\n",
		       					RdacInfo->ModuleName, controllerIndex));
						}
						else
						{
							/* Update RdacInfo information */
							MPP_LOCK(RdacInfo->Lock, LockLevel);
							RdacInfo->ControllerInfo[controllerIndex]->ControllerPresent = TRUE;
							RdacInfo->ControllerInfo[controllerIndex]->UTMLunExists = UTMEnabled;
							RdacInfo->ControllerInfo[controllerIndex]->UTMLunNumber = UTMEnabled ? UTMLun : -1;
							MPP_UNLOCK(RdacInfo->Lock, LockLevel);
							MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
								"mppLnx_ioctl: UTM Information Update for Array %s: UTM enabled %d, UTM LUN %d\n",
								RdacInfo->ModuleName,
								RdacInfo->ControllerInfo[controllerIndex]->UTMLunExists,
								RdacInfo->ControllerInfo[controllerIndex]->UTMLunNumber));
						}
						deviceObject = NULL;

					}
					else
					{
						MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 731,
							"mppLnx_ioctl: MPP_IOCTL_GET_UTMLUN_INFO: Cannot find a valid LUN for Array %s, controller %d\n",
		       				RdacInfo->ModuleName, controllerIndex));
					}
				}

			}
			break;
      case MPP_IOCTL_OPTIONCONTROL:
            ioctl_status = mppCmn_ProcessFeatureOptions((MppCmn_OptionControlIoctl_t *)arg);
            break;
      case MPP_IOCTL_FORCEREBALANCE:
            for(arrayIndex=0; arrayIndex<mpp_MaxArrayModules; ++arrayIndex)
            {
               if (!mpp_ModuleArray[arrayIndex].RdacInfo)
               {
                  continue;
               }
               else
               {
                  RdacInfo = mpp_ModuleArray[arrayIndex].RdacInfo;
                  mppCmn_RebalanceLuns(RdacInfo,TRUE);
               }
            }
            break;
		default:
			ioctl_status = -EINVAL;
			bSilent = TRUE;   // remain silent if we have no idea what to do
	}

	if( ioctl_status  && !bSilent)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_ALL, 732, "mpp_ioctl: ioctl failed with status %d, cmd 0x%lx\n",
		 ioctl_status, cmd_in));
	}

    	return(ioctl_status);


}




 /*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_Translate_Linux_Status
* SUMMARY:  Returns an equivalent MPP status code based on status returned by the target and low level driver .
* SCOPE:    public
*
*
* DESCRIPTION:
* This function has to translate the SCSI command status codes returned by the SCSI middle level
* layer to an appropriate MPP driver status code. It takes a SCSI command pointer as argument. The
* function has to translate the status of this SCSI command.
* There are four bytes of data in the result field of the SCSI command that is returned. The first
* byte is the SCSI_STATUS. If this field returns a valid status, it means the connection to the
* target is fine, only an error occurred in executing the SCSI command. It also means that we have
* the sense keys to make further sense out of any error that occurred. The second byte is the SCSI
* message. This gives more information about the status of the SCSI command, in which it failed.
* The third byte is the status returned by the low-level driver. The last byte is the error code
* inserted by the linux SCSI middle-layer code. If a SCSI error occurs, a SCSI sense key is returned
* and this also has to be checked so that more information about the errior may be obtained. This is
* done in the common code mpp_AnalyseIOError function. All these conditions have to be converted to
* MPP specific status codes.
 *      status byte = set by target device
 *      msg_byte    = return status from host adapter itself.
 *      host_byte   = set by low-level driver to indicate status.
 *      driver_byte = set by mid-level.
*
* RESTRICTIONS:
*
* RETURNS: An LWORD which is the equivalent MPP status, which are listed in MPP_Common.h
*
* ERRNO:
*
* NOTES: Called from the function mpp_SysdepSynchronousIo
*
* EXAMPLES:
*
* SEE ALSO:
*
*/


static LWORD mppLnx_Translate_Linux_Status (int result) {
	LWORD  retVal =  MPP_UNRECOGNIZED_OS_STATUS;

	MPP_DEBUGPRINT((MPP_SYNC_IO_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_Translate_Linux_Status: H0x%x S0x%x M0x%x D0x%x\n", host_byte(result), status_byte(result), msg_byte(result), driver_byte(result)));

	switch ( host_byte( result ) ) {
		/* No error */
		case DID_OK:
			switch ( status_byte( result ) ) {
				case GOOD:
					retVal = MPP_GOOD_STATUS;
					break;
				case COMMAND_TERMINATED:
					retVal =  MPP_UNRECOGNIZED_OS_STATUS;
	         			break;
				case CHECK_CONDITION:
					retVal = MPP_CHECK_CONDITION;
					break;
				case BUSY:
					retVal = MPP_SCSI_BUSY;
					break;
				case RESERVATION_CONFLICT:
					retVal = MPP_SCSI_RESERVATION;
					break;
				default:
					retVal = MPP_UNSUPPORTED_SCSI_STATUS;
			}
			break;
		/* SCSI SELECTION failed because there was no device at the address specified */
		/* DID_NO_CONNECT will abort the command */
		case DID_NO_CONNECT:
			retVal = MPP_SELECTION_TIMEOUT;
			break;
		/* SCSI ARBITRATION failed */
		/* DID_BUS_BUSY will force the command to be retried
		* This is case of command->result 0x20000 case. QLA driver returns this status even
		* in the case of command time-out. We translate this to MPP_SCSI_BUSY. This mpp status
		* will cause normal IO retry, sync IO retry and failover command retry.
		*/
		case DID_BUS_BUSY:
			retVal = MPP_SCSI_BUSY;
			break;
		/* A time-out occurred for some unknown reason, probably during SELECTION or while waiting for RESELECTION */
		case DID_TIME_OUT:
			retVal = MPP_SELECTION_TIMEOUT;
			break;
		/* The SCSI ID of the target was the same as the SCSI ID of the host adapter */
		case DID_BAD_TARGET:
			retVal = MPP_INVALID_REQUEST;
			break;
		/* The high-level code called the low-level abort() function */
		case DID_ABORT:
			retVal = MPP_COMMAND_ABORTED;
			break;
		/* A SCSI PARITY error was detected */
		case DID_PARITY:
			retVal = MPP_HARDWARE_ERROR;
			break;
		/* An error occurred which lacks a more appropriate error code (for example, an internal host adapter error) */

		case DID_ERROR:
			retVal = MPP_HARDWARE_ERROR;
			break;
		/* The high-level code called the low-level reset() function */
		case DID_RESET:
			retVal = MPP_HARDWARE_ERROR;
			break;
		/* An unexpected interrupt occurred and there is no appropriate way to handle this interrupt */
		case DID_BAD_INTR:
			retVal = MPP_HARDWARE_ERROR;
			break;
		default:
			retVal = MPP_UNRECOGNIZED_OS_STATUS;
	}
	/* Here the middle layer is telling us that a timeout has occurred on the SCSI command. */
	if (driver_byte(result)  == DRIVER_TIMEOUT)
		retVal = MPP_COMMAND_TIMEOUT;
	MPP_DEBUGPRINT((MPP_SYNC_IO_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_Translate_Linux_Status: %d \n", retVal));

	return retVal;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_isRemovedDevice
* SUMMARY:  Check whether or not the device has been removed before issuing a synch io to
the physical device.
* SCOPE:    local
*
*
*DESCRIPTION
Due to the race condition, we may issue an synch IO request to a device that is removed. This function will
check if the physical device has been removed. If so, it will return TRUE.
*
* RESTRICTIONS:
*
* RETURNS: TRUE if the physical device has been removed. Otherwise, FALSE.
*
* ERRNO:
*
* NOTES: Called from the function mpp_SynchronousIo
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
/*******************************************************************************/

BOOL mppLnx_isRemovedDevice(struct scsi_device *SDpnt)
{
   int numModules = 0;
	ArrayModuleEntry_t *mppModule = NULL;
   RdacDeviceInformation_t *pRdacDevice = NULL;
   RdacLunInformation_t    *pRdacLun = NULL;
	int  LunCont = 0;
	int  LunPath = 0;
   int lun = 0;
	LunPathObjects_t        *pLunPath;
	unsigned char			nullWWN[WWN_LENGTH];
   LunPathInfo_t        * pLunPathInfo;
   //LINT                 LockLevel;

   if(SDpnt->sdev_state == SDEV_RUNNING)
   {
      return FALSE;
   }

	memset(nullWWN, 0, WWN_LENGTH);


	for ( mppModule     = mpp_ModuleArray, numModules = 0;
                 numModules < mpp_MaxArrayModules; mppModule++, numModules++)
	{

		if (! (pRdacDevice = mppModule->RdacInfo))
		{
			/* There is no array at this location as yet */
			continue;
		}
		/* Go through each controller */
		for( LunCont = 0; LunCont < 2; LunCont++)
		{
         for (lun=0; lun<mpp_MaxLunsPerArray; lun++)
         {
           pRdacLun = &pRdacDevice->RdacLuns[lun];
           if ( !bcmp((BYTE *) pRdacLun->WWN, (BYTE *) nullWWN, 16) )
           {
			      /* That means that memory for the struct has been allocated and bzeroed but the lun has not been inserted as yet */
			      continue;
           }

			  pLunPath = pRdacLun->LunControllers[LunCont];

			  /* Go through each path */
			   for( LunPath = 0; LunPath <  mpp_MaxPathsPerController; LunPath++)
            {
               pLunPathInfo = &pLunPath->LunPaths[LunPath];
               if(pLunPathInfo->SavedLunPathDevice)
               {
                 MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
                    "SYNC 0x%lx %d:%d Check\n",(unsigned long)pLunPathInfo->SavedLunPathDevice,LunCont,LunPath));
               }
               if(pLunPathInfo->PathRemoveState)
               {
                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
                     "SYNC 0x%lx %d:%d PathRemoveState\n",(unsigned long)pLunPathInfo->SavedLunPathDevice,LunCont,LunPath));
               }
					if (pLunPathInfo->LunPathDevice == SDpnt)
               {


                  MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,"SYNC io %d:%d REMOVED\n",LunCont,LunPath));

					   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_isRemovedDevice : Physical device 0x%x \n",  SDpnt));
						return FALSE;

               }
            }
			}

		}

	}


	MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "SYNC io %d:%d:%d not removed\n",SDpnt->host->host_no,SDpnt->id,SDpnt->lun));
	return TRUE;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mpp_SysdepSynchronousIo
* SUMMARY:  This is the system dependent part of the Synchronous IO function .
* SCOPE:    public
*
*
*DESCRIPTION
* mpp_SysdepSynchronousIO starts the I/O indicated by the cdb on the device scsiDevice
* and waits for it to complete. bufferAddress and bufferLength define the data buffer, which will be
* returned by or sent to the scsi target. senseBuffer and senseLength define the sense data buffer.
* DataDirection indicates the direction of the data transfer - zero for output and non-zero for
* input. On return from this function, dataLengthXfered and senseLengthReturned will indicate the
* amount of data transferred and amount of sense data transferred.
* So this function takes the SCSI command, which is in the form of a cdb and converts it to a
* platform-based SCSI request and sends it down to the linux SCSI middle layer. The SCSI data, which
* is returned, in return is then passed back to the calling function. If the SCSI command results in
* an error, then the sense key data is sent back so that sense can be made of the error.
* For this we will use the scsi_request structure, which is essentially a slimmed down version of
* struct scsi_cmnd. Here we set its fields, like the actual SCSI command and the data buffer and the length
* of the data buffer.  Then we use the scsi_wait_req() function, exported by the middle layer, and
* pass the scsi command to the middle layer. To this function we also pass the MPP_SYNCTIMEOUT value
* as the timeout value for the command and mpp_SyncRetryCount for the number of times this command
* should be retried, both of which are defined in MPP_Common.h. Depending on whether the SCSI
* command was successful or not, we do or do not send any sense data back to the calling function.
* We also need to call mppLnx_Translate_Linux_Status to translate the status retuned by
* scsi_wait_req to an MPP specific status. If the status returned is MPP_CHECK_CONDITION, sense data
* would also have to be sent back.
*
* RESTRICTIONS:
*
* RETURNS: An LWORD which is the equivalent MPP status, which are listed in MPP_Common.h
*
* ERRNO:
*
* NOTES: Called from the function mpp_SynchronousIo
*
* EXAMPLES:
*
* SEE ALSO:
*
*/


LWORD mpp_SysdepSynchronousIo(MPP_HANDLE ScsiDevice, BYTE *cdb,	BYTE cdblen, void *BufferAddress,
								LWORD BufferLength,	void *SenseBuffer,
								LWORD SenseLength, TINY	DataDirection,
								LWORD *SenseLengthReturned,
								LWORD *DataLengthXfered)
{

	LWORD max_retrys = 0;
	LWORD mppStatus = 0;
	unsigned char cmd[cdblen];
   int result = 0;

   struct scsi_sense_hdr sshdr;
   int data_direction;
   SenseData_t * mpp_senseData = (SenseData_t *) SenseBuffer;
   if(cdblen > MAX_COMMAND_SIZE )
   {
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 734,"mpp_SysdepSynchronousIo: cdb length greater than can be assigned to the Scsi Request \n"));
		return (MPP_SCSI_BUSY); /* need to return the right value here */
	}
   if (DataDirection == MPPCMN_XFER_DEV_HOST)
	{
	   data_direction =  DMA_FROM_DEVICE;
	}
	else if(DataDirection == MPPCMN_XFER_HOST_DEV)
	{
		data_direction =  DMA_TO_DEVICE;
   }
   else if(DataDirection == MPPCMN_XFER_NONE)
   {
      data_direction = DMA_NONE;
   }else
   {
      /*
      * Log an error message
      */
      data_direction = DMA_BIDIRECTIONAL;
      MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 735,
			"Synchronous IO request with data transfer direction unknown:%d\n",
         DataDirection));
   }

    if(mppLnx_isRemovedDevice(ScsiDevice))
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 746, "Device(0x%x) is already removed, cannot send Synchronous IO request\n", ScsiDevice));

      /*
       * the selection timeout will cause the caller returning immediately without retries
       */

      return MPP_SELECTION_TIMEOUT;
   }

   if((ScsiDevice->sdev_state == SDEV_OFFLINE))
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 739,
			"Device is removed or marked offline: H%dT%dL%d\n",
         ScsiDevice->host->host_no, ScsiDevice->id, ScsiDevice->lun));

      /*
       * the selection timeout will cause the caller returning immediately without retries
       */

      return MPP_SELECTION_TIMEOUT;
   }
   get_device(&ScsiDevice->sdev_gendev);
   memset (cmd, 0, cdblen);
	memcpy(cmd, cdb, cdblen);
   result = mppLnx_scsi_execute_req(ScsiDevice, cmd,
		     data_direction, BufferAddress, BufferLength,
		     &sshdr, mpp_SynchTimeout * HZ, max_retrys);

   MPP_DEBUGPRINT((MPP_SYNC_IO_DEBUG+MPP_DEBUG_LEVEL_4, "mpp_SysdepSynchronousIo: Returned from mppLnx_scsi_execute_req \n"));

   if(ScsiDevice->sdev_state != SDEV_RUNNING)
   {
      result = DID_NO_CONNECT<<16;
   }
	/* Next we need to translate the status returned by the target, low level driver and middle layer to an equivalent MPP  status */
	mppStatus = mppLnx_Translate_Linux_Status(result);


	/* Need to check how much data is actually transfered */
	if(	mppStatus == MPP_GOOD_STATUS )
	{
		*DataLengthXfered = 1; /* some positive value right now, so that the code works */
	}
	else
		*DataLengthXfered = 0;


	/* We need to return the sense data back if the error condition is MPP_CHECK_CONDITION */
	if(	mppStatus == MPP_CHECK_CONDITION )
	{
      *SenseLengthReturned = MIN(SenseLength,	SCSI_SENSE_BUFFERSIZE);
      mpp_senseData->Sense_Key = (BYTE)sshdr.sense_key;
      mpp_senseData->ASC = (BYTE)sshdr.asc;
      mpp_senseData->ASCQ = (BYTE)sshdr.ascq;
      mpp_senseData->Additional_Len = (BYTE)sshdr.additional_length;
	}
	else
	{
		*SenseLengthReturned = 0;
	}

	/* Release the Scsi Request we allocated as we are doen with it */
   put_device(&ScsiDevice->sdev_gendev);

	MPP_DEBUGPRINT((MPP_SYNC_IO_DEBUG+MPP_DEBUG_LEVEL_3, "mpp_SysdepSynchronousIo: sync i/o done,  mppStatus %d, sense len %d, data len %d\n", mppStatus,*SenseLengthReturned,*DataLengthXfered));


	return (mppStatus);

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_sg_add
* SUMMARY:  MPP replacement detection routine for sg driver
* SCOPE:    local
*
* DESCRIPTION: sg class interface claims Engenio virtual devices / UTM LUN and
*	       non-Engenio devices.
* RESTRICTIONS:
*
* RETURNS:
*/

static int mppLnx_sg_add(struct class_device* cl_dev, struct class_interface *cl_intf)
{
	struct scsi_device *SDpnt = to_scsi_device(cl_dev->dev);
	ControllerAddress_t ctlAddress;
	LWORD device_type;
	LunPathInfo_t *lunPathInfo = NULL;
	RdacDeviceInformation_t	*pRdacDevice;
	void (*mppLnx_newDeviceAdded)(struct scsi_device * sdp);

	MPP_DEBUGPRINT ((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_sg_add invoked for H%dT%dL%d\n", SDpnt->host->host_no, SDpnt->id, SDpnt->lun));

   	/* check if virtual device
    	* If a virtual device is a data LUN, hand the virtual device to sg's add().
    	*/
   	if ( (SDpnt->host->hostt->proc_name != NULL) &&
            strcmp(SDpnt->host->hostt->proc_name, MPPLNX_PROC_NAME) == 0)
   	{
      		if(unlikely( !mpp_ModuleArray[(SDpnt->id)].RdacInfo->RdacLuns[SDpnt->lun].Present && SDpnt->lun == 0))
      		{
         		/*
         		* This is the dummy lun 0. We don't want give it to the sd driver.
         		* We would like to put it into the un-configured device list
         		*/
         		mppLnx_process_dummy_lun0(SDpnt);
         		return -ENODEV;
      		}
          return sg_old_add(cl_dev,cl_intf);
	}

	/* Fill in values to be sent to mppLnx_BuildRdacinfo */
	ctlAddress.ChannelId = SDpnt->channel;
	ctlAddress.TargetId  = SDpnt->id;
	ctlAddress.HostId    = SDpnt->host->host_no;
   	/*
   	* workaround for setting mppLnx_host_procName
   	*/
	mppLnx_host_procName = SDpnt->host->hostt->module->name;


	/* Call mppLnx_BuildRdacinfo */
	if(mppLnx_BuildRdacinfo(SDpnt, &ctlAddress, SDpnt->lun))
	{
		/* mppLnx_BuildRdacinfo returns true if it inserts the lun into the RdacInfo data structure or if the device has already been discovered */
		/* So if we come here it means that this struct scsi_device is a physical LUN and is part of an Engenio array */

		/* Find out what kind of Engenio device it is */
		if (( device_type = mppLnx_DeviceType(SDpnt, &lunPathInfo, &pRdacDevice)) ) {

			if (lunPathInfo != NULL)
			{
				lunPathInfo->refCount++;
			}

			/* if it is a UTMLUN */
			if ( device_type == MPP_UTM_LUN_DEVICE )
         {
            return sg_old_add(cl_dev,cl_intf);		     /* we want sg to claim this device */
			}


                        mppLnx_newDeviceAdded = (void *)symbol_get(mppLnx_newDeviceAdded);

                        if(mppLnx_newDeviceAdded) {
                                (*mppLnx_newDeviceAdded) (SDpnt);
                                symbol_put(mppLnx_newDeviceAdded);
                        }
                        else {
                                MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 744,
                                 "mppLnx_sg_add: symbol_get did not return a function pointer of mppLnx_newDeviceAdded\n"));
                        }

			/* It is a normal Engenio physical lun */
			return 0;
		}
		else
		{

			MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 736, "Host %d Target %d Lun %d Is a physical device but is an Unconfigured Device. \n",
					SDpnt->host->host_no,SDpnt->id,SDpnt->lun));
			return 0;

		}
	}

	/* Not an Engenio Device */
   return sg_old_add(cl_dev, cl_intf);   /* call original function */

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_sg_remove
* SUMMARY:  MPP replacement device removal routine for sg driver
* SCOPE:    local
*
* DESCRIPTION: sg class interface removes Engenio virtual devices / UTM LUN and
*	       non-Engenio devices.
* RESTRICTIONS:
*
* RETURNS:
*/

static void mppLnx_sg_remove(struct class_device* cl_dev, struct class_interface *cl_intf)
{
	struct scsi_device *SDpnt = to_scsi_device(cl_dev->dev);
	LunPathInfo_t  *lunPathInfo = NULL;
	LWORD device_type;
	mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
	RdacLunInformation_t	*pRdacLun;
	RdacDeviceInformation_t	*pRdacDevice;
	BOOL devUnconfigEntryExists = FALSE;
   void (*mppLnx_TestAndGetDevice) (struct scsi_device *,LunPathInfo_t *);
   //void (*mppLnx_cleanup_requests_per_scsi_device)( struct scsi_device *);


	MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_sg_remove invoked for H%dT%dL%d\n",
			SDpnt->host->host_no, SDpnt->id, SDpnt->lun));

	/* if the device is in the unconfigure device list, free related memory */
   	deviceEntry = mppLnx_getUnconfiguredDevice(SDpnt);
	if (deviceEntry)
	{
      		mppLnx_deleteUnconfiguredDevice(deviceEntry->physical_device);
            devUnconfigEntryExists = TRUE;
   	}

   	/* check if virtual device
    	* If a virtual device is a data LUN but not dummy lun, hand the virtual device to sg's remove().
    	*/
   	if ( (SDpnt->host->hostt->proc_name != NULL) &&
         strcmp(SDpnt->host->hostt->proc_name, MPPLNX_PROC_NAME) == 0)
   	{
    		//make sure you point to the right RdacInfo while locking and unlocking
		pRdacDevice = mpp_ModuleArray[(SDpnt->id)].RdacInfo;
		pRdacLun = &pRdacDevice->RdacLuns[SDpnt->lun];

               if (pRdacDevice == NULL || pRdacDevice->ModuleHandle == NULL) {
                        MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "Device H%dT%dL%d is already removed\n",
                        SDpnt->host->host_no, SDpnt->id, SDpnt->lun));
                        return;
                }

		if(unlikely(!pRdacLun->Present && SDpnt->lun == 0 && !pRdacLun->PseudoRemoveEligible))
      		{
         		/*
         		* This is the dummy lun 0. We don't want give it to sg_remove function
         		* since sg does not claim dummy lun 0.
         		*/
			pRdacLun->PseudoLunObject = NULL;
			MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "Device H%dT%dL%d is dummy LUN 0\n",
			SDpnt->host->host_no, SDpnt->id, SDpnt->lun));
			return;
      		}
	   /*
      pRdacLun->Present = 0;
      */

      MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3,
         "RVD-SG %d:%d:%d\n",SDpnt->host->host_no,SDpnt->id,SDpnt->lun));
      //scsi_device_cancel(SDpnt,0);
      /*
      * we must clean up the outstanding ios on the virtual device. Otherwise, during the
      sd remove time, the generic disk will wait for those IOs coming back. It will lock up
      the virtual device removing thread and also lock up the process that removing physical
      device.
      */
      sg_old_remove(cl_dev, cl_intf);

      /*
       *The sg driver has released the virtual device. We turn the virtual lun's
       *ReportedMissing flag back to false so that mppLnx_CheckRemoveModule() clean-up
       *RdacInfo resources.
       */
       pRdacLun->ReportedMissing = FALSE;
       mppLnx_CheckRemoveModule(pRdacDevice,SDpnt);

		return;
	}

	if (( device_type = mppLnx_DeviceType(SDpnt, &lunPathInfo, &pRdacDevice)) ) {

		/* remove the scsi_device from rdacInfo if we are the last one to be invoked */
		MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3,
         	"mppLnx_sg_remove: mppLnx_removeLun removes LUN H%dT%dL%d AD0x%lx\n",
		SDpnt->host->host_no,SDpnt->id,SDpnt->lun, (unsigned long)SDpnt));

		/*
		* This is the class interface remove. We have to support hot plug/remove.
		* 1. mark the device is being removed
		* 2. cancel all ios on the device
		*/
		//lunPathInfo->LunPathDevice = NULL:
		lunPathInfo->PathRemoveState = 1;
		lunPathInfo->SavedLunPathDevice = lunPathInfo->LunPathDevice;
		/* avoid the device being freed*/
		mppLnx_TestAndGetDevice = (void *) symbol_get(mppLnx_TestAndGetDevice);

		if(mppLnx_TestAndGetDevice) {
			(*mppLnx_TestAndGetDevice) (SDpnt,lunPathInfo);
			symbol_put(mppLnx_TestAndGetDevice);
		}
		else
		{
         		//new error number
			MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 742, 
			"mppLnx_TestAndGetDevice: symbol_get did not return a function pointer\n"));
      		}
      		//scsi_device_cancel(SDpnt, 0);

		mppLnx_removeLun(SDpnt);

                /* if it is a UTMLUN */
                if ( device_type == MPP_UTM_LUN_DEVICE ) {
                        MPP_DEBUGPRINT ((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_sg_remove: removes UTM LUN H%dT%dL%d\n",
                                SDpnt->host->host_no,SDpnt->id,SDpnt->lun));
                        sg_old_remove(cl_dev,cl_intf); /* we want sg to remove this device */
                }


	}
	else if (!devUnconfigEntryExists)
	{
		/* Not an Engenio Device */
      sg_old_remove(cl_dev,cl_intf);
	}
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_DeviceType
* SUMMARY:  MPP function that goes through RdacInfo data structure to find if a device has already been discovered
* and to return the type of the device whether it is a UTM LUN or Root/Boot device or just a Engenio physical device.
* SCOPE:    local
*
* DESCRIPTION: We need to send VPD Inquiry pages to the storage array to determine the type of a LUN presented to
* the detect function of an Upper level driver. A struct scsi_device could have already been discovered by another device
* driver. In this case we do not want to send a VPD Page Inquiry as this would have already been done and is
* expensive. So the solution is to go through the RdacInfo data structure and look for the device and check and see
* if it has already been discovered. The device can only be a Physical Device. We do not go through the entire list
* of luns on an array but instead index on the lun number. This Lun number is passed to mpp BuildRdacInfo from with* in which this function will be called. This function also returns the Device type of each device.

@SDpnt -- a physical scsi_device object for which this function will decide its device type
@lunPathInfo  output parameter. The LunPathInfo_t object which is associated with the physical scsi-device scsi device object SDpnt
@rdacinfo  output parameter. The RdacDeviceInformation_t object which is associated with the physical scsi device object SDpnt
*
* RESTRICTIONS:
*
* RETURNS: TRUE on finding a match for the device in the RdacInfo data structure, returns MPP_UTM_LUN_DEVICE or
* MPP_ROOT_BOOT_DEVICE depending on devicetype else returns FALSE
*/

LWORD mppLnx_DeviceType (MPP_HANDLE SDpnt, LunPathInfo_t **lunPathInfo,RdacDeviceInformation_t **rdacinfo)  {
	int numModules = 0;
	ArrayModuleEntry_t *mppModule = NULL;
        RdacDeviceInformation_t *pRdacDevice = NULL;
        RdacLunInformation_t    *pRdacLun = NULL;
	int  LunCont = 0;
	int  LunPath = 0;
	LunPathInfo_t  *pLunPathInfo = NULL;
	LunPathObjects_t        *pLunPath;
	unsigned char			nullWWN[WWN_LENGTH];
	*lunPathInfo = NULL;
	*rdacinfo = NULL;

	memset(nullWWN, 0, WWN_LENGTH);

	/* Handle unconfigured lun case */
	if( SDpnt->lun >= mpp_MaxLunsPerArray )
		return FALSE;

	/*should we lock here ?  */
	/*PP_LOCK(RdacInfo->Lock, LockLevel);  */
	for ( mppModule     = mpp_ModuleArray, numModules = 0;
                 numModules < mpp_MaxArrayModules; mppModule++, numModules++)
	{

		if (! (pRdacDevice = mppModule->RdacInfo))
		{
			/* There is no array at this location as yet */
			continue;
		}


		/* PR 96810 - UTM LUN proc node removal - The pRdacDevice->RdacLuns[SDpnt->lun]->UTMLun field
                   where pRdacDevice = mppModule->RdacInfo for that array is not set at this point and hence
		    the use of pRdacDevice->ControllerInfo[0]->UTMLunNumber was necessary.
		  We have to investogate why UTMLun is not set at this point. Also all this code
		  has to go before the bcmp code below, because the virtual LUN for the UTM LUN does not
		  have the WWN field set. */

                if (( (pRdacDevice->ControllerInfo[0]->ControllerPresent) && (pRdacDevice->ControllerInfo[0]->UTMLunNumber == SDpnt->lun)) ||((pRdacDevice->ControllerInfo[1]->ControllerPresent) && (pRdacDevice->ControllerInfo[1]->UTMLunNumber == SDpnt->lun)) )
		{
                        pRdacLun = &pRdacDevice->RdacLuns[SDpnt->lun];
                        for( LunCont = 0; LunCont < 2; LunCont++)
                        {
                                pLunPath = pRdacLun->LunControllers[LunCont];

                                // Go through each path
                                for( LunPath = 0; LunPath <  mpp_MaxPathsPerController; LunPath++)
                                {
                                        pLunPathInfo = &pLunPath->LunPaths[LunPath];
                                        if (!pLunPathInfo->LunPathDevice ||
                                                pLunPathInfo->PathRemoveState)
                                        {

                                                continue;
                                        }
                                        if ( pLunPathInfo->LunPathDevice == SDpnt)
                                        {
											*rdacinfo = pRdacDevice;
                                                *lunPathInfo = pLunPathInfo;
                                                return MPP_UTM_LUN_DEVICE;
                                        }
                                }

                        }
                }



		/* index on the Lun number passed to this function  */
		pRdacLun = &pRdacDevice->RdacLuns[SDpnt->lun];

		if ( !bcmp((BYTE *) pRdacLun->WWN, (BYTE *) nullWWN, 16) )
		{
			/* That means that memory for the struct has been allocated and bzeroed but the lun has not been inserted as yet */
			continue;

		}

		/* Go through each controller */
		for( LunCont = 0; LunCont < 2; LunCont++)
		{

			pLunPath = pRdacLun->LunControllers[LunCont];

			/* Go through each path */
			for( LunPath = 0; LunPath <  mpp_MaxPathsPerController; LunPath++)
			{
				pLunPathInfo = &pLunPath->LunPaths[LunPath];
				if (!pLunPathInfo->LunPathDevice ||
					pLunPathInfo->PathRemoveState)
				{

					continue;
				}

				if ( pLunPathInfo->LunPathDevice == SDpnt)
				{

					/* This is a physical device */

					MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_DeviceType : Physical device 0x%x \n",  SDpnt));
					*lunPathInfo = pLunPathInfo;
					*rdacinfo = pRdacDevice;

					/* if it is a UTMLUN */
					if (pRdacLun->UTMLun )
					{
						/*MPP_UNLOCK(RdacInfo->Lock, LockLevel);*/

						return MPP_UTM_LUN_DEVICE;
					}

					return TRUE;

				}
			}

		}

	}

	/* MPP_UNLOCK(RdacInfo->Lock, LockLevel); */

	return FALSE;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     LWORD mppLnx_BuildRdacinfo(MPP_HANDLE DeviceVertex, ControllerAddress_t *DeviceAddress,
                     LWORD Lun)
* SUMMARY:  Inserts a physical device to the RdacInfo data model
* SCOPE:    local
*
* DESCRIPTION: This function is a wrap of the mpp_BuildRdacinfo() common function. This function calls
mpp_BuildRdacinfo() to determine a device is a storage array device, a un-configured storage array device or not a
storage array device. For a un-configured storage array device, it will add the device to un-configured device list and manages its state changes
*
* @DeviceVertex-- A struct scsi_device object that represents the un-configured device
  @DeviceAddress -- A ControllerAddress_t object which represents the physical connection of the path
  @Lun -- the LUN number of the physical device object
* RESTRICTIONS:
*
* RETURNS: TRUE if the physical device is a storage array device (either configured or un-configured device) or FALSE
                   otherwise.
*/

LWORD
mppLnx_BuildRdacinfo(MPP_HANDLE DeviceVertex, ControllerAddress_t *DeviceAddress,
                     LWORD Lun)
{
   mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
   LWORD  deviceType;
   LWORD  retValue = FALSE;

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_BuildRdacinfo H:%dC:%dT:%dL:%d\n",
      DeviceVertex->host->host_no,DeviceVertex->channel,DeviceVertex->id,DeviceVertex->lun));

   if ( DeviceVertex->type == TYPE_ENCLOSURE ) {
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppLnx_BuildRdacinfo FALSE\n"));
        return FALSE;
   }

   deviceEntry = mppLnx_getUnconfiguredDevice(DeviceVertex);


   MPP_MUTEX_LOCK(mppLnx_rdac_mutex);
   deviceType = mpp_BuildRdacinfo(DeviceVertex,DeviceAddress, Lun);
   MPP_MUTEX_UNLOCK(mppLnx_rdac_mutex);

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "mppLnx_BuildRdacinfo H:%dC:%dT:%dL:%d type:%d\n",
      DeviceVertex->host->host_no,DeviceVertex->channel,
      DeviceVertex->id,DeviceVertex->lun,deviceType));

   switch (deviceType)
   {
      case TRUE:
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppLnx_BuildRdacinfo TRUE\n"));
         retValue = TRUE;
         /*it is a storage array device */
         if(deviceEntry)
         {
            /* it is in the unconfigured device list. We get here by hot-add */
            deviceEntry->isUnconfigured = FALSE;
         }
         break;
      case FALSE:
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppLnx_BuildRdacinfo FALSE\n"));
         /* it is not a storage array device */
         retValue = FALSE;
         break;
      case MPPCMN_UNCONFIGURED_LUN:
         /* it is a unconfigured device */
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppLnx_BuildRdacinfo MPPCMN_UNCONFIGURED_LUN\n"));
         if(!deviceEntry)
         {
            /* it is not in the un-configured device lis. Add it into the list */
            mppLnx_addUnconfiguredDevice(DeviceVertex);
         }
         retValue = TRUE;
         break;
	case MPPCMN_CONFIG_PARAMETER_OUT_OF_LIMIT:
         /* Max config parameter value exceeded add it to unconfigured list */
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppLnx_BuildRdacinfo MPPCMN_CONFIG_PARAMETER_OUT_OF_LIMIT\n"));
         if(!deviceEntry)
         {
            /* it is not in the un-configured device lis. Add it into the list */
            mppLnx_addUnconfiguredDevice(DeviceVertex);
         }
         retValue = TRUE;
         break;

      default:
         /* should not get here */
         break;
   }

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_BuildRdacinfo H:%dC:%dT:%dL:%d  return %d\n",
      DeviceVertex->host->host_no,DeviceVertex->channel,
      DeviceVertex->id,DeviceVertex->lun,retValue));
   return retValue;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_getUnconfiguredDevice()
* SUMMARY:  Gets a un-configured device from the global un-configured device list
* SCOPE:    local
*
* DESCRIPTION: A un-configured device is a un-configured LUN that either reported as
 a configured device by the controller firmware or a dummy LUN created by a physical
 HBA driver. The Linux MPP driver keeps a un-configured device list and treats
 a un-configured device differently than a real-configured device. This function
 queries a given Scsi_Deivce object from  the global un-configured device list
*
* @device -- A struct scsi_device object that represents the un-configured device
* RESTRICTIONS:
*
* RETURNS: An mppLnx_UnconfiguredDeviceEntry_t object if the specified device is in
  the global un-configured device list or NULL otherwise.
*/
static mppLnx_UnconfiguredDeviceEntry_t * mppLnx_getUnconfiguredDevice(struct scsi_device * device)
{
   unsigned long iflages;
   mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
   mppLnx_UnconfiguredDeviceEntry_t * foundDevice = NULL;
   struct list_head *listp;

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_getUnconfiguredDevice H:%dC:%dT:%dL:%d\n",
      device->host->host_no,device->channel,device->id,device->lun));


   spin_lock_irqsave(unconfiguredDeviceList.unconfiguredDeviceLock, iflages);
   for(listp = unconfiguredDeviceList.deviceEntryList->list.next;
      listp != &(unconfiguredDeviceList.deviceEntryList->list);
      listp = listp->next)
   {
      deviceEntry = list_entry(listp, mppLnx_UnconfiguredDeviceEntry_t, list);
      if(deviceEntry->physical_device == device)
      {
         foundDevice = deviceEntry;
         break;
      }
   }
   spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_getUnconfiguredDevice H:%dC:%dT:%dL:%d found?%d\n",
      device->host->host_no,device->channel,device->id,device->lun,
      foundDevice==NULL? 0:1));
   return foundDevice;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_deleteUnconfiguredDevice()
* SUMMARY:  Deletes a un-configured device from the global un-configured device list
* SCOPE:    local
*
* DESCRIPTION: A un-configured device is a un-configured LUN that either reported as
 a configured device by the controller firmware or a dummy LUN created by a physical
 HBA driver. The Linux MPP driver keeps a un-configured device list and treats
 a un-configured device differently than a real-configured device. This function
 deletes a given Scsi_Deivce object from  the global un-configured device list
*
* @device -- A struct scsi_device object that represents the un-configured device
* RESTRICTIONS:
*
* RETURNS:
*/

static void mppLnx_deleteUnconfiguredDevice(struct scsi_device * device)
{
   unsigned long iflages;
   mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_deleteUnconfiguredDevice H:%dC:%dT:%dL:%d\n",
      device->host->host_no,device->channel,device->id,device->lun));

   deviceEntry = mppLnx_getUnconfiguredDevice(device);
   if(deviceEntry)
   {
      spin_lock_irqsave(unconfiguredDeviceList.unconfiguredDeviceLock, iflages);
      list_del(&(deviceEntry->list));
      spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);
      MPP_FREE(deviceEntry,sizeof(mppLnx_UnconfiguredDeviceEntry_t));
   }
   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_deleteUnconfiguredDevice H:%dC:%dT:%dL:%d\n",
      device->host->host_no,device->channel,device->id,device->lun));

   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_addUnconfiguredDevice
* SUMMARY:  Adds a un-configured device to the global un-configured device list
* SCOPE:    local
*
* DESCRIPTION: A un-configured device is a un-configured LUN that either reported as
 a configured device by the controller firmware or a dummy LUN created by a physical
 HBA driver. The Linux MPP driver keeps a un-configured device list and treats
 a un-configured device differently than a real-configured device. This function
 adds a given Scsi_Deivce object to the global un-configured device list
*
* @device -- A struct scsi_device object that represents the un-configured device
* RESTRICTIONS:
*
* RETURNS:
*/
static void mppLnx_addUnconfiguredDevice(struct scsi_device * device)
{
   mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
   unsigned long iflages;
   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_addUnconfiguredDevice H:%dC:%dT:%dL:%d\n",
      device->host->host_no,device->channel,device->id,device->lun));

   deviceEntry = mppLnx_getUnconfiguredDevice(device);
   if(!deviceEntry)
   {
      deviceEntry = (mppLnx_UnconfiguredDeviceEntry_t *) MPP_INT_KMEM_ALLOC(sizeof(mppLnx_UnconfiguredDeviceEntry_t));

      if(NULL == deviceEntry)
      {
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 737,
            "Cannot allocate memory for unconfigured device entry\n"));
         return;
      }

      deviceEntry->isUnconfigured = TRUE;
      deviceEntry->isAttached = FALSE;
      deviceEntry->isDetected = FALSE;
      deviceEntry->physical_device = device;
      spin_lock_irqsave(unconfiguredDeviceList.unconfiguredDeviceLock, iflages);
      list_add_tail(&(deviceEntry->list), &(unconfiguredDeviceList.deviceEntryList->list));
      spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);
   }
   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_addUnconfiguredDevice H:%dC:%dT:%dL:%d\n",
      device->host->host_no,device->channel,device->id,device->lun));

   return;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_RescanUnconfiguredDevices
* SUMMARY:  Rescan un-configured devices in the global un-configured device list
* SCOPE:    local
*
* DESCRIPTION: A un-configured device is a un-configured LUN that either reported as
 a configured device by the controller firmware or a dummy LUN created by a physical
 HBA driver. The Linux MPP driver keeps a un-configured device list and treats
 a un-configured device differently than a real-configured device. This function
 is invoked when an end user issues "mppUtil -s busscan". This function will go through
 each un-configured device in the un-configured device list to check whether or not
 it become configured. If it is, the device will be added to RdacInfo data model.
*
* @device -- A struct scsi_device object that represents the un-configured device
* RESTRICTIONS:
*
* RETURNS:
*/
void mppLnx_RescanUnconfiguredDevices()
{
   unsigned long iflages;
   mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;
   struct list_head *listp;
   int               needScanBus = FALSE;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
   int    ret=0;
#endif

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_RescanUnconfiguredDevices()\n"));

   listp = unconfiguredDeviceList.deviceEntryList->list.next;
   while(1)
   {
      spin_lock_irqsave(unconfiguredDeviceList.unconfiguredDeviceLock, iflages);
      if( listp != &(unconfiguredDeviceList.deviceEntryList->list) )
      {
         deviceEntry = list_entry(listp, mppLnx_UnconfiguredDeviceEntry_t, list);
         listp = listp->next;
         spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);
         if(deviceEntry->isUnconfigured == TRUE)
         {
            needScanBus = TRUE;
            if ( (deviceEntry->physical_device->host->hostt->proc_name != NULL) &&
               strcmp(deviceEntry->physical_device->host->hostt->proc_name, MPPLNX_PROC_NAME) == 0)
            {
               /*
               * This is a dummy lun 0 virtual device
               */
               mppLnx_process_dummy_lun0(deviceEntry->physical_device);
            }

            /*
            * we found a un-configured device, feed it to all scsi drivers/interfaces
            */
            //mppLnx_sd_probe(&deviceEntry->physical_device->sdev_gendev);
            //mppLnx_driver_probe(&deviceEntry->physical_device->sdev_gendev);
            mppLnx_sg_add(&deviceEntry->physical_device->sdev_classdev, &mpp_interface);
            mppLnx_class_add(&deviceEntry->physical_device->sdev_classdev, &mpp_interface);
         }
      }
      else
      {
         spin_unlock_irqrestore(unconfiguredDeviceList.unconfiguredDeviceLock,iflages);
         break;
      }
   }

   /*
   * we know we have some unconfigured devices. They could be virtual device or physical
   * devices. Let the scsi bus to rescan those devices. The bus rescan will go over
   * all devices that doesn't have a driver.
   * The sd's probe() and mpp's probe() will be invoked.
   */
   if(needScanBus == TRUE)
   {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
      ret = bus_rescan_devices(mpp_template.gendrv.bus);
#else
      bus_rescan_devices(mpp_template.gendrv.bus);
#endif
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppLnx_RescanUnconfiguredDevices bus rescan \n"));
   }

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_RescanUnconfiguredDevices\n"));
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_process_dummy_lun0
* SUMMARY:  Handles the dummy virtual lun0
* SCOPE:    local
*
* When the LUN0 on the storage array is not mapped, the physical HBA drivers will
* create dummy lun 0 devices so that the inquiry and report lun commands for device
* discovery could go through. The virtual HBA has to do the same thing for the inquiry
* and report lun to go through but we don't want the device to be attached by
* the sd/sg driver.
* This function will add the virtual device to the un-configured device list.
*
* @sdp -- A struct scsi_device object that represents a dummy virtual lun 0
* RESTRICTIONS:
*
* RETURNS:
*/

static void mppLnx_process_dummy_lun0(struct scsi_device *sdp)
{
   mppLnx_UnconfiguredDeviceEntry_t * deviceEntry = NULL;

   deviceEntry = mppLnx_getUnconfiguredDevice(sdp);
   if(deviceEntry == NULL)
   {
    
      mppLnx_addUnconfiguredDevice(sdp);
   }
   else
   {
      /*
      * if the virtual device is becoming a real virtaul device,
      * we need to set its isUnconfigured to FALSE
      */
      if(mpp_ModuleArray[(sdp->id)].RdacInfo->RdacLuns[sdp->lun].Present && sdp->lun == 0)
      {
         deviceEntry->isUnconfigured = FALSE;
      }
   }
   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_CheckRemoveModule
* SUMMARY:  Check whether or not we are ready to free RdacInfo data structure memory.
* SCOPE:    local
*
* When all virtual and physical devices for a storage array are removed in a PnP
* environment, we would like to free resources of the mpp driver for that array
* and restore the internal state to a clean state. This function checks whether or not
* all physical and virtual scsi devices are removed. If so, it will clean-up mpp driver
* internal resources
*
* @RdacInfo  represents a storage array
* @DeviceVertex  Linux kernel object that represents a scsi device. The DeviceVertex
* 			is being removed.
* RESTRICTIONS:
*
* RETURNS:
*/

static void mppLnx_CheckRemoveModule(RdacDeviceInformation_t *RdacInfo, MPP_HANDLE DeviceVertex)
{
    LWORD i, VirtualLunExists;
    ArrayModuleEntry_t *ModPtr;
    unsigned long iflages;

    VirtualLunExists = 0;
    ModPtr = mpp_ModuleArray;
    ModPtr += RdacInfo->VirtualTargetId;


    for(i=0; i<mpp_MaxLunsPerArray; ++i)
    {
        if( RdacInfo->RdacLuns[i].Present  || RdacInfo->RdacLuns[i].PseudoLunObject )
        {
	        MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_2, 
                           "Array %s VirtualLun exists Lun %d %d %d\n", RdacInfo->ModuleName,i,
                            RdacInfo->RdacLuns[i].Present, RdacInfo->RdacLuns[i].PseudoLunObject ));
            VirtualLunExists = 1;
            break;
        }else if(RdacInfo->RdacLuns[i].ReportedMissing == TRUE)
        {
	        MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_2, 
                           "Array %s Reported Missing is TRUE Lun %d\n", RdacInfo->ModuleName,i));
            VirtualLunExists = 1;
            break;
        }
    }
    if( !VirtualLunExists &&
        !RdacInfo->ControllerInfo[0]->ControllerPresent &&
        !RdacInfo->ControllerInfo[1]->ControllerPresent )
    {
        mpp_RemoveModule(DeviceVertex, RdacInfo);

	    MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_2, 
                       "Array %s has been removed\n", RdacInfo->ModuleName));

        /* 
         * with out locking this area there was a panic noticed when sd and sg
         * remove device and both of them try to free resulting in a panic.
         * reusing unconfigured device lock instead of defining a new spinlock 
         */
		spin_lock_irqsave( &unconfiguredDevice_lock, iflages);
        if( ModPtr->RdacInfo == NULL )
        {
	        MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_2, 
                           "Array %s is already  removed\n", RdacInfo->ModuleName));
		    spin_unlock_irqrestore( &unconfiguredDevice_lock,iflages);

            return;
        }
        ModPtr->RdacInfo = NULL;
        MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 741, "Array %s has been removed\n",
           RdacInfo->ModuleName));
        mppCmn_FreeRdacInfo( RdacInfo );
		spin_unlock_irqrestore( &unconfiguredDevice_lock,iflages);
    }
}

/**
 * mppLnx_scsi_execute - insert request and wait for the result
 * @sdev:   scsi device
 * @cmd:    scsi command
 * @data_direction: data direction
 * @buffer: data buffer
 * @bufflen:    len of buffer
 * @sense:  optional sense buffer
 * @timeout:    request timeout in seconds
 * @retries:    number of times to retry request
 * @flags:  or into request flags;
 *
 * returns the req->errors value which is the the scsi_cmnd result
 * field.
 **/
int mppLnx_scsi_execute(struct scsi_device *sdev, const unsigned char *cmd,
         int data_direction, void *buffer, unsigned bufflen,
         unsigned char *sense, int timeout, int retries, int flags)
{
    struct request *req;
    int write = (data_direction == DMA_TO_DEVICE);
    int ret = DRIVER_ERROR << 24;

    req = blk_get_request(sdev->request_queue, write, __GFP_WAIT);

    if (bufflen &&  blk_rq_map_kern(sdev->request_queue, req,
                    buffer, bufflen, __GFP_WAIT))
        goto out;

    req->cmd_len = COMMAND_SIZE(cmd[0]);
    memcpy(req->cmd, cmd, req->cmd_len);
    req->sense = sense;
    req->sense_len = 0;
    req->retries = retries;
    req->timeout = timeout;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
    req->cmd_type = REQ_TYPE_BLOCK_PC;
    req->cmd_flags |= flags | REQ_QUIET | REQ_PREEMPT;
#else
    req->flags |= flags | REQ_BLOCK_PC | REQ_SPECIAL | REQ_QUIET;
#endif


    /*
     * head injection *required* here otherwise quiesce won't work
     */
    blk_execute_rq(req->q, NULL, req, 1);

    ret = req->errors;
 out:
    blk_put_request(req);

    return ret;
}


int mppLnx_scsi_execute_req(struct scsi_device *sdev, const unsigned char *cmd,
             int data_direction, void *buffer, unsigned bufflen,
             struct scsi_sense_hdr *sshdr, int timeout, int retries)
{
    char *sense = NULL;
    int result;
    
    if (sshdr) {
        sense = kzalloc(SCSI_SENSE_BUFFERSIZE, GFP_NOIO);
        if (!sense)
            return DRIVER_ERROR << 24;
    }
    result = mppLnx_scsi_execute(sdev, cmd, data_direction, buffer, bufflen,
                  sense, timeout, retries, 0);
    if (sshdr)
        scsi_normalize_sense(sense, SCSI_SENSE_BUFFERSIZE, sshdr);

    kfree(sense);
    return result;
}

module_init (mppLnx_init);
module_exit (mppLnx_exit);
MODULE_AUTHOR("IBM");
MODULE_DESCRIPTION ("MPP Upper Level Driver");
MODULE_LICENSE("GPL");
MODULE_VERSION(MPPLNX_VERSION);
MODULE_INFO(supported,"yes");

