/*******************************************************************************
* Copyright 2004-2007 LSI Logic Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************
NAME            mppLnx26p_vhbalib.c
SUMMARY         %description%
VERSION         %version: 7 %
UPDATE DATE     %date_modified: Thu May 08 17:53:43 2008 %
PROGRAMMER      %created_by:    sdachepa %


DESCRIPTION:
   This function contains the functions in the Linux MPP virtual HBA that manages
   the virtual HBA's internal data model. The functions include creation/deletion
   of list entries of different entry structs and manage the relations of virtual 
   SCSI commands and physical SCSI commands.

INCLUDE FILES:

NOTES:

RESTRICTIONS:

SEE ALSO:

REFERENCE:
  349-1033500 Linux MPP Driver Feature Architecture Specification
  349-1034610 Linux MPP Virtual HBA Driver Feature Design Document
  349-1034750 Linux MPP Upper Level Driver Feature Design Document
  349-1034730 Linux MPP Utility Feature Design Document
  349-1034620 Linux MPP Installation Feature Design Document
  349-1043860 Linux MPP driver for Linux 2.6 Kernel

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/

#define __SRCmppLnx26p_vhbalib_c

/***  INCLUDES  ***/  
/***  Linux kernel INCLUDES  ***/  
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/genhd.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <scsi/scsi_host.h>
#include <asm/io.h>

#include <linux/blkdev.h>

#include <linux/stat.h>

/***  MPP commomn API INCLUDES  ***/  
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "MPP_Sysdep.h"
#include "mppLnx_version.h"
/***  Linux MPP Virtual HBA INCLUDES  ***/  
#include "mppLnx26p_vhba.h"
#include "mppLnx26p_shared.h"


/***  CONSTANT DEFINITIONS  ***/


/***  MACRO DEFINITIONS  ***/


/***  TYPE DEFINITIONS  ***/


/***  LOCALS  ***/
#define MPPLNX_SLAB_NAME      "mpppq"
#define MPPLNX_ARSLAB_NAME    "mppaq"

mppLnx_VHBAContext_t mppLnxVHBAContext;
mppLnx_queueContext_t mppLnx_freeProxyRequestQ;
mppLnx_queueContext_t mppLnx_DPProxyRequestQ;
mppLnx_queueContext_t mppLnx_deferredProxyRequestQ;
mppLnx_queueContext_t mppLnx_queuedProxyRequestQ;
mppLnx_queueContext_t mppLnx_s2tos3ProxyRequestQ;
mppLnx_queueContext_t mppLnx_pathvalidationProxyRequestQ;
mppLnx_queueContext_t mppLnx_freeAsyncRequestQ;


/*
*declare a spinlock for protecting the context of the mppLnxVHBAContext
*/
static spinlock_t mppLnxVHBAContext_Lock = SPIN_LOCK_UNLOCKED;

/* list for adding the ProxyRequest blocks */
//static mppLnx_ProxyRequestEntry_t  	mppLnx_ProxyRequest_List;

mppLnx_vhbaMemPool_t          mppLnx_vhba_pqPool =
{
   .memCache = NULL,
   .size     = 0,
};

mppLnx_vhbaMemPool_t          mppLnx_vhba_arPool =
{
   .memCache = NULL,
   .size     = 0,
};


/***  PROCEDURES  ***/

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_init_queues
* SUMMARY:  Initialize the request queues
* SCOPE:   local
*
*
* DESCRIPTION:
      This function initializes request queues by initializing its list header, spin lock
      and queue type. This function should called only once before the virtual scsi host
      is ready to be allocated.
* RESTRICTIONS:
* 
* RETURNS:N/A
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_vhba_init_queues(void)
{
   
   INIT_LIST_HEAD(&mppLnx_freeProxyRequestQ.list);
   mppLnx_freeProxyRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
   mppLnx_freeProxyRequestQ.queueType = MPPLNX_QUEUE_FREE_LIST;

   INIT_LIST_HEAD(&mppLnx_DPProxyRequestQ.list);
   mppLnx_DPProxyRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
   mppLnx_DPProxyRequestQ.queueType = MPPLNX_QUEUE_DISPATCHED_LIST;

   INIT_LIST_HEAD(&mppLnx_deferredProxyRequestQ.list);
   mppLnx_deferredProxyRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
   mppLnx_deferredProxyRequestQ.queueType = MPPLNX_QUEUE_DEFERRED_LIST;

   INIT_LIST_HEAD(&mppLnx_queuedProxyRequestQ.list);
   mppLnx_queuedProxyRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
   mppLnx_queuedProxyRequestQ.queueType = MPPLNX_QUEUE_QUEUED_LIST;

   INIT_LIST_HEAD(&mppLnx_s2tos3ProxyRequestQ.list);
   mppLnx_s2tos3ProxyRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
   mppLnx_s2tos3ProxyRequestQ.queueType = MPPLNX_QUEUE_S2TOS3_LIST;

   INIT_LIST_HEAD(&mppLnx_pathvalidationProxyRequestQ.list);
   mppLnx_pathvalidationProxyRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
   mppLnx_pathvalidationProxyRequestQ.queueType = MPPLNX_QUEUE_PATHVALIDATION_LIST;

   INIT_LIST_HEAD(&mppLnx_freeAsyncRequestQ.list);
   mppLnx_freeAsyncRequestQ.queueLock = SPIN_LOCK_UNLOCKED;
  
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_mem_alloc
* SUMMARY:  allocate slab memory for proxy request
* SCOPE:   public 
*
*
* DESCRIPTION:
      This function allocates a number of mppLnx_ProxyRequestEntry_t object from 
      the slab. The memory for each  mppLnx_ProxyRequestEntry_t object include the 
      memory of the scsi_request object and "struct request". The mppLnx_ProxyRequestEntry_t
      object is initialized and added to the tail of the mppLnx_freeProxyRequestQ queue. 
* @size the  number of mppLnx_ProxyRequestEntry_t objects to be allocated
* RESTRICTIONS:
* 
* RETURNS:N/A
*
* ERRNO: 0 indicates success. -ENOMEM if no slab memory available.
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_vhba_mem_alloc(int size)
{
   struct kmem_cache                *mppCachep;
   int                              i;
   mppLnx_ProxyRequestEntry_t       *prp;
   int                              error = 0;
   unsigned long                    flags;
   int                              prSize;

   const int preOffset = ALIGN(sizeof(mppLnx_ProxyRequestEntry_t), sizeof(void *));
   
   prSize = preOffset;



   if(mppLnx_vhba_pqPool.memCache == NULL)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_ALL, 430,
         "mppLnx_vhba_mem_alloc request size = %d\n",prSize));
      mppCachep = kmem_cache_create(MPPLNX_SLAB_NAME,
            prSize, 0,
            0, NULL, NULL);
      if(NULL == mppCachep)
      {
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 432,
         "mppLnx_vhba_mem_alloc() kmem_cache_create failed\n"));
         error = -ENOMEM;
         goto alloc_fail1;
      }
      else
      {
         mppLnx_vhba_pqPool.memCache = mppCachep;
      }
   }

   
   for(i=0; i<size; i++)
   {
      prp = kmem_cache_alloc(mppLnx_vhba_pqPool.memCache,
         GFP_KERNEL);
      if(NULL == prp)
      {
         /*
         *
         */
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 433,
            "%s: kmem_cache_alloc failed\n", __FUNCTION__));
         error = -ENOMEM;
         goto alloc_fail2;
      }
      memset(prp,0,prSize);

      prp->state = MPPLNX_REQUEST_NOTUSED;
	  
      spin_lock_irqsave ( &mppLnx_freeProxyRequestQ.queueLock, flags);
      list_add_tail(&(prp->avail_list),
         &(mppLnx_freeProxyRequestQ.list)); 
      spin_unlock_irqrestore ( &mppLnx_freeProxyRequestQ.queueLock, flags);

   }
   mppLnx_vhba_pqPool.size +=size;
   

   return error;


alloc_fail2:
      /*
      * reset the free list
      */
      if(mppLnx_vhba_pqPool.memCache != NULL)
      {
         mppLnx_vhba_mem_free(mppLnx_vhba_pqPool.size);
      }
      INIT_LIST_HEAD(&mppLnx_freeProxyRequestQ.list);

alloc_fail1:
      return error;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_mem_free
* SUMMARY:  de-allocate slab memory for proxy request
* SCOPE:   public 
*
*
* DESCRIPTION:
      This function de-allocates mppLnx_ProxyRequestEntry_t object from the slab and 
      destroy the proxy request's slab cache object.
* @size the  number of mppLnx_ProxyRequestEntry_t objects to be allocated
* RESTRICTIONS:
* 
* RETURNS:N/A
*
* ERRNO: 0 indicates the number of mppLnx_ProxyRequestEntry_t objects be freed is equal 
         to the "size". 1 Otherwise
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_vhba_mem_free(int size)
{
   int count =0;
   mppLnx_ProxyRequestEntry_t    *preq;
   unsigned long                 flags;
   if(mppLnx_vhba_pqPool.memCache != NULL)
   {
      /*
      * free all proxy requests
      */
      spin_lock_irqsave ( &mppLnx_freeProxyRequestQ.queueLock, flags);

      while (!list_empty(&(mppLnx_freeProxyRequestQ.list))) 
      {
         count++;
         preq = list_entry(mppLnx_freeProxyRequestQ.list.next, mppLnx_ProxyRequestEntry_t,avail_list);
         list_del_init(&preq->avail_list);
         kmem_cache_free(mppLnx_vhba_pqPool.memCache, preq);
      }

      spin_unlock_irqrestore ( &mppLnx_freeProxyRequestQ.queueLock, flags);

      kmem_cache_destroy(mppLnx_vhba_pqPool.memCache);
   }

   return (size != count);
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_arMem_alloc
* SUMMARY:  allocate slab memory for async request
* SCOPE:   public 
*
*
* DESCRIPTION:
      This function allocates a number of mppLnx_AsyncRequestEntry_t object from 
      the slab. The memory for each  mppLnx_AsyncRequestEntry_t object include the 
      memory of the struct scsi_cmnd object and data buffer. The mppLnx_AsyncRequestEntry_t
      object is initialized and added to the tail of the mppLnx_freeAsyncRequestQ queue. 
* @size the  number of mppLnx_AsyncRequestEntry_t objects to be allocated
* RESTRICTIONS:
* 
* RETURNS:N/A
*
* ERRNO: 0 indicates success. -ENOMEM if no slab memory available.
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_vhba_arMem_alloc(int size)
{
   struct kmem_cache                *mppCachep;
   int                              i;
   mppLnx_AsyncRequestEntry_t       *are;
   int                              error = 0;
   unsigned long                    flags;
   int                              arSize = 0;

   arSize = sizeof(mppLnx_AsyncRequestEntry_t)+sizeof(struct scsi_cmnd)+
                sizeof(InquiryData_t);
   if(mppLnx_vhba_arPool.memCache == NULL)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_ALL,434,
         "mppLnx_vhba_arMem_alloc request size = %d\n",arSize));
      mppCachep = kmem_cache_create(MPPLNX_ARSLAB_NAME,
            arSize, 0,
            0, NULL, NULL);
      if(NULL == mppCachep)
      {
        MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 435,
            "%s: kmem_cache_create failed\n", __FUNCTION__));
         error = -ENOMEM;
         goto alloc_fail1;
      }
      else
      {
         mppLnx_vhba_arPool.memCache = mppCachep;
      }
   }

   
   for(i=0; i<size; i++)
   {
      are = kmem_cache_alloc(mppLnx_vhba_arPool.memCache,
         GFP_KERNEL);
      if(NULL == are)
      {
         /*
         *
         */
         MPP_ERRORLOGPRINT((MPP_ERR_ALL,440,
            "%s: kmem_cache_alloc failed\n", __FUNCTION__));
         error = -ENOMEM;
         goto alloc_fail2;
      }
      memset(are,0,arSize);

      are->cmnd = (struct scsi_cmnd*)( (( char *)are) + sizeof(mppLnx_AsyncRequestEntry_t));
      are->buffer = (void *)(((char *)are->cmnd) + sizeof(struct scsi_cmnd));
      spin_lock_irqsave ( &mppLnx_freeAsyncRequestQ.queueLock, flags);
      list_add_tail(&(are->free_list),
         &(mppLnx_freeAsyncRequestQ.list));
      spin_unlock_irqrestore ( &mppLnx_freeAsyncRequestQ.queueLock, flags);

   }
   mppLnx_vhba_arPool.size +=size;
  

   return error;


alloc_fail2:
      /*
      * reset the free list
      */
      if(mppLnx_vhba_arPool.memCache != NULL)
      {
         mppLnx_vhba_arMem_free(mppLnx_vhba_arPool.size);
      }

      INIT_LIST_HEAD(&mppLnx_freeAsyncRequestQ.list);
      

alloc_fail1:
      return error;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_vhba_arMem_free
* SUMMARY:  de-allocate slab memory for proxy request
* SCOPE:   public 
*
*
* DESCRIPTION:
      This function de-allocates mppLnx_AsyncRequestEntry_t object from the slab and 
      destroy the proxy request's slab cache object.
* @size the  number of mppLnx_AsyncRequestEntry_t objects to be allocated
* RESTRICTIONS:
* 
* RETURNS:N/A
*
* ERRNO: 0 indicates the number of mppLnx_ProxyRequestEntry_t objects be freed is equal 
         to the "size". 1 mppLnx_AsyncRequestEntry_t
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_vhba_arMem_free(int size)
{
   int count =0;
   mppLnx_AsyncRequestEntry_t    *are;
   unsigned long                 flags;
   if(mppLnx_vhba_arPool.memCache != NULL)
   {
      /*
      * free all proxy requests
      */
      spin_lock_irqsave ( &mppLnx_freeAsyncRequestQ.queueLock, flags);

      while (!list_empty(&(mppLnx_freeAsyncRequestQ.list))) 
      {
         count++;
         are = list_entry(mppLnx_freeAsyncRequestQ.list.next, mppLnx_AsyncRequestEntry_t,free_list);
         list_del_init(&are->free_list);
         kmem_cache_free(mppLnx_vhba_arPool.memCache, are);
      }

      spin_unlock_irqrestore ( &mppLnx_freeAsyncRequestQ.queueLock, flags);

      kmem_cache_destroy(mppLnx_vhba_arPool.memCache);
   }

   return (size != count);
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_initVHBAContext
* SUMMARY:  Initialize the internal data model of the Linux MPP virtual HBA 
         driver module.
* SCOPE:   local
*
*
* DESCRIPTION:
      This function is invoked by the mppLnx_detect() function, which is the 
      first function in the Linux MPP virtual HBA driver being invoked by the
      Linux SCSI middle level. This function initializes the internal data model, 
      walks through the MPP common data model and creates entry lists of the 
      virtual HBA's data model and /proc entries for all virtual and physical devices
* RESTRICTIONS:
* 
* RETURNS:0 indicates the proc leaf node is successfully 
*           created or 1 otherwise
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_initVHBAContext()
{
   int                           vTarget =0;
   int                           vLun = 0;
   int                           cntl = 0;
   int                           path = 0;
   int rtn  = 0;
   RdacDeviceInformation_t       *rdacInfo;
   RdacLunInformation_t          *rdaclun;
   LunPathObjects_t              * lunPathObject;
   struct scsi_device                   *sdp;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_initVHBAContext()\n"));
   /*
   * assign the lock to the data struct
   */
   mppLnxVHBAContext.vhbaContextLock = &mppLnxVHBAContext_Lock;
   /*
   * insert physical devices to the internal data struct - mppLnxVHBAContext
   */
   for(vTarget = 0; vTarget < mppLnx_Config_Parameters.mpp_MaxArrayModules; vTarget++)
   {
      if( (rdacInfo = mpp_ModuleArray[vTarget].RdacInfo) == NULL)
      {
         MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
            "no storage array at virtual target %d\n",vTarget));
         continue;
      }
      
      for(vLun = 0; vLun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; vLun++)
      {
         rdaclun = &(rdacInfo->RdacLuns[vLun]);
         if(!rdaclun->Present)
         {
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "No device at the addresses virtual target %d lun %d\n",
               vTarget, vLun));
            continue;
         }
         /*
         * the virtual device entries are created during slave_config.
         * only the physical entries are created here
         */

         for(cntl = 0; cntl < 2; cntl++)
         {
            lunPathObject = rdaclun->LunControllers[cntl];
            for(path = 0; path < mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
            {
               sdp = lunPathObject->LunPaths[path].LunPathDevice;
               if(sdp != NULL)
               {
                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
                     "found a physical device at h%dc%dt%dl%d\n",
                     sdp->host->host_no, sdp->channel, sdp->id, sdp->lun));
                  MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
                     "From array %s controller %d path %d\n",
                     (char *)rdacInfo->ModuleName, cntl,path));
                  mppLnx_insert_PdeviceEntry(sdp, rdacInfo, cntl, path);
			
               }
            }
         }
      }
   }
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_initVHBAContext()\n"));
   return rtn;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_cleanVHBAContext
* SUMMARY:  Cleans up the Linux MPP virtual HBA driver internal data model and 
      release system resources.
* SCOPE:   public
*
*
* DESCRIPTION:
   This function is invoked by the mppLnx_release() function, which is the last
   function of the Linux MPP virtual HBA driver being invoked by the Linux SCSI
   middle level at the time of the virtual HBA driver module unloading time.

   This function cleans-up the Linux MPP virtual HBA driver's internal data model 
   and release system resources. The system resources include the memory allocated
   for the internal model, and the /proc entries of virtual devices and physical
   devices.

* RESTRICTIONS:
* 
* RETURNS:SUCCESS indicates the proc leaf node is successfully 
*           created or FAILED otherwise
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_cleanVHBAContext()
{
   int                        lun = 0;
   struct list_head           *phlistPt;
   struct list_head           *ptlistPt;

   mppLnx_PDeviceEntry_t      *pde;
   mppLnx_PhostEntry_t        *phe;
   mppLnx_PtargetEntry_t      *pte;
   unsigned long              iflag;
   
   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_cleanVHBAContext()\n"));
  

   if(NULL == mppLnxVHBAContext.virtualDeviceList )
   {
      /* no virtual device */
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
         "No virtual device in the virtual device list\n"));
   }
   else
   {
      /**
      * the virtual mppLnx_VDeviceEntry_t objects are removed individually
      * during slave_destroy() time. 
      */
      /* finally, release the empty list header itself */
      MPP_FREE(mppLnxVHBAContext.virtualDeviceList,sizeof(mppLnx_VDeviceEntry_t));
      mppLnxVHBAContext.virtualDeviceList = NULL;
   }
   /*
   * remove/free physical device list
   */
   if(NULL == mppLnxVHBAContext.phostList )
   {
      /* no physical device in the internal data model */
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
         "No physical device in the internal data model\n"));
   }else
   {
      while(1)
      {
         /*
         * all physical Scsi_Host entry
         */
         spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
         if((phlistPt = mppLnxVHBAContext.phostList->list.next)
            != &(mppLnxVHBAContext.phostList->list))
         {
            phe = list_entry(phlistPt, mppLnx_PhostEntry_t, list);
            spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);

            MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
               "De-initializing on hostNo %d\n", phe->hostNo));
            /*
            * physical target entry
            */
            spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
            if((ptlistPt = phe->targetList->list.next)
               != &(phe->targetList->list))
            {
               pte = list_entry(ptlistPt, mppLnx_PtargetEntry_t, list);
               spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);

               MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
                  "De-initializing target %d\n", pte->targetId));
               /*
               * Lun entry of each target
               */
               for(lun = 0; lun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lun++)
               {
                  pde = pte->pdevices[lun];
                  if(pde != NULL)
                  {
                     MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
                        "deleting mppLnx_PDeviceEntry h%dc%dt%dl%d\n",
                        pde->pdevice->host->host_no,
                        pde->pdevice->channel,
                        pde->pdevice->id,
                        pde->pdevice->lun));
                     if(mppLnx_delete_PdeviceEntry(pde->pdevice) != 0 )
                     {
                        /* memory for pte is freed in mppLnx_delete_PdeviceEntry() for the last configured lun*/   
                        break;      
                     }
                  }
                  else
                  {
                     MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
                        "No mppLnx_PDeviceEntry at targetId %d lun %d\n",
                        pte->targetId, lun));
                  }
               } /* for lun loop */
            }/* for target if */
            else
            {
            
               /* free the mppLnx_PhostEntry_t's targetList */
               MPP_FREE(phe->targetList,sizeof(mppLnx_PtargetEntry_t));
               /* delete the mppLnx_PhostEntry_t object from its list */
               list_del(&(phe->list)); 
               spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
               /* free the memory of the mppLnx_PhostEntry_t object */
               MPP_FREE(phe,sizeof(mppLnx_PhostEntry_t));
            }

         } /* for host loop*/ 
         else
         {
            spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
            MPP_FREE(mppLnxVHBAContext.phostList,sizeof(mppLnx_PhostEntry_t)); /* free the empty list head*/
            break; /*break the while (1) */
         }
      }/* while(1) */
      
   }

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_cleanVHBAContext()\n"));
   return 0;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_get_VdeviceEntry
* SUMMARY:  Gets a virtual device entry object by a virtual device
* SCOPE:    public
*
* DESCRIPTION:
   This function looks up the mppLnx_VHBAContext for a mppLnx_VdeviceEntry_t 
   object that represent the virtual struct scsi_device object of "vdevice".
* @vdevice A virtual struct scsi_device object whose mppLnx_VDeviceEntry_t is requested.
* RESTRICTIONS:
* 
* RETURNS: Returns a mppLnx_VDeviceEntry_t object that represents the virtual
            scsi device or null if fails.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_VDeviceEntry_t * mppLnx_get_VdeviceEntry(struct scsi_device *vdevice)
{
   mppLnx_VDeviceEntry_t   *vDEp = NULL;
   struct list_head        * listPt;
   unsigned long           iflag;

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering mppLnx_get_VdeviceEntry()\n"));

   if(NULL == vdevice)
   {
      MPPLNX_VHBA_BUG();
   }

   if(NULL == mppLnxVHBAContext.virtualDeviceList)
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "Exiting mppLnx_get_VdeviceEntry()- Null virtualDeviceList \n"));
      return NULL;

   }
   else
   {
      spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
      for(listPt = mppLnxVHBAContext.virtualDeviceList->list.next;
         listPt != &(mppLnxVHBAContext.virtualDeviceList->list);
         listPt = listPt->next)
      {
         vDEp = list_entry(listPt, mppLnx_VDeviceEntry_t, list);
         if(vDEp->rdacInfo->VirtualTargetId == vdevice->id &&
            vDEp->lun == vdevice->lun  && vDEp->vdevice->host->host_no == vdevice->host->host_no)
         {
            /*
            * If this function is called first time from a virtual command
            * completion function, the vdevuce is not set yet. The vdevice 
            * is used later, we must set it here.
            */
            if(vDEp->vdevice == NULL)
            {
               vDEp->vdevice = vdevice;
            }
            break;
         }
         else
         {
            vDEp = NULL;
         }
      }
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
   }
  
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_getVdeviceEntry()\n"));
   return vDEp;
     
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_insert_VdeviceEntry
* SUMMARY:  creates and insert a mppLnx_VDeviceEntry_t object to mppLnx_VHNAContext
* SCOPE:   public
*
*
* DESCRIPTION:
   This function creates a "mppLnx_VdeviceEntry_t object" based on the input arguments
   and inserts the object to the MPP virtual HBA driver's internal data model.
   If an object mppLnx_VdeviceEntry_t, which represents the input "vdevice",  already 
   exists in the internal data model, this function doesn't insert the object 
   to the list again.

*  @lun the LUN number of a  virtual struct scsi_device object to be inserted
*  @rdacInfo a RdacDevuceInformation_t object that represents a storage array where the
*           vdevice is a virtual LUN of the array* 
RESTRICTIONS:
* 
* RETURNS: An mppLnx_VdeviceEntry_t object
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_VDeviceEntry_t * mppLnx_insert_VdeviceEntry
(
   int                        lun, 
   RdacDeviceInformation_t    * rdacInfo
)
{
   mppLnx_VDeviceEntry_t            *vde;
   struct list_head                 *listp;
   int                              size = 0;
   unsigned long                    iflag;
   
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Enterting mppLnx_insert_VdeviceEntry()\n"));

   if(NULL == rdacInfo)
   {
      MPPLNX_VHBA_BUG();
   }

   if(mppLnxVHBAContext.virtualDeviceList == NULL)
   {
      /*
      * create an empty entry 
      */
      size = sizeof(mppLnx_VDeviceEntry_t);
      vde = MPP_INT_KMEM_ALLOC(size);
      if(NULL == vde)
      {
         /* error. Cannot get kernel memory for mppLnx_VDeviceEntry_t*/
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 420,
            "Cannot get kernel memory for mppLnx_VDeviceEntry_t\n"));
         return NULL;
      }
      memset(vde, 0 , size);
      mppLnxVHBAContext.virtualDeviceList = vde;
	   listp = &vde->list;
      INIT_LIST_HEAD(listp);
   }

   /*
   * check whether or not the entry is already in the list
   */
   vde = NULL;
   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   for(listp = mppLnxVHBAContext.virtualDeviceList->list.next;
      listp != &(mppLnxVHBAContext.virtualDeviceList->list);
      listp = listp->next)
   {
      vde = list_entry(listp, mppLnx_VDeviceEntry_t, list);
      if(vde->rdacInfo->VirtualTargetId == rdacInfo->VirtualTargetId &&
         vde->lun == lun)
      {
         MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
            "Found the virtual device entry for vTarget %d vLun %d\n",
            rdacInfo->VirtualTargetId, lun));
         break;
      }else
      {
         vde = NULL;
      }
   }
   spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);


   /*
   * it is not in. Creates it and adds it to the list
   */
   if(NULL == vde)
   {
      /* this is a new vritual device */
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         " add a new virtual device to virtual device list vTarget %d vLun %d\n",
            rdacInfo->VirtualTargetId, lun));
      vde = mppLnx_createVDeviceEntry(lun, rdacInfo);
      spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
      list_add_tail(&(vde->list), &(mppLnxVHBAContext.virtualDeviceList->list));
      
      vde->rdacInfo = rdacInfo;
      vde->ioCount = 0;
      vde->longestTrip = 0;
      vde->shortestTrip = 0;
      vde->numCntlFailures = 0;
      vde->numPathFailures = 0;
      vde->lun = lun;
      vde->vdevice = rdacInfo->RdacLuns[lun].PseudoLunObject;
      /*
      * create a cross reference in the common data struct. It is used for updating virtual
      * device IO statistics during IO completion time.
      */
      rdacInfo->RdacLuns[lun].PlatformPrivateData = (VOID *) vde;

      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);

      mppLnx_createVDeviceProcNode(vde);
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppLnx_insert_VdeviceEntry() creates a new mppLnx_VDeviceEntry_t entry\n"));
   }
   else
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppLnx_insert_VdeviceEntry() the entry is already in the list\n"));
   }
   
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_insert_VdeviceEntry()\n"));
   
   return vde;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_delete_VdeviceEntry
* SUMMARY:  remove a mppLnx_VDeviceEntry_t object from mppLnx_VHNAContext,
            delete its proc node, and free its memory
* SCOPE:   public
*
* DESCRIPTION:
   This function remove a "mppLnx_VdeviceEntry_t object", based on the input
   Scsi virtual device, from the MPP virtual HBA driver's internal data
   model.
   It removes this mppLnx_VDeviceEntry_t object's /proc file and frees this 
   object's memory.
*  @vdevice a virtual struct scsi_device object to be deleted.

* RESTRICTIONS:
* 
* RETURNS: 0 if the resources are successfully released. Otherwise 1.
*
* ERRNO:
*
* NOTES:
*       This function performs a reverse operation as the function of
*       mppLnx_insert_VdeviceEntry does
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_delete_VdeviceEntry(mppLnx_VDeviceEntry_t * vde)
{
   unsigned long              iflag;

   if(NULL == vde)
   {
      MPPLNX_VHBA_BUG();
   }

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_delete_VdeviceEntry() %s:Lun %d\n",
      (char *)vde->rdacInfo->ModuleName,vde->lun));

   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   list_del(&(vde->list));
   spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
   /*
    * remove its proc node
    */
    mppLnx_removeVDeviceProcNode(vde);
   /*
   * before freeing the memory, set its cross reference in the common data struct to NULL
   */
    vde->rdacInfo->RdacLuns[vde->lun].PlatformPrivateData = NULL;
   /*
   * free its memory
   */
   MPP_FREE(vde,sizeof(mppLnx_VDeviceEntry_t));

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_delete_VdeviceEntry()\n"));   
   return 0;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_get_PdeviceEntry
* SUMMARY:  Gets a physical device entry object by a physical device
* SCOPE:   public
*
*
* DESCRIPTION:
   This function looks up the mppLnx_VHBAContext for a mppLnx_PdeviceEntry_t 
   object that represent the physical struct scsi_device object of "pdevice".
* @pdevice A physical struct scsi_device object whose mppLnx_PDeviceEntry_t object is
*          requested.
* RESTRICTIONS:
* 
* RETURNS:Returns a mppLnx_PDeviceEntry_t object that represents the physical
            scsi device or null if fails.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_PDeviceEntry_t * mppLnx_get_PdeviceEntry(struct scsi_device *pdevice)
{
   int                        hostNo = 0;
   int                        targetId =0;
   int                        lun = 0;
   int                        channelId = 0;
   struct                     list_head * listPrt;
   mppLnx_PhostEntry_t        * pHostEntry;
   mppLnx_PtargetEntry_t      * pTargetEntry=NULL;
   mppLnx_PDeviceEntry_t      * pDeviceEntry;
   unsigned long              iflag;
  
   if(NULL == pdevice)
   {
      MPPLNX_VHBA_BUG();
   }

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering mppLnx_get_PdeviceEntry() h%dc%dt%dl%d\n",
      pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun));
   
   hostNo = pdevice->host->host_no;
   targetId = pdevice->id;
   lun = pdevice->lun;
   channelId = pdevice->channel;
 
   /*
   * get the lock first
   */
   mppLnxVHBAContext.vhbaContextLock = &mppLnxVHBAContext_Lock;
   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   pHostEntry= mppLnxVHBAContext.phostList;
   if(pHostEntry == NULL)
   {
       MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
          "mppLnx_get_PdeviceEntry: did not have phostList\n"));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return NULL;
   }
   
   for(listPrt = mppLnxVHBAContext.phostList->list.next; 
         listPrt != &(mppLnxVHBAContext.phostList->list); 
		 listPrt =listPrt->next)
   {
      pHostEntry = list_entry(listPrt, mppLnx_PhostEntry_t, list);
      if(pHostEntry->hostNo == hostNo)
      {
         break;
      }
	  else
      {
         pHostEntry = NULL;
      }
   }
   /*
   * the device is from a new host
   */
   if(pHostEntry == NULL)
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppLnx_get_PdeviceEntry: did not have host entry\n"));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return NULL;
   }
   /*
   * The device is from new target
   */
   if(pHostEntry->targetList == NULL)
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppLnx_get_PdeviceEntry: did not have targetList\n"));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return NULL;
   }
   /*
   * find the target for this device
   */
   for(listPrt = pHostEntry->targetList->list.next; 
         listPrt != &(pHostEntry->targetList->list); 
		 listPrt =listPrt->next)
   {
      pTargetEntry = list_entry(listPrt, mppLnx_PtargetEntry_t, list);
      if((pTargetEntry->targetId == targetId) && (pTargetEntry->channelId == channelId))
      {
         break;
      }else
      {
         pTargetEntry = NULL;
      }
   }
   /*
   * the device is from a new target
   */
   if(pTargetEntry == NULL)
   {
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppLnx_get_PdeviceEntry: did not have target entry\n"));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return NULL;
   }
   
   pDeviceEntry =  pTargetEntry->pdevices[lun];
   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_get_PdeviceEntry() h%dc%dt%dl%d\n",
      pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun)); 
   spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
   return pDeviceEntry;
 
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_insert_PdeviceEntry
* SUMMARY:  creates and insert a mppLnx_PDeviceEntry_t object to mppLnx_VHNAContext
* SCOPE:   public
*
* DESCRIPTION:
   This function creates a "mppLnx_PdeviceEntry_t object" based on the input 
   arguments and inserts the object to the MPP virtual HBA driver's 
   internal data model.
   If an object mppLnx_PdeviceEntry_t, which represents the input "pdevice",  already 
   exists in the internal data model, this function doesn't insert the object to 
   the list again.

* @pdevice A physical struct scsi_device object
* @rdacInfo the RdacDeviceInformation_t object that represents the arrary where the physical
            device is discovered
* @controllerSlot The controller slot number from where the physcial device is discovered
* @pathId   The path ID number of the physical device in its LunPathObjects
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_insert_PdeviceEntry
(
   struct scsi_device * pdevice, 
   RdacDeviceInformation_t *rdacInfo, 
   int controllerSlot, 
   int pathId
   )
{
   int                        size = 0;
   int                        hostNo = 0;
   int                        targetId =0;
   int                        lun = 0;
   int                        channelId = 0;
   struct list_head           * listPrt;
   mppLnx_PhostEntry_t        * pHostEntry = NULL;
   mppLnx_PtargetEntry_t      * pTargetEntry = NULL;
   unsigned long              iflag;

   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_insert_PdeviceEntry()\n"));

   hostNo = pdevice->host->host_no;
   targetId = pdevice->id;
   lun = pdevice->lun;
   channelId = pdevice->channel;
   /*
   * get the lock first
   */
   mppLnxVHBAContext.vhbaContextLock = &mppLnxVHBAContext_Lock;
   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   /*check if the phostList is initialized
   */
   if(mppLnxVHBAContext.phostList == NULL)
   {
      /*
      * create an empty entry 
      * this entry is only used as list_head
      */
      size = sizeof(mppLnx_PhostEntry_t);
      mppLnxVHBAContext.phostList = MPP_INT_KMEM_ALLOC(size);
      if(mppLnxVHBAContext.phostList == NULL)
      {
         /* error case*/
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 421,
            "Can not get kernel memory for allacating mppLnx_PhostEntry_t\n"));
         spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
         return;
      }
      else
      {
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "create an empty mppLnx_PhostEntry_t as list_head\n"));
      }

      memset(mppLnxVHBAContext.phostList, 0, size);
      
      INIT_LIST_HEAD(&(mppLnxVHBAContext.phostList->list));
  
   }

   /*
   * find its host
   */
   for(listPrt = mppLnxVHBAContext.phostList->list.next; 
         listPrt != &(mppLnxVHBAContext.phostList->list); 
		 listPrt =listPrt->next)
   {
      pHostEntry = list_entry(listPrt, mppLnx_PhostEntry_t, list);
      if(pHostEntry->hostNo == hostNo)
      {
         MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "found the mppLnx_PhostEntry_t for the device host_no %d\n",
            hostNo));
         break;
      }else
      {
         pHostEntry = NULL;
      }
   }
   /*
   * the device is from a new host
   */
   if(pHostEntry == NULL)
   {
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
         "Did not found the mppLnx_PhostEntry_t for the device host_no %d\n",
            hostNo));
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
         "Create a new mppLnx_PhostEntry_t\n"));
      pHostEntry = mppLnx_createPhostEntry(pdevice);
      list_add_tail(&(pHostEntry->list), &(mppLnxVHBAContext.phostList->list));
   }
   /*
   * check if its target is created
   */
   if(pHostEntry->targetList == NULL)
   {
      /**
      * there is no mppLnx_PtargetEntry object in the host's target list
      * create an empty one and it will be the list_head
      */
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
         " create a list_head for mppLnx_PtargetEntry_t \n"));
      size = sizeof(mppLnx_PtargetEntry_t);
      pHostEntry->targetList = MPP_INT_KMEM_ALLOC(size);
      if(NULL == pHostEntry->targetList)
      {
         /* error case. Cannot get kernel memory*/
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 422,
            "Cannot get kernel memory for allacating mppLnx_PtargetEntry_t\n"));
         spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
         return;
      }
      memset(pHostEntry->targetList, 0, size);

      INIT_LIST_HEAD(&(pHostEntry->targetList->list));
   }
   /*
   * find its target
   */
   for(listPrt = pHostEntry->targetList->list.next; 
         listPrt != &(pHostEntry->targetList->list);
		 listPrt =listPrt->next)
   {
      pTargetEntry = list_entry(listPrt, mppLnx_PtargetEntry_t, list);
      if((pTargetEntry->targetId == targetId) && (pTargetEntry->channelId == channelId))
      {
         break;
      }else
      {
         pTargetEntry = NULL;
      }
   }
   /*
   *From a new target
   */
   if(NULL == pTargetEntry)
   {
      /*
      * did not find the target entry for this device
      * create one
      */
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
         "create an mppLnx_PtargetEntry_t for device h%dc%dt%dl%d\n",
         pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun));
      pTargetEntry = mppLnx_createPtargetEntry(pdevice);
      if(NULL == pTargetEntry)
      {
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 423,
            "Error in geting memory for mppLnx_PtargetEntry_t\n"));
         spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
         return;
      }
      list_add_tail(&(pTargetEntry->list),&(pHostEntry->targetList->list));
      size = sizeof(mppLnx_PDeviceEntry_t *)* mppLnx_Config_Parameters.mpp_MaxLunsPerArray;
      pTargetEntry->pdevices = 
			(mppLnx_PDeviceEntry_t * *) MPP_INT_KMEM_ALLOC(size);
      if(pTargetEntry->pdevices == NULL)
      {
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 424,
            "Cannot allocate memory for mppLnx_PDeviceEntry_t **)\n"));
         spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
         return;
      }
      memset(pTargetEntry->pdevices, 0, size);
   }
   
   /*
   * Check whether or not the physical device entry is alreay created
   */
   if(pTargetEntry->pdevices[lun] != NULL)
   {
      MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
            "The mppLnx_PDeviceEntry is already in the list\n"));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return;
   }

   /* 
   * get the target entry. Create mppLnx_PDeviceEntry_t
   */
   size = sizeof(mppLnx_PDeviceEntry_t);
   pTargetEntry->pdevices[lun] = MPP_INT_KMEM_ALLOC(size);

   if(pTargetEntry->pdevices[lun] == NULL)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 425,
            "Cannot allocate memory for mppLnx_PDeviceEntry_t)\n"));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return;
   }

   memset(pTargetEntry->pdevices[lun], 0 ,size);
   pTargetEntry->pdevices[lun]->pdevice = pdevice;
   pTargetEntry->pdevices[lun]->rdacInfo = rdacInfo;
   pTargetEntry->pdevices[lun]->controllerSlot = controllerSlot;
   pTargetEntry->pdevices[lun]->pathId = pathId;
   /*
   * create a cross reference in the common data struct. It is used for updating physical device
   * IO statistics during IO completion time.
   */
   rdacInfo->RdacLuns[lun].LunControllers[controllerSlot]->LunPaths[pathId].PlatformPrivateData
      = (VOID *) pTargetEntry->pdevices[lun];
   
   
   /*
   * release the lock
   */
   spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);

   mppLnx_createPDeviceProcNode(pTargetEntry->pdevices[lun]);
   
   MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_insert_PdeviceEntry()\n"));
   
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_delete_PdeviceEntry
* SUMMARY:  remove a mppLnx_PDeviceEntry_t object from mppLnx_VHNAContext,
            delete its proc node, and free its memory
* SCOPE:   public
*
* DESCRIPTION:
   This function remove a "mppLnx_PDeviceEntry_t object", based on the input
   Scsi physical device, from the MPP virtual HBA driver's internal data
   model.
   It removes this mppLnx_PDeviceEntry_t object's /proc file and frees this 
   object's memory.
*  @pdevice a physical struct scsi_device object to be deleted.

* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*       This function performs a reverse operation as the function of
*       mppLnx_insert_PdeviceEntry does
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_delete_PdeviceEntry(struct scsi_device * pdevice)
{
   int                        hostNo = 0;
   int                        targetId =0;
   int                        lun = 0;
   struct list_head *         listPrt;
   mppLnx_PhostEntry_t        * pHostEntry = NULL;
   mppLnx_PtargetEntry_t      * pTargetEntry = NULL;
   mppLnx_PDeviceEntry_t      * pDeviceEntry = NULL;
   unsigned long              iflag;
   mppLnx_PDeviceEntry_t      * pde;
   int                        needRemoveTarget = 1;
   int                        size;
  
   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_delete_PdeviceEntry() h%dc%dt%dl%d\n",
      pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun));
   
   hostNo = pdevice->host->host_no;
   targetId = pdevice->id;
   lun = pdevice->lun;
 
   /*
   * get the lock first
   */
   mppLnxVHBAContext.vhbaContextLock = &mppLnxVHBAContext_Lock;
   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   pHostEntry = mppLnxVHBAContext.phostList;

   if(NULL == pHostEntry)
   {
      /* why no Host device
      * the phsyical HBA driver must not be loaded
      */
     MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
        "progaming error:pHostEntry= mppLnxVHBAContext.phostList is NULL\n"));
     MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
        "Exiting mppLnx_delete_PdeviceEntry() h%dc%dt%dl%d\n",
              pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun));
     spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
     return -1;
   }
   
   
   if(( mppLnxVHBAContext.phostList->list.next) == &(mppLnxVHBAContext.phostList->list)) {

       MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,"mppLnx_delete_PdeviceEntry Last phost list head entry has to be removed \n"));

       spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);

       MPP_FREE(mppLnxVHBAContext.phostList,sizeof(mppLnx_PhostEntry_t)); /* free the empty list head*/

       mppLnxVHBAContext.phostList = NULL;

       return -1;
    }
   
   for(listPrt = mppLnxVHBAContext.phostList->list.next; 
         listPrt != &(mppLnxVHBAContext.phostList->list); 
		 listPrt =listPrt->next)
   {
      pHostEntry = list_entry(listPrt, mppLnx_PhostEntry_t, list);
      if(pHostEntry->hostNo == hostNo)
      {
         MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
            "mppLnx_delete_PdeviceEntry found the device's host\n"));
         break;
      }
	  else
      {
         pHostEntry = NULL;
      }
   }

   
   if(pHostEntry == NULL)
   {
      /* why no physical host
      * the phsyical HBA driver must not be loaded.
      * the upper level driver messed up?
      */
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppLnx_delete_PdeviceEntry did not have the device's host\n"));
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "Exiting mppLnx_get_PdeviceEntry() h%dc%dt%dl%d\n",
              pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun));
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return -1;
   }
   

   if(pHostEntry->targetList == NULL)
   {
      /* why no target
      * the upper level driver messed up?
      */
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppLnx_delete_PdeviceEntry: targetList is NULL\n"));

      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "Exiting mppLnx_delete_PdeviceEntry() h%dc%dt%dl%d\n",
         pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun)); 

      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return -1;
   }



   if(( pHostEntry->targetList->list.next) == &(pHostEntry->targetList->list)) {

         MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
            "mppLnx_delete_PdeviceEntry: Last ptarget list head entry has to be removed \n"));

       /* free the mppLnx_PhostEntry_t's targetList */

       MPP_FREE(pHostEntry->targetList,sizeof(mppLnx_PtargetEntry_t));

       /* delete the mppLnx_PhostEntry_t object from its list */

       list_del(&(pHostEntry->list));

       spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);

       /* free the memory of the mppLnx_PhostEntry_t object */

       MPP_FREE(pHostEntry,sizeof(mppLnx_PhostEntry_t));

       return -1;

   }
   /*
   *
   */
   for(listPrt = pHostEntry->targetList->list.next; 
         listPrt != &(pHostEntry->targetList->list); 
		 listPrt =listPrt->next)
   {
      pTargetEntry = list_entry(listPrt, mppLnx_PtargetEntry_t, list);
      if(pTargetEntry->targetId == targetId)
      {
         MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
            "mppLnx_delete_PdeviceEntry: found the device's target object\n"));
         break;
      }else
      {
         pTargetEntry = NULL;
      }
   }
   

   if(pTargetEntry == NULL)
   {
      /* why no target
      * the upper level driver messed up?
      */
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppLnx_delete_PdeviceEntry: cannot find the device's target object\n"));
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "Exiting mppLnx_get_PdeviceEntry() h%dc%dt%dl%d\n",
         pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun)); 
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return -1;
   }
   
   pDeviceEntry =  pTargetEntry->pdevices[lun];
   if(NULL == pDeviceEntry)
   {
      /* why no the physical device
      */
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppLnx_delete_PdeviceEntry: cannot find the device's mppLnx_PDeviceEntry_t\n"));
      MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
         "Exiting mppLnx_get_PdeviceEntry() h%dc%dt%dl%d\n",
         pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun)); 
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      return -1;
   }

   /**
   * clean its proc node and free its memory
   */
   mppLnx_removePDeviceProcNode(pDeviceEntry);
   
   /*
   * before we free the memory, set its reference in the common data struct to NULL
   */
   pDeviceEntry->rdacInfo->RdacLuns[pDeviceEntry->pdevice->lun].
      LunControllers[pDeviceEntry->controllerSlot]->LunPaths[pDeviceEntry->pathId].PlatformPrivateData = NULL;

   pTargetEntry->pdevices[lun] = NULL;

   MPP_FREE(pDeviceEntry,sizeof(mppLnx_PDeviceEntry_t));

  /*
   * remove the target 
   */
   for(lun = 0; lun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; lun++)
   {
      pde = pTargetEntry->pdevices[lun];
      if(pde != NULL)
      {
         needRemoveTarget = 0;
         break;
      }           
   } /* for lun loop */

   if(needRemoveTarget)
   {
      list_del(&(pTargetEntry->list)); /* delete the target from the list */
      spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
      size = sizeof(mppLnx_PDeviceEntry_t *)* mppLnx_Config_Parameters.mpp_MaxLunsPerArray;
      MPP_FREE(pTargetEntry->pdevices,size); /* free its pdevice ( array of pointers */
      MPP_FREE(pTargetEntry, sizeof(mppLnx_PtargetEntry_t)); /* free the mppLnx_PtargetEntry_t itself */
      return -1;
   }else
   {
       spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
   }

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_get_PdeviceEntry() h%dc%dt%dl%d\n",
      pdevice->host->host_no, pdevice->channel, pdevice->id, pdevice->lun)); 
   return 0;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_createPtargetEntry
* SUMMARY:  Creates and initializes an mppLnx_PtargetEntry_t object.
* SCOPE:   local
*
* DESCRIPTION:
   This function creates and initializes an mppLnx_PtargetEntry_t object. 
   The object is not linked/added to its list.
* @ pdevice A physical struct scsi_device object for which its mppLnx_PtargetEntry_t
   object needs to be created
* RESTRICTIONS:
* 
* RETURNS: An mppLnx_PtargetEntry_t object which is newly created or NULL if we
*           have memory allocation problem
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_PtargetEntry_t * mppLnx_createPtargetEntry(struct scsi_device * pdevice)
{
	int                        size = 0;
   mppLnx_PtargetEntry_t      * ptargetEntry;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_createPtargetEntry()\n"));
     /*
      * create an entry 
      */
      size = sizeof(mppLnx_PtargetEntry_t);
      ptargetEntry = MPP_INT_KMEM_ALLOC(size);
      
      if(ptargetEntry == NULL)
      {
         /* error. Cannot get kernel memory */
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 427,
            "Cannot get kernel memory for mppLnx_PtargetEntry_t\n"));
         return NULL;
      }
      memset((void *) ptargetEntry, 0, size);
      ptargetEntry->targetId = pdevice->id;
      ptargetEntry->channelId = pdevice->channel;
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_createPtargetEntry()\n"));
   return ptargetEntry;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_createVDeviceEntry
* SUMMARY:  Creates and initializes an mppLnx_VDeviceEntry_t object.
* SCOPE:   local
*
* DESCRIPTION:
   This function creates and initializes an mppLnx_VDeviceEntry_t object. The object 
   is not linked/added to its list.

* @ lun The Lun number of the virtual struct scsi_device
* @rdacInfo A RdacDeviceInformat_t object which represents a storage array,
* RESTRICTIONS:
* 
* RETURNS: A newly created mppLnx_VDeviceEntry_t object or NULL if memory allocation 
            error.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_VDeviceEntry_t * mppLnx_createVDeviceEntry(
    int lun,
    RdacDeviceInformation_t *rdacInfo)
{
   int                        size = 0;
   mppLnx_VDeviceEntry_t      * vDeviceEntry;
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_createVDeviceEntry()\n"));
   /*
    * create an entry 
   */
   size = sizeof(mppLnx_VDeviceEntry_t);
   vDeviceEntry = MPP_INT_KMEM_ALLOC(size);
      
   if(vDeviceEntry == NULL)
   {
      /* error cannot get kernel memory */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 428,
           "Cannot get kernel memory for mppLnx_VDeviceEntry_t\n"));
      return NULL;
   }
   memset((void *) vDeviceEntry, 0, size);
   vDeviceEntry->lun = lun;
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
       "Exiting mppLnx_createVDeviceEntry()\n"));
   return vDeviceEntry;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_get_free_proxyRequest
* SUMMARY:  get a proxy request object from the free proxy request list.
* SCOPE:   public
*
* DESCRIPTION:
*  detach a proxy request object from the free proxy request list. If the free proxy 
   request list is empty, try to allocate a new proxy request object from the slab. 
* RETURNS: An mppLnx_ProxyRequestEntry_t object or NULL if memory allocation error
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_ProxyRequestEntry_t * mppLnx_get_free_proxyRequest( void )
{
   mppLnx_ProxyRequestEntry_t       *pre=NULL;
   struct list_head                 *avail_listp;
   unsigned long                    flags;
   const int preOffset = ALIGN(sizeof(mppLnx_ProxyRequestEntry_t), sizeof(void *));

   const int prSize = preOffset;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "Entering mppLnx_get_free_proxyRequest()\n"));

   /* Acquire Lock */
   spin_lock_irqsave ( &mppLnx_freeProxyRequestQ.queueLock, flags);

   /* get the next free entry from the head of available list */
   list_for_each( avail_listp, &(mppLnx_freeProxyRequestQ.list))
   {
      pre = list_entry( avail_listp, mppLnx_ProxyRequestEntry_t, avail_list );
      if( pre->state == MPPLNX_REQUEST_NOTUSED )
      {
         break;
      }
   }

   if(likely(pre))
   {
      /* delete it from the availble list */
      list_del(&(pre->avail_list));
   }

   /* Release Lock */
   spin_unlock_irqrestore ( &mppLnx_freeProxyRequestQ.queueLock, flags);
   /*
   * We should have a free slot. If not, try to allocate another one
   */
   if(unlikely(!pre))
   {
      /*
      * print a warning here. we may have problem in managing the queues.
      */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,436,
         "%s: NEED more free slot\n", __FUNCTION__));
      pre = kmem_cache_alloc(mppLnx_vhba_pqPool.memCache,
                  GFP_ATOMIC);
      if ( unlikely(!pre ))
      {
         /* cannot get memory */
        
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 426,
            "Error cannot get kernel memory for mppLnx_ProxyRequestEntry_t\n"));
         return (pre);
      }

      mppLnx_vhba_pqPool.size++;
      
      memset(pre,0, prSize);

      pre->state = MPPLNX_REQUEST_NOTUSED;

   }

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "Exiting mppLnx_get_free_proxyRequest()\n"));

   return ( pre );
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_put_free_proxyRequest
* SUMMARY:  put a mppLnx_ProxyRequestEntry_t object back to the free proxy request list.
* SCOPE:   local
*
* DESCRIPTION:
 Add the object to available list.
 Set the state of the object to MPPLNX_REQUEST_NOTUSED.

* @mppLnx_ProxyRequestEntry_t object that is to be added to the available  list.
* 
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
void	mppLnx_put_free_proxyRequest( mppLnx_ProxyRequestEntry_t *pre)
{

   unsigned long  flags;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_put_free_proxyRequest()\n"));

   
   /* Set the state to free */
   pre->state 	= MPPLNX_REQUEST_NOTUSED ;
   /*
   *reset some fields
   */
   pre->expire_time                    = 0;
   pre->errorRecovery                  = 0;
   pre->ArrayIoWaitTime                = 0;
   pre->CommandTimeoutRetryCount       = 0;
   pre->ControllerIoWaitTime           = 0;
   pre->rdacInfo                       = NULL;
   pre->RetryCount                     = 0;
   pre->SelectionTimeoutRetryCount     = 0;
   pre->startTime                      = 0;
   pre->UaRetryCount                   = 0;
   pre->req                            = NULL;

    /* Acquire Lock */
   spin_lock_irqsave ( &mppLnx_freeProxyRequestQ.queueLock, flags);

   /* add the element to the available list */
   list_add_tail( &(pre->avail_list) , &(mppLnx_freeProxyRequestQ.list));

   /* Release Lock */
   spin_unlock_irqrestore ( &mppLnx_freeProxyRequestQ.queueLock, flags);

      MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
            "Exiting mppLnx_put_free_proxyRequest()\n"));

   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_get_free_asyncRequest
* SUMMARY:  get a async request object from the free async request list.
* SCOPE:   public
*
* DESCRIPTION:
*  detach a async request object from the free async request list. If the free async 
   request list is empty, try to allocate a new async request object from the slab. 
* RETURNS: An mppLnx_AsyncRequestEntry_t object or NULL if memory allocation error
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_AsyncRequestEntry_t * mppLnx_get_free_asyncRequest( void )
{
   mppLnx_AsyncRequestEntry_t       *are = NULL;
   struct list_head                 *free_listp;
   unsigned long                    flags;
   int                              arSize = 0;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "Entering mppLnx_get_free_asyncRequest()\n"));
   arSize = sizeof(mppLnx_AsyncRequestEntry_t)+sizeof(struct scsi_cmnd)+
                sizeof(InquiryData_t);

   /* Acquire Lock */
   spin_lock_irqsave ( &mppLnx_freeAsyncRequestQ.queueLock, flags);

   /* get the next free entry from the head of available list */
   list_for_each( free_listp, &(mppLnx_freeAsyncRequestQ.list))
   {
      are = list_entry( free_listp, mppLnx_AsyncRequestEntry_t, free_list );
      if(likely(are) )
      {
         break;
      }
   }

   if(likely(are))
   {
      /* delete it from the availble list */
      list_del(&(are->free_list));
   }

   /* Release Lock */
   spin_unlock_irqrestore ( &mppLnx_freeAsyncRequestQ.queueLock, flags);
   /*
   * We should have a free slot. If not, try to allocate another one
   */
   if(unlikely(!are))
   {
      /*
      * print a warning here. we may have problem in managing the queues.
      */
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE,439,
         "%s: NEED more free slot\n", __FUNCTION__));

      are = kmem_cache_alloc(mppLnx_vhba_arPool.memCache,
                  GFP_ATOMIC);
      if ( unlikely(!are ))
      {
         /* cannot get memory */
        MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 429,
            "%s: kmem_cache_alloc failed\n", __FUNCTION__));

         return (are);
      }

      mppLnx_vhba_arPool.size++;
      
      memset(are,0,arSize);

      are->cmnd = (struct scsi_cmnd*)( (( char *)are) + sizeof(mppLnx_AsyncRequestEntry_t));
      are->buffer = (void *)(((char *)are->cmnd) + sizeof(struct scsi_cmnd));
      
   }

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "Exiting mppLnx_get_free_asyncRequest()\n"));

   return ( are );
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_put_free_asyncRequest
* SUMMARY:  put a mppLnx_AsyncRequestEntry_t object back to the free async request list.
* SCOPE:   local
*
* DESCRIPTION:
 Add the object to free list.
 
* @mppLnx_AsyncRequestEntry_t object that is to be added to the available  list.
* 
* RETURNS: N/A
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*/
void	mppLnx_put_free_asyncRequest( mppLnx_AsyncRequestEntry_t *are)
{

   unsigned long  flags;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_put_free_asyncRequest()\n"));

   /* Acquire Lock */
   spin_lock_irqsave ( &mppLnx_freeAsyncRequestQ.queueLock, flags);

   

   /* add the element to the available list */
   list_add_tail( &(are->free_list) , &(mppLnx_freeAsyncRequestQ.list));

   /* Release Lock */
   spin_unlock_irqrestore ( &mppLnx_freeAsyncRequestQ.queueLock, flags);

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
            "Exiting mppLnx_put_free_asyncRequest()\n"));

   return;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_insert_proxyRequest_to_list
* SUMMARY:  Inserts a proxy request entry to a specified queue
* SCOPE:    public
*
* DESCRIPTION:
   This function inserts mppLnx_ProxyRequestEntry_t object to a specified queue. 
   The queue type is specified by "type" and the mppLnx_ProxyRequestEntry_t object 
   is specified by "pre"

*  @request A mppLnx_ProxyRequestEntry_t object which represents the association of
*           a virtual Scsi command and its proxy Scsi command.
*  @type the type of the queue
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void  mppLnx_insert_proxyRequest_to_list( 
       mppLnx_ProxyRequestEntry_t *pre,
       mppLnx_queue_type_t type)
{

   unsigned long  flags;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_insert_proxyRequest_to_list()\n"));

   switch (type)
   {
      case  MPPLNX_QUEUE_FREE_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "mppLnx_insert_proxyRequest_to_list() MPPLNX_QUEUE_FREE_LIST \n"));
            /* Acquire Lock */
            spin_lock_irqsave ( &mppLnx_freeProxyRequestQ.queueLock, flags);
            /* add the element to the available list */
            list_add_tail( &(pre->avail_list) , &(mppLnx_freeProxyRequestQ.list));
            pre->state = MPPLNX_REQUEST_NOTUSED;
            spin_unlock_irqrestore ( &mppLnx_freeProxyRequestQ.queueLock, flags);
            break;

      case  MPPLNX_QUEUE_DEFERRED_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "mppLnx_insert_proxyRequest_to_list() MPPLNX_QUEUE_DEFERRED_LIST \n"));
            spin_lock_irqsave ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
            list_add_tail( &(pre->pending_list) , &(mppLnx_deferredProxyRequestQ.list));
            pre->state = MPPLNX_REQUEST_DEFERRED;
            spin_unlock_irqrestore ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
            break;

      case  MPPLNX_QUEUE_QUEUED_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "mppLnx_insert_proxyRequest_to_list() MPPLNX_QUEUE_QUEUED_LIST \n"));
            spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
            list_add_tail( &(pre->queued_list) , &(mppLnx_queuedProxyRequestQ.list));
            pre->state = MPPLNX_REQUEST_DISPATCHED;
            spin_unlock_irqrestore ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
            break;
      case  MPPLNX_QUEUE_DISPATCHED_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "mppLnx_insert_proxyRequest_to_list() MPPLNX_QUEUE_DISPATCHED_LIST \n"));
            spin_lock_irqsave ( &mppLnx_DPProxyRequestQ.queueLock, flags);
            list_add_tail( &(pre->dispatch_list) , &(mppLnx_DPProxyRequestQ.list));
            if(pre->dispatchType == MPPLNX_QUEUED_CMND_TYPE)
            {
               pre->state = MPPLNX_REQUEST_DPQUEUED;
            }
            else if((pre->dispatchType == MPPLNX_COMPLETED_CMND_TYPE) ||
		    (pre->dispatchType == MPPLNX_FAILOVER_CMND_TYPE) ||
		    (pre->dispatchType == MPPLNX_VALIDATE_PATH_TYPE))
            {
               pre->state = MPPLNX_REQUEST_COMPLETING;
            }
            else if(pre->dispatchType  == MPPLNX_DELAY_RESPONSE_TYPE)
            {
               pre->state = MPPLNX_REQUEST_DPQUEUED;
            }
            else
            {
               MPPLNX_VHBA_BUG();
            }
            spin_unlock_irqrestore ( &mppLnx_DPProxyRequestQ.queueLock, flags);
            break;
      case MPPLNX_QUEUE_S2TOS3_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "mppLnx_insert_proxyRequest_to_list() MPPLNX_QUEUE_S2TOS3_LIST \n"));
            spin_lock_irqsave ( &mppLnx_s2tos3ProxyRequestQ.queueLock, flags);
            list_add_tail( &(pre->s2tos3_list) , &(mppLnx_s2tos3ProxyRequestQ.list));
            pre->state = MPPLNX_REQUEST_DISPATCHED;
            spin_unlock_irqrestore ( &mppLnx_s2tos3ProxyRequestQ.queueLock, flags);
            break;
      case MPPLNX_QUEUE_PATHVALIDATION_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
               "mppLnx_insert_proxyRequest_to_list() MPPLNX_QUEUE_PATHVALIDATION_LIST \n"));
            spin_lock_irqsave ( &mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
            list_add_tail( &(pre->pathvalidation_list) , &(mppLnx_pathvalidationProxyRequestQ.list));
            pre->state = MPPLNX_REQUEST_COMPLETING;
            spin_unlock_irqrestore ( &mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
            break;
      default:
         /*
         * print an error
         */
         break;
   }
     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_insert_proxyRequest_to_list()\n"));
   return;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_remove_proxyRequest_from_list
* SUMMARY:  remove a proxy request entry to a specified queue
* SCOPE:    public
*
* DESCRIPTION:
   This function removes mppLnx_ProxyRequestEntry_t object to a specified queue. 
   The queue type is specified by "type" and the mppLnx_ProxyRequestEntry_t object 
   is specified by "pre"

*  @request A mppLnx_ProxyRequestEntry_t object which represents the association of
*           a virtual Scsi command and its proxy Scsi command.
*  @type the type of the queue
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void  mppLnx_remove_proxyRequest_from_list(
      mppLnx_ProxyRequestEntry_t *pre,
      mppLnx_queue_type_t type)
{

   unsigned long  flags;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_remove_proxyRequest_from_list()\n"));

   switch (type)
   {
      case  MPPLNX_QUEUE_FREE_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppLnx_remove_proxyRequest_from_list() MPPLNX_QUEUE_FREE_LIST\n"));
            /* Acquire Lock */
            spin_lock_irqsave ( &mppLnx_freeProxyRequestQ.queueLock, flags);
            /* add the element to the available list */
            list_del( &(pre->avail_list));
            spin_unlock_irqrestore ( &mppLnx_freeProxyRequestQ.queueLock, flags);
            break;

      case  MPPLNX_QUEUE_DEFERRED_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppLnx_remove_proxyRequest_from_list() MPPLNX_QUEUE_DEFERRED_LIST\n"));
            spin_lock_irqsave ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
            list_del( &(pre->pending_list));
            spin_unlock_irqrestore ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
            break;

      case  MPPLNX_QUEUE_QUEUED_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppLnx_remove_proxyRequest_from_list() MPPLNX_QUEUE_QUEUED_LIST\n"));
            spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
            list_del( &(pre->queued_list));
            spin_unlock_irqrestore ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
            break;
      case  MPPLNX_QUEUE_DISPATCHED_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppLnx_remove_proxyRequest_from_list() MPPLNX_QUEUE_DISPATCHED_LIST\n"));
            spin_lock_irqsave ( &mppLnx_DPProxyRequestQ.queueLock, flags);
            list_del( &(pre->dispatch_list));
            spin_unlock_irqrestore ( &mppLnx_DPProxyRequestQ.queueLock, flags);
            break;
      case  MPPLNX_QUEUE_S2TOS3_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppLnx_remove_proxyRequest_from_list() MPPLNX_QUEUE_S2TOS3_LIST\n"));
            spin_lock_irqsave ( &mppLnx_s2tos3ProxyRequestQ.queueLock, flags);
            list_del( &(pre->s2tos3_list));
            spin_unlock_irqrestore ( &mppLnx_s2tos3ProxyRequestQ.queueLock, flags);
            break;
      case  MPPLNX_QUEUE_PATHVALIDATION_LIST:
            MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppLnx_remove_proxyRequest_from_list() MPPLNX_QUEUE_PATHVALIDATION_LIST\n"));
            spin_lock_irqsave ( &mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
            list_del( &(pre->pathvalidation_list));
            spin_unlock_irqrestore ( &mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
            break;
      default:
         /*
         * print an error
         */
         break;
   }
     MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_remove_proxyRequest_from_list()\n"));
   return;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_getAnyPathByVDev
* SUMMARY:  Gets a virtual devive's physical device
* SCOPE:   local
*
*
* DESCRIPTION:
   This function gets a physical path of a virtual scsi device.

* @ vsdp the input argument is a virtual struct scsi_device object for whom its
      physical path to be queried
* RESTRICTIONS:
* 
* RETURNS: any available physical path of the virtual device
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
struct scsi_device * mppLnx_getAnyPathByVDev(struct scsi_device * vsdp)
{
   struct scsi_device                   *psdp = NULL;
   RdacDeviceInformation_t       *rdacInfo;
   RdacLunInformation_t          *rdacLun;
   int                           path = 0;
   int                           controller = 0;
   
   if( vsdp == NULL )
   {
       return NULL;
   }  
   rdacInfo = mpp_ModuleArray[vsdp->id].RdacInfo;
   if( rdacInfo == NULL )
   {
	MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
            "mppLnx_getAnyPathByVDev()- rdacInfo is NULL.\n" ));

	return NULL;
   }

   rdacLun  = & rdacInfo->RdacLuns[vsdp->lun];
   for(controller = 0; controller < 2; controller++)
   {
      for(path = 0; path < mppLnx_Config_Parameters.mpp_MaxPathsPerController; path++)
      {
         psdp = rdacLun->LunControllers[controller]->LunPaths[path].LunPathDevice;
         if(psdp != NULL)
         {
            break;
         }else
         {
            psdp = NULL;
         }
      }

      if(psdp != NULL)
      {
         break;
      }
   }
   return psdp;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_update_PdeviceStat
* SUMMARY:  Updates a mppLnx_PDeviceEntry_t object's statistics.
* SCOPE:   local
*
*
* DESCRIPTION:
   This function updates a mppLnx_PDeviceEntry_t object's statistics information.
   This function is called after an IO is completed via a physical device.

* @ vde The mppLnx_PDeviceEntry_t object whose statistics needs to be updated
* @ioStatus The completion status of an IO
* @roundTrip the time spent on the IO in the unit of Linux jiffies.
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_update_PdeviceStat(
        mppLnx_PDeviceEntry_t *pde,
        int ioStatus, 
        unsigned long roundTrip)
{
   unsigned long        iflag;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_update_PdeviceStat()\n"));

   if(unlikely(!pde))
   {
      return;
      MPPLNX_VHBA_BUG();
   }

   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   switch (ioStatus)
   {
      case MPPLNX_IO_SUCCESS:
         pde->ioCount++;
         if(pde->shortestTrip == 0)
         {
            pde->shortestTrip = roundTrip;
         }else if(roundTrip < pde->shortestTrip)
         {
            pde->shortestTrip = roundTrip;
         }

         if(roundTrip > pde->longestTrip)
         {
            pde->longestTrip = roundTrip;
         }
         break;
      case MPPLNX_IO_PATHFAILOVER:
         pde->numPathFailures++;
         break;
      case MPPLNX_IO_FAILED:
         pde->ioFails++;
         break;
      case MPPLNX_IO_CTRLFAILOVER:
      default:
         /* 
         *The physical device should not receive CTRLFAILOVER IO state
         */
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL,437,
            "Unsupported IO completion state:%d\n",ioStatus));
         break;
   }
   spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_update_PdeviceStat()\n"));
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_update_VdeviceStat
* SUMMARY:  update a mppLnx_VDeviceEntry_t object's statistics
* SCOPE:   local
*
*
* DESCRIPTION:
   Updates a mppLnx_VDeviceEntry_t object's statistics information.

* @vde a mppLnx_VDeviceEntry_t object whose statistics information to be updated.
* @ioStatus The status of an IO to the virtual device. Value can be MPPLNX_IO_SUCESS
      MPPLNX_IO_FAILED, MPPLNX_IO_PATHFAILOVER or MPPLNX_IO_CTRLFAILOVER
  @roundTrip the time spent on the IO in the unit of Linux jiffies.
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_update_VdeviceStat(
       mppLnx_VDeviceEntry_t * vde, 
       int ioStatus, 
       unsigned long roundTrip)
{
   unsigned long        iflag;
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_update_VdeviceStat()\n"));
   if(unlikely(!vde))
   {
      /*for the first couple of IOs, the vde object has not created.
      *the first couple of IOs are for virtual device discovery
      */
      return;
   }
   spin_lock_irqsave(mppLnxVHBAContext.vhbaContextLock,iflag);
   switch (ioStatus)
   {
      case MPPLNX_IO_SUCCESS:
         vde->ioCount++;
         if(vde->shortestTrip == 0)
         {
            vde->shortestTrip = roundTrip;
         }else if(roundTrip < vde->shortestTrip)
         {
            vde->shortestTrip = roundTrip;
         }

         if(roundTrip > vde->longestTrip)
         {
            vde->longestTrip = roundTrip;
         }
         break;
      case MPPLNX_IO_PATHFAILOVER:
         vde->numPathFailures++;
         break;
      case MPPLNX_IO_FAILED:
         vde->ioFails++;
         break;
      case MPPLNX_IO_CTRLFAILOVER:
         vde->numCntlFailures++;
         if(vde->shortestCntlFailoverTime == 0)
         {
            vde->shortestCntlFailoverTime = roundTrip;
         }
         else if(roundTrip < vde->shortestCntlFailoverTime)
         {
            vde->shortestCntlFailoverTime = roundTrip;
         }

         if(roundTrip > vde->longestCntlFailoverTime)
         {
            vde->longestCntlFailoverTime = roundTrip;
         }

         break;
      default:
         /* 
         *The physical device should not receive CTRLFAILOVER IO state
         */
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL,438,
            "Unsupported IO completion state %d\n",ioStatus));
         break;
   }
   
   spin_unlock_irqrestore(mppLnxVHBAContext.vhbaContextLock,iflag);
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_update_VdeviceStat()\n"));

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_createPhostEntry
* SUMMARY:  Creates and initializes an mppLnx_PhostEntry_t object.
* SCOPE:   local
*
*
* DESCRIPTION:
   This function creates and initializes mppLnx_PhostEntry_t object. The object
   is not linked/added to its list.

* @pdevice A physical struct scsi_device object whose mppLnx_PhostEntry_t object needs
   to be created.

* RESTRICTIONS:
* 
* RETURNS: A newly created mppLnx_PhostEntry_t object.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
mppLnx_PhostEntry_t * mppLnx_createPhostEntry(struct scsi_device * pdevice)
{
   mppLnx_PhostEntry_t     * phostEntry;
   int                     size = 0;
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_createPhostEntry()\n"));
   /*
   * create an entry 
   */
   size = sizeof(mppLnx_PhostEntry_t);
   phostEntry = MPP_INT_KMEM_ALLOC(size);
   if(phostEntry == NULL)
   {
      /* error in getting memory
      */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 431,
            "Cannot get kernel memory for mppLnx_PhostEntry_t\n"));
      return NULL;
   }
   memset((void *) phostEntry, 0, size);
   phostEntry->host = pdevice->host;
   phostEntry->hostNo = pdevice->host->host_no;
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_createPhostEntry()\n"));
   return phostEntry;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_checkPdevEntry
* SUMMARY:  Check if the mppLnx_PdeviceEntry objects need to be created.
* SCOPE:   local
*
*
* DESCRIPTION:
   After the virtual host is initialized, the physical host rescan may discover new devices. Since the new device/driver model doesn't tell the virtual HBA driver or the upper level driver that a new physical device is added to the RdacInfo, the virtual HBA driver has no way to create the proc file for the physical device after the virtual host is initialized. This function is invoked when the virtual host scsi completion function could not find the physical device's mppLnx_PdeviceEntry object.

This function will go over all physical devices for the specified storage array and creates mppLnx__PdeviceEntry if it is not created yet.
* @rdacInfo The RdacInformation_t object that represents a storage array.

* RESTRICTIONS:
* 
* RETURNS: N/A.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

void mppLnx_checkPdevEntry(RdacDeviceInformation_t *rdacInfo)
{
   mppLnx_PDeviceEntry_t         * pde = NULL;
   MPP_HANDLE                    device;
   BYTE                          Controller;
   BYTE                          Path;
   LWORD                         Lun;

   for(Controller = 0; Controller<2; Controller++)
   {
      for(Lun=0; Lun <mppLnx_Config_Parameters.mpp_MaxLunsPerArray; Lun++)
      {
         for(Path = 0; Path <mppLnx_Config_Parameters.mpp_MaxPathsPerController; Path++)
         {
            pde = (mppLnx_PDeviceEntry_t *) rdacInfo->RdacLuns[Lun].
               LunControllers[Controller]->LunPaths[Path].PlatformPrivateData;
            device = rdacInfo->RdacLuns[Lun].
               LunControllers[Controller]->LunPaths[Path].LunPathDevice;

            if(device && (!pde))
            {
               /*
               *if the physical device is there but the PlatformPrivateData is not there,
               *we need to create the proc file 
               */
               MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_checkPdevEntry- new device %s:%d:%d:%d\n", rdacInfo->ModuleName, Controller,Path,Lun));
               mppLnx_newDeviceAdded(device);
            }
         }
      }
   }
    
}
