/*******************************************************************************
* Copyright 2004-2007 LSI Logic Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppLnx26p_vhbaproc.c
SUMMARY         %description%
VERSION         %version: 2 %
UPDATE DATE     %date_modified: Wed Dec 05 15:56:00 2007 %
PROGRAMMER      %created_by:    sdachepa %


DESCRIPTION:This source file contains Linux MPP virtual HBA driver functions 
   that are related to the creation/deletion/management/information-output of 
   the device topology tree. In the Linux MPP environment, the device topology 
   tree is implemented in the Linux /proc file system. The functions in this file
   manages the creation of the leaf node of the topology tree. The Linux MPP upper
   level driver manages the directory nodes of the device topology tree.


INCLUDE FILES:

NOTES:

RESTRICTIONS:

SEE ALSO:

REFERENCE:
  349-1033500 Linux MPP Driver Feature Architecture Specification
  349-1034610 Linux MPP Virtual HBA Driver Feature Design Document
  349-1034750 Linux MPP Upper Level Driver Feature Design Document
  349-1034730 Linux MPP Utility Feature Design Document
  349-1034620 Linux MPP Installation Feature Design Document
  349-1043860 Linux MPP driver for Linux 2.6 Kernel
  349-1046490 On-The-Fly Path Validation

IMPLEMENTATION:

MODIFICATION HISTORY:

  Error numbers start from 950

*******************************************************************************/

#define __SRCmppLnx26p_vhbaproc_c

/***  INCLUDES  ***/  
/***  Linux kernel INCLUDES  ***/  
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/genhd.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/stat.h>
#include <asm/io.h>
#include <scsi/scsi_host.h>

#include <linux/blkdev.h>
/*
* Linux SCSI middle level include
*/


/***  MPP commomn API INCLUDES  ***/  
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "MPP_Sysdep.h"
#include "mppLnx_version.h"

/***  Linux MPP Virtual HBA INCLUDES  ***/  
#include "mppLnx26p_vhba.h"
#include "mppLnx26p_shared.h"


extern mppLnx_VHBAContext_t mppLnxVHBAContext;
/***  CONSTANT DEFINITIONS  ***/
#define MPPLNX_VIRTUALLUN_PREFIX       "virtualLun"
#define MPPLNX_PHYSICALLUN_PREFIX      "LUN"
#define MPPLNX_UTMLUN_PREFIX           "UTM_"

/***  MACRO DEFINITIONS  ***/


/***  TYPE DEFINITIONS  ***/


/***  LOCALS  ***/


/***  PROCEDURES  ***/
/*
* map the states of mppLnx_proxyRequestState_t's enum value to a string
*/
static char * mppLnx_proxyReqyestStateStrings[] =
{
   "NOTUSED    ",
   "DPQUEUED   ",
   "DISPATCHED ",
   "COMPLETING ",
   "DEFERRED   "
};

static char * mppLnx_pathStateStrings[] =
{
   "OPTIMAL           ",
   "FAILED				 ",
   "OPTIMAL_NEED_CHECK",
   "OPTIMAL_CHECKING  ",
   "FAILED_NEED_CHECK ",
   "FAILED_CHECKING   ",
   "SYSTEM_SPECIFIC   "
};

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_createVDeviceProcNode
* SUMMARY:  Creates a virtual Scsi_Device's proc node
* SCOPE:    public
   
*
* DESCRIPTION:
   This function creates a virtual device /proc leaf node. It is invoked at the 
   time of building the virtual HBA driver internal data model by the mppLnx_detect() 
   function. 
   This function uses Linux /proc file system APIs  "proc_mkdir()" and 
   "create_proc_entry()" to create virtual device proc leaf node. The proc entry's 
   read_proc function pointer will point to mppLnx_Vdevice_read_proc() .
* 
*  @vde the virtual device entry object for which the /proc node needs to be created
* RESTRICTIONS:
*     This function can not be invoked from an interrupt handler since proc entry creation
*     uses the memory with GFP_KERNEL flag
* 
* RETURNS: SUCCESS indicates the proc leaf node is successfully 
*           created or FAILED otherwise
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_createVDeviceProcNode(mppLnx_VDeviceEntry_t * vde)
{
   int                        rv = FAILED;
   struct proc_dir_entry      * parent = NULL;
   RdacDeviceInformation_t    *rdacInfo = NULL;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_createVDeviceProcNode()\n"));
   
   if(NULL == vde)
   {
      MPPLNX_VHBA_BUG();
   }

   rdacInfo = vde->rdacInfo;

   if(NULL == rdacInfo)
   {
      MPPLNX_VHBA_BUG();
   }

   parent = rdacInfo->ModuleHandle;

   if(NULL == parent)
   {
      MPPLNX_VHBA_BUG();
   }

   vde->procNode.parent = parent;
   sprintf(vde->procNode.nodeName,"%s%d",
      MPPLNX_VIRTUALLUN_PREFIX,
      vde->lun);
   vde->procNode.procMode =  PROC_LEAF_NODE_MODE;
  
   vde->procNode.thisNode = create_proc_read_entry(
         vde->procNode.nodeName,
         vde->procNode.procMode,
         vde->procNode.parent,
         mppLnx_VDevice_read_proc, 
         vde
         );
   
   if(vde->procNode.thisNode == NULL)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,950,
         "Not enough memory to reigster MPP virtual HBA in /proc/scsi/mpp !\n"));
      rv = FAILED;
   }else
   {
      vde->procNode.thisNode->owner = THIS_MODULE;
      rv = 0;
      vde->procNode.isCreated = 1;
      vde->procNode.needsClean = 0;
      vde->procNode.nodeType = MPPLNX_VDEVICE_NODE;
      rv = SUCCESS;
   }
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exitering mppLnx_createVDeviceProcNode() for virtual device T%dL%d\n",
      vde->rdacInfo->VirtualTargetId,
      vde->lun));
   return rv;

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_createPDeviceProcNode
* SUMMARY:  Creates physical scsi device's /proc node
* SCOPE:   local
*
*
* DESCRIPTION:
   This function creates a physical device /proc leaf node. It is invoked at the time 
   of building the virtual HBA driver internal data model by the mppLnx_detect() function. 
   This function uses Linux /proc file system APIs  "proc_mkdir()" and "create_proc_entry()"
   to create virtual device proc leaf node. The proc entry's read_proc function pointer 
   will point to mppLnx_Pdevice_read_proc()
* @pde the physical device entry object for which the /proc node needs to be created.
*
* RESTRICTIONS:RICTIONS:
*     This function can not be invoked from an interrupt handler since proc entry creation
*     uses the memory with GFP_KERNEL flag
* 
* RETURNS:SUCCESS indicates the proc leaf node is successfully 
*           created or FAILED otherwise
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_createPDeviceProcNode(mppLnx_PDeviceEntry_t * pde)
{
   RdacDeviceInformation_t       *rdacInfo =NULL;
   struct proc_dir_entry         *parent = NULL;
   int                           rv = FAILED;
   int                           ControllerIndex;
   int                           PathIndex;

   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_createPDeviceProcNode()\n"));
   rdacInfo = pde->rdacInfo;
   ControllerIndex = pde->controllerSlot;
   PathIndex = pde->pathId;
   parent = rdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[PathIndex].DirectoryVertex;
   if(NULL == parent)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,951,
         "the physical device does not have a parent array %s controller %d path %d\n",
         (char *) rdacInfo->ModuleName, ControllerIndex, PathIndex));
      MPPLNX_VHBA_BUG();
   }

   pde->procNode.parent = parent;
   
   /*
   * check if the device is UTM device. If so, its proc node name will be
   * UTM_LUNn
   */
   if(!rdacInfo->RdacLuns[pde->pdevice->lun].UTMLun)
   {
      sprintf(pde->procNode.nodeName,"%s%d",
         MPPLNX_PHYSICALLUN_PREFIX,
         pde->pdevice->lun);
   }else
   {
      sprintf(pde->procNode.nodeName,"%s%s%d",
         MPPLNX_UTMLUN_PREFIX,
         MPPLNX_PHYSICALLUN_PREFIX,
         pde->pdevice->lun);
   }

   pde->procNode.procMode =  PROC_LEAF_NODE_MODE;
   pde->procNode.thisNode = create_proc_read_entry(
         pde->procNode.nodeName,
         pde->procNode.procMode,
         pde->procNode.parent,
         mppLnx_PDevice_read_proc, 
		   (void *)pde
         );
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Made PDevice Proc Node\n"));
   
 
   if(pde->procNode.thisNode == NULL)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,952,
         "Not enough memory to reigster MPP virtual HBA in /proc/scsi/mpp !\n"));
      rv = FAILED;
   }else
   {
      rv = SUCCESS;
      pde->procNode.isCreated = 1;
      pde->procNode.needsClean = 0;
      pde->procNode.nodeType = MPPLNX_PDEVICE_NODE;
      pde->procNode.thisNode->owner = THIS_MODULE;
   }
   MPP_DEBUGPRINT((MPP_INIT_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_createPDeviceProcNode()\n"));
   return rv;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_removeVDeviceProcNode
* SUMMARY:  Removes virtual device /proc node.
* SCOPE:     local
*
*
* DESCRIPTION:
   This function removes a virtual device /proc leaf node. It is invoked at the time
   of the virtual HBA driver unloading time by the mppLnx_release() function. 
   This function uses Linux /proc file system APIs  "remove_proc_entry()" to remove
   virtual device proc leaf node. 
* @vde the virtual device entry for which its /proc node will be removed
* RESTRICTIONS:
* 
* RETURNS:SUCCESS indicates the proc leaf node is successfully 
*           removed or FAILED otherwise
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_removeVDeviceProcNode(mppLnx_VDeviceEntry_t * vde)
{
  

   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_removeVDeviceProcNode()%s:Lun %d\n",
      (char *)vde->rdacInfo->ModuleName,vde->lun));
   remove_proc_entry(vde->procNode.nodeName, vde->procNode.parent);
   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exitng mppLnx_removeVDeviceProcNode()\n"));

   
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_removePDeviceProcNode
* SUMMARY:  Removes a physical device's /proc node
* SCOPE: local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function removes a physical device /proc leaf node. It is invoked at the 
   time of the virtual HBA driver unloading time by the mppLnx_release() function. 
   This function uses Linux /proc file system APIs  "remove_proc_entry()" to remove 
   virtual device proc leaf node. 
   
*  @pde the physical device entry for which its /proc node will be removed
* RESTRICTIONS:
* 
* RETURNS:SUCCESS indicates the proc leaf node is successfully 
*           removed or FAILED otherwise
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_removePDeviceProcNode(mppLnx_PDeviceEntry_t * pde)
{
   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_removePDeviceProcNode()h%dc%dt%dl%d\n",
      pde->pdevice->host->host_no, pde->pdevice->channel, pde->pdevice->id, pde->pdevice->lun));
   remove_proc_entry(pde->procNode.nodeName, pde->procNode.parent);
   MPP_DEBUGPRINT((MPP_DETACH_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_removePDeviceProcNode()\n"));
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_VDevice_read_proc
* SUMMARY:  Wrtie a virtual device's info to the user space via /proc file system
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function is assigned to /proc file system's proc_dir_entry 
   data structure's (*read_proc) function pointer at a virtual device /proc leaf
   node creation time. The "void *data" is an object of "mppLnx_VdeviceEntry_t" 
   object. This function is invoked when an end user types 
   "cat  /proc/scsi/mpp/arrayName/virtualLun#".
   
*  @page the pointer of write-buffer. 
*  @start 
*  @offset the function should start writing at an offset of "offset"   
*  @count  at most "count" bytes that the function can write
*  @eof  write 1 to indicate End Of File
*  @data  driver specific data pointer. It points to an mppLnx_VDeviceEntry_t object
*           this function
* RESTRICTIONS:
* 
* RETURNS: This function returns the number of bytes written into the "page
*
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_VDevice_read_proc
(
      char *page, 
      char ** start, 
      off_t offset, 
      int count, 
      int *eof, 
      void *data
    )
{
   mppLnx_VDeviceEntry_t      *vde =(mppLnx_VDeviceEntry_t *) data;
   struct scsi_device         *sdev = vde->vdevice;
   int                        len = 0;
   int                        position = 0;
   int                        begin;
   char                       asciiWWN[34];
   RdacDeviceInformation_t    * rdacInfo = vde->rdacInfo;
   RdacLunInformation_t       * lunInfo = &(rdacInfo->RdacLuns[vde->lun]);
   static char                vDeviceReadBuffer[MPPLNX_PROC_READ_BUFFSIZE];
   struct scsi_cmnd                  *SCpnt;
   int                        i=0;
   mppLnx_ProxyRequestEntry_t *preq;
   unsigned long              flags;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_VDevice_read_proc()\n"));
   memset(vDeviceReadBuffer,0, MPPLNX_PROC_READ_BUFFSIZE);

   len = sprintf(&vDeviceReadBuffer[position],"Linux MPP driver. Version:%s Build:%s\n",
         MPPLNX_VERSION,MPPLNX_BUILDTIMESTAMP);

   position = position +len;
   mpp_ConvertWWNtoAscii(asciiWWN,&lunInfo->WWN[0]);
   len = sprintf(&vDeviceReadBuffer[position],"Lun WWN:%s\n",asciiWWN);

   if(vde->vdevice == NULL)
   {
      position = position +len;
      len = sprintf(&vDeviceReadBuffer[position],
         "The virtual device for array %s Lun %d is not available\n",
         (char *)rdacInfo->ModuleName, vde->lun);
   }
   else
   {
      position = position +len;
      len = sprintf(&vDeviceReadBuffer[position],
         "Virtual Scsi Address: host_no:%d channel:%d target:%d Lun:%d\n",
         vde->vdevice->host->host_no,
         vde->vdevice->channel,
         vde->vdevice->id,
         vde->vdevice->lun);
   }
  
   /* print the queue depth of the device */
   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "Queue Depth = %d\n", vde->vdevice->queue_depth);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "I/O Statistics:\n");

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tNumber of IOs:%d\n",vde->ioCount);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tLongest trip of all I/Os:%d\n",vde->longestTrip);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tShortest trip of all I/Os:%d\n",vde->shortestTrip);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tNumber of occurrences of path failover events:%d\n",vde->numPathFailures);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tNumber of occurrences of controller failover events:%d\n",
      vde->numCntlFailures);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tThe longest Controller Failover Time:%ld\n",
      vde->longestCntlFailoverTime);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "\tThe shortest Controller Failover Time:%ld\n",
      vde->shortestCntlFailoverTime);

   /*
   * print out how many virtual commands are queued up
   */
   /*
   * check the dispatch queue  
   */
   i = 0;
   spin_lock_irqsave ( &mppLnx_DPProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_DPProxyRequestQ.list, dispatch_list)
   {
      if(preq->vcmnd->device == sdev)
      {
         if(position >count)
         {
            spin_unlock_irqrestore ( &mppLnx_DPProxyRequestQ.queueLock, flags);
            goto output_too_long;
         }

         position = position +len;
         SCpnt = preq->vcmnd;
   	   len = sprintf(&vDeviceReadBuffer[position],
            "scsi_cmnd[%d]::cmd=0x%x::sn=%ld::addr=0x%p::st=%s::in=%s\n",
            i,
            SCpnt->cmnd[0],
            SCpnt->serial_number,
            SCpnt,
            mppLnx_proxyReqyestStateStrings[preq->state],
            "DP");
         i++;
      }
   }
   spin_unlock_irqrestore ( &mppLnx_DPProxyRequestQ.queueLock, flags);
   /*
   * check the queued list
   */

   spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_queuedProxyRequestQ.list, queued_list)
   {
      if(preq->vcmnd->device == sdev)
      {
         if(position >count)
         {
            spin_unlock_irqrestore ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
            goto output_too_long;
         }
         position = position +len;
         SCpnt = preq->vcmnd;
   	   len = sprintf(&vDeviceReadBuffer[position], 
		      "scsi_cmnd[%d]::cmd=0x%x::sn=%ld::addr=0x%p::st=%s::in=%s\n",
		      i,
            SCpnt->cmnd[0],
            SCpnt->serial_number,
            SCpnt,
            mppLnx_proxyReqyestStateStrings[preq->state],
            "QD");
         i++;
      }
   }
   spin_unlock_irqrestore ( &mppLnx_queuedProxyRequestQ.queueLock, flags);

   /*
   * check the deffered list
   */

   spin_lock_irqsave ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_deferredProxyRequestQ.list, pending_list)
   {
      if(preq->vcmnd->device == sdev)
      {
          if(position >count)
         {
            spin_unlock_irqrestore ( &mppLnx_deferredProxyRequestQ.queueLock, flags);
            goto output_too_long;
         }

         position = position +len;
         SCpnt = preq->vcmnd;
   	   len = sprintf(&vDeviceReadBuffer[position], 
		      "scsi_cmnd[%d]::cmd=0x%x::sn=%ld::addr=0x%p::st=%s::in=%s\n",
		      i,
            SCpnt->cmnd[0],
            SCpnt->serial_number,
            SCpnt,
            mppLnx_proxyReqyestStateStrings[preq->state],
            "DF");
         i++;
      }
   }
   spin_unlock_irqrestore ( &mppLnx_deferredProxyRequestQ.queueLock, flags);

   position = position +len;
   len = sprintf(&vDeviceReadBuffer[position],
      "total size:%d\n",position+20);
   position = position +len;

output_too_long:
   /*
   * let's copy the buffer to the page
   */
   if(position < offset)
   {
      len = 0;
      begin = position;
   }
   else
   {
      begin = offset;
      len = position - offset;
      if(len > count)
      {
         len = count;
      }
   }

   if(len < count)
   {
      (*eof) = 1;
   }else
   {
      (*eof) = 0;
   }
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
   	"position %d  begin %d len %d offset %d count %d eof %d\n", position, begin,len,offset,count,*eof));  

   if(len > 0)
   {
      memcpy(page, &vDeviceReadBuffer[begin],len);
   }

   *start = page;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_VDevice_read_proc()\n"));
   return len;


}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_PDevice_read_proc
* SUMMARY:  Wrtie a physical device's info to the user space via /proc file system
* SCOPE:    public 
*
* DESCRIPTION:
   This function is assigned to /proc file system's proc_dir_entry 
   data structure's (*read_proc) function pointer at a physical device /proc 
   leaf node creation time. The "void *data" is an object of "mppLnx_PdeviceEntry_t" 
   object. This function is invoked when an end user types 
   "cat  /proc/scsi/mpp/arrayName/Controller#/path#/Lun#".

   
* @page the pointer of write-buffer. 
*  @start
*  @offset    
*  @count
*  @eof
*  @data  driver specific data pointer. It points to an mppLnx_PDeviceEntry_t object
*           this function
* RESTRICTIONS:
* 
* RETURNS:This function returns the number of bytes written into the "page".
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_PDevice_read_proc
(
      char *page, 
      char ** start, 
      off_t offset, 
      int count, 
      int *eof, 
      void *data
      )
{
   mppLnx_PDeviceEntry_t         *pde =(mppLnx_PDeviceEntry_t *) data;
   int                           len = 0;
   int                           position = 0;
   int                           begin;
   char                          asciiWWN[34];
   RdacDeviceInformation_t       * rdacInfo = pde->rdacInfo;
   RdacLunInformation_t          * lunInfo = &(rdacInfo->RdacLuns[pde->pdevice->lun]);
   LunPathInfo_t                 * lunPathInfo = &rdacInfo->RdacLuns[pde->pdevice->lun].LunControllers[pde->controllerSlot]->LunPaths[pde->pathId];
   RdacControllerPath_t          * controllerPath = &rdacInfo->ControllerInfo[pde->controllerSlot]->ControllerPath[pde->pathId];
   mppLnx_ProxyRequestEntry_t    * preq;
   static char                   pDeviceReadBuffer[MPPLNX_PROC_READ_BUFFSIZE];
   unsigned long                 flags;
   struct scsi_cmnd			 	         *vSCpnt;
   int				 			      i=0;
   struct scsi_device            *sdev = pde->pdevice;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Entering mppLnx_PDevice_read_proc()\n"));
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Input arguments offset %d count %d \n",
       offset, count));
   memset(pDeviceReadBuffer,0, MPPLNX_PROC_READ_BUFFSIZE);

   len = sprintf(&pDeviceReadBuffer[position],"Linux MPP driver. Version:%s Build:%s\n",
         MPPLNX_VERSION,MPPLNX_BUILDTIMESTAMP);

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "wrote version info to the buffer\n"));
   position = position +len;
   if(NULL == lunInfo)
   {
      /* programming error 
      */
      MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
         "Cannot get WWN from array %s lun %d\n",
         (char *) rdacInfo->ModuleName, pde->pdevice->lun));
      MPPLNX_VHBA_BUG();
      return 0;
   }
   mpp_ConvertWWNtoAscii(asciiWWN,&lunInfo->WWN[0]);
   len = sprintf(&pDeviceReadBuffer[position],"Lun WWN:%s\n",asciiWWN);
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "put wwn to the buufer\n"));
   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "Physical HBA driver: %s\n",
      pde->pdevice->host->hostt->module->name);
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "put host proc_name to the buffer\n"));

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "Device Scsi Address: host_no:%d channel:%d target:%d Lun:%d\n",
      pde->pdevice->host->host_no,
      pde->pdevice->channel,
      pde->pdevice->id,
      pde->pdevice->lun);
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "put device ID to the buufer\n"));

	/* print the queue depth of the device */
   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "Queue Depth = %d\n", pde->pdevice->queue_depth);

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "I/O Statistics:\n");

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "\tNumber of IOs:%d\n",pde->ioCount);

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "\tLongest trip of all I/Os:%d\n",pde->longestTrip);

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "\tShortest trip of all I/Os:%d\n",pde->shortestTrip);

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "\tNumber of occurences of IO failed events:%d\n",pde->ioFails);

   
   /*
   * Report device, path, controller status
   */
   for(i=0; i<MPPCMN_MAX_STATE_HISTORY; i++)
   {
      position = position +len;
      len = sprintf(&pDeviceReadBuffer[position],
         "Device state: [%d] %s\n",(lunPathInfo->DeviceStateIndex + i)%MPPCMN_MAX_STATE_HISTORY,
         mppLnx_pathStateStrings[lunPathInfo->DeviceState[(lunPathInfo->DeviceStateIndex + i)%MPPCMN_MAX_STATE_HISTORY]]);
   }

   for(i=0; i<MPPCMN_MAX_STATE_HISTORY; i++)
   {
      position = position +len;
      len = sprintf(&pDeviceReadBuffer[position],
         "Path state:[%d] %s\n",(controllerPath->PathStateIndex+i)%MPPCMN_MAX_STATE_HISTORY,
         mppLnx_pathStateStrings[controllerPath->PathState[(controllerPath->PathStateIndex+i)%MPPCMN_MAX_STATE_HISTORY]]);
   }

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "Controller Failed? %d\n",
      rdacInfo->ControllerInfo[pde->controllerSlot]->Failed);


   /*
   * check the queued list
   */
   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "Outstanding IOs on this device:\n");
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "put I/O statics to the buffer\n"));

   /*
   * check the dispatch queue. The command in the physical device's queue must be the
   * proxy requests that are in the queued queue.
   */
   i = 0;
   spin_lock_irqsave ( &mppLnx_queuedProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_queuedProxyRequestQ.list, queued_list)
   {
      if(preq->pdev == sdev) 
      {
         if(position > count)
         {
            spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);
            goto output_too_long;
         }

         position = position +len;
         vSCpnt = preq->vcmnd;
   	     len = sprintf(&pDeviceReadBuffer[position], 
		      "scsi_cmnd[%d]::cmd=0x%x::vsn=%ld:st=%s::in=%s\n",
		      i,
            vSCpnt->cmnd[0],
            vSCpnt->serial_number,
            mppLnx_proxyReqyestStateStrings[preq->state],
            "QD");
         i++;
      }
   }
   spin_unlock_irqrestore (&mppLnx_queuedProxyRequestQ.queueLock, flags);

   /*
   * check the path validation command queue
   */
   i = 0;
   spin_lock_irqsave (&mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
   list_for_each_entry(preq, &mppLnx_pathvalidationProxyRequestQ.list, pathvalidation_list)
   {
      if(preq->pdev == sdev)
      {
         if(position > count)
         {
            spin_unlock_irqrestore (&mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
            goto output_too_long;
         }
         position = position +len;
         vSCpnt = preq->vcmnd;
   	   len = sprintf(&pDeviceReadBuffer[position], 
		      "scsi_cmnd[%d]::cmd=0x%x::vsn=%ld::st=%s::in=%s\n",
		      i,
            vSCpnt->cmnd[0],
            vSCpnt->serial_number,
            mppLnx_proxyReqyestStateStrings[preq->state],
            "PV");
         i++;
      }
   }
   spin_unlock_irqrestore (&mppLnx_pathvalidationProxyRequestQ.queueLock, flags);
   

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "put outstanding I/O to the buffer\n"));
 

   position = position +len;
   len = sprintf(&pDeviceReadBuffer[position],
      "total size:%d\n",position+20);

   position = position +len;

output_too_long:
   /*
   * let's copy the buffer to the page
   */
   if(position < offset)
   {
      len = 0;
      begin = position;
   }
   else
   {
      begin = offset;
      len = position - offset;
      if(len > count)
      {
         len = count;
      }
   }

   if(len < count)
   {
      (*eof) = 1;
   }else
   {
      (*eof) = 0;
   }
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "put outstanding I/O to the buufer\n"));
   
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
   	"position %d  begin %d len %d offset %d count %d eof %d\n", position, begin,len,offset,count,*eof));  

   if(len >0 )
   {
      memcpy(page, &pDeviceReadBuffer[begin], len);
   }

   *start = page;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_PDevice_read_proc()\n"));
   return len;
    
}
