/*******************************************************************************
* Copyright 2004-2008 LSI Corporation
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppCmn_s2tos3.c
SUMMARY         %description%
VERSION         %version: 47 %
UPDATE DATE     %date_modified: Wed Jul 02 11:03:53 2008 %
PROGRAMMER      %created_by:    harshi %



DESCRIPTION: 
   This file contains common source code for performing Scsi2  reserve/release 
   to Scsi3 persistent reservation translation. The translation operations are
   performed asynchronously. A platform mpp driver implementation should call 
   APIs in this file via a separate task so that the translation operation 
   will not block system dispatching I/Os.

INCLUDE FILES:

NOTES:

  For Error Log purposes this module has been assigned numbers 350-499.  the last number 
  currently used is 403.

RESTRICTIONS:

SEE ALSO:

REFERENCE:
    

  349-1031000 Yuma Software Functional Specification
  349-1033500 Linux MPP Driver Feature Architecture Specification
  349-1034610 Linux MPP Virtual HBA Driver Feature Design Document
  349-1034750 Linux MPP Upper Level Driver Feature Design Document
  349-1034730 Linux MPP Utility Feature Design Document
  349-1034620 Linux MPP Installation Feature Design Document

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/

#define __SRCmppCmn_s2tos3_c

/***  INCLUDES  ***/
/*TTTTTTTTTTTT*/ 
/* don't remove the lines with "TTTTTTTTTTTTT". It is used by the test makefile
* for extracting key-generation code
*/
#include "MPP_Sysdep.h"
/*TTTTTTTTTTTT*/
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "copyright.h"
#include "scsiio.h"
#include "mppCmn_s2tos3.h"
#include "mppCmn_SysInterface.h"


/***  CONSTANT DEFINITIONS  ***/

#define	 SCSI_PR_IN    		0x5E
#define   SCSI_PR_OUT         0x5F

/*
* PROUT service action codes
*/
#define  SCSI_PROUT_REGISTER                 0x00
#define  SCSI_PROUT_RESERVE                  0x01
#define  SCSI_PROUT_RELEASE                  0x02
#define  SCSI_PROUT_CLEAR                    0x03
#define  SCSI_PROUT_PREEMPT                  0x04
#define  SCSI_PROUT_PREEMPT_AND_ABORT        0x05
#define  SCSI_PROUT_REGISTER_AND_IGNORE_EXISTING_KEY  0x06
/*
* PRIN service action codes
*/
#define  SCSI_PRIN_READ_KEYS                    0x00
#define  SCSI_PRIN_READ_RESERVATION             0x01
#define  SCSI_PRIN_REPORT_CAPABILITIES          0x02

/*
* Persistent reservation scope codes
*/
#define  SCSI_PR_LU_SCOPE                       0x00
#define  SCSI_PR_ELEMENT_SCOPE                  0x02
/*
* Persistent reservation type codes
*/
#define  SCSI_PR_WRITE_EXCL                     0x01
#define  SCSI_PR_EXCL_ACCESS                    0x02
#define  SCSI_PR_WRITE_EXCL_RO                  0x05
#define  SCSI_PR_EXCL_ACCESS_RO                 0x06
#define  SCSI_PR_WRITE_EXCL_AR                  0x07
#define  SCSI_PR_EXCL_ACCESS_AR                 0x08  


/***  MACRO DEFINITIONS  ***/

#ifdef MPP_DEBUG
static VOID mppCmn_dumpPRINReadKeysParamData(PRINReadKeysParamData_t *rkParamData);
static VOID mppCmn_dumpPRINReadResvParamData(PRINReadResevParamData_t *rsParamData);
static VOID mppCmn_dumpSenseData(SenseData_t *senseData);

#define MPPCMN_DUMP_READKEY(x) mppCmn_dumpPRINReadKeysParamData(x)
#define MPPCMN_DUMP_RESVKEY(x) mppCmn_dumpPRINReadResvParamData(x)
#define MPPCMN_DUMP_SENSEDATA(x) mppCmn_dumpSenseData(x)

#else

#define  MPPCMN_DUMP_READKEY(x)
#define  MPPCMN_DUMP_RESVKEY(x)
#define  MPPCMN_DUMP_SENSEDATA(x)

#endif
/*TTTTTTTTTTTT*/
/*
--------------------------------------------------------------------
mix -- mix 3 32-bit values reversibly.
For every delta with one or two bits set, and the deltas of all three
  high bits or all three low bits, whether the original value of a,b,c
  is almost all zero or is uniformly distributed,
* If mix() is run forward or backward, at least 32 bits in a,b,c
  have at least 1/4 probability of changing.
* If mix() is run forward, every bit of c will change between 1/3 and
  2/3 of the time.  (Well, 22/100 and 78/100 for some 2-bit deltas.)
mix() was built out of 36 single-cycle latency instructions in a 
  structure that could supported 2x parallelism, like so:
      a -= b; 
      a -= c; x = (c>>13);
      b -= c; a ^= x;
      b -= a; x = (a<<8);
      c -= a; b ^= x;
      c -= b; x = (b>>13);
      ...
  Unfortunately, superscalar Pentiums and Sparcs can't take advantage 
  of that parallelism.  They've also turned some of those single-cycle
  latency instructions into multi-cycle latency instructions.  Still,
  this is the fastest good hash I could find.  There were about 2^^68
  to choose from.  I only looked at a billion or so.
--------------------------------------------------------------------
*/
#define mix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

/***  TYPE DEFINITIONS  ***/


/***  LOCALS  ***/
static         BYTE    MPP_SIGNATURE[]={0x31,0x41,0x56,0x92,}; 
/*TTTTTTTTTTTT*/

/***  PROCEDURES  ***/



/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MapScsi2Reserve
* SUMMARY:  Translate a Scsi2 reserve command to Scsi3 persistent reservation commands
* SCOPE:    public
*
*
* DESCRIPTION:
   This function performs the mapping of Scsi-2 reserve command to Scsi-3 
   persistent reservation. The "reservationKey" is a 8 byte long 
   "generated reservation key".

  @rdacInfo The object which represents a storage array
  @Lun   The logical unit where a Scsi2 reserve command is addressed to
  @uniqueId The 8-byte long unique ID of this host.
  @reservationKey The 8-bytes generated reservation key. The reservationKey are generated
      by mppCmn_GenReservationKey() API
  @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
      
*
* RESTRICTIONS:
* 
* RETURNS:The return value can be:
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT    
MPPCMN_PRTN_RETURN_ERROR
MPPCMN_PRTN_CHECK_CONDITION            

*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MapScsi2Reserve(RdacDeviceInformation_t *rdacInfo, 
                             LWORD Lun, 
                             BYTE *uniqueId,
                             BYTE *reservationKey,
                             BOOL enableCC,
                             SenseData_t *sensedata)
{
   LWORD                               mpp_status;
   PRINReadKeysParamData_t             *keyParamData;
   PRINReadResevParamData_t            * resParamData;
   LWORD                               additional_len;
   BYTE                                isReservationHolder = 1;
   LWORD                               i;
   
   MPP_RW_LOCK_SHARED(rdacInfo->RWLock);

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering mppCmn_MapScsi2Reserve Lun %d\n", Lun));
   
   /*
   * read the registration key
   *    we may get reservation conflict if the LUN is already reserved by SCSI-2 
   *    reserve. The firmware doesn't handle the co-existence of SCSI-2/SCSI-3
   *    reservation protocol.
   */
   keyParamData = (PRINReadKeysParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadKeysParamData_t));
   if(NULL == keyParamData)
   {
      /* error */
      MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 350, 
         "Error in allocating PRINReadKeysParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(keyParamData,sizeof(PRINReadKeysParamData_t));
   }

   mpp_status = mppCmn_GetPRKeys(rdacInfo, Lun, keyParamData, enableCC,sensedata);
   
   /*
   * we don't want go any further if we get reservation conflict or check-condition
   */

   if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT || 
      mpp_status == MPPCMN_PRTN_RETURN_ERROR ||
      mpp_status == MPPCMN_PRTN_CHECK_CONDITION )
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 351, 
         "S2 to S3: Failed to read PR keys Lun %d - status %d\n",
          Lun,mpp_status));
      MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
      MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
      return mpp_status;
   }
   
   additional_len = MPP_GET_LWORD_FROM_4BYTES(keyParamData->Additional_Len);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "mppCmn_MapScsi2Reserve() keyParam addtional_len %d\n",additional_len));

   MPPCMN_DUMP_READKEY(keyParamData);

   MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));

   if(0 == additional_len)
   {
      /*
      * empty registration key list
      * perform the registration 
      */
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppCmn_MapScsi2Reserve- empty registration key\n"));
      /*
      * The enableCC is hard coded to FALSE. We would like to perform retries during
      * registration pahse.
      */
      mpp_status = mppCmn_MkPRRegistrationForAll(rdacInfo, Lun,reservationKey,FALSE,sensedata);    

   }
   else
   {
      /*
      * read reservation 
      */
      resParamData = ( PRINReadResevParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadResevParamData_t));
      if(NULL == resParamData)
      {
         /* error */
      	 MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 352, 
            "mppCmn_MapScsi2Reserve- cannot allocate PRINReadResevParamData_t\n"));
         return MPPCMN_PRTN_RETURN_ERROR;
      }
      else
      {
         bzero(resParamData,sizeof(PRINReadResevParamData_t));
      }
      mpp_status = mppCmn_GetPRReservations(rdacInfo, Lun, resParamData, enableCC,sensedata);
      /*
      * We should not failed here because we just got the reservation keys.
      * just check
      */
      if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT ||
         mpp_status == MPPCMN_PRTN_RETURN_ERROR ||
         mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
      {
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 353, 
            "Exiting mppCmn_MapScsi2Reserve with error failed to read reservation Lun:%d status:0x%x\n",
                  Lun,mpp_status));
      	 MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
         MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
         return mpp_status;
      }

      additional_len = MPP_GET_LWORD_FROM_4BYTES(resParamData->Additional_Len);
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppCmn_MapScsi2Reserve() resParamData addtional_len %d\n",additional_len));
      
      MPPCMN_DUMP_RESVKEY(resParamData);

      if(0 == additional_len)
      {
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
            "mppCmn_MapScsi2Reserve() mppCmn_MapScsi2Reserve empty reservation list\n"));
         mppCmn_MkPRRegistrationForAll(rdacInfo, Lun,reservationKey,FALSE,sensedata);
         MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
      }
      else
      {
         /*
         * check if we are the persistent reservation holder
         * because the controller firmware support only LU_SCOPE, the list
         * should have only one reservaton descriptor.
         */
         for(i = 0; i < (additional_len)/(sizeof(ReservationDescriptor_t)); i++)
         {
            isReservationHolder = mppCmn_IsKeyForThisHost(
               uniqueId,
               resParamData->Reservation_Descriptor[i].Reservation_Key);

            if(!isReservationHolder)
            {
      	       MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
               MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
               MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 354, 
                   "Exiting mppCmn_MapScsi2Reserve with error-Lun is reserved by others\n"));
               return (MPPCMN_PRTN_RESERVATION_CONFLICT);
            }
         }
         /*
         * we are the reservation holder
         * return good status
         * 06/15/04 - Set the reservation flag for this lun.
         */
		 if(rdacInfo->RdacLuns[Lun].NeedsReservationCheck == 0)
		 {
	         	rdacInfo->RdacLuns[Lun].NeedsReservationCheck = 1;
		 }

      	 MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
         MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
            "Exiting mppCmn_MapScsi2Reserve -- already reserved\n"));
         return MPPCMN_PRTN_GOOD_STATUS;
      }
   }

   /*
   * Make reservation. If we get reservation conflict. We must roll our registrations
   * back.
   * the make-reservation is a critical step. We should set enableCC to FALSE
   */

   mpp_status = mppCmn_MkPRReservation(rdacInfo, Lun,reservationKey,FALSE,sensedata);
   
   if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT )
   {
      /*
      * we get here because another node wins the reserve race-condition
      * the MkPRReservation function has performed the roll back
      */
      /* 06/15/04 - Clear the reservation flag for this lun.  */
      rdacInfo->RdacLuns[Lun].NeedsReservationCheck = 0;
      MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 355, 
         "mppCmn_MapScsi2Reserve -- loss racing. roll back out registration\n"));
   }
   else if(mpp_status == MPPCMN_PRTN_GOOD_STATUS)
   {
      /* 06/15/04 - Set the reservation flag for this lun. */
       rdacInfo->RdacLuns[Lun].NeedsReservationCheck = 1;
   }
   
   MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppCmn_MapScsi2Reserve mpp_status 0x%x\n", mpp_status));
   return mpp_status;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MapScsi2Release
* SUMMARY:  Translate a Scsi2 release command to Scsi3 persistent reservation commands
* SCOPE:    public
*
*
* DESCRIPTION:
   This function performs the mapping of Scsi-2 release command to
   Scsi-3 persistent reservation. The "reservationKey" is a 8 byte long 
   "generated reservation key".

   @rdacInfo The object which represents a storage array
   @Lun   The logical unit where a Scsi2 release command is addressed to
   @uniqueId The 8-byte unique Id of this host.
   @reservationKey The 8-bytes generated reservation key. The reservationKey are generated
      by mppCmn_GenReservationKey() API
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
   @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
*
* RESTRICTIONS:
* 
* RETURNS:The return value can be:
MPPCMN_PRTN_GOOD_STATUS
MPPCMN_PRTN_RETURN_ERROR
MPPCMN_PRTN_CHECK_CONDITION
* ERRNO:
*
* NOTES:
* Based on SCSI-2 specification ( Clause 9.2.11, page 187.Information Technology 
* Small Computer System Interface -2 X3T9.2 Project 375D Revision 10L 09/07.03),
* it is not an error for an initator to attempt to release a reservation that is
* not currently valid, or is held by another initiator. In this case, the target
* should return GOOD status without altering any other reservation.

* This function will never return scsi reservation conflict status
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MapScsi2Release(RdacDeviceInformation_t *rdacInfo, 
                             LWORD Lun, 
                             BYTE * uniqueId, 
                             BYTE *reservationKey,
                             BOOL enableCC,
                             SenseData_t * sensedata)
{
   LWORD                            mpp_status;
   PRINReadKeysParamData_t          *keyParamData;
   PRINReadResevParamData_t         *resParamData;
   LWORD                            additional_len;
   LWORD                            i;
   LINT                             isReservationHolder;

   MPP_RW_LOCK_SHARED(rdacInfo->RWLock);

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering mppCmn_MapScsi2Release Lun %d\n",Lun));
   /*
   * read the registration key
   *    we may get reservation conflict if the LUN is already reserved by SCSI-2 
   *    reserve. The firmware doesn't handle the co-existence of SCSI-2/SCSI-3
   *    reservation protocol. We will return good status in the case that the controller
   *    rejects the request with reservation conflict status.
   */
   keyParamData = (PRINReadKeysParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadKeysParamData_t));
   
   if(NULL == keyParamData)
   {
      /* error */
      MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 356,
         "mppCmn_MapScsi2Release- Error cannot allocate PRINReadKeysParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(keyParamData,sizeof(PRINReadKeysParamData_t));
   }

   mpp_status = mppCmn_GetPRKeys(rdacInfo, Lun, keyParamData,enableCC,sensedata);

   if(mpp_status == MPPCMN_PRTN_RETURN_ERROR ||
      mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 357, 
         "Scsi-2 to Scsi-3: Failed to read registration key Lun:%d status:%d\n",
         Lun, mpp_status)); 
      MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
      MPP_FREE(keyParamData, sizeof(PRINReadKeysParamData_t));
      return mpp_status;
   }
   else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
   {
      /*
      * the LUN is already reserved by SCSI-2
      * return good status. SCSI-2 release allways return good status
      */
      MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
      MPP_FREE(keyParamData, sizeof(PRINReadKeysParamData_t));
      MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 358, 
          "Exiting mppCmn_MapScsi2Release with Error MPP_SCSI_RESERVATION\n"));
      return MPPCMN_PRTN_GOOD_STATUS;
   }

   additional_len = MPP_GET_LWORD_FROM_4BYTES(keyParamData->Additional_Len);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "keyParamData addtional_len %d\n",additional_len));
   MPPCMN_DUMP_READKEY(keyParamData);

   if(0 == additional_len)
   {
      /*
      * there is no registration keys for this LUN. Nothing to release
      * return good status
      */
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "Exiting mppCmn_MapScsi2Release - no registration keys\n"));
      MPP_FREE(keyParamData, sizeof(PRINReadKeysParamData_t));
      
      /* 06/15/04 - Clear the reservation setting */
      if(rdacInfo->RdacLuns[Lun].NeedsReservationCheck == 1)
      {
          rdacInfo->RdacLuns[Lun].NeedsReservationCheck = 0;
      }
      MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
      return MPPCMN_PRTN_GOOD_STATUS;
   }
   else
   {
      MPP_FREE(keyParamData, sizeof(PRINReadKeysParamData_t));
      /*
      * read reservation key
      */
      resParamData = ( PRINReadResevParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadResevParamData_t));
      if(NULL == resParamData)
      {
         /* error */
         MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 359,
            "mppCmn_MapScsi2Release- cannot allocate PRINReadResevParamData_t\n"));
         return MPPCMN_PRTN_RETURN_ERROR;
      }
      else
      {
         bzero(resParamData,sizeof(PRINReadResevParamData_t));
      }

      mpp_status = mppCmn_GetPRReservations(rdacInfo, Lun, resParamData,enableCC,sensedata);
      /*
      * We should not failed here because we just got the reservation keys.
      * just check
      */
      if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT ||
         mpp_status == MPPCMN_PRTN_RETURN_ERROR ||
         mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
      {
         MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 360, 
            "Scsi-2 to Scsi-3: failed to read reservation Lun:%d status:%d\n",
            Lun, mpp_status)); 
         MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
         return mpp_status;
      }

      additional_len = MPP_GET_LWORD_FROM_4BYTES(resParamData->Additional_Len);
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "resParamData addtional_len %d\n",additional_len));
      
      MPPCMN_DUMP_RESVKEY(resParamData);

      if(0 != additional_len)
      {
         /*
         * check if we are the persistent reservation holder
         */
         for(i = 0; i < (additional_len)/(sizeof(ReservationDescriptor_t)); i++)
         {
            isReservationHolder = mppCmn_IsKeyForThisHost(
               uniqueId,
               resParamData->Reservation_Descriptor[i].Reservation_Key);

            if(!isReservationHolder)
            {
               MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
               MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
               MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 361, 
                  "Exiting mppCmn_MapScsi2Release with error-Lun is reserved by others Lun:%d\n",
                  Lun));
               return (MPPCMN_PRTN_GOOD_STATUS);
            }
         }
         /*
         * we are the reservation holder
         */
         MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
      }
      else
      {
         MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
      }
   }

   /*
   * Clear all registrations
   * We get here either this node is the reservation holder or there is no reservation
   * holder but some registration keys are there. Anyway we need to clear all registration
   * keys and reservation
   */
   mpp_status = mppCmn_MkPRClear(rdacInfo, Lun, reservationKey,enableCC, sensedata);
   
   if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
   {
      /*
      * the return status can only be either good ot failed
      */
      mpp_status = MPPCMN_PRTN_GOOD_STATUS;
   }

   /* 06/15/04 - Clear the reservation setting */
   rdacInfo->RdacLuns[Lun].NeedsReservationCheck = 0;

   MPP_RW_UNLOCK_SHARED(rdacInfo->RWLock);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppCmn_MapScsi2Release - mpp_status 0x%x\n",mpp_status));
   return mpp_status;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MapBusDeviceReset
* SUMMARY:  Translate a scsi bus device reset message to Scsi3 persistent reservation
            command for clearing persistent reservation made by the translation layer.
* SCOPE:    public
*
*
* DESCRIPTION:
   This function performs the mapping of "bus device reset" for cleaning up 
   persistent reservations/registrations that are made by mpp driver's 
   translation layer. The "reservationKey" is a 8 byte long "generated reservation key".


   @rdacInfo The object which represents a storage array
   @Lun   The logical unit where a bus device reset message is addressed to
*
* RESTRICTIONS:
* 
* RETURNS:
         MPPCMN_PRTN_GOOD_STATUS or MPPCMN_PRTN_RETURN_ERROR 
* ERRNO:
*
* NOTES:
   This function will clear reservations for all logical units mapped in the host partition.

   If the request message is a "Logical Unit Reset"(see SAM2) or "logical device reset", 
   this function should not be invoked. The correct logical unit reset function is 
   "mppCmn_MapDeviceReset()".
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MapBusDeviceReset(
      RdacDeviceInformation_t *rdacInfo,
      LWORD Lun)
{
   LWORD                         virtualLun;
   RdacLunInformation_t          *rdacLun;
   PRINReadKeysParamData_t       *keyParamData;
   LWORD                         mpp_status;
   LWORD                         additional_len;
   LWORD                         i;
   SenseData_t                   *sensedata;

   MPP_RW_LOCK_EXCLUSIVE(rdacInfo->RWLock);

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_MapBusDeviceReset Array %s Lun %d\n",
      (BYTE *)rdacInfo->ModuleName,
      Lun));
   /*
   *Check whether or not the array is a pre-Sonoran 4 array. If it is, return
   */
   if(!rdacInfo->FWSonoran4OrLater)
   {
      MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "mppCmn_MapBusDeviceReset Array %s does not support Persistent reservation\n",
         (BYTE *)rdacInfo->ModuleName));
      return MPPCMN_PRTN_GOOD_STATUS;
   }


   /*
   * loop through all virtual device in the storage array and clear 
   * persistent reservations made by a MPP driver's translation layer
   * (not only by this host)
   */

   keyParamData = (PRINReadKeysParamData_t *) MPP_INT_KMEM_ALLOC(sizeof(PRINReadKeysParamData_t));
   if(NULL == keyParamData)
   {
      /* error */
      MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 362,
         "mppCmn_MapBusDeviceReset - Error cannot allocate PRINReadKeysParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(keyParamData,sizeof(PRINReadKeysParamData_t));
   }

   sensedata = (SenseData_t *) MPP_INT_KMEM_ALLOC(sizeof(SenseData_t));
   if(NULL == sensedata)
   {
      /* error */
      MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
      MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 403,
         "mppCmn_MapBusDeviceReset - Error cannot allocate SenseData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(sensedata,sizeof(SenseData_t));
   }

   for(virtualLun = 0; virtualLun < mpp_MaxLunsPerArray; virtualLun++)
   {
      rdacLun = &rdacInfo->RdacLuns[virtualLun];
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "Lun %d Present %d PseudoLunObject 0x%p UTMLun %d\n",
            virtualLun,
            rdacLun->Present,
            rdacLun->PseudoLunObject,
            rdacLun->UTMLun));

      // 09/09/05 CR93190.  Removed check for NULL PseudoLunObject.  With Microsoft MPIO we don't have a
      // virtual device object to maintain.
      if(rdacLun->Present && !rdacLun->UTMLun)
      {
         /*
         * read registration keys and decide whether or not we need to 
         * clear the reservation
         */
         bzero(keyParamData, sizeof(PRINReadKeysParamData_t));
         bzero(sensedata,sizeof(SenseData_t));
         /*
         *enableCC is hard-coded to FALSE. During BDR time, we don't want to return
         *a check-condition back.
         */
         mpp_status = mppCmn_GetPRKeys(rdacInfo, virtualLun,keyParamData,FALSE,sensedata);

         if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT ||
            mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
         {
            /*
            * This lun is reserved by SCSI-2 reserve. It is not our responsibility
            * to clear it. We clear only the SCSI-3 reservation that are made by 
            * ANY MPP drivers.
            */
            MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
               "Get scsi reservation conflict from reading keys or CC. Lun:%d status:%d Skip\n",
               virtualLun,mpp_status));
            continue;
         }
         else if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
         {
            MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 363,
               "couldnot reading keys. Lun %d... Skip\n",
               virtualLun));
            continue;
         }

         additional_len = MPP_GET_LWORD_FROM_4BYTES(keyParamData->Additional_Len);

         if(0 == additional_len)
         {
            MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
               "0 registration key for Lun %d ... skip \n", virtualLun));
            continue;
         }

         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
            "Has registration keys for Lun %d \n", virtualLun));
         /*
         * check whether or not those keys has MPP signiture
         */
         for(i = 0; i < (additional_len/sizeof(ReservationKey)); i++)
         {
            if( mppCmn_IsMppKey(keyParamData->Reservation_Key[i]) )
            {
               /*
               * Has mpp signiture in the key. Clear the reservation
               */
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_MapBusDeviceReset() clear reservation for Lun %d\n",
                  virtualLun));
               bzero(sensedata,sizeof(SenseData_t));
               /*
               *enableCC is hard-coded to FALSE. During BDR time, we don't want to return
               *a check-condition back.
               */
               mpp_status = mppCmn_MkPRClear(rdacInfo, 
                  virtualLun,keyParamData->Reservation_Key[i],FALSE,sensedata);

               /* 06/15/04 - Clear the reservation flag. */
		       rdacInfo->RdacLuns[virtualLun].NeedsReservationCheck = 0;
               break;
            }
            else
            {
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_MapBusDeviceReset() Lun %d has SCSI-3 registrants but doesn't have MPP signature\n",
                  virtualLun));
            }
         }
      }
   }
   
   MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
   MPP_FREE(sensedata,sizeof(SenseData_t));
   MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MapBusDeviceReset Lun %d\n", Lun));
   return MPPCMN_PRTN_GOOD_STATUS;

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MapBusDeviceResetVs
* SUMMARY:  Issue a vendor-specific request to clear all outstanding Persistent Reservations
* 			for volumes belonging to this hosts' storage partition.
* SCOPE:    public
*
*
* DESCRIPTION:
*  This function locates a valid device to send a vendor-unique request to clear all outstanding
*  Persistent Reservations for those volumes in the same host group as this host.  Used as an
*  alternate to mppCmn_MapBusDeviceReset(), which walks through each lun issuing several
*  requests to only clear reservation requests originated from the MPP driver.  This function
*  is used in environments where there are low tolerances for clearing outstanding reservations,
*  such as MS Cluster environments.

* RESTRICTIONS:
* 
* RETURNS:
         MPPCMN_PRTN_GOOD_STATUS or MPPCMN_PRTN_RETURN_ERROR 
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MapBusDeviceResetVs(
      RdacDeviceInformation_t *RdacInfo)
{
	MPP_HANDLE			device = NULL;
	RdacLunInformation_t            *rdacLun;
	BYTE				*cdb;
	BYTE				cdblen = 10;
	BYTE				controllerIndex, pathIndex; 
	LWORD				lunIndex;
	LWORD				stateIndex;
	LWORD				mpp_status = MPPCMN_PRTN_RETURN_ERROR;
	unsigned int			controller;
   	SenseData_t                     *sensedata;

        MPP_RW_LOCK_EXCLUSIVE(RdacInfo->RWLock);

	sensedata = (SenseData_t *) MPP_INT_KMEM_ALLOC(sizeof(SenseData_t));
	if(sensedata == NULL)
	{
		/* error */
		MPP_RW_UNLOCK_EXCLUSIVE(RdacInfo->RWLock);
		return MPPCMN_PRTN_RETURN_ERROR;
	}

	cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
	if(cdb == NULL)
	{
		/* error */
		MPP_RW_UNLOCK_EXCLUSIVE(RdacInfo->RWLock);
		MPP_FREE(sensedata, sizeof(SenseData_t));
		return MPPCMN_PRTN_RETURN_ERROR;
	}
	else
	{
		bzero(cdb, cdblen);
	}

	// First, find a valid path to send the reset request to.  Start with the first lun.
	for(lunIndex = 0; lunIndex < mpp_MaxLunsPerArray; lunIndex++)
	{
		rdacLun = &RdacInfo->RdacLuns[lunIndex];
		if(rdacLun->Present && !rdacLun->UTMLun)
		{

			// Start with the owning controller for this lun first.
			controller = RdacInfo->RdacLuns[lunIndex].PreferredPath;
			for(controllerIndex = 0; controllerIndex < 2; controllerIndex++)
			{
				if(controllerIndex == 1)
				{
					controller = controller^1;
				}

				for(pathIndex = 0; pathIndex < mpp_MaxPathsPerController; pathIndex++)
				{
					stateIndex = RdacInfo->ControllerInfo[controller]->ControllerPath[pathIndex].PathStateIndex;
					if( RdacInfo->ControllerInfo[controller]->ControllerPath[pathIndex].Present &&
					   (RdacInfo->ControllerInfo[controller]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL ||
						RdacInfo->ControllerInfo[controller]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
						RdacInfo->ControllerInfo[controller]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL_CHECKING) &&
						RdacInfo->RdacLuns[lunIndex].LunControllers[controller]->LunPaths[pathIndex].LunPathDevice &&
					   !RdacInfo->RdacLuns[lunIndex].LunControllers[controller]->LunPaths[pathIndex].PathRemoveState)
					{
						device = RdacInfo->RdacLuns[lunIndex].LunControllers[controller]->LunPaths[pathIndex].LunPathDevice;

						// Found a device to send the request to.
						bzero(sensedata,sizeof(SenseData_t));
						*cdb = SCSI_ENGENIO_BUS_RESET;

						mpp_status = mppCmn_SynchronousPRIo(device, cdb, cdblen, NULL, 0, MPPCMN_XFER_NONE, FALSE, sensedata);
						if(mpp_status == MPPCMN_PRTN_GOOD_STATUS || mpp_status == MPPCMN_PRTN_COMMAND_NOT_SUPPORTED)
						{
							// Reset was either succesful or not supported by the controller.  Exit this function.
							MPP_RW_UNLOCK_EXCLUSIVE(RdacInfo->RWLock);
							MPP_FREE(sensedata, sizeof(SenseData_t));
							MPP_FREE(cdb, cdblen);
							return(mpp_status);
						}
					}
				}
			}
		}
	}

	// Either no devices were found or there were errors on every attempt.  Go ahead and return the last status.
	MPP_FREE(sensedata, sizeof(SenseData_t));
	MPP_FREE(cdb, cdblen);
	MPP_RW_UNLOCK_EXCLUSIVE(RdacInfo->RWLock);
	return(mpp_status);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MapDeviceReset
* SUMMARY:  Translate a scsi "logical unit reset" task management function to Scsi3 persistent
                      reservation command for clearing persistent reservation made by the translation layer for a specific
                      logical unit. 
* SCOPE:    public
*
*
* DESCRIPTION:
   This function performs the mapping of "logical unit reset" task management function" for cleaning up 
   persistent reservations/registrations that are made by mpp driver's 
   translation layer. 

   @rdacInfo The object which represents a storage array
   @Lun   The logical unit where a bus device reset message is addressed to
*
* RESTRICTIONS:
* 
* RETURNS:
         MPPCMN_PRTN_GOOD_STATUS or MPPCMN_PRTN_RETURN_ERROR 
* ERRNO:
*
 NOTES:
    From SPC2 (T10/1236-D Revision 20 Clause 5.6 Multiple port and multiple initiator behavior)
    the LOGICAL UNIT RESET task management function releases reservations established 
    by the reserve/release method and removes all tasks for all initiators for the 
    addressed logical unit and any logical units issuing from it in a hierarchical addressing
    structure (see SAM-2).
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

LWORD mppCmn_MapDeviceReset(
      RdacDeviceInformation_t *rdacInfo,
      LWORD Lun)
{
   RdacLunInformation_t          *rdacLun;
   PRINReadKeysParamData_t       *keyParamData;
   LWORD                         mpp_status;
   LWORD                         additional_len;
   LWORD                         i;
   SenseData_t                   *sensedata;

   MPP_RW_LOCK_EXCLUSIVE(rdacInfo->RWLock);

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_MapDeviceReset Array %s Lun %d\n",
      (BYTE *)rdacInfo->ModuleName,
      Lun));
   /*
   *Check whether or not the array is a pre-Sonoran 4 array. If it is, return
   */
   if(!rdacInfo->FWSonoran4OrLater)
   {
      MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "mppCmn_MapDeviceReset Array %s does not support Persistent reservation\n",
         (BYTE *)rdacInfo->ModuleName));
      return MPPCMN_PRTN_GOOD_STATUS;
   }


   keyParamData = (PRINReadKeysParamData_t *) MPP_INT_KMEM_ALLOC(sizeof(PRINReadKeysParamData_t));
   if(NULL == keyParamData)
   {
      /* error */
      MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 406,
         "mppCmn_MapDeviceReset - Error cannot allocate PRINReadKeysParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(keyParamData,sizeof(PRINReadKeysParamData_t));
   }

   sensedata = (SenseData_t *) MPP_INT_KMEM_ALLOC(sizeof(SenseData_t));
   if(NULL == sensedata)
   {
      /* error */
      MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
      MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 414,
         "mppCmn_MapDeviceReset - Error cannot allocate SenseData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(sensedata,sizeof(SenseData_t));
   }

   rdacLun = &rdacInfo->RdacLuns[Lun];
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "Lun %d Present %d PseudoLunObject 0x%p UTMLun %d\n",
         Lun,
         rdacLun->Present,
         rdacLun->PseudoLunObject,
         rdacLun->UTMLun));

      // 09/09/05 CR93190.  Removed check for NULL PseudoLunObject.  With Microsoft MPIO we don't have a
      // virtual device object to maintain.
      if(rdacLun->Present && !rdacLun->UTMLun)
   {
         /*
         * read registration keys and decide whether or not we need to 
         * clear the reservation
         */
         bzero(keyParamData, sizeof(PRINReadKeysParamData_t));
         bzero(sensedata,sizeof(SenseData_t));
         /*
         *enableCC is hard-coded to FALSE. During BDR time, we don't want to return
         *a check-condition back.
         */
         mpp_status = mppCmn_GetPRKeys(rdacInfo, Lun,keyParamData,FALSE,sensedata);

         if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT ||
            mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
         {
            /*
            * This lun is reserved by SCSI-2 reserve. It is not our responsibility
            * to clear it. We clear only the SCSI-3 reservation that are made by 
            * ANY MPP drivers.
            */
      	    MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
            MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
            MPP_FREE(sensedata,sizeof(SenseData_t));
            MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
               "Get scsi reservation conflict from reading keys or CC. Lun:%d status:%d Skip\n",
               Lun,mpp_status));
	    return(mpp_status);
         }
         else if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
         {
      	    MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
            MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
            MPP_FREE(sensedata,sizeof(SenseData_t));
            MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 407,
               "couldnot reading keys. Lun %d... Skip\n",
               Lun));
	    return(mpp_status);
         }

         additional_len = MPP_GET_LWORD_FROM_4BYTES(keyParamData->Additional_Len);

         if(0 == additional_len)
         {
      	    MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
            MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
               "0 registration key for Lun %d ... skip \n", Lun));
            MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
            MPP_FREE(sensedata,sizeof(SenseData_t));
	    return(mpp_status);
         }

         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
            "Has registration keys for Lun %d \n", Lun));
         /*
         * check whether or not those keys has MPP signiture
         */
         for(i = 0; i < (additional_len/sizeof(ReservationKey)); i++)
         {
            if( mppCmn_IsMppKey(keyParamData->Reservation_Key[i]) )
            {
               /*
               * Has mpp signiture in the key. Clear the reservation
               */
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_MapDeviceReset() clear reservation for Lun %d\n",
                  Lun));
               bzero(sensedata,sizeof(SenseData_t));
               /*
               *enableCC is hard-coded to FALSE. During BDR time, we don't want to return
               *a check-condition back.
               */
               mpp_status = mppCmn_MkPRClear(rdacInfo, Lun,keyParamData->Reservation_Key[i],FALSE,sensedata);

               /* 06/15/04 - Clear the reservation flag. */
	      rdacInfo->RdacLuns[Lun].NeedsReservationCheck = 0;
               break;
            }
            else
            {
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_MapDeviceReset() Lun %d has SCSI-3 registrants but doesn't have MPP signature\n",
                  Lun));
            }
         }
   }
   
   MPP_RW_UNLOCK_EXCLUSIVE(rdacInfo->RWLock);
   MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
   MPP_FREE(sensedata,sizeof(SenseData_t));
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MapDeviceReset Lun %d\n", Lun));
   return MPPCMN_PRTN_GOOD_STATUS;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MkPRClear
* SUMMARY:  Clear Scsi3 persistent reservation made by the translation layer for
            this host node
* SCOPE:    local
*
*
* DESCRIPTION:
   This function issues a synchronous IO of PROUT service action = clear to 
   the an available path. This function is invoked by mppCmn_MapScsi2Release(). 
   
   @RdacInfo The object which represents a storage array.
   @Lun   The logical unit where a bus device reset message is addressed to
   @reservationKey The 8-bytes generated reservation key. The reservationKey are generated
      by mppCmn_GenReservationKey() API
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
   @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
* RESTRICTIONS:
* 
* RETURNS:
   MPPCMN_PRTN_GOOD_STATUS             
   MPPCMN_PRTN_RESERVATION_CONFLICT    
   MPPCMN_PRTN_RETURN_ERROR
   MPPCMN_PRTN_CHECK_CONDITION

*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MkPRClear(RdacDeviceInformation_t *RdacInfo, LWORD Lun, BYTE * reservationKey,
                       BOOL enableCC,SenseData_t *sensedata)
{
   MPP_HANDLE           device = NULL;
   BYTE                 ControllerIndex;
   unsigned int         Controller;
   BYTE                 PathIndex;
   LWORD                mpp_status = MPPCMN_PRTN_RETURN_ERROR;
   LWORD                BufferLength, StateIndex;
   BYTE                 *cdb;
   BYTE                 cdblen;
   PROUTParamData_t     *proutParamData; 

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_MkPRClear Lun %d\n",Lun));
   /*
   * we need to perform a registration first and then perform the clear
   */
   Controller = RdacInfo->RdacLuns[Lun].PreferredPath;

   for(ControllerIndex = 0; ControllerIndex <2; ControllerIndex++)
   {
      /*
      * Try the current owning controller first
      */
      if(ControllerIndex == 1)
      {
         Controller = Controller^1;
      }

      for(PathIndex = 0; PathIndex <mpp_MaxPathsPerController; PathIndex++)
      {
         StateIndex = RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathStateIndex;
         if(   RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].Present &&
              (RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL ||
               RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
               RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_CHECKING) &&
               RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].LunPathDevice &&
               !RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].PathRemoveState) 
         {
            device = RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].LunPathDevice;
            mpp_status = mppCmn_MkPRRegistration(device, 
               MPPCMD_PR_REGISTER_AND_IGNORE_EXISTING_KEY, 
               reservationKey,enableCC,sensedata);
            if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
            {
               MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 364,
                  "mppCmn_MkPRClear failed to make registration via 0x%p",
                  device));
                  
               continue;
            }
            else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
            {
               MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 365,
                  "Exiting mppCmn_MkPRClear -- mk registration error MPPCMN_PRTN_RESERVATION_CONFLICT\n"));
               return MPPCMN_PRTN_RESERVATION_CONFLICT;
            }
            break;
         }
      }
      if(mpp_status != MPPCMN_PRTN_RETURN_ERROR)
      {
         break;
      }
   }
   
   /* 
   * we made the registration. Now we need to send the clear command
   */
   BufferLength = sizeof(PROUTParamData_t);
   proutParamData = (PROUTParamData_t *) MPP_INT_KMEM_ALLOC( BufferLength );
   if(NULL == proutParamData)
   {
      /* error */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 366,
         "mppCmn_MkPRClear - cannot allocate PROUTParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(proutParamData, BufferLength);
   }

   cdblen = 10;
   cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
   *cdb = SCSI_PR_OUT; /*Persistent Reservation out command */
   *(cdb+1) = (BYTE) SCSI_PROUT_CLEAR; /* service action 0x03 clear */
   *(cdb+7) = (BYTE)(BufferLength>>8 &0x00FF);
   *(cdb+8) = (BYTE)(BufferLength    &0x00FF); /* bytex 7 and 8 are 
                                               * allocation length in MSB */
   /*
   * From SPC3. 
   * The RESERVATION KEY field contains an 8-byte value provided by the application 
   * client to the device server to identify the I_T nexus that is the source of 
   * the PERSISTENT RESERVE OUT command. The device server shall
   * verify that the contents of the RESERVATION KEY field in a PERSISTENT RESERVE OUT
   * command parameter data matches the registered reservation key for the I_T nexus 
   * from which the task was received
   */
   bcopy(reservationKey,proutParamData->Reservation_Key,8);
   mpp_status = mppCmn_SynchronousPRIo(device,cdb,cdblen, proutParamData,BufferLength,
                     MPPCMN_XFER_HOST_DEV,enableCC,sensedata);

   MPP_FREE(cdb, cdblen);
   MPP_FREE(proutParamData,sizeof(PROUTParamData_t));
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MkPRClear -- mpp status %d\n",mpp_status));
   return mpp_status;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_GetPRKeys
* SUMMARY:  Read reservation keys from any path of a specified LUN
* SCOPE:    public
*
* DESCRIPTION:
   This function issues a synchronous IO of PRIN service action = "READ KEYS" 
   to obtain the reservation keys. The reservation keys are returned via 
   "keyParamData". The data structure PRINReadKeysParamData_t is defined 
   in "scsiio.h".
   @RdacInfo The object which represents a storage array.
   @Lun   The logical unit where a bus device reset message is addressed to
   @keyParamData a pointer of PRINReadKeysParamData_t object. If this function is
   successfully returned, keyParamData contains SCSI-3 
   "PERSISTENT RESERVE IN parameter data for READ KEYS". PRINReadKeysParamData_t 
   data struct is defined in scsiio.h
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
*
* RESTRICTIONS:
* 
* RETURNS:The function return value can be:
	scsi status " reservation conflict" that indicates that Scsi-2 reservation
    has been used to reserve this LUN
MPPCMN_PRTN_GOOD_STATUS        
MPPCMN_PRTN_RESERVATION_CONFLICT    
MPPCMN_PRTN_RETURN_ERROR    
MPPCMN_PRTN_CHECK_CONDITION        

*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_GetPRKeys(RdacDeviceInformation_t *RdacInfo, 
                       LWORD Lun, 
                       PRINReadKeysParamData_t *keyParamData,
                       BOOL enableCC,
                       SenseData_t *sensedata)
{
   BYTE                 *cdb, cdblen=10;
   LWORD                BufferLength;  /*the length of PRINReadKeyParamData_t */
   LWORD		mppStatus = MPPCMN_PRTN_RETURN_ERROR;
   MPP_HANDLE           device = NULL;
   unsigned int         Controller;
   LWORD		retryOnceOnAllCntrl = 0;
   MPP_TIME		startTime;
   MPP_TIME		currentTime;

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3, "Entering mppCmn_GetPRKeys Lun %d\n", Lun));

   MPP_GETTIME(startTime);
   cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
   BufferLength = sizeof(PRINReadKeysParamData_t);

   if( !cdb ) 
   {
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 367, 
          "Unable to allocate memory for PRIN cdb\n")); 
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(cdb,cdblen);
   }

   *cdb = SCSI_PR_IN; /*Persistent Reservation IN command */
   *(cdb+1) = (BYTE) SCSI_PRIN_READ_KEYS; /* service action 00h READ KEYS */
   *(cdb+7) = (BYTE)(BufferLength>>8 &0x00FF);
   *(cdb+8) = (BYTE)(BufferLength    &0x00FF); /* bytex 7 and 8 are 
                                               * allocation length in MSB */

   // 01/21/08 - CR135691.  Re-factored logic to complete the request in as timely a
   // manner as possible by first trying the owning controller then moving to the
   // alternate.  If the alternate controller is not present or returns an error
   // condition deemed non-retryable the request will be re-attempted to the owning
   // controller.
   Controller = RdacInfo->RdacLuns[Lun].CurrentOwningPath;
   for(;;)
   {
	MPP_GETTIME(currentTime);
	if((currentTime - startTime) > mppCmn_PRWaitTime)
	{
             MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 173, "%s - PR wait time exceeded on lun %d\n", RdacInfo->ModuleName, Lun));
	     return(mppStatus);
	}

        mppStatus = mppCmn_SelectPRPath(RdacInfo, Controller, Lun, &device);
        if(mppStatus == MPPCMN_PRTN_RETURN_ERROR)
        { 
	     if(retryOnceOnAllCntrl == 1)
	     {
                  MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 368,
	 	      "mppCmn_GetPRKeys -- failed to read reg keys from device 0x%p\n", device));
                  return(mppStatus);
	     }

	     // Path not found for this controller.  Try the alternate.
	     Controller = Controller^1;
	     mppStatus = mppCmn_SelectPRPath(RdacInfo, Controller, Lun, &device);
	     if(mppStatus == MPPCMN_PRTN_RETURN_ERROR)
	     {
	          // No paths left.
                  MPP_FREE(cdb, cdblen);
                  MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 369,
                      "Exiting mppCmn_GetPRKeys- error LUN:%d status:%d\n", Lun, mppStatus));
                  return mppStatus;
	     }
        }    

       // Re-use mppStatus variable.  Also, always set the enableCC parameter to true so the request comes back immediately on
       // CHECK_CONDITION's or BUSY conditions.  We need this because SynchronousPRIo() will attempt retries to the same controller
       // and path originally passed in.  We want to try a different controller each time, even if it was tried during a previous attempt.
       mppStatus = mppCmn_SynchronousPRIo(device, cdb, cdblen, keyParamData,BufferLength, MPPCMN_XFER_DEV_HOST, TRUE, sensedata);
       switch(mppStatus)
       {
            case MPPCMN_PRTN_GOOD_STATUS:
            default:
            {
	         // Nothing more to do.
	         MPP_FREE(cdb, cdblen);
	         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
		     "Exiting mppCmn_GetPRKeys- mpp status %d\n", mppStatus));
	         return(mppStatus);
            }

	    case MPPCMN_PRTN_RESERVATION_CONFLICT:
	    case MPPCMN_PRTN_COMMAND_NOT_SUPPORTED:
	    {
	         MPP_FREE(cdb, cdblen);
	         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 408,
		     "Exiting mppCmn_GetPRKeys- error LUN:%d status:%d\n", Lun, mppStatus));
	         return mppStatus;
	    }

	    case MPPCMN_PRTN_CHECK_CONDITION:
	    {
	         if(enableCC == TRUE)
	         {
		     // If the callee set 'enableCC' to true then don't evaluate the error, just return it.
		     MPP_FREE(cdb, cdblen);
		     MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
		         "Exiting mppCmn_GetPRKeys- mpp status %d\n", mppStatus));
		     return(mppStatus);
	         }

	         // Retry on alternate controller, even if the request has already been tried on it.
	         Controller = Controller^1;
	         break;
	    }

	    case MPPCMN_PRTN_RETURN_ERROR:
	    {
		// Retry at least once on the alternate controller provided it is available.
		if(retryOnceOnAllCntrl == 1)
		{
		    MPP_FREE(cdb, cdblen);
		    MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 409,
		    	"Exiting mppCmn_GetPRKeys- error LUN:%d status:%d\n", Lun, mppStatus));
		    return mppStatus;
		}

		retryOnceOnAllCntrl = 1;
		Controller = Controller^1;
		break;
	    }
       }
   }

   MPP_FREE(cdb, cdblen);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
       "Exiting mppCmn_GetPRKeys- mpp status %d (Lun-0x%x)\n", mppStatus, Lun));
   return(mppStatus);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_GetPRReservations
* SUMMARY:  Read Scsi-3 reservation parameter from any path of
             a specified LUN
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function issues a synchronous IO of PRIN service action = "READ RESERVATION" 
   to obtain the reservation parameter of this LUN. The reservation parameter 
   is returned via "ResParamData". The data structure PRINReadResevParamData_t 
   will be defined in "scsiio.h".
   @RdacInfo The object which represents a storage array.
   @Lun   The logical unit where a bus device reset message is addressed to
   @ResParamData  a pointer of PRINReadResevParamData_t object. If this function is
   successfully returned, ResParamData contains SCSI-3 
   "ERSISTENT RESERVE IN parameter data for READ RESERVATION". PRINReadResevParamData_t 
   data struct is defined in scsiio.h
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
*
* RESTRICTIONS:
* 
* RETURNS:The function return value can be:
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT    
MPPCMN_PRTN_RETURN_ERROR 
MPPCMN_PRTN_CHECK_CONDITION           
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_GetPRReservations(RdacDeviceInformation_t *RdacInfo, 
                               LWORD Lun, 
                               PRINReadResevParamData_t *ResParamData,
                               BOOL enableCC,
                               SenseData_t * sensedata)
{
   BYTE                 *cdb, cdblen=10;
   LWORD                BufferLength;  /*the length of PRINReadResevParamData_t */
   LWORD                mppStatus = MPPCMN_PRTN_RETURN_ERROR;
   MPP_HANDLE           device = NULL;
   unsigned int         Controller;
   LWORD		retryOnceOnAllCntrl = 0;
   MPP_TIME		startTime;
   MPP_TIME		currentTime;
	
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
       "Entering mppCmn_GetPRReservations Lun %d\n",Lun));

   MPP_GETTIME(startTime);
   cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
   BufferLength = sizeof(PRINReadResevParamData_t);

   if( !cdb ) {
	MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 370, "Unable to allocate memory\n")); 
	return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
        bzero(cdb,cdblen);
   }

   *cdb = SCSI_PR_IN; /*Persistent Reservation IN command */
   *(cdb+1) = (BYTE) SCSI_PROUT_RESERVE; /* service action 01h READ RESERVATION */
   *(cdb+7) = (BYTE)(BufferLength>>8 &0x00FF);
   *(cdb+8) = (BYTE)(BufferLength    &0x00FF); /* bytex 7 and 8 are 
                                               * allocation length in MSB */
   bzero(ResParamData, BufferLength);

   // 01/21/08 - CR135691.  Re-factored logic to complete the request in as timely a
   // manner as possible by first trying the owning controller then moving to the
   // alternate.  If the alternate controller is not present or returns an error
   // condition deemed non-retryable the request will be re-attempted to the owning
   // controller.
   Controller = RdacInfo->RdacLuns[Lun].CurrentOwningPath;
   for(;;)
   {
        MPP_GETTIME(currentTime);
	if((currentTime - startTime) > mppCmn_PRWaitTime)
	{
             MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 174, "%s - PR wait time exceeded on lun %d\n", RdacInfo->ModuleName, Lun));
	     return(mppStatus);
	}

        mppStatus = mppCmn_SelectPRPath(RdacInfo, Controller, Lun, &device);
        if(mppStatus == MPPCMN_PRTN_RETURN_ERROR)
        { 
	     if(retryOnceOnAllCntrl == 1)
	     {
                  MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 371,
	 	      "mppCmn_GetPRReservations -- failed to read reg keys from device 0x%p\n", device));
                  return(mppStatus);
	     }

	     // Path not found for this controller.  Try the alternate.
	     Controller = Controller^1;
	     mppStatus = mppCmn_SelectPRPath(RdacInfo, Controller, Lun, &device);
	     if(mppStatus == MPPCMN_PRTN_RETURN_ERROR)
	     {
	          // No paths left.
                  MPP_FREE(cdb, cdblen);
                  MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 372,
                      "Exiting mppCmn_GetPRReservations- error LUN:%d status:%d\n", Lun, mppStatus));
                  return mppStatus;
	     }
        }    

       // Re-use mppStatus variable.  Also, always set the enableCC parameter to true so the request comes back immediately on
       // CHECK_CONDITION's or BUSY conditions.  We need this because SynchronousPRIo() will attempt retries to the same controller
       // and path originally passed in.  We want to try a different controller each time, even if it was tried during a previous attempt.
       mppStatus = mppCmn_SynchronousPRIo(device,cdb,cdblen, ResParamData, BufferLength, MPPCMN_XFER_DEV_HOST, TRUE, sensedata);
       switch(mppStatus)
       {
            case MPPCMN_PRTN_GOOD_STATUS:
            default:
            {
	         // Nothing more to do.
	         MPP_FREE(cdb, cdblen);
	         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
		     "Exiting mppCmn_GetPRReservations- mpp status %d\n", mppStatus));
	         return(mppStatus);
            }

	    case MPPCMN_PRTN_RESERVATION_CONFLICT:
	    case MPPCMN_PRTN_COMMAND_NOT_SUPPORTED:
	    {
	         MPP_FREE(cdb, cdblen);
	         MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 410,
		     "Exiting mppCmn_GetPRReservations- error LUN:%d status:%d\n", Lun, mppStatus));
	         return mppStatus;
	    }

	    case MPPCMN_PRTN_CHECK_CONDITION:
	    {
	         if(enableCC == TRUE)
	         {
		     // If the callee set 'enableCC' to true then don't evaluate the error, just return it.
		     MPP_FREE(cdb, cdblen);
		     MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
			"Exiting mppCmn_GetPRReservations- mpp status %d\n", mppStatus));
		     return(mppStatus);
	         }

	         // Retry on alternate controller, even if the request has already been tried on it.
	         Controller = Controller^1;
	         break;
	    }

	    case MPPCMN_PRTN_RETURN_ERROR:
	    {
		// Retry at least once on the alternate controller provided it is available.
		if(retryOnceOnAllCntrl == 1)
		{
		    MPP_FREE(cdb, cdblen);
		    MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 411,
		    	"Exiting mppCmn_GetPRReservations- error LUN:%d status:%d\n", Lun, mppStatus));
		    return mppStatus;
		}

		retryOnceOnAllCntrl = 1;
		Controller = Controller^1;
		break;
	    }
       }
   }

   MPP_FREE(cdb, cdblen);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
	"Exiting mppCmn_GetPRReservations- mpp status %d (Lun-0x%x)\n", mppStatus, Lun));
   return(mppStatus);
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MkPRRegistration
* SUMMARY:  Make Scsi3 registration to all paths of the specified LUN
* SCOPE:    public 
*
* DESCRIPTION:
   This function issues a synchronous IO of PROUT service action = "REGISTER"
   to register all initiator paths with the LUN. 
   
   @RdacInfo The object which represents a storage array.
   @Lun   The logical unit where a bus device reset message is addressed to
   @reservationKey The 8-bytes generated reservation key. The reservationKey are generated
      by mppCmn_GenReservationKey() API
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
*
* RESTRICTIONS:
* 
* RETURNS:The function return value can be:
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT   
MPPCMN_PRTN_RETURN_ERROR 
MPPCMN_PRTN_CHECK_CONDITION           
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MkPRRegistrationForAll(RdacDeviceInformation_t *RdacInfo, 
                                    LWORD Lun, 
                                    BYTE *reservationKey,
                                    BOOL enableCC,
                                    SenseData_t * sensedata)
{
   LWORD	               status = 0;
   BYTE                 ControllerIndex;
   BYTE                 PathIndex;
   MPP_HANDLE           device = NULL;
   LINT                 anySuccess = 0;
   LWORD                availablePath;
   LWORD		StateIndex;

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_MkPRRegistrationForAll Lun %d\n", Lun));
   /*
   * the registration should be made on all not-failed paths
   */
   for(ControllerIndex = 0; ControllerIndex <2; ControllerIndex++)
   {
      for(PathIndex = 0; PathIndex <mpp_MaxPathsPerController; PathIndex++)
      {
         StateIndex = RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[PathIndex].PathStateIndex;
         if(   RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[PathIndex].Present &&
		        (RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL ||
		         RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
		         RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_CHECKING) &&
		         RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[PathIndex].LunPathDevice &&
		         !RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[PathIndex].PathRemoveState) 
         {
            device = RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[PathIndex].LunPathDevice;
			   status = mppCmn_MkPRRegistration(device, MPPCMN_PR_REGISTER, reservationKey,
                              enableCC, sensedata);
            if( status != MPPCMN_PRTN_GOOD_STATUS ) 
            {
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3, 
                     "mppCmn_MkPRRegistrationForAll: data returned with mpp_status %d \n",
                     status));
               /*
               * The path is not available, we must mark it as failed so that
               * the failback scan thread has a chance to re-register when the path
               * moves its state from failed to unfailed
               */
               if(status == MPPCMN_PRTN_RETURN_ERROR)
               {
                  MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 373,
                     "unable to make a registration for the Array %s controller %d path %d\n",
                     (BYTE *)RdacInfo->ModuleName,
                     ControllerIndex,
                     PathIndex));

                  availablePath = mpp_FailPath(RdacInfo,ControllerIndex, PathIndex);
                  if(!availablePath)
                  {
                     /*
                     * no path to the controller. But it is not this code's job to fail
                     * a controller because this code doesn't know which controller owns
                     * a volume. Let's the normal io code to take care of the controller
                     * failover.
                     */
                     MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 374,
                        "No un-failed paths for the Array %s controller %d \n",
                           (BYTE *)RdacInfo->ModuleName,
                           ControllerIndex));

                  }
               }
            }
            else
            {
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_MkPRRegistrationForAll: data returned with mpp_status %d for controller %d path %d\n",
                     status,ControllerIndex,PathIndex));
               anySuccess++;
            }
         }
      }
   }

   if(anySuccess)
   {
      status = MPPCMN_PRTN_GOOD_STATUS;
   }
   else
   {
      status = MPPCMN_PRTN_RETURN_ERROR;
   }

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MkPRRegistrationForAll status %d\n",status));

   return(status);
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MkPRRegistration
* SUMMARY:  Make Scsi3 registration to a specified LUN via the given physical path
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function issues a synchronous IO of PROUT service action = "REGISTER"  or 
   "REGISTER AND IGNORE EXISTING KEY" to register this initiator with the LUN. 
   The service action value is dependent on "reservationType" argument. 
   
   @node a MPP_HANDLE object which represents a physical path of a storage array
      volume.
   @reservationType either "REGISTER"  or "REGISTER AND IGNORE EXISTING KEY" 
   @reservationKey The 8-bytes generated reservation key. The reservationKey are generated
      by mppCmn_GenReservationKey() API
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
*
* RESTRICTIONS:
* 
* RETURNS:The function return value can be:
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT    
MPPCMN_PRTN_RETURN_ERROR  
MPPCMN_PRTN_CHECK_CONDITION          
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MkPRRegistration(MPP_HANDLE node, BYTE reservationType, BYTE * reservationKey,
                              BOOL enableCC,
                              SenseData_t * sensedata)
{
   LWORD                         status;
   BYTE	                        *cdb, cdblen=10;
   LWORD                         BufferLength;  /*the length of PROUTParamData_t */
   PROUTParamData_t              *proutParamData; 

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_MkPRRegistration for device 0x%p\n",node));

	cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);

   BufferLength = sizeof(PROUTParamData_t);
   proutParamData = (PROUTParamData_t *)MPP_INT_KMEM_ALLOC(BufferLength);

   if( !proutParamData ) 
   {
	   	MPP_FREE(cdb, cdblen);
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 375,
         "Unable to allocate memory\n"));
		return MPPCMN_PRTN_RETURN_ERROR;
	}

   /*
   * Based on SPC3, when the cdb's service action is "REGISTER" (00h) and 
   * "REGISTER AND IGNORE EXISTING KEY" (06h), the SCOPE and TYPE fields are
   * ignored by the target device server.
   */
	if( !cdb ) {
		MPP_FREE(proutParamData, BufferLength);
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 376,
         "Unable to allocate memory\n")); 
		return MPPCMN_PRTN_RETURN_ERROR;
	}

	*cdb = SCSI_PR_OUT; /*Persistent Reservation OUT command */
   if(reservationType == MPPCMN_PR_REGISTER)
   {
	   *(cdb+1) = (BYTE) SCSI_PROUT_REGISTER; /* service action 00h REGISTER */
   }
   else
   {
      bcopy(reservationKey, proutParamData->Reservation_Key,8);
      *(cdb+1) = (BYTE) SCSI_PROUT_REGISTER_AND_IGNORE_EXISTING_KEY; /* service action 06h REGISTER AND IGNORE EXISTING KEY*/
   }
   /*
   * Based on the SPC3, if SPEC_I_PT bit is zero in PERSISTENT RESERVATION OUT parameter list
   * the parameter list shall be 24 bytes in length and the PARAMETER LIST LENGTH field should
   * contain 24 (18h). Otherwise, we will get check condition with ASC=Illegal request and
   * ASCQ= Parameter list length error
   */
	//(cdb+7) is 0x0000 by default
   *(cdb+8) = (BYTE)(24   &0x00FF); /* bytex 7 and 8 are 
                                     * allocation length in MSB */
	
   /*
   * With a register service action, the valid fields in the Persisten Reserve Out
   * parameter list are "reservation key" and "service action reservation key"
   * For initial registration, reservation key field set to zero
   */
   bcopy(reservationKey, proutParamData->Srv_Act_Reservation_Key,8);
   proutParamData->All_Tg_Pt = 0;
   proutParamData->Spec_I_Pt = 0;
   proutParamData->Aptpl = 0;

   /*
    * 01/30/07 - CR 116355.  Changed the buffer length passed in from the size
    * of the proutParamData structure to a hard-coded 24 bytes.  A problem was
    * found under iSCSI where the buffer length was being checked against the
    * CDB length, and the command was rejected because of a mismatch in the length.
    */
	status = mppCmn_SynchronousPRIo(node, cdb, cdblen, proutParamData, 24,
                                    MPPCMN_XFER_HOST_DEV, enableCC,sensedata);
	MPP_FREE(cdb, cdblen);
   MPP_FREE(proutParamData,BufferLength);
	if( status != MPPCMN_PRTN_GOOD_STATUS)
   {
       MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
            "mppCmn_GetPRReservations: data returned with mpp_status %d\n",
            status));
	}

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MkPRRegistration stauts %d\n",status));
	return(status);
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MkPRPreempt
* SUMMARY:  issues a synchronous IO of PROUT service action = "PREEMPT" 
            command to the LUN.
* SCOPE:    public
*
*
* DESCRIPTION:
   This function issues a synchronous IO of PROUT service action = "PREEMPT" 
   command to the LUN. This function could be used to un-register all previous 
   registrations of this host or preempts other host's reservation
   and registrations.
   The reservationKey is the reservation key in the command and the 
   serviceActionKey is the "service action reservation key" in the command.
   
   @RdacInfo The object which represents a storage array.
   @Lun   The logical unit where a bus device reset message is addressed to
   @reservationKey The reservationKey is the reservation key in the Persistent 
         Reservation Out command. 8-bytes long.
   @serviceActionKey The serviceActionKey is the "service action reservation key" 
         in Persistent Reservation Out command. 8-bytes long.
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
*
* RESTRICTIONS:
* 
* RETURNS:
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT   
MPPCMN_PRTN_RETURN_ERROR 
MPPCMN_PRTN_CHECK_CONDITION           
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MkPRPreempt(RdacDeviceInformation_t *RdacInfo, 
                         LWORD Lun,BYTE * reservationKey, 
                         BYTE * serviceActionKey,
                         BOOL enableCC,
                         SenseData_t * sensedata)
{
   LWORD	               mpp_status = MPPCMN_PRTN_RETURN_ERROR;
	BYTE	               *cdb, cdblen=10;
   LWORD                BufferLength;  /*the length of PROUTParamData_t */
   LWORD		StateIndex;
   BYTE                 ControllerIndex;
   unsigned int         Controller;
   BYTE                 PathIndex;
   MPP_HANDLE           device = NULL;
   PROUTParamData_t     *proutParamData;
	
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
        "Entering mppCmn_MkPRPreempt Lun %d\n", Lun));
	
	cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
   BufferLength = sizeof(PROUTParamData_t);

   proutParamData = (PROUTParamData_t *) MPP_INT_KMEM_ALLOC(BufferLength);

   if( !proutParamData ) 
   {
	   	MPP_FREE(cdb, cdblen);
		MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 377, 
         "Unable to allocate memory\n")); 
		return MPPCMN_PRTN_RETURN_ERROR;
	}

	if( !cdb ) 
   {
	   	MPP_FREE(proutParamData, BufferLength);
		MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 378, 
         "Unable to allocate memory\n")); 
		return(MPPCMN_PRTN_RETURN_ERROR);
	}
	*cdb = SCSI_PR_OUT; /*Persistent Reservation OUT command */
	*(cdb+1) = (BYTE) SCSI_PROUT_PREEMPT; /* service action 04h preempt */
   *(cdb+2) = (BYTE) (SCSI_PR_EXCL_ACCESS_RO); 
	*(cdb+7) = (BYTE)(BufferLength>>8&0x00FF);
   *(cdb+8) = (BYTE)(BufferLength   &0x00FF); /* bytex 7 and 8 are 
                                               * allocation length in MSB */

   bcopy(reservationKey,proutParamData->Reservation_Key,8);
   bcopy(serviceActionKey,proutParamData->Srv_Act_Reservation_Key,8);
   

  /*
   * 
   */
   Controller = RdacInfo->RdacLuns[Lun].PreferredPath;
   for(ControllerIndex = 0; ControllerIndex <2; ControllerIndex++)
   {
     /*
      * Try the current owning controller first
      */
      if(ControllerIndex == 1)
      {
         Controller = Controller^1;
      }

      for(PathIndex = 0; PathIndex <mpp_MaxPathsPerController; PathIndex++)
      {
         StateIndex = RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathStateIndex;
         if(   RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].Present &&
		        (RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL ||
		         RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
		         RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_CHECKING) &&
		         RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].LunPathDevice &&
		         !RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].PathRemoveState) 
         {
            device = RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].LunPathDevice;
			   mpp_status = mppCmn_SynchronousPRIo(device, cdb,cdblen, proutParamData,BufferLength,
                           MPPCMN_XFER_HOST_DEV,enableCC,sensedata);
            if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
            {
               MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 379,
                  "mppCmn_MkPRPreempt -- failed to preempt device 0x%p\n",
                  device));
               continue;
            }
            else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
            {
               MPP_FREE(cdb, cdblen);
               MPP_FREE(proutParamData,BufferLength);
               MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 380,
                  "Exiting mppCmn_MkPRPreempt -- MPP_SCSI_RESERVATION\n"));
               return MPPCMN_PRTN_RESERVATION_CONFLICT;
            }
			   break;
         }
      }

      if(mpp_status != MPPCMN_PRTN_RETURN_ERROR)
      {
         break;
      }
	}

   MPP_FREE(cdb, cdblen);
   MPP_FREE(proutParamData,BufferLength);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MkPRPreempt -- status %d\n",mpp_status));
   return mpp_status;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_MkPRReservation
* SUMMARY:  Issues a synchronous IO of PROUT service action = "RESERVE" 
      command to the LUN via the given physical path.
* SCOPE:    public 
*
* DESCRIPTION:
   This function issues a synchronous IO of PROUT service action = "RESERVE" 
   command to the LUN.
   
   @node a MPP_HANDLE object which represents a physical path of a storage array
      volume.
   @reservationKey The 8-bytes generated reservation key. The 
      reservationKey are generated by mppCmn_GenReservationKey() API
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
* RESTRICTIONS:
* 
* RETURNS:The function return value can be:
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT    
MPPCMN_PRTN_RETURN_ERROR   
MPPCMN_PRTN_CHECK_CONDITION         
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_MkPRReservation(RdacDeviceInformation_t *RdacInfo,
                             LWORD Lun, 
                             BYTE * reservationKey,
                             BOOL enableCC,
                             SenseData_t * sensedata)
{
   LWORD	        mpp_status = MPPCMN_PRTN_RETURN_ERROR;
	BYTE	               *cdb, cdblen=10;
   LWORD                BufferLength;  /*the length of PROUTParamData_t */
   LWORD		StateIndex;
   BYTE                 ControllerIndex;
   unsigned int         Controller;
   BYTE                 PathIndex;
   MPP_HANDLE           device = NULL;
   PROUTParamData_t     *proutParamData;
	
	MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_MkPRReservation Lun %d\n", Lun));

	cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
   BufferLength = sizeof(PROUTParamData_t);
   proutParamData = (PROUTParamData_t *)MPP_INT_KMEM_ALLOC(BufferLength);

   if( !proutParamData ) 
   {
	   	MPP_FREE(cdb, cdblen);
		MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 381, "Unable to allocate memory\n")); 
		return MPPCMN_PRTN_RETURN_ERROR;
	}

	if( !cdb ) 
   {
	   	MPP_FREE(proutParamData, BufferLength);
		MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 382, "Unable to allocate memory\n")); 
		return MPPCMN_PRTN_RETURN_ERROR;
	}
	*cdb = (BYTE) SCSI_PR_OUT; /*Persistent Reservation IN command */
	*(cdb+1) = (BYTE) SCSI_PROUT_RESERVE; /* service action 01h RESERVE */
   *(cdb+2) = (BYTE) SCSI_PR_EXCL_ACCESS_RO; /* type 06h Exclusive access registrants only */
	/*
   * Based on the SPC3, if SPEC_I_PT bit is zero in PERSISTENT RESERVATION OUT parameter list
   * the parameter list shall be 24 bytes in length and the PARAMETER LIST LENGTH field should
   * contain 24 (18h). Otherwise, we will get check condition with ASC=Illegal request and
   * ASCQ= Parameter list length error
   */
	//(cdb+7) is 0x0000 by default
   *(cdb+8) = (BYTE)(24    &0x00FF); /* bytex 7 and 8 are 
                                      * allocation length in MSB */

   bcopy(reservationKey,proutParamData->Reservation_Key,8);

   Controller = RdacInfo->RdacLuns[Lun].PreferredPath;
   for(ControllerIndex = 0; ControllerIndex <2; ControllerIndex++)
   {
           /*
      * Try the current owning controller first
      */
      if(ControllerIndex == 1)
      {
         Controller = Controller^1;
      }

      for(PathIndex = 0; PathIndex <mpp_MaxPathsPerController; PathIndex++)
      {
         StateIndex = RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathStateIndex;
         if(   RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].Present &&
		        (RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL ||
		         RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
		         RdacInfo->ControllerInfo[Controller]->ControllerPath[PathIndex].PathState[StateIndex] == MPPCMN_OPTIMAL_CHECKING) &&
		         RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].LunPathDevice &&
		         !RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].PathRemoveState) 
         {
            device = RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[PathIndex].LunPathDevice;
            mpp_status = mppCmn_SynchronousPRIo(device, cdb,cdblen, proutParamData,24,
                              MPPCMN_XFER_HOST_DEV,enableCC,sensedata);
            if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
            {
               MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 383,
                  "mppCmn_MkPRReservation -- failed to reserve device 0x%p ctrl %d path %d\n",
                  device,Controller,PathIndex));
               continue;
            }
            else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
            {
               MPP_FREE(cdb, cdblen);
               MPP_FREE(proutParamData,BufferLength);
               
               mppCmn_MkPRPreempt(RdacInfo, 
                         Lun,
                         reservationKey, 
                         reservationKey,
                         enableCC,
                         sensedata);
               MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 384,
                  "Exiting mppCmn_MkPRReservation -- MPP_SCSI_RESERVATION_CONFLICT for device 0x%p ctrl %d path %d\n",
                  device,Controller,PathIndex));
               return MPPCMN_PRTN_RESERVATION_CONFLICT;
            }
			   break;
         }
      }

      if(mpp_status != MPPCMN_PRTN_RETURN_ERROR)
      {
         break;
      }
	}

   MPP_FREE(cdb, cdblen);
   MPP_FREE(proutParamData,BufferLength);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_MkPRReservation -- status %d\n",mpp_status));
   return mpp_status;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_CheckPRRegistration
* SUMMARY:  Issues a synchronous IO of PROUT service action = "RESERVE" 
      command to the LUN via the given physical path.
* SCOPE:    public 
*
* DESCRIPTION:
   This function is invoked when a failed path become un-failed. The purpose of 
   this function is to make a persistent reservation registration if this host 
   owns the reservation of the LUN. The input argument "device" points to 
   a physical Scsi Device object.
  
   This function performs synchronous I/Os.

   @device a MPP_HANDLE object which represents a physical path of a storage array
      volume.
* RESTRICTIONS:
* 
* RETURNS:The function return value can be:
MPPCMN_PRTN_GOOD_STATUS            

* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
LWORD mppCmn_CheckPRRegistration(RdacDeviceInformation_t * rdacInfo, MPP_HANDLE device, LWORD virtualLun)
{
   LWORD                               mpp_status;
   BYTE                                *reservationKey = NULL;
   PRINReadKeysParamData_t             *keyParamData;
   PRINReadResevParamData_t            * resParamData;
   LWORD                               additional_len;
   BYTE                                *cdb;
   BYTE                                cdblen=10;
   LINT                                BufferLength;
   BYTE                                isReservationHolder = 1;
   LWORD                               i;
   SenseData_t                         *sensedata;
   
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppCmn_CheckPRRegistration 0x%p\n", device));
  /*
   *Check whether or not the array is a pre-Sonoran 4 array. If it is, return
   */
   if(!rdacInfo->FWSonoran4OrLater)
   {
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "Eixting mppCmn_CheckPRRegistration() Array %s does not support Persistent reservation\n",
         (BYTE *)rdacInfo->ModuleName));

      return MPPCMN_PRTN_GOOD_STATUS;
   }

   /*
    * 06/15/04 - Check whether a reservation has already been acquired for this device.
    */
   if(rdacInfo->RdacLuns[virtualLun].NeedsReservationCheck == 0)
   {
       MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4, "Lun %d doesn't need a reservation check\n", virtualLun));
       return MPPCMN_PRTN_GOOD_STATUS;
   }

   cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);
   BufferLength = sizeof(PRINReadKeysParamData_t);

	if( !cdb ) 
   {
		MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 385,
         "Unable to allocate memory\n")); 
		return MPPCMN_PRTN_RETURN_ERROR;
	}
   else
   {
      bzero(cdb,cdblen);
   }
	*cdb = SCSI_PR_IN; /*Persistent Reservation IN command */
	*(cdb+1) = (BYTE) SCSI_PRIN_READ_KEYS; /* service action 00h READ KEYS */
	*(cdb+7) = (BYTE)(BufferLength>>8 &0x00FF);
   *(cdb+8) = (BYTE)(BufferLength    &0x00FF); /* bytex 7 and 8 are 
                                               * allocation length in MSB */
	
   /*
   * 1. read the reservation key
   *    we may get reservation conflict if the LUN is already reserved by SCSI-2 
   *    reserve. The firmware doesn't handle the co-existence of SCSI-2/SCSI-3
   *    reservation protocol
   */
   keyParamData = (PRINReadKeysParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadKeysParamData_t));
   if(NULL == keyParamData)
   {
      /* error */
      MPP_FREE(cdb, cdblen);
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 386,
         "Error in allocating PRINReadKeysParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(keyParamData,sizeof(PRINReadKeysParamData_t));
   }

   sensedata = (SenseData_t *)MPP_INT_KMEM_ALLOC(sizeof(SenseData_t));
   if(NULL == sensedata)
   {
      /* error */
      MPP_FREE(cdb, cdblen);
      MPP_FREE(keyParamData, sizeof(PRINReadKeysParamData_t));
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 415,
         "Error in allocating SenseData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(sensedata,sizeof(SenseData_t));
   }
   /*
   *enableCC is hard-coded to FALSE. We would like retries to make the registration.
   */
   mpp_status = mppCmn_SynchronousPRIo(device,cdb,cdblen, keyParamData,BufferLength,
                        MPPCMN_XFER_DEV_HOST,FALSE,sensedata);

   if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 387,
         "mppCmn_GetPRKeys -- failed to read reg keys from device 0x%p\n",
                  device));
      MPP_FREE(cdb, cdblen);
      MPP_FREE(keyParamData,BufferLength);
      MPP_FREE(sensedata, sizeof(SenseData_t));
      /*
      * this case should not happen because we know the path status changed from
      * failed to unfailed.
      * Anyway, return a good status
      */
      return MPPCMN_PRTN_GOOD_STATUS;
   }
   else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
   {
      /*
      * we get a reservation conflict scsi status
      * this means that the device is reserved by a SCSI-2 reserve command.
      * we don't need to worry about making-up a SCSI-3 reservation registration
      * for a failed->unfailed path in this case.
      */
      MPP_FREE(cdb, cdblen);
      MPP_FREE(keyParamData,BufferLength);
      MPP_FREE(sensedata, sizeof(SenseData_t));
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 388,
         "Exiting mppCmn_CheckPRRegistration- error MPP_SCSI_RESERVATION while reading keys\n"));
      return MPPCMN_PRTN_GOOD_STATUS;
   }

   additional_len = MPP_GET_LWORD_FROM_4BYTES(keyParamData->Additional_Len);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "keyParam addtional_len %d\n",additional_len));

   MPPCMN_DUMP_READKEY(keyParamData);

   if(0 == additional_len)
   {
      /*
      * empty registration key list
      */
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "mppCmn_MapScsi2Reserve- empty registration key\n"));
      /*There is no registration keys with this LUN, this means this
      * LUN is not reserved by any initiators. We can leave safely.
      */
      MPP_FREE(cdb, cdblen);
      MPP_FREE(keyParamData,BufferLength);
      MPP_FREE(sensedata, sizeof(SenseData_t));
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "Exiting mppCmn_CheckPRRegistration- Empty reservation keys\n"));
      return MPPCMN_PRTN_GOOD_STATUS;

   }
   else
   {
      /* 
      * the registration keys are not empty. Let's check whether or not this host
      * owns the reservation.
      */
      bzero(cdb,cdblen);
      BufferLength = sizeof(PRINReadResevParamData_t);
	   *cdb = SCSI_PR_IN; /*Persistent Reservation IN command */
	   *(cdb+1) = (BYTE) SCSI_PRIN_READ_RESERVATION; /* service action 01h READ RESERVATION */
	   *(cdb+7) = (BYTE)(BufferLength>>8 &0x00FF);
      *(cdb+8) = (BYTE)(BufferLength   &0x00FF); /* bytex 7 and 8 are 
                                               * allocation length in MSB */
      resParamData = ( PRINReadResevParamData_t *)MPP_INT_KMEM_ALLOC(BufferLength);
      if(NULL == resParamData)
      {
         /* error */
         MPP_FREE(cdb, cdblen);
         MPP_FREE(keyParamData,BufferLength);
         MPP_FREE(sensedata, sizeof(SenseData_t));
         MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 389,
            "mppCmn_CheckPRRegistration- cannot allocate PRINReadResevParamData_t\n"));
         return MPPCMN_PRTN_RETURN_ERROR;
      }
   
		mpp_status = mppCmn_SynchronousPRIo(device, cdb, cdblen, resParamData, BufferLength,
                              MPPCMN_XFER_DEV_HOST, FALSE, sensedata);
      if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
      {
         MPP_FREE(cdb, cdblen);
         MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
         MPP_FREE(resParamData, BufferLength);
         MPP_FREE(sensedata, sizeof(SenseData_t));
         MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 390,
            "Exiting mppCmn_CheckPRRegistration -- failed to read reservation from device 0x%p\n",
                  device));
         return MPPCMN_PRTN_GOOD_STATUS;
      }  
      else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
      {
         MPP_FREE(cdb, cdblen);
         MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
         MPP_FREE(resParamData, BufferLength);
         MPP_FREE(sensedata, sizeof(SenseData_t));
         MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 391,
            "Exiting mppCmn_CheckPRRegistration- error MPP_SCSI_RESERVATION\n"));
         return MPPCMN_PRTN_GOOD_STATUS;
      }
      
      /*we got the reservation holder info. Let's see who is the holder*/
      additional_len = MPP_GET_LWORD_FROM_4BYTES(resParamData->Additional_Len);
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
         "resParamData addtional_len %d\n",additional_len));
      
      MPPCMN_DUMP_RESVKEY(resParamData);

      if(0 == additional_len)
      {
         MPP_FREE(cdb, cdblen);
         MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
         MPP_FREE(resParamData, BufferLength);
         MPP_FREE(sensedata, sizeof(SenseData_t));
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
            "Exiting mppCmn_CheckPRRegistration- No reservation holder\n"));
         return MPPCMN_PRTN_GOOD_STATUS;
         
      }
      else
      {
         /*
         * check if we are the persistent reservation holder
         */
         for(i = 0; i < (additional_len)/(sizeof(ReservationDescriptor_t)); i++)
         {
            reservationKey = resParamData->Reservation_Descriptor[i].Reservation_Key;
            isReservationHolder = mppCmn_IsKeyForThisHost(
               mpp_S2ToS3Key,
               reservationKey);

            if(!isReservationHolder)
            {
               MPP_FREE(cdb, cdblen);
               MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
               MPP_FREE(resParamData, BufferLength);
               MPP_FREE(sensedata, sizeof(SenseData_t));
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "Exiting mppCmn_CheckPRRegistration:Lun is reserved by others\n"));
               return MPPCMN_PRTN_GOOD_STATUS;
            }
         }/* for loop*/
      }/* else read reservation holder*/
   } /* else read keys */

  
   mpp_status = mppCmn_MkPRRegistration(device, 
               MPPCMD_PR_REGISTER_AND_IGNORE_EXISTING_KEY, 
               reservationKey,FALSE,sensedata);

   
   if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT )
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 392,
         "mppCmn_CheckPRRegistration:RESERVATION_CONFLICT while making registration\n"));
      mpp_status = MPPCMN_PRTN_GOOD_STATUS;
   }
   else if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 393,
         "mppCmn_CheckPRRegistration:IO error while making registration\n"));
      mpp_status = MPPCMN_PRTN_GOOD_STATUS;
   }

   /* 06/15/04 - Set the reservation flag. */
   if(rdacInfo->RdacLuns[virtualLun].NeedsReservationCheck == 0)
   {
        MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
            "mppCmn_CheckPRRegistration: Setting reservation flag for Lun %d\n", virtualLun));
   	rdacInfo->RdacLuns[virtualLun].NeedsReservationCheck = 1;
   }

   MPP_FREE(cdb, cdblen);
   MPP_FREE(keyParamData,sizeof(PRINReadKeysParamData_t));
   MPP_FREE(resParamData, BufferLength);
   MPP_FREE(sensedata,sizeof(SenseData_t));
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppCmn_CheckPRRegistration mpp_status 0x%x\n", mpp_status));
   return mpp_status;
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_S2toS3SynchCmnd
* SUMMARY:  This function is used by the native platform driver code to issue special scsi 
            commands.
* SCOPE:    public 
*
*
* DESCRIPTION:  If a LUN is reserved by SCSI-2 reserve command, any other initiator attempts
    to perform any command on the reserved logical unit other than an INQUIRY, REQUEST SENSE,
    PREVENT ALLOW MEDIUM REMOVEAL or a RELEASE command the command shall be rejected with
    RESERVATION CONFLICT status.

    With the SCSI-2 to SCSI-3 reservation translation layer, the MPP driver emulates Scsi2
    reserve with SCSI-3 persistent reservation by using persistent reservation type 
    Exclusive-Access Registrants-only. Based on the SPC-3, the following commands are 
    allowed if the LUN is reserved. 
      SPC-3 commands 
         ACCESS CONTROL IN 
         ACCESS CONTROL OUT 
         INQUIRY 
         LOG SENSE 
         PREVENT/ALLOW 
         READ MEDIA SERIAL NUMBER 
         REPORT ALIASES 
         REPORT DEVICE IDENTIFIER 
         REPORT LUNS 
         REPORT TARGET PORT GROUPS 
         REQUEST SENSE 
         TEST UNIT READY 
      SBC-2 Commands 
         READ CAPACITY (10)/(16) 
         START/STOP UNIT with START=1 and POWER CONDITION=0 
*
* The MPP driver must take special cares for the following commands:

      SPC-3 commands 
         ACCESS CONTROL IN 
         ACCESS CONTROL OUT 
         LOG SENSE 
         READ MEDIA SERIAL NUMBER 
         REPORT ALIASES 
         REPORT DEVICE IDENTIFIER 
         REPORT LUNS 
         REPORT TARGET PORT GROUPS 
         TEST UNIT READY 
      SBC-2 Commands 
         READ CAPACITY (10)/(16) 
         START/STOP UNIT with START=1 and POWER CONDITION=0 
      Engenio vendor specific command
         QUIESCE
         SAFE-PASS-THROUGH
*When the native platform MPP driver receives a virtual command that is one of the above command, it should call this function to complete the request.
* RESTRICTIONS: 
* @RdacInfo The RdacDeviceInformation_t object which represents a storage array
* @Lun    The Logic Unit Number of the addressed device
* @cdb    The CDB of a SCSI command
* @cdblen The length of the cdb
* @BufferAddress The buffer for data transfer between the host and the device
* @BufferLength The length of the buffer
* @Direction 0 -send data to device (data write); 1 - read data from the device (data read)
  @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
* RETURNS: 
   MPPCMN_PRTN_GOOD_STATUS    if this node is  the reservation holder, and a synchronous 
                              command is successfully issued.   
   MPPCMN_PRTN_RESERVATION_CONFLICT    if this node is not the reservation holder 
   MPPCMN_PRTN_RETURN_ERROR  If can not talk to the device though all paths.  
   MPPCMN_PRTN_CHECK_CONDITION
*
* ERRNO: 
*
* NOTES:
*
* EXAMPLES:        
*
* SEE ALSO: 
*
*/
LWORD
mppCmn_S2toS3SynchCmnd(RdacDeviceInformation_t *RdacInfo, LWORD Lun, BYTE *cdb, 
                       BYTE cdblen, void *BufferAddress,LWORD BufferLength,
                       TINY Direction,
                       BOOL enableCC,
                       SenseData_t * sensedata) 
{
   LWORD                               mpp_status;
   PRINReadResevParamData_t            * resParamData;
   LWORD                               additional_len;
   BYTE                                isReservationHolder = 1;
   BYTE                                isMppKey = 1;
   LWORD                               i;
   LWORD			       StateIndex;
   MPP_HANDLE                          DeviceObject;
   unsigned int                        Controller;
   BYTE                                ControllerIndex, path;
   
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering mppCmn_S2toS3SynchCmnd Lun %d command %x\n", Lun, cdb[0])); 

   /*
   * read reservation 
   */
   resParamData = ( PRINReadResevParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadResevParamData_t));
   if(NULL == resParamData)
   {
      /* error */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 394, 
            "mppCmn_S2toS3SyncCmnd - cannot allocate PRINReadResevParamData_t\n"));
      return MPPCMN_PRTN_RETURN_ERROR;
   }
   else
   {
      bzero(resParamData,sizeof(PRINReadResevParamData_t));
   }

   mpp_status = mppCmn_GetPRReservations(RdacInfo, Lun, resParamData,enableCC,sensedata);
   /*
   * Check the return status
   */
   if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT ||
      mpp_status == MPPCMN_PRTN_RETURN_ERROR ||
      mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
   {
      MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 397, 
         "Exiting mppCmn_S2toS3SynchCmnd with error failed to read reservation Lun:%d status:0x%x\n",
               Lun,mpp_status));
      MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
      return mpp_status;
   }

   additional_len = MPP_GET_LWORD_FROM_4BYTES(resParamData->Additional_Len);
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "mppCmn_S2toS3SynchCmnd() resParamData addtional_len %d\n",additional_len));
      
   MPPCMN_DUMP_RESVKEY(resParamData);

   if(0 == additional_len)
   {
      MPP_FREE(resParamData, sizeof(PRINReadResevParamData_t));
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "mppCmn_S2toS3SynchCmnd() mppCmn_S2toS3SynchCmnd empty reservation list\n"));
   }
   else
   {
      /*
      * check if we are the persistent reservation holder
      * because the controller firmware support only LU_SCOPE, the list
      * should have only one reservaton descriptor.
      */
      for(i = 0; i < (additional_len)/(sizeof(ReservationDescriptor_t)); i++)
      {
         isReservationHolder = mppCmn_IsKeyForThisHost(
            mpp_S2ToS3Key,
            resParamData->Reservation_Descriptor[i].Reservation_Key);
         isMppKey = mppCmn_IsMppKey(resParamData->Reservation_Descriptor[i].Reservation_Key);
         /* 
         * return reservation conflict if the reservation key is an mpp driver's 
         * reservation key and this node is not the reservation holder
         */
         if(isMppKey && !isReservationHolder)
         {
            MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
            MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 395, 
                "Exiting mppCmn_S2toS3SynchCmnd with error-Lun is reserved by others\n"));
            return (MPPCMN_PRTN_RESERVATION_CONFLICT);
         }
      }
      /*
      * we are the reservation holder
      */
      MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
            "Exiting mppCmn_S2toS3SynchCmnd -- This node is the reservation holder\n"));
   }
   /*
   * We get here because either the device is not reserved or this node is the reservation
   * holder. We can issue the request command done to the device
   */
   Controller = RdacInfo->RdacLuns[Lun].PreferredPath;
   for( ControllerIndex=0; ControllerIndex<2; ++ControllerIndex ) 
   {
      /*
      * try the current owning controller first
      */
      if(ControllerIndex == 1)
      {
         Controller = Controller^1;
      }

      if( !RdacInfo->ControllerInfo[Controller]->ControllerPresent || 
            RdacInfo->ControllerInfo[Controller]->Failed )
      {
         continue;
      }

      for( path=0; path<mpp_MaxPathsPerController; ++path ) 
      {
         StateIndex = RdacInfo->ControllerInfo[Controller]->ControllerPath[path].PathStateIndex;
         if( !RdacInfo->ControllerInfo[Controller]->ControllerPath[path].Present ||
              RdacInfo->ControllerInfo[Controller]->ControllerPath[path].PathState[StateIndex] == MPPCMN_FAILED ||
              RdacInfo->ControllerInfo[Controller]->ControllerPath[path].PathState[StateIndex] == MPPCMN_FAILED_NEED_CHECK ||
              RdacInfo->ControllerInfo[Controller]->ControllerPath[path].PathState[StateIndex] == MPPCMN_FAILED_CHECKING ||
              RdacInfo->ControllerInfo[Controller]->ControllerPath[path].PathState[StateIndex] == MPPCMN_SYSTEM_SPECIFIC ||
             !RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[path].LunPathDevice)
         {
            continue;
         }

         DeviceObject = RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[path].LunPathDevice;
         
         mpp_status = mppCmn_SynchronousPRIo(DeviceObject, cdb, cdblen, 
               BufferAddress, BufferLength, Direction, enableCC, sensedata);
         if(mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
         {
             /* controller is in 06/08/02 or 02/04/01 condition */
            MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 399,
               "mppCmn_S2toS3SynchCmnd: Device %s:%d - get CC for special command %x\n",
               RdacInfo->ModuleName, Controller,cdb[0]));
            return MPPCMN_PRTN_CHECK_CONDITION;
         }
         else if(mpp_status == MPPCMN_PRTN_RETURN_ERROR)
         {
            MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 396,
               "mppCmn_S2toS3SynchCmnd: Device %s:%d - Unable to send special command %x\n",
               RdacInfo->ModuleName, Controller,cdb[0]));
            continue;
         }else
         {
            return MPPCMN_PRTN_GOOD_STATUS;
         }
      }
   }

   return MPPCMN_PRTN_RETURN_ERROR;

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_CheckForReservationHolder
* SUMMARY:  This function determines whether a given lun is currently reserved by
* 	    this node.
* SCOPE:    public 
*
*
* DESCRIPTION: 
* 	This function retrieves the Persistent Reservation information from a controller
* 	to determine if the provided Lun is currently reserved by this node and returns
* 	an appropriate status that may be used by the caller of this function to
* 	determine whether a given command should be routed.
*
* RETURNS: 
   MPPCMN_PRTN_GOOD_STATUS    if this node is the reservation holder.
   MPPCMN_PRTN_RESERVATION_CONFLICT    if this node is not the reservation holder 
   MPPCMN_PRTN_RETURN_ERROR   if can not talk to the device through all paths.  
   MPPCMN_PRTN_CHECK_CONDITION
*/
LWORD
mppCmn_CheckForReservationHolder(RdacDeviceInformation_t *RdacInfo, LWORD Lun, BOOL enableCC, SenseData_t *senseData)
{
	PRINReadResevParamData_t            * resParamData;
	LWORD                               mpp_status;
	LWORD                               additional_len;
	LWORD                               i;
	BYTE                                isReservationHolder = 1;
	BYTE                                isMppKey = 1;

   
	/*
	* read reservation 
	*/
	resParamData = ( PRINReadResevParamData_t *)MPP_INT_KMEM_ALLOC(sizeof(PRINReadResevParamData_t));
	if(NULL == resParamData)
	{
		/* error */
		MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 412, 
		      "mppCmn_CheckForReservationHolder - cannot allocate PRINReadResevParamData_t\n"));
		return(MPPCMN_PRTN_RETURN_ERROR);
	}
	else
	{
		bzero(resParamData,sizeof(PRINReadResevParamData_t));
	}

	mpp_status = mppCmn_GetPRReservations(RdacInfo, Lun, resParamData,enableCC,senseData);

	/*
	* Check the return status
	*/
	if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT ||
	   mpp_status == MPPCMN_PRTN_RETURN_ERROR ||
	   mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
	{
		MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 405, 
			"Exiting mppCmn_CheckForReservationHolder with error failed to read reservation Lun:%d status:0x%x\n",
			Lun, mpp_status));
		MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
		return mpp_status;
	}

	additional_len = MPP_GET_LWORD_FROM_4BYTES(resParamData->Additional_Len);
	MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
		"mppCmn_CheckForReservationHolder() resParamData addtional_len %d\n",additional_len));
      
//	MPPCMN_DUMP_RESVKEY(resParamData);

	if(0 == additional_len)
	{
		MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
		MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
			"mppCmn_CheckForReservationHolder() mppCmn_S2toS3SynchCmnd empty reservation list\n"));
	}
	else
	{
		/*
		 * check if we are the persistent reservation holder
		 * because the controller firmware support only LU_SCOPE, the list
		 * should have only one reservaton descriptor.
		 */
		for(i = 0; i < (additional_len)/(sizeof(ReservationDescriptor_t)); i++)
		{
			isReservationHolder = mppCmn_IsKeyForThisHost(mpp_S2ToS3Key,
				resParamData->Reservation_Descriptor[i].Reservation_Key);
			isMppKey = mppCmn_IsMppKey(resParamData->Reservation_Descriptor[i].Reservation_Key);
			/* 
			 * return reservation conflict if the reservation key is an mpp driver's 
			 * reservation key and this node is not the reservation holder
			 */
			if(isMppKey && !isReservationHolder)
			{
				MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
				MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 413, 
					"Exiting mppCmn_CheckForReservationHolder with error-Lun is reserved by others\n"));
				return (MPPCMN_PRTN_RESERVATION_CONFLICT);
			}
		}

		/*
		 * we are the reservation holder
		 */
		MPP_FREE(resParamData,sizeof(PRINReadResevParamData_t));
		MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
			"Exiting mppCmn_CheckForReservationHolder -- This node is the reservation holder\n"));
	}

	return (MPPCMN_PRTN_GOOD_STATUS);
}
 

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_SynchronousPRIo
* SUMMARY:  
* SCOPE:    private 
*
*
* DESCRIPTION: This routine is used by the MPP driver to initiate synchronous
*              Persistent Reservation IN/OUT I/O.
*
* RESTRICTIONS: 
* 
   @DeviceVertex A MPP_HANDLE object which repesents a scsi device object
   @cdb          Scsi Command Description Block
   @cdblen       The length of the CDB in byte
   @BufferAddress The buffer address of transfered data
   @BufferLength  The length of the buffer
   @Direction     The direction of data transfer. 1 - data in and 0- data out
   @enableCC If TRUE, this function may return MPPCMN_PRTN_CHECK_CONDITION and the "sensedata"
            will be filled. If FALSE, this function will perform retries and will not return
            MPPCMN_PRTN_CHECK_CONDITION.
  @sensedata A pointer to a SenseData_t object. Called must allocate memory for this object.
            If MPPCMN_PRTN_CHECK_CONDITION is returned, the sensedata will contain valid 
            sense data.
* RETURNS: 
MPPCMN_PRTN_GOOD_STATUS             
MPPCMN_PRTN_RESERVATION_CONFLICT    
MPPCMN_PRTN_RETURN_ERROR  
MPPCMN_PRTN_CHECK_CONDITION          
*
* ERRNO: 
*
* NOTES:
*
* EXAMPLES:  					 
*
* SEE ALSO: 
*
*/
LWORD 
mppCmn_SynchronousPRIo(MPP_HANDLE DeviceVertex,
                       BYTE *cdb, BYTE cdblen, 
                       void *BufferAddress, 
                       LWORD BufferLength, 
                       TINY Direction,
                       BOOL enableCC,
                       SenseData_t * SenseData)
{
	LWORD SenseLengthReturned, DataLengthXfered, mppStatus, retry;
	LWORD retryCount;
	MPP_TIME StartTime;
	BYTE SelectionTimeoutRetryCount = 0;
	BYTE CommandTimeoutRetryCount = 0;
	BYTE UaRetryCount = 0;

	
	MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
      "mppCmn_SynchronousPRIo:  called with vertex 0x%x, cdb 0x%lx, cdblen %d, BufferAddress 0x%lx, BufferLength 0x%x\n",
	    DeviceVertex, cdb, cdblen, BufferAddress, BufferLength));
   
   MPP_GETTIME(StartTime); 
   bzero(SenseData,sizeof(SenseData_t));

	for( retryCount = mpp_SyncRetryCount; retryCount > 0;  ) 
   {
    	MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppCmn_SynchronousPRIo: calling mpp_SysdepSynchronousIo, vertex 0x%lx, retryCount %d\n",
		      DeviceVertex, retryCount));

		mppStatus = mpp_SysInterface->mpp_SysdepSynchronousIo( DeviceVertex, cdb, cdblen, 
						BufferAddress, BufferLength, SenseData, sizeof(SenseData_t), 
			Direction, &SenseLengthReturned, &DataLengthXfered);

      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
         "mppCmn_SynchronousPRIo: mpp_SysdepSynchronousIo returned status %d\n", mppStatus));
      
      if(mppStatus == MPP_SCSI_RESERVATION)
      {
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "Exiting mppCmn_SynchronousPRIo() with MPPCMN_PRTN_RESERVATION_CONFLICT\n"));
         return MPPCMN_PRTN_RESERVATION_CONFLICT;
      }

      /*
      * If the controller is in 06/8b/02 or 02/04/01 condtion or SCSI BUSY status, and 
      * endbleCC is true, this function will return MPPCMN_PRTN_CHECK_CONDITION
      */
      if(mppStatus == MPP_CHECK_CONDITION && enableCC == TRUE)
      {
         MPPCMN_DUMP_SENSEDATA(SenseData);

         switch( SenseData->Sense_Key ) 
         {
            case SENSE_KEY_NOT_READY:
               if( SenseLengthReturned < 12 ) 
               {
                  MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 400, "SynchPR IO: Not ready but no ASC/ASCQ\n")); 
                  break;
               }
               else
               {
                  MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 401, "SynchPR IO: Not ready:ASC/ASCQ 0x%x/0x%x\n",
                     SenseData->ASC, SenseData->ASCQ));  
                  if( (SenseData->ASC == ASC_NOT_READY) && (SenseData->ASCQ == ASCQ_BECOMING_READY) ) 
                  {
                     return(MPPCMN_PRTN_CHECK_CONDITION);
                  }
               }
               break;
            case SENSE_KEY_UNIT_ATTENTION:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4, 
                  "mppCmn_SynchronousPRIo: UA ASC/ASCQ 0x%x/0x%x\n", 
                  SenseData->ASC, SenseData->ASCQ));
               if( (SenseLengthReturned > 12) &&
                   (SenseData->ASC == ASC_QUIESCENCE) && (SenseData->ASCQ == ASCQ_QUIESCENCE_IN_PRGS) ) 
               {
                  return(MPPCMN_PRTN_CHECK_CONDITION);
               }
               break;
            case SENSE_KEY_ILLEGAL_REQUEST:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
                  "mppCmn_SynchronousPRIo: Illegal Request ASC/ASCQ 0x%x/0x%x\n",
                  SenseData->ASC, SenseData->ASCQ));
               if (cdb[0] == SCSI_ENGENIO_BUS_RESET)
               {
                 if((SenseLengthReturned > 12) && (SenseData->ASC == ASC_INVALID_COMMAND &&
               	     SenseData->ASCQ == ASCQ_INVALID_COMMAND))
                 {
                   return(MPPCMN_PRTN_COMMAND_NOT_SUPPORTED);
                 }
               }
               break;
            default:
               break;
         }
      }
      else if(mppStatus == MPP_SCSI_BUSY && enableCC == TRUE)
      {
         MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 402, "SynchPR IO: SCSI BUSY\n"));
         return(MPPCMN_PRTN_CHECK_CONDITION);
      }
      else if(mppStatus == MPP_CHECK_CONDITION && enableCC == FALSE && 
                SenseData->Sense_Key == SENSE_KEY_ILLEGAL_REQUEST)
      {

           MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
              "mppCmn_SynchronousPRIo: Illegal Request ASC/ASCQ 0x%x/0x%x\n",
              SenseData->ASC, SenseData->ASCQ));
           if (cdb[0] == SCSI_ENGENIO_BUS_RESET)
           {
               if((SenseLengthReturned > 12) && (SenseData->ASC == ASC_INVALID_COMMAND &&
                     SenseData->ASCQ == ASCQ_INVALID_COMMAND))
               {
                   return(MPPCMN_PRTN_COMMAND_NOT_SUPPORTED);
               }
           }
      }


      retry = mpp_AnalyseSyncError(
         mppStatus, SenseData, 
         SenseLengthReturned, DataLengthXfered, 
         StartTime, &SelectionTimeoutRetryCount,
         &CommandTimeoutRetryCount, 
         &UaRetryCount );
      

		if( (retry == MPP_NO_ERROR) || (retry == MPP_RETURN_ERROR) ) 
      {
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "Exiting mppCmn_SynchronousPRIo()\n"));
         return(retry == MPP_NO_ERROR ? MPPCMN_PRTN_GOOD_STATUS: MPPCMN_PRTN_RETURN_ERROR);
		}
      
		if( retry == MPP_RETRY_DEC_COUNT )
			--retryCount;
	}


   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
      "Exiting mppCmn_SynchronousPRIo()\n"));
	return(MPPCMN_PRTN_RETURN_ERROR);
	
}


LWORD
mppCmn_SelectPRPath(
	RdacDeviceInformation_t *RdacInfo,
	LWORD ControllerIndex,
	LWORD Lun,
	MPP_HANDLE *DeviceVertex)
{
	LWORD	stateIndex;
	LWORD	pathIndex;

	*DeviceVertex = NULL;
    for(pathIndex = 0; pathIndex < mpp_MaxPathsPerController; pathIndex++)
	{
		stateIndex = RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[pathIndex].PathStateIndex;
		if( RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[pathIndex].Present &&
		   (RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL ||
		    RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
		    RdacInfo->ControllerInfo[ControllerIndex]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL_CHECKING) && 
			RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[pathIndex].LunPathDevice &&
			!RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[pathIndex].PathRemoveState)
		{
				*DeviceVertex = RdacInfo->RdacLuns[Lun].LunControllers[ControllerIndex]->LunPaths[pathIndex].LunPathDevice;
				return(MPPCMN_PRTN_GOOD_STATUS);
		}
	}

	return(MPPCMN_PRTN_RETURN_ERROR);
}


/*TTTTTTTTTTTT*/
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_GenReservationKey
* SUMMARY:  <one-line description>
* SCOPE:    public | private | local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function will perform the operations as specified in the feature design
   document. This function takes "uniqueID" as input and generate a 8 byte 
   long reservation key which is returned to the "key".The 4-byte mpp signature 
   and "mpp key" are predefined. The "hash function" is the mppCmn_GenHashKey() 
   function. The "symmetric key encoding" function is the mppCmn_KeyEncode() function.
  @uniqueID  a 8-bytes long unique ID of this host
  @key       a 8-bytes long buffer. The generated reservation key will be returned 
      through the key argument
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
VOID  mppCmn_GenReservationKey(BYTE *uniqueID, BYTE *key)
{
   BYTE     convertedId[4];
   BYTE     clearKey[8];


   mppCmn_GenHashKey(uniqueID,convertedId);
   memcpy(clearKey, convertedId, 4);
   memcpy(&(clearKey[4]), MPP_SIGNATURE,4);
   mppCmn_KeyEncode(clearKey,key);
   return ;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_IsKeyForThisHost
* SUMMARY:  Verify whether or not a reservation key belongs to this host node.
* SCOPE:    public | private | local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function will perform the operations of verifying reservation key as specified 
   in Linux MPP virtual HBA driver feature design document. This function takes 
   a 8 byte  "uniqueID" and a 8 byte "reservation key" as input arguments and 
   returns "TRUE" or "FALSE" that indicates whether or not the reservation key 
   belongs to this host.
   The 4 byte mpp signature and "mpp key" are predefined.
   The "hash function" is the mppCmn_GenHashKey() function.
   The "symmetric key encoding" function is the mppCmn_KeyDecode() function.
   @uniqueID      The unique ID of this node. 8-bytes long
   @reservationKey The reservation key to be verified. 8-bytes long.
*
* RESTRICTIONS:
* 
* RETURNS:"TRUE" or "FALSE" that indicates whether or not the reservation key 
   belongs to this host.
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
TINY  mppCmn_IsKeyForThisHost(BYTE *uniqueID, BYTE * reservationKey)
{
   BYTE  hostId[4];
   BYTE  clearKey[8];
   int i;

   mppCmn_GenHashKey(uniqueID,hostId);
   mppCmn_KeyDecode(reservationKey, clearKey);
   for(i=0; i<4; i++)
   {
      if(hostId[i] != clearKey[i])
      {
         return 0;
      }
   }
   return 1;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_IsMppKey
* SUMMARY:  Verify whether or not a reservation key belongs to MPP driver
            translation layer.
* SCOPE:    public | private | local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
   @uniqueID      The unique ID of this node. 8-bytes long
   @reservationKey The reservation key to be verified. 8-bytes long.
*
*
* DESCRIPTION:
   This function will perform the operations of verifying whether or 
   not a reservation key belongs to MPP driver as specified in the Linux
   MPP virtual HBA driver feature design document. This function takes 
   a 8 byte  "uniqueID" and a 8 byte "reservation key" as input arguments 
   and returns "TRUE" or "FALSE" that indicates whether or not the reservation
   key belongs to the MPP driver.
   The 4 byte mpp signature and "mpp key" are predefined.
   The "hash function" is the mppCmn_GenHashKey() function.
   The "symmetric key encoding" function is the mppCmn_KeyEncode() function.
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
TINY  mppCmn_IsMppKey(BYTE * reservationKey)
{
   BYTE  clearKey[8];
   int i;

   mppCmn_KeyDecode(reservationKey, clearKey);
   for(i=0; i<4; i++)
   {
      if(MPP_SIGNATURE[i] != clearKey[i+4])
      {
         return 0;
      }
   }
   return 1;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_GenHashKey
* SUMMARY:  A hash function that takes a 8-bytes unique ID and outputs a 4-bytes
            hash key.
* SCOPE:    public | private | local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function is one-way hash function. It takes a 8-byte "uniqueID" as input
     and generate a 4-byte hash key. The hash key is returned to called 
   via the "hashKey".

  @uniqueID    A 8-bytes unique ID (input)
  @hashKey     A 4-bytes hashKey of the 8 bytes unique ID (output)
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
/*
--------------------------------------------------------------------
hash() -- hash a variable-length key into a 32-bit value
  k       : the key (the unaligned variable-length array of bytes)
  len     : the length of the key, counting by bytes
  initval : can be any 4-byte value
Returns a 32-bit value.  Every bit of the key affects every bit of
the return value.  Every 1-bit and 2-bit delta achieves avalanche.
About 6*len+35 instructions.

The best hash table sizes are powers of 2.  There is no need to do
mod a prime (mod is sooo slow!).  If you need less than 32 bits,
use a bitmask.  For example, if you need only 10 bits, do
  h = (h & hashmask(10));
In which case, the hash table should have hashsize(10) elements.

If you are hashing n strings (unsigned char **)k, do it like this:
  for (i=0, h=0; i<n; ++i) h = hash( k[i], len[i], h);

By Bob Jenkins, 1996.  bob_jenkins@burtleburtle.net.  You may use this
code any way you wish, private, educational, or commercial.  It's free.

See http://burtleburtle.net/bob/hash/evahash.html
Use for hash table lookup, or anything where one collision in 2^^32 is
acceptable.  Do NOT use for cryptographic purposes.
--------------------------------------------------------------------
*/
VOID  mppCmn_GenHashKey(BYTE *uniqueID, BYTE *hashKey)
{
   unsigned char        *k = uniqueID;
   unsigned int         length = 8;
   unsigned int         initval = 0xfad1ec98;
   unsigned int         a,b,c,len;

   /* Set up the internal state */
   len = length;
   a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
   c = initval;         /* the previous hash value */

   /*------------------------------------- handle the last 11 bytes */
   c += length;
   switch(len)              /* all the case statements fall through */
   {
      /* the first byte of c is reserved for the length */
      case 8 : b+=((unsigned int)k[7]<<24);
      case 7 : b+=((unsigned int)k[6]<<16);
      case 6 : b+=((unsigned int)k[5]<<8);
      case 5 : b+=k[4];
      case 4 : a+=((unsigned int)k[3]<<24);
      case 3 : a+=((unsigned int)k[2]<<16);
      case 2 : a+=((unsigned int)k[1]<<8);
      case 1 : a+=k[0];
     /* case 0: nothing left to add */
   }

   mix(a,b,c);
   //printf("a 0x%016x b 0x%016x c 0x%016x\n",a,b,c);
   /*-------------------------------------------- report the result */

   /* c is the value to be returned
   *  copy c to a 4 byte char array
   */
   hashKey[0] = (unsigned char) (c >>24) & 0x000000FF;
   hashKey[1] = (unsigned char) (c >>16) & 0x000000FF;
   hashKey[2] = (unsigned char) (c >>8)  & 0x000000FF;
   hashKey[3] = (unsigned char) c        & 0x000000FF;
   return;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_KeyEncode
* SUMMARY:  Encode a 8 bytes key.
* SCOPE:    public 
*
* DESCRIPTION:
   This function takes a 8-byte "clearKey" as input and encodes the "clearKey" to
   a 8-byte "cipherKey". The "cipher key" is returned to the called via the 
   "cipherKey" argument. 
   @clearKey      An 8-bytes clear key. (input)
   @cipherKey     An 8-bytes cipher key (output)  
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
VOID  mppCmn_KeyEncode(BYTE *clearKey, BYTE *cipherKey)
{
   unsigned char mixedClearKey[8];
   unsigned int  first4;
   unsigned int  second4;
   unsigned int  temp;
   unsigned int  mppKey = 0x9e3779b9;
   mixedClearKey[0] = clearKey[0];
   mixedClearKey[1] = clearKey[4];
   mixedClearKey[2] = clearKey[1];
   mixedClearKey[3] = clearKey[5];
   mixedClearKey[4] = clearKey[2];
   mixedClearKey[5] = clearKey[6];
   mixedClearKey[6] = clearKey[3];
   mixedClearKey[7] = clearKey[7];
   
   first4 = 0;
   temp = (unsigned int) mixedClearKey[0];
   temp = temp << 24;
   first4 += temp;

   temp = (unsigned int) mixedClearKey[1];
   temp = temp << 16;
   first4 += temp;

   temp = (unsigned int) mixedClearKey[2];
   temp = temp << 8;
   first4 += temp;

   temp = (unsigned int) mixedClearKey[3];
   first4 += temp;


   second4 = 0;
   temp = (unsigned int) mixedClearKey[4];
   temp = temp << 24;
   second4 += temp;

   temp = (unsigned int) mixedClearKey[5];
   temp = temp << 16;
   second4 += temp;

   temp = (unsigned int) mixedClearKey[6];
   temp = temp << 8;
   second4 += temp;

   temp = (unsigned int) mixedClearKey[7];
   second4 += temp;

   first4 = first4 ^ mppKey;
   second4 = second4 ^ mppKey;

   cipherKey[0] = (unsigned char) (first4 >>24) & 0x000000FF;
   cipherKey[1] = (unsigned char) (first4 >>16) & 0x000000FF;
   cipherKey[2] = (unsigned char) (first4 >>8)  & 0x000000FF;
   cipherKey[3] = (unsigned char) first4        & 0x000000FF;

   cipherKey[4] = (unsigned char) (second4 >>24) & 0x000000FF;
   cipherKey[5] = (unsigned char) (second4 >>16) & 0x000000FF;
   cipherKey[6] = (unsigned char) (second4 >>8)  & 0x000000FF;
   cipherKey[7] = (unsigned char) second4        & 0x000000FF;
  
   return;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_KeyDecode
* SUMMARY:  Takes an 8-bytes cipher key and decodes it to an 8-bytes clear key
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function takes a 8-byte "cipherKey" as input and decodes the "cipherKey" 
   to a 8-byte "clearKey". The "clear key" is returned to the called via the 
   "clearKey" argument. 
     
   @cipherKey     An 8-bytes cipher key (input) 
   @clearKey      An 8-bytes clear key. (output)
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
VOID  mppCmn_KeyDecode(BYTE *cipherKey, BYTE *clearKey)
{
   unsigned char mixedClearKey[8];
   unsigned int  first4;
   unsigned int  second4;
   unsigned int  temp;
   unsigned int  mppKey = 0x9e3779b9;
   
   
   first4 = 0;
   temp = (unsigned int) cipherKey[0];
   temp = temp << 24;
   first4 += temp;

   temp = (unsigned int) cipherKey[1];
   temp = temp << 16;
   first4 += temp;
   temp = (unsigned int) cipherKey[2];
   temp = temp << 8;
   first4 += temp;

   temp = (unsigned int) cipherKey[3];
   first4 += temp;


   second4 = 0;
   temp = (unsigned int) cipherKey[4];
   temp = temp << 24;
   second4 += temp;

   temp = (unsigned int) cipherKey[5];
   temp = temp << 16;
   second4 += temp;

   temp = (unsigned int) cipherKey[6];
   temp = temp << 8;
   second4 += temp;

   temp = (unsigned int) cipherKey[7];
   second4 += temp;

   first4 = first4 ^ mppKey;
   second4 = second4 ^ mppKey;

   mixedClearKey[0] = (unsigned char) (first4 >>24) & 0x000000FF;
   mixedClearKey[1] = (unsigned char) (first4 >>16) & 0x000000FF;
   mixedClearKey[2] = (unsigned char) (first4 >>8)  & 0x000000FF;
   mixedClearKey[3] = (unsigned char) first4        & 0x000000FF;

   mixedClearKey[4] = (unsigned char) (second4 >>24) & 0x000000FF;
   mixedClearKey[5] = (unsigned char) (second4 >>16) & 0x000000FF;
   mixedClearKey[6] = (unsigned char) (second4 >>8)  & 0x000000FF;
   mixedClearKey[7] = (unsigned char) second4        & 0x000000FF;
  
   clearKey[0] = mixedClearKey[0];
   clearKey[4] = mixedClearKey[1];
   clearKey[1] = mixedClearKey[2];
   clearKey[5] = mixedClearKey[3];
   clearKey[2] = mixedClearKey[4];
   clearKey[6] = mixedClearKey[5];
   clearKey[3] = mixedClearKey[6];
   clearKey[7] = mixedClearKey[7];
   return;
}
/*TTTTTTTTTTTT*/

#ifdef MPP_DEBUG
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_dumpPRINReadResvParamData
* SUMMARY:  Dump the content of a PRINReadResevParamData_t object
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function dumps the content of a PRINReadResevParamData_t object. It is only 
   compiled and available when MPP_DEBUG is on
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static VOID mppCmn_dumpPRINReadResvParamData(PRINReadResevParamData_t *rsParamData)
{
   LWORD           i,j,size;
   LINT           position;
   LINT            len;
   BYTE           *buffer;
   TINY           print_buffer[1024];

   bzero(&print_buffer[0],1024);
   position = 0;


   len = sprintf(&print_buffer[position],"PRINReadResevParamData_t dump\n");
   size = sizeof(PRINReadResevParamData_t);
   /*
   * dump the buffer 
   */
   buffer = (BYTE *) rsParamData;
   for(i = 0; i < size; i++)
   {
      position += len;
      len = sprintf(&print_buffer[position],"%02x ",buffer[i]);
   }

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));

   bzero(&print_buffer[0],1024);
   position = 0;
   len      = 0;

   len = sprintf(&print_buffer[position],"\n");

   position += len;
   len = sprintf(&print_buffer[position],
      "PRgenetration %d \n",MPP_GET_LWORD_FROM_4BYTES(rsParamData->PRgeneration));

   position += len;
   len = sprintf(&print_buffer[position],
      "Additional_len %d\n",MPP_GET_LWORD_FROM_4BYTES(rsParamData->Additional_Len));

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));

   
   for(i = 0; i <(MPP_GET_LWORD_FROM_4BYTES(rsParamData->Additional_Len))/(sizeof(ReservationDescriptor_t)); i++)
   {
      bzero(&print_buffer[0],1024);
      position = 0;

      len = sprintf(&print_buffer[position],"reservation descriptor [%d]\n",i);

      position += len;
      len = sprintf(&print_buffer[position],
         "\t scop %d\n",rsParamData->Reservation_Descriptor[i].Scope);

      position += len;
      len = sprintf(&print_buffer[position],
         "\t Type 0x%x\n",rsParamData->Reservation_Descriptor[i].Type);

      position += len;
      len = sprintf(&print_buffer[position],"\t reservation key:");

      for(j = 0; j <8; j++)
      {
         position += len;
         len = sprintf(&print_buffer[position],
            "%02x ",rsParamData->Reservation_Descriptor[i].Reservation_Key[j]);
      }
      position += len;
      len = sprintf(&print_buffer[position],"\n");
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));
   }
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_dumpPRINReadKeysParamData
* SUMMARY:  Dump the content of a PRINReadKeysParamData_t object
* SCOPE:    public 
 
*
* DESCRIPTION:
   This function dumps the content of a PRINReadKeysParamData_t object. It is only 
   compiled and available when MPP_DEBUG is on
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static VOID mppCmn_dumpPRINReadKeysParamData(PRINReadKeysParamData_t *rkParamData)
{
   LWORD          i,j,size;
   BYTE           *buffer;
   LINT           position;
   LINT            len;
   TINY           print_buffer[1024];

   
   bzero(&print_buffer[0],1024);
   position = 0;

   len = sprintf(&print_buffer[position],"PRINReadKeysParamData_t dump\n");

   size = sizeof(PRINReadKeysParamData_t);
   /*
   * dump the buffer 
   */
   buffer = (BYTE *) rkParamData;
   for(i = 0; i < size; i++)
   {
       position += len;
       len = sprintf(&print_buffer[position],"%02x ",buffer[i]);
   }
   position += len;
   len = sprintf(&print_buffer[position],"\n");
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));


   bzero(&print_buffer[0],1024);
   position = 0;
   len = sprintf(&print_buffer[position],
         "PRgenetration %d \n",MPP_GET_LWORD_FROM_4BYTES(rkParamData->PRgeneration));
   position += len;
   len = sprintf(&print_buffer[position],
      "Additional_len %d\n",MPP_GET_LWORD_FROM_4BYTES(rkParamData->Additional_Len));
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));

   bzero(&print_buffer[0],1024);
   position = 0;
   len = 0;
   for(i = 0; i < (MPP_GET_LWORD_FROM_4BYTES(rkParamData->Additional_Len)/(sizeof(ReservationKey))); i++)
   {
      bzero(&print_buffer[0],1024);
      position = 0;
      len = 0;
      len = sprintf(&print_buffer[position],"\t reservation key: [%d]",i);
      for(j = 0; j <8; j++)
      {
         position += len;
         len = sprintf(&print_buffer[position],"%02x ",rkParamData->Reservation_Key[i][j]);
      }
      position += len;
      len = sprintf(&print_buffer[position],"\n");
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));
   }

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppCmn_dumpSenseData
* SUMMARY:  Dump the content of a SenseData_t object
* SCOPE:    public 
*
*
* DESCRIPTION:
   This function dumps the content of a SenseData_t object. It is only 
   compiled and available when MPP_DEBUG is on
*
* RESTRICTIONS:
* 
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
static VOID mppCmn_dumpSenseData(SenseData_t *senseData)
{
   LINT           position;
   LINT           len;
   TINY           print_buffer[1024];

   
   bzero(&print_buffer[0],1024);
   position = 0;

   len = sprintf(&print_buffer[position],"Sense Data Dump:\n");
   position += len;
   len = sprintf(&print_buffer[position],
      "\t Response Code 0x%02x\n",senseData->Error_Code);
   position += len;
   len = sprintf(&print_buffer[position],
      "\t Sense Key 0x%02x\n",senseData->Sense_Key);
   position += len;
   len = sprintf(&print_buffer[position],
      "\t ASC 0x%02x\n",senseData->ASC);
   position += len;
   len = sprintf(&print_buffer[position],
      "\t ASCQ 0x%02x\n",senseData->ASCQ);

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
               "%s", print_buffer));  
}

#endif
