/*******************************************************************************
* Copyright 2004-2008 LSI Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppLnx26p_vhbatask.c
SUMMARY         %description%
VERSION         %version: 17 %
UPDATE DATE     %date_modified: Wed Jul 23 14:07:08 2008 %
PROGRAMMER      %created_by:    harshi %

DESCRIPTION:
   This source file contains function that are related to Linux MPP virtual HBA
   driver's kernel thread management and tasklet management. The kernel threads
   include:
   1. The fail back scan kernel thread. The fail back scan kernel thread wakes-up
   in a specified time period and performs fail back scan function that is provided
   by the MPP common code
   2.The sync IO kernel thread. The sync IO kernel thread performs sync IOs that
   are generated during the virtual HBA's IO routing. Currently, the sync IOs are
   those IOs related to the SCSI-2 reserve/release translation layer

   The tasklet is used for performing delayed IOs and dispatching proxy command to
   the Linux SCSI middle level.


INCLUDE FILES:

NOTES: Error number range for this file is 450 to 479

RESTRICTIONS:

SEE ALSO:

REFERENCE:
  349-1033500 Linux MPP Driver Feature Architecture Specification
  349-1034610 Linux MPP Virtual HBA Driver Feature Design Document
  349-1034750 Linux MPP Upper Level Driver Feature Design Document
  349-1034730 Linux MPP Utility Feature Design Document
  349-1034620 Linux MPP Installation Feature Design Document
  349-1043860 Linux MPP driver for Linux 2.6 Kernel
  349-1046490 On-The-Fly Path Validation

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/

#define __SRCmppLnx26p_vhbatask_c

/***  INCLUDES  ***/
/***  Linux kernel INCLUDES  ***/
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/timer.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/genhd.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_eh.h>

#include <asm/io.h>

#include <linux/blkdev.h>


#include <linux/stat.h>

/***  MPP commomn API INCLUDES  ***/
#include "MPP_Common.h"
#include "MPP_RdacInfo.h"
#include "MPP_ProtoTypes.h"
#include "MPP_Sysdep.h"
#include "mppLnx_version.h"
#include "mppCmn_s2tos3.h"

/***  Linux MPP Virtual HBA INCLUDES  ***/
#include "mppLnx26p_vhba.h"
#include "mppLnx26p_shared.h"



/***  MACRO DEFINITIONS  ***/
#define MPPLNX_FAILBACK_PROC_NAME           "mppFailback"
#define MPPLNX_PATHVALIDATE_PROC_NAME       "mppPathValidate"
#define MPPLNX_SYNCIO_WORKER_PROC_NAME      "mppWorker"
#define BUFFER_LENGTH                       24
#define MPPLNX_REBALANCE_DELAY_TIME          600 /* 10 minutes */


/***  TYPE DEFINITIONS  ***/


/***  LOCALS  ***/
extern void (*middle_level_timeoutfunc)(struct scsi_cmnd *);
extern mppLnx_dpcQueueContext_t   mppLnxDpcQueueContext;
extern spinlock_t taskQueue_lock;
extern mppLnx_failbackScanContext_t mppLnxFailbackScanContext;
extern mppLnx_pathvalidateContext_t mppLnxPathValidateContext;
extern mppLnx_workerQueueContext_t mppLnxWorkerQueueContextContext;
extern struct workqueue_struct *mppLnx_wq;

/***  PROCEDURES  ***/
void mppLnx_scanOfflinedPath(void);


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_pathvalidate_prcheck
* SUMMARY:  routine to check the PR Registration check
* SCOPE:    public
*
*
* DESCRIPTION:
   This function is called from the pathvalidate kernel thread function.
      This thread performs the PR Registration checks for the LUNS when path validation command
      is completed.
*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void
mppLnx_pathvalidate_prcheck( mppLnx_ProxyRequestEntry_t	*preq)
{
   RdacDeviceInformation_t	*RdacInfo=NULL;
   BYTE				Controller;
   BYTE				Path;
   LWORD			reservationStatus;
   LWORD			lun;
   LINT              		LockLevel;
   BOOL				flushDeferredQueue=FALSE;
   WORD				host,target;
   MPP_HANDLE			pSDp;

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_pathvalidate_prcheck()\n"));

   if(unlikely(preq == NULL))
   {
      MPPLNX_VHBA_BUG();
   }

   RdacInfo     = preq->rdacInfo;
   Controller	= preq->controller;
   Path			= preq->path;

   // Physical Host
   host 		= preq->phost->host_no; 

   // physical Target
   target 		= preq->pdev->id; 

	for(lun = 0; lun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lun)
	{
		if(!RdacInfo->RdacLuns[lun].Present
		|| !RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice
		||   RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].PathRemoveState)
		{
			continue;
		}

		/* Perform the PR Registration check only if the DEVICE was previously in MPPCMN_FAILED_CHECKING */
		if(RdacInfo->RdacLuns[lun].NeedsReservationCheck)
		{
			pSDp = RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice;
			reservationStatus = mppCmn_CheckPRRegistration(RdacInfo, pSDp, lun);
			if(	(reservationStatus == MPPCMN_PRTN_RETURN_ERROR) ||
				(reservationStatus == MPPCMN_PRTN_CHECK_CONDITION))
			{
				MPP_DEBUGPRINT((0, "Error in checking PR registration\n"));
			}
			else
			{
				// Registration checked.  Restore device PathState
				MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
					"DeviceValidation: %s:%d:%d:%d Successful, Check PR Registration Success.\n",
					RdacInfo->ModuleName, Controller, Path, lun));
				MPP_LOCK(RdacInfo->Lock, LockLevel);
				MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo,Controller,Path, lun, MPPCMN_OPTIMAL);
				MPP_UNLOCK(RdacInfo->Lock, LockLevel);

				/* We can restart the queued requests as the virtual device is optimal */
				flushDeferredQueue = TRUE;
			}

		}
		else
		{
			MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
					"DeviceValidation: %s:%d:%d:%d Successful, Check PR Registration not required.\n",
					RdacInfo->ModuleName, Controller, Path, lun));

		}
	} /* for */

	if( flushDeferredQueue == TRUE )
	{

		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3, "State-0x%x\n",
			MPPCMN_OPTIMAL));

		// Flush the deferred commands
		mppLnx_flushdeferredqueue(host, target);

	}

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_pathvalidate_prcheck()\n"));

}

/************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_pathvalidate_regcheck
* SUMMARY:  routine to check the PR Registration and if not, register
* SCOPE:    public
*
*
* DESCRIPTION:
   This function is called from the pathvalidate kernel thread function.
      This thread performs the PR Registration only (no Reservation) for the LUNS when path 
      validation command is completed.
*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

void
mppLnx_pathvalidate_regcheck(mppLnx_ProxyRequestEntry_t *preq)
{

   RdacDeviceInformation_t     *RdacInfo=NULL;
   mppLnx_PDeviceEntry_t       *pde = NULL;
   mppLnx_VDeviceEntry_t       *vde = NULL;
   SenseData_t                  sensedata;
   PROUTParamData_t            *proutParamData;
   BYTE                        *cdb, cdblen=10;
   LWORD                        BufferLength;  /*the length of PROUTParamData_t */
   BYTE                         Controller;
   BYTE                         Path;
   LWORD                        lun;
   LWORD                        status;
   LINT                         LockLevel;
   BOOL                         enableCC = FALSE;
   BOOL                         flushDeferredQueue = FALSE;
   WORD                         host,target;
   MPP_HANDLE                   pSDp;

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "Entering mppLnx_pathvalidate_regcheck()\n"));

   if(unlikely(preq == NULL))
   {
      MPPLNX_VHBA_BUG();
   }

   RdacInfo     = preq->rdacInfo;
   Controller   = preq->controller;
   Path         = preq->path;

   // Physical Host
   host         = preq->phost->host_no;

   // physical Target
   target       = preq->pdev->id;

	for(lun = 0; lun < mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++lun)
        {
           if(!RdacInfo->RdacLuns[lun].Present
              || !RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice
              || RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].PathRemoveState)
            {
               continue;
            }

            vde = RdacInfo->RdacLuns[lun].PlatformPrivateData;
            pde = RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].PlatformPrivateData;

            if(RdacInfo->RdacLuns[lun].UTMLun == TRUE)
               continue;

            if(vde->PRKeyValid == FALSE)
               continue;

            //Since this LUN is already Registered, we don't need to Register again. Go to the next lun
             if(pde->PRKeyRegistered == TRUE)
                continue;

            /* setup (allocate mem) scsi command */
            cdb = (BYTE *) MPP_INT_KMEM_ALLOC(cdblen);

            /* set up prout command's buffer */
            BufferLength = sizeof(PROUTParamData_t);
            proutParamData = (PROUTParamData_t *) MPP_INT_KMEM_ALLOC(BufferLength);

            if( !proutParamData){
               MPP_FREE(proutParamData, BufferLength);
               MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 450,
                  "Unable to allocate memory\n"));
            }

            if(!cdb){
               MPP_FREE(proutParamData, BufferLength);
               MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 451,
                  "Unable to allocate memory\n"));
            }   
                                
            /* setup the whole command before sending */
            *cdb = SCSI_PR_OUT;
            *(cdb+1) = (BYTE) vde->srvActCode;
            //(cdb+7) is 0x00 by default
            *(cdb+8) = (BYTE) (24 &0xFF);

            /* Copy the Srv_action_key to the buffer */
            bcopy(&(vde->PRKey), proutParamData->Srv_Act_Reservation_Key , 8);

            pSDp = RdacInfo->RdacLuns[lun].LunControllers[Controller]->LunPaths[Path].LunPathDevice;
            status = mppCmn_SynchronousPRIo(pSDp, cdb, cdblen, proutParamData, BUFFER_LENGTH, MPPCMN_XFER_HOST_DEV, enableCC, &sensedata);

            if(status == MPPCMN_PRTN_GOOD_STATUS)
            {
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_SynchronousPRIo: Registration successful, data returned with mpp_status:%d\n",
                   status));
                   pde->PRKeyRegistered = TRUE;

               MPP_FREE(cdb, cdblen);
               MPP_FREE(proutParamData, BufferLength);

               MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
                  "DeviceValidation: %s:%d:%d:%d Successful, PR Registration Success.\n",
                   RdacInfo->ModuleName, Controller, Path, lun));
               MPP_LOCK(RdacInfo->Lock, LockLevel);
               MPPCMN_SET_DEVICE_PATH_STATE(RdacInfo,Controller,Path, lun, MPPCMN_OPTIMAL);
               MPP_UNLOCK(RdacInfo->Lock, LockLevel);

               /* We can restart the queued requests as the virtual device is optimal */
               flushDeferredQueue = TRUE;
            }
            else
            {
               //set the flag as FALSE
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_3,
                  "mppCmn_SynchronousPRIo: Registration unsuccessful, data returned with mpp_status:%d\n",
                   status));
                   pde->PRKeyRegistered = FALSE;
               MPP_FREE(cdb, cdblen);
               MPP_FREE(proutParamData, BufferLength);
            }
			
        }/* for loop */


        if( flushDeferredQueue == TRUE )
        {

           MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3, "State-0x%x\n",
              MPPCMN_OPTIMAL));

           // Flush the deferred commands
           mppLnx_flushdeferredqueue(host, target);
        }

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_3,
      "Exiting mppLnx_pathvalidate_regcheck()\n"));
}



/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_wakeup_pathvalidate_thread
* SUMMARY:  The function that wakes up the mpp virtual HBA driver pathvalidate scan thread
* SCOPE:    public
*
*
* DESCRIPTION:
* This function is called from the mpp_RemoveLun function in mppLnx26p_sysdep.c.
* The purpose of this function is to wake the path validate thread if it has
* not been active ( ie, it is sleeping ).
* RESTRICTIONS:
*
* RETURNS: void
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*
*/
void mppLnx_wakeup_pathvalidate_thread(void )
{
      if ( atomic_read(&mppLnxPathValidateContext.scan_active) == 0)
      {
   	      up(mppLnxPathValidateContext.pathvalidate_sem);
      }

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_pathvalidate_handler
* SUMMARY:  The entry function of the virtual HBA driver pathvalidate scan thread
* SCOPE:    public
*
*
* DESCRIPTION:
   This function is the pathvalidate kernel thread entry function. The input
   argument "void *data" is not used. This function performs the following operations:
      The process name of this kernel thread in the process table will be "mpp_pathvalidate".
      This kernel thread will be demonized so that it has no associated control terminal
         and its stdin, stdout and stderr are closed.
      Its parent process id be init's process id (1).
      If the virtual HBA driver is loaded as a module, this kernel thread ignores
         all signals except SIGHUP that is used for system shutdown and module unloading.
         Signal SIGHUP will cause the failback kernel thread terminated.
      This thread performs the PR Registration checks for the LUNS when path validation command
      is completed.
*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_pathvalidate_handler(void * data)
{
   mppLnx_ProxyRequestEntry_t	*preq;
   unsigned long		iflages;
   struct list_head		*listPt;
   mpp_hot_plug_event * mpp_hot_plug_info  = NULL;
   spinlock_t *temp = NULL;

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_pathvalidate_handler\n"));

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_pathvalidate_handler Locks the kernel\n"));
   /*
   * the 2.6 way of handling the kernel thread
   *  1. lock the kernel
   *  2. daemonize
   *  3. allow certain signal
   *  4. set a nice value
   *
   */
	lock_kernel();
	daemonize(MPPLNX_PATHVALIDATE_PROC_NAME);
	allow_signal(SIGHUP);
   	set_user_nice(current,-20);

	/*
	 * Set the name of this process and initialize its internal data fields
	 */
   	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      		"mppLnx_pathvalidate_handler release the kernel\n"));

	unlock_kernel();

	/*
	 * Wake up the thread that created us.
	 */
	MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_pathvalidate_handler--Wake up parent from PathValidate scan\n"));

	up(mppLnxPathValidateContext.detect_sem);

   	atomic_set(&mppLnxPathValidateContext.scan_active, 0);


	while (1) {
		/*
		 * If we get a signal, it means we are supposed to go
		 * away and die.  This typically happens if the user is
		 * trying to unload a module.
		 */
		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
         "path validate scan sleeping\n"));

      /*
       * Note - we always use down_interruptible with the semaphore
       * even if the module was loaded as part of the kernel.  The
       * reason is that down() will cause this thread to be counted
       * in the load average as a running process, and down
       * interruptible doesn't.  Given that we need to allow this
       * thread to die if the driver was loaded as a module, using
       * semaphores isn't unreasonable.
       */
      if(mppLnxPathValidateContext.isStopped)
      {
         break;
      }

      if( down_interruptible(mppLnxPathValidateContext.pathvalidate_sem))
      {
         break;
      }



      if ( atomic_read(&mppLnxPathValidateContext.scan_active) == 0)
      {
	      atomic_set(&mppLnxPathValidateContext.scan_active, 1);

	      MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
	         "PathValidate scan handler-- PathValidationScan waking up\n"));

		/* CR 139596 - Flush the workqueue from our own thread rather
		 * than in interrupt context.
		 */
		flush_workqueue(mppLnx_wq);

		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
			"PathValidate scan handler-- PathValidationScan before looping for Proxyrequest\n"));
	        /*
		 *Loop through all commands in the command queue. It breaks when the queue is empty
		 */
		while ( 1 )
		{

		      /*
			* lock the queue and pick up one cmnd entry at a time. Remove the cmnd entry
			* from the list and unlock. In this way, we don't block any new entry being
			* added to the list.
			*/
		      preq = NULL;
		      spin_lock_irqsave(&mppLnx_pathvalidationProxyRequestQ.queueLock, iflages);
		      listPt = mppLnx_pathvalidationProxyRequestQ.list.next;
		      if(listPt != &(mppLnx_pathvalidationProxyRequestQ.list))
		      {
				preq = list_entry(listPt, mppLnx_ProxyRequestEntry_t,pathvalidation_list);
				spin_unlock_irqrestore(&mppLnx_pathvalidationProxyRequestQ.queueLock, iflages);
				MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
					"PathValidate scan handler-- PathValidationScan.Path validation completion for C %d P %d.\n",
					preq->controller,preq->path));
			}
			else
			{
				spin_unlock_irqrestore(&mppLnx_pathvalidationProxyRequestQ.queueLock, iflages);
				/* No more completed path validation's in the queue */
				MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
				"PathValidate scan handler-- PathValidationScan. No more Path validation completions.\n"));
				break;
			}

			/* Remove the proxy request from the list */
			mppLnx_remove_proxyRequest_from_list(preq, MPPLNX_QUEUE_PATHVALIDATION_LIST);

                        /* call the completion handler for the path validation command */
                        //__mppLnx_validatepath_done(preq->pcmnd);

                        /* This routine is invoked only during SCSI3 PROUT */
                        mppLnx_pathvalidate_regcheck( preq );

                        /* This routine is invoked during SCSI2 to SCSI3 */
                        mppLnx_pathvalidate_prcheck( preq );

                        /* remove the proxy request element */
                        mppLnx_put_free_proxyRequest(preq);

		}

		while( ! (list_empty(&(mpp_hot_plug_list)) )) {

			spin_lock_irqsave(&mpp_hot_plug_spin_lock, iflages);
			mpp_hot_plug_info = list_entry(mpp_hot_plug_list.next, mpp_hot_plug_event, list );
      			spin_unlock_irqrestore(&mpp_hot_plug_spin_lock, iflages);
			//MPP_DEBUGPRINT((MPP_ATTACH_DEBUG+MPP_DEBUG_LEVEL_3, "mppLnx_pathvalidate - handled hot plug event \n"));
			if (mpp_hot_plug_info->event_type == DEL_DEVICE) {

				if (mpp_hot_plug_info->device == NULL) {
					spin_lock_irqsave(mpp_hot_plug_info->hot_plug_lock, iflages);
					temp = mpp_hot_plug_info->hot_plug_lock;
					list_del(&mpp_hot_plug_info->list);
					spin_unlock_irqrestore(temp, iflages);
					kfree(mpp_hot_plug_info);
					continue;
				}
				scsi_device_set_state(mpp_hot_plug_info->device, SDEV_OFFLINE);

				MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
				"mppLnx_pathvalidate - set scsi device state to offline H%dT%dL%d\n", mpp_hot_plug_info->device->host->host_no,mpp_hot_plug_info->device->id,  mpp_hot_plug_info->device->lun));
				scsi_remove_device(mpp_hot_plug_info->device);
				//scsi_device_put(mpp_hot_plug_info->device);

				spin_lock_irqsave(mpp_hot_plug_info->hot_plug_lock, iflages);
				temp = mpp_hot_plug_info->hot_plug_lock;
				list_del(&mpp_hot_plug_info->list);
				spin_unlock_irqrestore(temp, iflages);
				kfree(mpp_hot_plug_info);

			}
		}

		atomic_set(&mppLnxPathValidateContext.scan_active, 0);

		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
		"PathValidate scan handler-- performed PathValidationScan\n"));

	}else
	{
		MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
		"PathValidate Scan is still running\n"));
	}
   } /*while (1) */

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      "PathValidate scan exiting\n"));

   /*
    * Make sure that nobody tries to wake us up again.
    */
   mppLnxPathValidateContext.pathvalidate_sem       = NULL;
   mppLnxPathValidateContext.pathvalidate_pid = -1;
   atomic_set(&mppLnxPathValidateContext.scan_active, 0);

   /*
    * If anyone is waiting for us to exit (i.e. someone trying to unload
    * a driver), then wake up that process to let them know we are on
    * the way out the door.  This may be overkill - I *think* that we
    * could probably just unload the driver and send the signal, and when
    * the error handling thread wakes up that it would just exit without
    * needing to touch any memory associated with the driver itself.
    */
   if (mppLnxPathValidateContext.isStopped)
   {
      if(mppLnxPathValidateContext.detect_sem)
      up(mppLnxPathValidateContext.detect_sem);
      mppLnxPathValidateContext.detect_sem = NULL;
   }

   mppLnxPathValidateContext.isStopped = 1;

   MPP_DEBUGPRINT((MPP_PATH_VALIDATION_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_pathvalidate_handler\n"));

   return 0;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_failback_handler
* SUMMARY:  The entry function of the virtual HBA driver failback scan thread
* SCOPE:    public
*
*
* DESCRIPTION:
   This function is the failback rescan kernel thread entry function. The input
   argument "void *data" is not used. This function performs the following operations:
      The process name of this kernel thread in the process table will be "mpp_failback".
      This kernel thread will be demonized so that it has no associated control terminal
         and its stdin, stdout and stderr are closed.
      Its parent process id be init's process id (1).
      If the virtual HBA driver is loaded as a module, this kernel thread ignores
         all signals except SIGHUP that is used for system shutdown and module unloading.
         Signal SIGHUP will cause the failback kernel thread terminated.
      Starts an infinity loop and uses Linux kernel timer and Linux semaphore to schedule
         failback rescan.
*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_failback_handler(void * data)
{
	unsigned long        		currentTime;

	MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
				"Entering mppLnx_failback_handler\n"));

	MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
				"mppLnx_failback_handler Locks the kernel\n"));
	/*
	 * the 2.6 way of handling the kernel thread
	 *  1. lock the kernel
	 *  2. daemonize
	 *  3. allow certain signal
	 *  4. set a nice value
	 *
	 */
	lock_kernel();
	daemonize(MPPLNX_FAILBACK_PROC_NAME);
	allow_signal(SIGHUP);
	set_user_nice(current,-20);

	/*
	 * Set the name of this process and initialize its internal data fields
	 */
	/*
	 *set the time-out function
	 */
	mppLnxFailbackScanContext.fbs_timer.function = mppLnx_rescan_timer_handler;
	MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
				"mppLnx_failback_handler release the kernel\n"));

	unlock_kernel();

	/*
	 * Wake up the thread that created us.
	 */
	MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
				"mppLnx_failback_handler--Wake up parent from failback scan\n"));

	up(mppLnxFailbackScanContext.detect_sem);

	/*
	 * Start the failback thread, 10 minutes after the boot. We have seen problems if we send the rebalance 
	 * command during the system reboot if AVT is enabled(on boot volume). This is only for the first time. 
	 * Schedule the callback only after 10 minutes
	 */
	mppLnxFailbackScanContext.fbs_timer.expires = jiffies + MPPLNX_REBALANCE_DELAY_TIME * HZ;
	mppLnxFailbackScanContext.fbs_timer.data = (unsigned long)NULL;
	atomic_set(&mppLnxFailbackScanContext.scan_active, 0);

	/*
	 * the timer must be initialized first
	 * Then schedule the timer and wait
	 */
	init_timer(&mppLnxFailbackScanContext.fbs_timer);
	add_timer(&mppLnxFailbackScanContext.fbs_timer);
	down_interruptible(mppLnxFailbackScanContext.failback_sem);

	while (1)
	{
		/*
		 * For the failback scan thread, we want the thread to be terminated immediately
		 * at the mpp virtual HBA driver unloading time
		 */
		if(mppLnxFailbackScanContext.isStopped)
		{
			break;
		}


		/*
		 * perform the fail back scan only if the scanning is not active
		 */
		if(atomic_read(&mppLnxFailbackScanContext.scan_active) == 0)
		{
			atomic_set(&mppLnxFailbackScanContext.scan_active,1);
			mpp_FailBackScan();
			mpp_CheckAVTState();
			mppLnx_scanOfflinedPath();
			atomic_set(&mppLnxFailbackScanContext.scan_active,0);
			MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
						"Failback scan handler-- performed failbackScan\n"));
		}else
		{
			MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
						"Failback is still running\n"));
		}

		/**
		 * add timer. This thread will wake up after the timer goes off
		 * If the timer is still alive, don't add it again. Just modify
		 * the expires.
		 */

		currentTime = jiffies;
		if(currentTime < mppLnxFailbackScanContext.fbs_timer.expires)
		{
			mppLnxFailbackScanContext.fbs_timer.data = (unsigned long)NULL;
			mppLnxFailbackScanContext.fbs_timer.expires = jiffies +
				mppLnx_Config_Parameters.mpp_ScanInterval*HZ;
			MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
						"mppLnx_failback_handler -- mod rescan timer\n"));
			mod_timer(&mppLnxFailbackScanContext.fbs_timer,
					jiffies + mppLnx_Config_Parameters.mpp_ScanInterval*HZ);
		}
		else
		{
			mppLnxFailbackScanContext.fbs_timer.data = (unsigned long)NULL;
			mppLnxFailbackScanContext.fbs_timer.expires = jiffies +
				mppLnx_Config_Parameters.mpp_ScanInterval*HZ;
			MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
						"mppLnx_failback_handler -- add rescan timer\n"));
			add_timer(&mppLnxFailbackScanContext.fbs_timer);
		}
		/*
		 * For the failback scan thread, we want the thread to be terminated immediately
		 * at the mpp virtual HBA driver unloading time
		 */
		MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
					"failback scan sleeping\n"));
		if( down_interruptible(mppLnxFailbackScanContext.failback_sem))
		{
			break;
		}

		MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
					"Failback scan handler waking up\n"));

	} /*while (1) */

	MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
				"Failback scan exiting\n"));

	/*
	 * Make sure that nobody tries to wake us up again.
	 */
	mppLnxFailbackScanContext.failback_sem       = NULL;
	mppLnxFailbackScanContext.failback_pid       = -1;
	atomic_set(&mppLnxFailbackScanContext.scan_active, 0);
	/*
	 * delete the timer
	 */
	del_timer(&(mppLnxFailbackScanContext.fbs_timer));
	/*
	 * The isStopped flag is turned on by the mppLnx_release_virtual_host().
	 * If the flag is on, we do a up(). Otherwise, the kernel thread is killed by
	 * outside user process, the flag will not be on.
	 */
	if (mppLnxFailbackScanContext.isStopped)
	{
                if(mppLnxFailbackScanContext.detect_sem)
		up(mppLnxFailbackScanContext.detect_sem);
		mppLnxFailbackScanContext.detect_sem = NULL;
	}

	mppLnxFailbackScanContext.isStopped = 1;

	MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
				"Exiting mppLnx_failback_handler\n"));

	return 0;

}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_worker_handler
* SUMMARY:  The entry function of the virtual HBA driver sync IO worker thread
* SCOPE:    public
*
*
* DESCRIPTION:
  This function is the sync IO worker thread entry function. The input
   argument "void *data" is not used. This function performs the following operations:
      The process name of this kernel thread in the process table will be "mppWorker".
      This kernel thread will be demonized so that it has no associated control terminal
         and its stdin, stdout and stderr are closed.
      Its parent process id be init's process id (1).
      If the virtual HBA driver is loaded as a module, this kernel thread ignores
         all signals except SIGHUP that is used for system shutdown and module unloading.
         Signal SIGHUP will cause the failback kernel thread terminated.
      Starts an infinity loop, sleeps on a semaphore and wakes up when a sysn command
      needs to be processed
*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_worker_handler(void * data)
{
   struct list_head             *listPt;
   unsigned long                 iflages;
   RdacDeviceInformation_t      *rdacInfo=NULL;
   unsigned char                 reservationKey[8];
   LWORD                         mpp_status;
   LWORD                         status; 
   struct scsi_cmnd             *vcmnd;
   void                          (*vcmnd_timeoutfunc)(struct scsi_cmnd *);
   SenseData_t                   sensedata;
   TINY                          dataDirection;
   mppLnx_ProxyRequestEntry_t   *preq;
   unsigned char                *cmd;
   BYTE                          serviceAction;
   PRINReadKeysParamData_t      *keyParamData = NULL;
   PROUTParamData_t             *proutParamData;
   struct scatterlist           *sgel = NULL;

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_worker_handler\n"));
   /*
   * the 2.6 way of handling the kernel thread
   *  1. lock the kernel
   *  2. daemonize
   *  3. allow certain signal
   *  4. set a nice value
   *
   */
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_worker_handler() lock the kernel\n"));
	lock_kernel();

	daemonize(MPPLNX_SYNCIO_WORKER_PROC_NAME);
   allow_signal(SIGHUP);
   set_user_nice(current,-20);

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_worker_handler released the kernel\n"));

	unlock_kernel();

	/*
	 * Wake up the thread that created us.
	 */
	MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "mppLnx_worked_handler--Wake up parent from syncIO thread\n"));

	up(mppLnxWorkerQueueContextContext.detect_sem);

   /*
   * the thread is not busy
   */
   atomic_set(&mppLnxWorkerQueueContextContext.worker_active,0);


	while (1)
   {

      /*
      * perform our real work here
      */
      /*
      * lock the queue and pick up one cmnd entry at a time. Remove the cmnd entry
      * from the list and unlock. In this way, we don't block any new entry being
      * added to the list.
      */
      preq = NULL;
      spin_lock_irqsave(&mppLnx_s2tos3ProxyRequestQ.queueLock, iflages);
      listPt = mppLnx_s2tos3ProxyRequestQ.list.next;
      if(listPt != &(mppLnx_s2tos3ProxyRequestQ.list))
      {
         preq = list_entry(listPt, mppLnx_ProxyRequestEntry_t,s2tos3_list);
      }
      spin_unlock_irqrestore(&mppLnx_s2tos3ProxyRequestQ.queueLock, iflages);
      /*
      * process the command
      */
      if(unlikely(preq != NULL))
      {
         /*
         *mark the thread is busy
         */
         atomic_set(&mppLnxWorkerQueueContextContext.worker_active,1);
         vcmnd = preq->vcmnd;
         /*
         * delete the timer for this command
         */
         vcmnd_timeoutfunc = (void (*)(struct scsi_cmnd *))preq->vcmd_timeoutfunc;
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
            "before delete timer vcmnd 0x%lx SNno 0x%lx \n",
            (unsigned long)vcmnd, vcmnd->serial_number));

         switch(vcmnd->cmnd[0])
         {
            case RESERVE:
            case RESERVE_10:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "received Reserve command\n"));

               mppCmn_GenReservationKey(mppLnx_Config_Parameters.mpp_S2ToS3Key
                  ,reservationKey);
               rdacInfo = mpp_ModuleArray[vcmnd->device->id].RdacInfo;
               mpp_status = mppCmn_MapScsi2Reserve(rdacInfo,
                  vcmnd->device->lun,
                  mppLnx_Config_Parameters.mpp_S2ToS3Key,
                  reservationKey,
                  TRUE,
                  &sensedata);

               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "mppCmn_MapScsi2Reserve returned mpp_status %d\n",mpp_status));

               if(mpp_status == MPPCMN_PRTN_GOOD_STATUS)
               {
                  vcmnd->result = 0x00;
               }
               else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
               {
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,460,
                     "%s:0/1:any:%d: mppVhba reserve failure - RESERVATION CONFLICT vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                  vcmnd->result = (RESERVATION_CONFLICT <<1);
               }
               else if(mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
               {
                  /*
                  * 1 log the check condition
                  * 2 re-wrap the check condition to 02/04/09
                  * 3. complete the command with check-condition
                  */
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,468,
                     "%s:0/1:any:%d: mppVhba reserve failure - Check-Condition. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                  mppLnx_setCheckCondition(vcmnd,&sensedata);

               }else
               {
                  /* return an error. It will be translated to selection time-out*/
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,461,
                     "%s:0/1:any:%d: mppVhba reservation failure - Cannot talk to the deivice. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                  vcmnd->result = (DID_NO_CONNECT << 16);
               }

               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "Ready to add timer back vcmnd 0x%lx, timeout 0x%x timeout_func 0x%lx\n",
                  (unsigned long)vcmnd,
                  vcmnd->timeout_per_command,
                  (unsigned long)vcmnd_timeoutfunc));
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "before add timer vcmnd 0x%lx SNno 0x%lx \n",
                     (unsigned long)vcmnd, vcmnd->serial_number));
               mppLnx_scsi_add_timer(vcmnd,5*HZ,vcmnd_timeoutfunc);

               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "before call scsi_done done 0x%lx \n",
                  (unsigned long)vcmnd->scsi_done));

               vcmnd->scsi_done(vcmnd);

               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "completed Reserve command\n"));
               break;

            case PERSISTENT_RESERVE_IN:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "received SCSI3 PRIN  command\n")); 

               rdacInfo = mpp_ModuleArray[vcmnd->device->id].RdacInfo;
               cmd = (unsigned char *)vcmnd->cmnd;
               serviceAction = (BYTE)cmd[1] & 0x1F;

               /* Getting the correct buf address using scatter-gather */
               if(vcmnd->use_sg)
               {
                  /* scatter-gather list is used for the this SCSI command */
                  sgel = (struct scatterlist *)vcmnd->request_buffer;
                  keyParamData = page_address(sgel->page) + sgel->offset;
               }
               else
               {
                  keyParamData = vcmnd->request_buffer;
               }

               status = mppLnx_PersistentReserveIn(rdacInfo, vcmnd, serviceAction, (unsigned char*) keyParamData);
		
               if(status == MPPCMN_PRTN_GOOD_STATUS || MPPCMN_PRTN_RETURN_ERROR ||
                  status == MPPCMN_PRTN_RESERVATION_CONFLICT || MPPCMN_PRTN_CHECK_CONDITION ||
                  status == MPPCMN_PRTN_COMMAND_NOT_SUPPORTED)
               {
                  /* Add the timer back to the command and send the scsi_done status */

                  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
                     "PROut(%s:%d;Action-0x%x;Status-0x%x): NewRequestComplete\n",
                      rdacInfo->ModuleName, vcmnd->device->lun, serviceAction, status));

                  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                     "Ready to add timer back vcmd 0x%lx, timeout 0x%x timeout_func 0x%lx, SNno 0x%1x \n",
                      (unsigned long)vcmnd,
                      vcmnd->timeout_per_command,
                      (unsigned long)vcmnd_timeoutfunc,
                      vcmnd->serial_number));

                  mppLnx_scsi_add_timer(vcmnd,5*HZ,vcmnd_timeoutfunc);

                  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                     "before call scsi_done\n"));

                  vcmnd->scsi_done(vcmnd);
               }
		
               break;

            case PERSISTENT_RESERVE_OUT:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "received SCSI3 PROUT command\n"));
	
               rdacInfo = mpp_ModuleArray[vcmnd->device->id].RdacInfo;
               cmd = (unsigned char *)vcmnd->cmnd;
               serviceAction = (BYTE)cmd[1] & 0x1F;

               /* Getting the correct buf address using scatter-gather */
               if(vcmnd->use_sg)
               {
                  /* scatter-gather list is used for the this SCSI command */
                  sgel = (struct scatterlist *)vcmnd->request_buffer;
                  proutParamData = page_address(sgel->page) + sgel->offset;

               }
               else
               {
                  proutParamData = vcmnd->request_buffer;
               }

               status = mppLnx_PersistentReserveOut(rdacInfo, vcmnd, serviceAction, (unsigned char *) proutParamData);

               if(status == MPPCMN_PRTN_GOOD_STATUS || MPPCMN_PRTN_RETURN_ERROR || 
                  status == MPPCMN_PRTN_RESERVATION_CONFLICT || MPPCMN_PRTN_CHECK_CONDITION ||
                  status == MPPCMN_PRTN_COMMAND_NOT_SUPPORTED)
               {	
                  /* Add the timer back to the command and send the scsi_done status */

                  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
                     "PROut(%s:%d;Action-0x%x;Status-0x%x): NewRequestComplete\n",
                      rdacInfo->ModuleName, vcmnd->device->lun, serviceAction, status));
			
                  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                     "Ready to add timer back vcmd 0x%lx, timeout 0x%x timeout_func 0x%lx, SNno 0x%1x \n",
                      (unsigned long)vcmnd,
                      vcmnd->timeout_per_command,
                      (unsigned long)vcmnd_timeoutfunc,
                      vcmnd->serial_number));

                  mppLnx_scsi_add_timer(vcmnd,5*HZ,vcmnd_timeoutfunc);

                  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                     "before call scsi_done\n"));

                  vcmnd->scsi_done(vcmnd);
               }

               break;

            case RELEASE:
            case RELEASE_10:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "received Release command\n"));

               mppCmn_GenReservationKey(mppLnx_Config_Parameters.mpp_S2ToS3Key
                  ,reservationKey);
               rdacInfo = mpp_ModuleArray[vcmnd->device->id].RdacInfo;
               mpp_status = mppCmn_MapScsi2Release(rdacInfo,
                  vcmnd->device->lun,
                  mppLnx_Config_Parameters.mpp_S2ToS3Key,
                  reservationKey,
                  FALSE,
                  &sensedata);

               if(mpp_status == MPPCMN_PRTN_GOOD_STATUS)
               {
                  vcmnd->result = 0x00;
               }
               else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
               {
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,464,
                     "%s:0/1:any:%d: S2toS3 translation : Mapping release gets a RESERVATION_CONFLICT return-code. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                  vcmnd->result = (RESERVATION_CONFLICT <<1);
               }
               else if(mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
               {
                  /*
                  * 1 log the check condition
                  * 2 re-wrap the check condition to 02/04/09
                  * 3. complete the command with check-condition
                  */
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,469,
                     "%s:0/1:any:%d: mppVhba release failure - Check-Condition. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                  mppLnx_setCheckCondition(vcmnd,&sensedata);

               }else
               {
                  /* return an error. It will be translated to selection time-out*/
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,462,
                     "%s:0/1:any:%d: mppVhba release failure - Cannot talk to the deivice. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                  vcmnd->result = (DID_NO_CONNECT << 16);
               }
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "Ready to add timer back vcmnd 0x%lx, timeout 0x%x timeout_func 0x%lx\n",
                  (unsigned long)vcmnd,
                  vcmnd->timeout_per_command,
                  (unsigned long)vcmnd_timeoutfunc));
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "before add timer vcmnd 0x%lx SNno 0x%lx \n",
                     (unsigned long)vcmnd, vcmnd->serial_number));
               mppLnx_scsi_add_timer(vcmnd,5*HZ,vcmnd_timeoutfunc);
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "before call scsi_done\n"));
               vcmnd->scsi_done(vcmnd);
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "completed Release command\n"));
               break;
            case MPPLNX_BDR_CMND_OPCODE:
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "received bus device reset  command. PRResetSupported = %d\n",rdacInfo->PRResetSupported));
               rdacInfo = mpp_ModuleArray[vcmnd->device->id].RdacInfo;

		/* Jan 26 2006 - Use Engenio vendor specific command to clear reservations
		  *   for Amethyst or later firmware versions.
		  */
		mpp_status = mppCmn_MapBusDeviceResetVs(rdacInfo);
		if ( mpp_status == MPPCMN_PRTN_COMMAND_NOT_SUPPORTED )
               		mpp_status = mppCmn_MapBusDeviceReset(rdacInfo,vcmnd->device->lun);

               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "completed bus device reset command. status:%d\n",mpp_status));
               /*
               * the virtual command is a copy of the orignal command from the middle
               * level. We need to free the virtual command
               */
               mppLnx_put_free_asyncRequest((mppLnx_AsyncRequestEntry_t *)
                  (((char *) vcmnd) - sizeof(mppLnx_AsyncRequestEntry_t)));
               /*
               * The bus device reset message is not a real command. We don't need to
               * complete it. The scsi middle level has a dummy scsi_done function.
               */
               break;

            default:
               rdacInfo = mpp_ModuleArray[vcmnd->device->id].RdacInfo;
               if(vcmnd->sc_data_direction==DMA_FROM_DEVICE)
               {
                  dataDirection = MPPCMN_XFER_DEV_HOST;
               }
               else if(vcmnd->sc_data_direction==DMA_TO_DEVICE)
               {
                  dataDirection = MPPCMN_XFER_HOST_DEV;
               }
               else if(vcmnd->sc_data_direction==DMA_NONE)
               {
                  dataDirection = MPPCMN_XFER_NONE;
               }
               else
               {
                  dataDirection = MPPCMN_XFER_UNKNOWN;
               }

               mpp_status = mppCmn_CheckForReservationHolder(rdacInfo,
                  vcmnd->device->lun,
                  TRUE,
                  &sensedata);

               if(mpp_status == MPPCMN_PRTN_GOOD_STATUS)
               {
                  /*
                  * Either this node is the persistent reservation holder of this LUN or the LUN
                  * is not reserved. Let's dispatch the special command as a normal command.
                  * Note we break the "switch" to avoid the command being completed.
                  */
                  mppLnx_schedule_queuecmd(vcmnd);
                  break;
               }
               else if(mpp_status == MPPCMN_PRTN_RESERVATION_CONFLICT)
               {
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,470,
                     "%s:0/1:any:%d: mppVhba command %x failure - RESERVATION CONFLICT. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->cmnd[0],
                     vcmnd->serial_number));


                  vcmnd->result = (RESERVATION_CONFLICT <<1);
               }else if(mpp_status == MPPCMN_PRTN_CHECK_CONDITION)
               {
                  /*
                  * 1 log the check condition
                  * 2 re-wrap the check condition to 02/04/09
                  * 3. complete the command with check-condition
                  */
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,471,
                     "%s:0/1:any:%d: mppVhba command %x failure - Check-Condition. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->cmnd[0],
                     vcmnd->serial_number));

                  mppLnx_setCheckCondition(vcmnd,&sensedata);

               }else
               {
                  /* return an error. It will be translated to selection time-out*/
                  MPP_ERRORLOGPRINT((MPP_ERR_FATAL,463,
                     "%s:0/1:any:%d: mppVhba command %x failure - cannot talk to the deivice. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->cmnd[0],
                     vcmnd->serial_number));

                  vcmnd->result = (DID_NO_CONNECT << 16);
               }
               mppLnx_scsi_add_timer(vcmnd,5*HZ,vcmnd_timeoutfunc);
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "before call scsi_done\n"));
               vcmnd->scsi_done(vcmnd);
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
                  "completed command %x\n",vcmnd->cmnd[0]));
               break;
         }
         /*
         * put the request back to the free list
         */
         mppLnx_remove_proxyRequest_from_list(preq,MPPLNX_QUEUE_S2TOS3_LIST);
         mppLnx_put_free_proxyRequest(preq);
      }
      else
      {
         /*
		   * If we get a signal, it means we are supposed to go
		   * away and die.  This typically happens if the user is
		   * trying to unload a module.
		   */
		  MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
           "mpp worker thread sleeping\n"));

        if(mppLnxWorkerQueueContextContext.isStopped)
        {
           break;
        }

        /*
        * before going to sleep, mark the thread not-busy
        */
        atomic_set(&mppLnxWorkerQueueContextContext.worker_active,0);

         /*
		   * Note - we always use down_interruptible with the semaphore
		   * even if the module was loaded as part of the kernel.  The
		   * reason is that down() will cause this thread to be counted
		   * in the load average as a running process, and down
		   * interruptible doesn't.  Given that we need to allow this
		   * thread to die if the driver was loaded as a module, using
		   * semaphores isn't unreasonable.
		   */

        if(down_interruptible(mppLnxWorkerQueueContextContext.workerQueue_sem))
        {
           break;
        }

		   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
            "SyncIO worker handler waking up\n"));
      }

	}
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "syncIO worker thread exiting\n"));

	/*
	 * Make sure that nobody tries to wake us up again.
	 */
	mppLnxWorkerQueueContextContext.workerQueue_sem = NULL;
   mppLnxWorkerQueueContextContext.workerQueue_pid = -1;

	/*
	 * If anyone is waiting for us to exit (i.e. someone trying to unload
	 * a driver), then wake up that process to let them know we are on
	 * the way out the door.  This may be overkill - I *think* that we
	 * could probably just unload the driver and send the signal, and when
	 * the error handling thread wakes up that it would just exit without
	 * needing to touch any memory associated with the driver itself.
	 */
	if (mppLnxWorkerQueueContextContext.isStopped)
        {
                if(mppLnxWorkerQueueContextContext.detect_sem) 
		up(mppLnxWorkerQueueContextContext.detect_sem);
                mppLnxWorkerQueueContextContext.detect_sem = NULL;
        }

   mppLnxWorkerQueueContextContext.isStopped = 1;

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_worker_handler\n"));

	return 0;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_rescan_timer_handler
* SUMMARY:  The failback scan thread's timer-callback function
* SCOPE:    local
*
*
* DESCRIPTION:
   This is the failback rescan thread's timer callback function. This function
   will be assigned to rescan timer's timeout callback function "(*function)".
   The input data is the address of "mppLnx_sem", which is a Linux semaphore.
   This function simply calls Linux kernel semaphore API  "up()" to raise the
   semaphore.

*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_rescan_timer_handler(unsigned long data)
{
   MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_rescan_timer_handler\n"));
   /*we only wake up the failback snan thread if it is sleep
    */
   if(atomic_read(&mppLnxFailbackScanContext.scan_active) == 0)
   {
      MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
         "mppLnx_rescan_timer_handler -- wake up failback scan thread\n"));
      up(mppLnxFailbackScanContext.failback_sem);
   }
   else
   {
      MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
         "mppLnx_rescan_timer_handler -- failback scan is in progress\n"));
   }
   MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_rescan_timer_handler.\n"));
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_failbackRescan
* SUMMARY:  Scan failback one.
* SCOPE:    public
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function is called by the Linux MPP upper level driver after receiving
   an failback rescan IOCTL from mppUtil program.
   This function simply calls Linux kernel semaphore API  "up()" to raise
   the rescan semaphore. That will cause the rescan kernel thread to perform
   failback rescan.

*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

void mppLnx_failbackRescan()
{
   MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
      "Enterting mppLnx_failbackRescan()\n"));
   mppLnx_rescan_timer_handler((unsigned long) NULL);
   MPP_DEBUGPRINT((MPP_FAILOVERSCAN_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_failbakcRescan()\n"));
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_dpc_handler
* SUMMARY:  The entry function of the virtual HBA driver command dispatching/commpletion/retires
 thread
* SCOPE:    public
*
*
* DESCRIPTION:
   This function is the command dispatch/completion/retries entry function. The input
   argument "void *data" is the virtual host.
   This function performs the following operations:
      The process name of this kernel thread in the process table will be "mpp_dcr".
      This kernel thread will be demonized so that it has no associated control terminal
         and its stdin, stdout and stderr are closed.
      Its parent process id be init's process id (1).
      If the virtual HBA driver is loaded as a module, this kernel thread ignores
         all signals except SIGHUP that is used for system shutdown and module unloading.
         Signal SIGHUP will cause the DCR kernel thread terminated.
      Starts an infinity loop and uses Linux kernel timer and Linux semaphore to schedule
         its dispatching, completion and retries tasks. The thread will be waked up by
        1. a new virtual command is added to command queue for dispatching
        2. a proxy command is completed and added to command queue for
*
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

int mppLnx_dpc_handler(void * data)
{
   unsigned long                    currentTime;
   mppLnx_dpcQueueContext_t         * taskQueuep = &mppLnxDpcQueueContext;

   mppLnx_ProxyRequestEntry_t       * preq;
   struct list_head                 * listPt;
   unsigned long                    iflages;
   MPP_TIME                         ControllerIoWaitTime;
   MPP_TIME                         ArrayIoWaitTime;

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Entering mppLnx_dpc_handler\n"));

   /*
   * the 2.6 way of handling the kernel thread
   *  1. lock the kernel
   *  2. daemonize
   *  3. allow certain signal
   *  4. set a nice value
   *
   */

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "mppLnx_dpc_handler Locks the kernel\n"));

	lock_kernel();
   daemonize("mpp_dcr");
   allow_signal(SIGHUP);
   set_user_nice(current,-20);
	unlock_kernel();

	/*
	 * Wake up the thread that created us.
	 */
	MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "mppLnx_dpc_handler--Wake up parent from failback scan\n"));

	up(mppLnxDpcQueueContext.detect_sem);

   /*
   * Set the command queue to need schedule
   */
   atomic_set(&mppLnxDpcQueueContext.needSchedule,1);

   /*
   * The infinity loop. It can only break by SIGHUP
   */
	while (1)
   {
      /*
      *Loop through all commands in the command queue. It breaks when the queue is empty
      */
      while(1)
      {
         spin_lock_irqsave(&mppLnx_DPProxyRequestQ.queueLock, iflages);
         atomic_set(&mppLnxDpcQueueContext.needSchedule,0);
         if( (listPt = mppLnx_DPProxyRequestQ.list.next)
           != &(mppLnx_DPProxyRequestQ.list) )
         {
            preq = list_entry(listPt, mppLnx_ProxyRequestEntry_t, dispatch_list);
            spin_unlock_irqrestore(&mppLnx_DPProxyRequestQ.queueLock, iflages);
            if(preq->dispatchType == MPPLNX_DELAY_RESPONSE_TYPE)
            {
               /*
               * completing delayed response command
               */
               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "Completing scsi delayed virtual scsi_cmnd vtarget %d vlun %d\n",
                  preq->vcmnd->device->id, preq->vcmnd->device->lun));
               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "the virtual command serial_number %ld command %x\n",
                  preq->vcmnd->serial_number, preq->vcmnd->cmnd[0]));
               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "preq->result %d\n", preq->vcmnd->result));
               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "mppLnx_dpc_handler cmnd %lx serial_number %ld\n",
                  (unsigned long)preq->vcmnd, preq->vcmnd->serial_number));

              MPP_DEBUGPRINT((MPP_RETRY_DEBUG+MPP_DEBUG_LEVEL_1,
                  "mppLnxIOT SN=%ld OPC=%02x VD=H:%dC:%dT:%dL:%d LCP=LV EX=No-device CS=%08x\n",
                  preq->vcmnd->serial_number,
                  preq->vcmnd->cmnd[0],
                  preq->vcmnd->device->host->host_no,
                  preq->vcmnd->device->channel,
                  preq->vcmnd->device->id,
                  preq->vcmnd->device->lun,
                  preq->vcmnd->result));


               mppLnx_remove_proxyRequest_from_list(preq,MPPLNX_QUEUE_DISPATCHED_LIST);
               preq->vcmnd->scsi_done(preq->vcmnd);
               /*
               * remove the request from the dispatching queue
               * put the request back to free list
               */
               mppLnx_put_free_proxyRequest(preq);
            }
            else if(preq->dispatchType == MPPLNX_QUEUED_CMND_TYPE)
            {
               /*dispatch the proxy command to the physical HBA
               */
               MPP_GETTIME(currentTime);
               MPP_GETTIME(ControllerIoWaitTime);
               MPP_GETTIME(ArrayIoWaitTime);

               MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
                  "queue the command to the physical HBA driver cmnd %02x %02x\n",
                  preq->vcmnd->cmnd[0],
                  preq->vcmnd->cmnd[1]));
               /*
               * remove the request from the dispatching list
               */
               mppLnx_remove_proxyRequest_from_list(preq, MPPLNX_QUEUE_DISPATCHED_LIST);
               mppLnx_do_queuecommand(preq);
            }
            else if(preq->dispatchType == MPPLNX_COMPLETED_CMND_TYPE)
            {
               mppLnx_remove_proxyRequest_from_list(preq, MPPLNX_QUEUE_DISPATCHED_LIST);
                __mppLnx_scsi_done(preq); 
            }
            else if(preq->dispatchType == MPPLNX_VALIDATE_PATH_TYPE)
            {
               mppLnx_remove_proxyRequest_from_list(preq, MPPLNX_QUEUE_DISPATCHED_LIST);
	       		/* call __mppLnx_validate_done */
               __mppLnx_validatepath_done(preq); 
            }
            else if(preq->dispatchType == MPPLNX_FAILOVER_CMND_TYPE)
            {
               mppLnx_remove_proxyRequest_from_list(preq, MPPLNX_QUEUE_DISPATCHED_LIST);
	       		/* call __mppLnx_failoverCmd_done */
               __mppLnx_failoverCmd_done(preq); 
            }
            else
            {
               /*
               * we should not get here
               */
               MPPLNX_VHBA_BUG();
            }
         }else
         {
            atomic_set(&taskQueuep->needSchedule,1);
            spin_unlock_irqrestore(&mppLnx_DPProxyRequestQ.queueLock, iflages);
            break;

         }
      } /*the command loop */
      /*
      * If we get a signal, it means we are supposed to go
      * away and die.  This typically happens if the user is
      * trying to unload a module.
		*/
      MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
               "mpp_dpc thread sleeping\n"));
      /*
      * Note - we always use down_interruptible with the semaphore
      * even if the module was loaded as part of the kernel.  The
      * reason is that down() will cause this thread to be counted
      * in the load average as a running process, and down
      * interruptible doesn't.  Given that we need to allow this
      * thread to die if the driver was loaded as a module, using
      * semaphores isn't unreasonable.
      */

      if(mppLnxDpcQueueContext.isStopped)
      {
         break;
      }

      if(down_interruptible(mppLnxDpcQueueContext.dpc_sem))
      {
         break;
      }


		MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
            "dpc handler waking up\n"));

   } /*while (1) */

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "dpc thread exiting\n"));

   /*
    * Make sure that nobody tries to wake us up again.
    */
   mppLnxDpcQueueContext.dpc_sem       = NULL;
   mppLnxDpcQueueContext.dpcQueue_pid  = -1;
   atomic_set(&mppLnxDpcQueueContext.needSchedule, 0);

   /*
    * If anyone is waiting for us to exit (i.e. someone trying to unload
    * a driver), then wake up that process to let them know we are on
    * the way out the door.  This may be overkill - I *think* that we
    * could probably just unload the driver and send the signal, and when
    * the error handling thread wakes up that it would just exit without
    * needing to touch any memory associated with the driver itself.
    */
   if (mppLnxDpcQueueContext.isStopped)
   {
      if(mppLnxDpcQueueContext.detect_sem)
      up(mppLnxDpcQueueContext.detect_sem);
      mppLnxDpcQueueContext.detect_sem = NULL;
   }

   mppLnxDpcQueueContext.isStopped  =1;

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting dpc_handler\n"));

   return 0;

}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_Scsi2toScsi3_handler
* SUMMARY:  Handles Scsi-2 release/reserve and also SCSI3 PRIN/PROUT
*           virtual struct scsi_cmnd
* SCOPE:    public
*
* DESCRIPTION:
   This function is called by mppLnx_queuecommand() function when a Scsi-2 reserve/release
   command is received by the mppLnx_queuecommand().
   The input argument is the Scsi-2 reserve/release or prin/prout (scsi3) command in the format of struct scsi_cmnd
   dispatched by the Scsi middle level.
   This function simply creates a mppLnx_CmndEntry_t and inserts the object to
   mppLnxWorkerQueueContextContex. Then calls wakes up the sync IO worker thread to process
   the command

* @scp the virtual struct scsi_cmnd that is either Scsi-2 reserve or Scsi-2 release virtual command
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_Scsi2toScsi3_handler( struct scsi_cmnd * scp)
{
   mppLnx_ProxyRequestEntry_t    *preq;

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "Entering mppLnx_Scsi2toScsi3_handler()\n"));
   preq = mppLnx_get_free_proxyRequest();
   if(unlikely(preq == NULL))
   {
      /* error. Cannot get memory */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,465,
         "%s:0/1:any:%d: Error could not allocate mppLnx_ProxyRequestEntry_t in mppLnx_Scsi2toScsi3_handler. vcmnd SN %ld \n",
         (char *)mpp_ModuleArray[scp->device->id].RdacInfo->ModuleName,
         scp->device->lun,
         scp->serial_number));
      /*
      * complete the command with error
      */
      scp->result = DRIVER_ERROR << 24;
      scp->scsi_done(scp);
      return;
   }
   else
   {
      preq->vcmnd = scp;
      /*
       * delete the virtual command's timer so that the command will not time-out
       * We need to add the timer back before complete the virtual command
       */
      if(scp->eh_timeout.function != NULL)
      {
         preq->vcmd_timeoutfunc = (void (*)(struct scsi_cmnd *))scp->eh_timeout.function;
         mppLnx_scsi_delete_timer(scp);
      }else
      {
         /*
         * this should be the case of the BDR message.
         */
         preq->vcmd_timeoutfunc = NULL;
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
            "the vcmnd has no timer 0x%lx SNno 0x%lx cdb[0]= %x\n",
            (unsigned long)scp, scp->serial_number, scp->cmnd[0]));
      }
      preq->dispatchType = MPPLNX_SCSI2TOSCSI3_TRAN_TYPE;
      mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_S2TOS3_LIST);

      if( atomic_read(&mppLnxWorkerQueueContextContext.worker_active) == 0)
      {
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_Scsi2toScsi3_handler()- wake up worker thread\n"));
         up(mppLnxWorkerQueueContextContext.workerQueue_sem);
      }
      else
      {
         MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
            "mppLnx_Scsi2toScsi3_handler()- worker thread is running\n"));
      }
   }
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_1,
      "Exiting mppLnx_Scsi2toScsi3_handler()\n"));

}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_schedule_resp
* SUMMARY:  Schedule a Scsi Command to be completed at a late time
* SCOPE:   local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function is called by mppLnx_queuecommand() function when a non-dispatch-able
   virtual command is received. The mppLnx_queuecommand() uses this function to defer
   the non-dispatch-able command's completion.
   The input argument is the non-dispatch-able command in the format of struct scsi_cmnd.
   This function simply creates a mppLnx_CmndEntry_t and inserts the object to
   mppLnx_taskletQueue. It calls Linux kernel API tasklet_schedule() to schedule
   the virtual HBA driver's tasklet.

* @ scp the input argument is the non-dispatch-able command in the format of struct scsi_cmnd.
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_schedule_resp( struct scsi_cmnd * scp)
{
   mppLnx_ProxyRequestEntry_t       *preq;

   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "mppLnx_schedule_resp received cmnd %lx serial_number %ld\n",
      (unsigned long)scp, scp->serial_number));
   /*
   * get a free request
   */
   preq = mppLnx_get_free_proxyRequest();
   if( unlikely(!preq ))
   {
      /* error. Cannot get memory */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,466,
         "%s:0/1:any:%d: mppLnx_schedule_resp() cannot get memory for mppLnx_ProxyRequestEntry_t. vcmnd SN %ld \n",
         (char *)mpp_ModuleArray[scp->device->id].RdacInfo->ModuleName,
         scp->device->lun,
         scp->serial_number));
      /*
      * complete the command with error
      */
      scp->result = DRIVER_ERROR << 24;
      scp->scsi_done(scp);
      return;
   }
   else
   {

      /*
      * set the dispatch type of the request
      * insert the request to the dispatching queue
      * wake up the dpc thread
      */
      preq->vcmnd = scp;
      preq->dispatchType = MPPLNX_DELAY_RESPONSE_TYPE;
      preq->state = MPPLNX_REQUEST_DISPATCHED;
      preq->done = scp->scsi_done; 
      mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DISPATCHED_LIST);
      if(atomic_read(&mppLnxDpcQueueContext.needSchedule))
      {
         up(mppLnxDpcQueueContext.dpc_sem);
      }
   }
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_4,
      "Exiting mppLnx_schedule_resp\n"));
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_schedule_queuecmd
* SUMMARY:  Schedule a Scsi Command to be completed at a late time
* SCOPE:   local
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION:
   This function is called by mppLnx_queuecommand() function when a dispatch-able
   virtual command is received. The mppLnx_queuecommand() uses this function to dispatch
   the virtual command to the physical HBA driver.
   The input argument is the virtual command in the format of struct scsi_cmnd.
   This function simply creates a mppLnx_CmndEntry_t and inserts the object to
   mppLnx_taskletQueue. It calls Linux kernel API tasklet_schedule() to schedule
   the virtual HBA driver's tasklet.

* @ scp the input argument is the non-dispatch-able command in the format of struct scsi_cmnd.
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_schedule_queuecmd( struct scsi_cmnd * scp)
{
   mppLnx_ProxyRequestEntry_t    *preq;
   MPP_TIME                      currentTime;
   MPP_TIME                      ControllerIoWaitTime;
   MPP_TIME                      ArrayIoWaitTime;

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "mppLnx_schedule_queuecmd received cmnd %lx serial_number %ld\n",
      (unsigned long)scp, scp->serial_number));

   preq = mppLnx_get_free_proxyRequest();
   if( unlikely(!preq))
   {
      /* error. Cannot get memory */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,467,
         "%s:0/1:any:%d: mppLnx_schedule_queuecmd cannot get memory for mppLnx_ProxyRequestEntry_t. vcmnd SN %ld \n",
         (char *)mpp_ModuleArray[scp->device->id].RdacInfo->ModuleName,
         scp->device->lun,
         scp->serial_number));

      /*
      * complete the command with error
      */
      scp->result = DRIVER_ERROR << 24;
      scp->scsi_done(scp);
      return;
   }
   else
   {
      /*
      * insert the proxy request to the dispatching queue
      */
      preq->vcmnd = scp;
      preq->dispatchType = MPPLNX_QUEUED_CMND_TYPE;
       /*
       * This is a brand new virtual command. Set all fields related to
       * timing and retries
       */
      MPP_GETTIME(currentTime);
      MPP_GETTIME(ControllerIoWaitTime);
      MPP_GETTIME(ArrayIoWaitTime);
      preq->vcmd_timeoutfunc = middle_level_timeoutfunc;
      preq->done = scp->scsi_done; 
      preq->controller = UNKNOWN_PREVIOUS_CONTROLLER;
      preq->path = UNKNOWN_PREVIOUS_PATH;
      preq->CommandTimeoutRetryCount = 0;
      preq->RetryCount = mppLnx_Config_Parameters.mpp_SyncRetryCount;
      preq->SelectionTimeoutRetryCount = 0;
      preq->UaRetryCount = 0;
      preq->vcmndStartTime = currentTime;
      preq->ControllerIoWaitTime = ControllerIoWaitTime;
      preq->ArrayIoWaitTime = ArrayIoWaitTime;
      /*
      * put the proxy request to the dispatching queue and wake-up the
      * dispatching thread
      */
      mppLnx_insert_proxyRequest_to_list(preq,MPPLNX_QUEUE_DISPATCHED_LIST);
      if(atomic_read(&mppLnxDpcQueueContext.needSchedule))
      {
         up(mppLnxDpcQueueContext.dpc_sem);
      }
   }

   MPP_DEBUGPRINT((MPP_HBA_COMMAND_DEBUG+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_schedule_queuecmd\n"));
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_setCheckCondition
* SUMMARY:  Prepare a SCSI command to return check-condition status
* SCOPE:   local
*
*
* DESCRIPTION:
   This function will set the command's result to CHECK_CONDITION and command's sense buffer
   to return 02/04/09 (Sense-key/ASC/ASCQ)

* @ vcmd the input argument. The virtual command needs to be completed
* @ sensedata - a SenseData_t object.
* RESTRICTIONS:
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_setCheckCondition(struct scsi_cmnd * vcmd, SenseData_t *sensedata)
{
   int senseLen = 0;

   senseLen = (sizeof(SenseData_t) < SCSI_SENSE_BUFFERSIZE)?
      sizeof(SenseData_t):SCSI_SENSE_BUFFERSIZE;
   /*
   *
   * only the first 18 bytes ( the SCSI standard sense data will be filled
   typedef struct SenseData {
	0 BYTE		Error_Code;
	1 BYTE		Segment_Number; // not applicable to DAC
	2 BYTE		Sense_Key;
	3 BYTE		Information[ 4 ];
	7 BYTE	 	Additional_Len;
	8 LWORD		Command_Specific_Info;
	12 BYTE		ASC;
	13 BYTE		ASCQ;
   14 BYTE		Field_Replaceable_Unit;
	15 BYTE		Sense_Key_Specific_Info[ 3 ];
   */
   bzero(sensedata,sizeof(SenseData_t));
   sensedata->Error_Code = 0x70;
   sensedata->Segment_Number = 0x00;
   sensedata->Sense_Key = 0x02;
   /*
   * Information -> 4 bytes, all 0s
   */
   sensedata->Additional_Len = 0x98; /* 152 */
   /*
   *Command_Specific_Info. all 0s
   */
   sensedata->ASC = 0x04;
   sensedata->ASCQ = 0x09;
   /*
   *Field_Replaceable_Unit, 0. Sense_Key_Specific_Info, all 0s
   */

   /*
   * struct scsi_cmnd->results's status byte -> check-condition
   */
   vcmd->result = (CHECK_CONDITION<<1);

   bcopy (sensedata,vcmd->sense_buffer, senseLen );
}
/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_addWorkQueue
* SUMMARY:  Add an work task to the Linux work queue
* SCOPE:    Global
*
* DESCRIPTION:
*  This function will schedule an immediate work task or delayed work task based on the "type".
*
* @ context - the input argument.  A void point of work task handler's context data struct
* @ type - The type of work task. The task is used to determine immediate work task schedule,
*          delayed work task schedule and which work task handler function will be registered
* RESTRICTIONS:
*
* RETURNS: 0 - success. Otherwise, failed.
*
* ERRNO:
*
* NOTES: The work task handle must free the memory of work_struct and caller's work task context
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
int mppLnx_addWorkQueue(void *context, mppLnx_workqType_t type )
{
   int rtv = 0;
   mppLnx_workQueueWrapper_t *wqw =
      MPP_INT_KMEM_ALLOC(sizeof(mppLnx_workQueueWrapper_t));
    
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Entering mppLnx_addWorkQueue\n"));

   if (unlikely(!wqw))
   {
     /*
      * log an error message
      */
      MPP_ERRORLOGPRINT((MPP_ERR_FATAL,472,
         "Hot-Plug mppLnx_addWorkQueue cannot get memory for mppLnx_workQueueWrapper_t.\n"));
      return 1;
   }
   wqw->work_context = context;
   switch (type)
   {
      case SCAN_VHOST:
         if(mppLnx_startVhbaModRemove)
         {
            MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
            rtv = 1;
         }else
         {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
            INIT_WORK(&wqw->work, mppLnx_delayHostScanWorkHandler);
#else
            INIT_WORK(&wqw->work, mppLnx_delayHostScanWorkHandler, wqw);
#endif
            schedule_work(&wqw->work);
            mppLnx_workTaskRefCnt++;
            MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
               "mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
         }
         break;
      case ADD_VDEV:
         if(mppLnx_startVhbaModRemove)
         {
            MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
            rtv = 1;
         }else
         {
            /*
            *delay 2 second first and give HBA and scsi driver some time to discover the
            *rest of devices.
            */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
            INIT_DELAYED_WORK(&wqw->dwork, mppLnx_vdAddWorkHandler);
            schedule_delayed_work(&wqw->dwork, 2*HZ);
#else
            INIT_WORK(&wqw->work, mppLnx_vdAddWorkHandler, wqw);
            schedule_delayed_work(&wqw->work, 2*HZ);
#endif
            mppLnx_workTaskRefCnt++;
            MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
                  "mppLnx_workTaskRefCnt%d\n", mppLnx_workTaskRefCnt));
         }
         break;
      default:
         MPP_ERRORLOGPRINT((MPP_ERR_FATAL,473,
            "Hot-Plug mppLnx_addWorkQueue. Unknown Work task type: %d.\n",type));
         MPP_FREE(wqw,sizeof(mppLnx_workQueueWrapper_t));
         rtv = 1;
         break;
   }
   MPP_DEBUGPRINT((MPP_DEBUG_LOG_ONLY+MPP_DEBUG_LEVEL_2,
      "Exiting mppLnx_addWorkQueue\n"));
   return rtv;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_QueueRequest
* SUMMARY: Queue & schedule a callback to a specified function utilizing the
*          specified context information.
*/
LWORD mppLnx_QueueRequest(MPPCMN_QUEUE_REQUEST_CALLBACK RequestCallback, void *Context, LWORD DelaySecs)
{
    /*
     * 01/28/08 - CR136027.  Added new argument that can be used to specify whether the 
     * callee wants to delay the queued request.  For now Linux will ignore this option.
     */

    mppLnx_workQueueWrapper_t *wqw = MPP_INT_KMEM_ALLOC(sizeof(mppLnx_workQueueWrapper_t));

    if (unlikely(!wqw))
    {
        /*
        * log an error message
        */
        MPP_ERRORLOGPRINT((MPP_ERR_FATAL,475,
           "mppLnx_QueueRequest cannot get memory for mppLnx_workQueueWrapper_t.\n"));
        return 0;
    }
    wqw->callback_fcn = RequestCallback;
    wqw->work_context = Context;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
    INIT_WORK(&wqw->work, mppLnx_genericCallback);
#else
    INIT_WORK(&wqw->work, mppLnx_genericCallback, wqw);
#endif
   
    /* Adding to the private work queue */
    queue_work(mppLnx_wq, &wqw->work);
    /* CR 139596 - We can't flush the workqueue from interrupt
     * context, so we will wake up the Path Validation Handler
     * Whenever we queue up work. 
     */
    mppLnx_wakeup_pathvalidate_thread();

    mppLnx_workTaskRefCnt++;
    return 1;
}


/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_genericCallback
* SUMMARY: Function that handles work queue mechanics for "queued callback" work.
*/
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
void mppLnx_genericCallback(struct work_struct *ws)
#else
void mppLnx_genericCallback(void *arg)
#endif
{

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
    mppLnx_workQueueWrapper_t *wqw = container_of(ws, mppLnx_workQueueWrapper_t, work);
#else
	mppLnx_workQueueWrapper_t *wqw = (mppLnx_workQueueWrapper_t *) arg;
#endif

	(wqw->callback_fcn)(wqw->work_context);
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
    MPP_FREE(wqw, sizeof(mppLnx_workQueueWrapper_t));
#else
	MPP_FREE(arg,sizeof(mppLnx_workQueueWrapper_t));
#endif
	mppLnx_workTaskRefCnt--;
}

/*******************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_scanOfflinedPath
* SUMMARY:  Checking if any physical devices become offlined and bring then back
*		online if possible
* SCOPE:    Local
*
* DESCRIPTION:
   This function will be invoked from the failback scan thread. When a IO gets lost for whatever
   reason, the scsi error recovery operation will be started. If the error recovery could not
   recover the lost IO, a scsi-device will be marked as offline by the kernel. After a scsi-device
   is marked offline, the scsi-device could not be used for routing IO requested until a system
   reboot or system administrators manually mark the device online through sys fs interface.
   This function will move an offlined device online to workaround some issues in certain versions
   of iSCSI software initiators.

* RESTRICTIONS: For the kernel versions 2.6.5 and earlier,
*
* RETURNS:
*
* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/
void mppLnx_scanOfflinedPath(void)
{
    RdacDeviceInformation_t *RdacInfo;
    MPP_HANDLE DeviceObject;
    BYTE Controller, path, i;
    LWORD Lun;
    ArrayModuleEntry_t *ModPtr;

    MPP_DEBUGPRINT((MPP_AVTSCAN_DEBUG+MPP_DEBUG_LEVEL_4,
        "mppLnx_scanOfflinedPath: Called\n"));

    ModPtr = mpp_ModuleArray;
    for(i=0; i<mppLnx_Config_Parameters.mpp_MaxArrayModules; ++i, ++ModPtr)
    {
        RdacInfo = ModPtr->RdacInfo;
        if( !RdacInfo )
            continue;
            for( Controller=0; Controller<2; ++Controller )
            {
                CHECK_RDACINFO_EXIST(RdacInfo);
                if( !RdacInfo->ControllerInfo[Controller]->ControllerPresent )
                    continue;
                for( path=0; path<mppLnx_Config_Parameters.mpp_MaxPathsPerController; ++path )
                {
                    CHECK_RDACINFO_EXIST(RdacInfo);
                    DeviceObject = NULL;
                    if( !RdacInfo->ControllerInfo[Controller]->ControllerPath[path].Present )
                        continue;
                    for( Lun=0; Lun<mppLnx_Config_Parameters.mpp_MaxLunsPerArray; ++Lun )
                    {
                        CHECK_RDACINFO_EXIST(RdacInfo);
                        if( !RdacInfo->RdacLuns[Lun].Present ||
                                !RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[path].LunPathDevice ||
                                RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[path].PathRemoveState )
                            continue;

                        DeviceObject = RdacInfo->RdacLuns[Lun].LunControllers[Controller]->LunPaths[path].LunPathDevice;

                        if (DeviceObject->sdev_state == SDEV_OFFLINE)
                        {
                            MPP_ERRORLOGPRINT((MPP_ERR_RETRYABLE, 474,
                                "Moved offlined device %s:%d:%d:%d pdev H%d:C%d:T%d:L%d online\n",
                                RdacInfo->ModuleName, Controller, path, Lun,
                                DeviceObject->host->host_no,
                                DeviceObject->channel,
                                DeviceObject->id,
                                DeviceObject->lun));
                            scsi_device_set_state(DeviceObject, SDEV_RUNNING);
                        }else
                        {
                            MPP_DEBUGPRINT((MPP_AVTSCAN_DEBUG+MPP_DEBUG_LEVEL_4,
                                "Check device state %s:%d:%d:%d pdev H%d:C%d:T%d:L%d is online\n",
                                RdacInfo->ModuleName, Controller, path, Lun,
                                DeviceObject->host->host_no,
                                DeviceObject->channel,
                                DeviceObject->id,
                                DeviceObject->lun));
                        }


                    }
                }
            }
    }
}



/**************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_PersistentReserveIn
* SUMMARY:  Handles LnxMPP Failover Cluster PRIN (SCSI3) commands.
* SCOPE:    local
* ARGUMENTS:
@rdacInfo               - The object which represents a storage array
@vcmnd                  - The virtual struct scsi_cmnd's data structure
@serviceAction          - This is the opcode which identifes what function to be performed
@keyParamData           - This is the Buffer address where the Reservation Key is present
*
* DESCRIPTION:
*      This routine will determine which devices to send the PRIN request to based on the
*      Service Action code.  This routine sends the orginal I/O request down unlike the
*      common code s2tos3 code that creates its own requests.
*
*      On READ_KEYS it will send down one path.  In case of failure the other
*      paths will be tried until one succeeds.  Failure is returned only if
*      it fails down all paths.
*
*      On READ_RESERVATION/REPORT_CAPABILITIES the command is sent down one
*      path.  Failed requests are not retried.
*
* RESTRICTIONS:
*      None.
*
* RETURNS:The return value can be:
*       MPPCMN_PRTN_GOOD_STATUS
*       MPPCMN_PRTN_RESERVATION_CONFLICT
*       MPPCMN_PRTN_RETURN_ERROR
*       MPPCMN_PRTN_CHECK_CONDITION
*	MPPCMN_PRTN_COMMAND_NOT_SUPPORTED
*
*************************************************************************/

LWORD mppLnx_PersistentReserveIn(RdacDeviceInformation_t *rdacInfo,
                                 struct scsi_cmnd *vcmnd,
                                 BYTE serviceAction,
                                 unsigned char *keyParamData)

{
   LunPathInfo_t     *lunPathInfo = NULL;
   SenseData_t        sensedata;
   BYTE               controllerIndex;
   BYTE               pathIndex;
   LWORD              BufferLength = 0;
   BYTE               path;
   BOOL               retryFailedOnAnotherPath = FALSE;
   BOOL               enableCC = TRUE;
   MPP_HANDLE         pSDp = NULL;
   LWORD              mpp_status = MPPCMN_PRTN_GOOD_STATUS;
	
   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "PRIn(%s:%d;Action-0x%x): NewRequest\n",
       (char *)rdacInfo->ModuleName,
       vcmnd->device->lun,
       serviceAction));

		
   switch(serviceAction)
   {
      case SCSI_PRIN_READ_KEYS:
      {
         BufferLength = sizeof(PRINReadKeysParamData_t);
         retryFailedOnAnotherPath = TRUE;
         break;
      }

      case SCSI_PRIN_READ_RESERVATION:
      case SCSI_PRIN_REPORT_CAPABILITIES:
      {
         BufferLength = sizeof(PRINReadResevParamData_t);
         break;
      }

      default:
      {
         /* Unknown Service action. So,investigate it with Debug build */
         vcmnd->result = (DID_ERROR <<16);
         return MPPCMN_PRTN_COMMAND_NOT_SUPPORTED;
      }
   }

         /* 
          * Send the request. Currently none of the PRIN commands require that the
          * command be explicitly sent to every path so use mppCmn_NewSelectPath()
          * to find one to send the request.
          */

          for(path= 0; path < (mppLnx_Config_Parameters.mpp_MaxPathsPerController); path++)
          {
             lunPathInfo = mppCmn_NewSelectPath(rdacInfo, vcmnd->device->lun, &controllerIndex, &pathIndex);
             if(lunPathInfo == NULL)
             {
                vcmnd->result = (DID_NO_CONNECT <<16);
                return MPPCMN_PRTN_RETURN_ERROR;
             }

             pSDp = lunPathInfo->LunPathDevice;
             mpp_status = mppCmn_SynchronousPRIo(pSDp, vcmnd->cmnd, vcmnd->cmd_len, keyParamData, BufferLength, MPPCMN_XFER_DEV_HOST, enableCC, &sensedata);

             switch(mpp_status)
             {
                case MPPCMN_PRTN_GOOD_STATUS:
                {
                   /* The request succeeded & so we need not try it on the alternate path. We're done. */
                   vcmnd->result = 0x00;
                   return MPPCMN_PRTN_GOOD_STATUS;
                   break;

#ifdef MPP_DEBUG
 {
    LWORD     size;
    LWORD     idx;
    TINY      printBuffer[256];
    BYTE     *buffer;
    LINT      position;
    LINT      length;

   buffer = (BYTE *)vcmnd->request_buffer;
   bzero(&printBuffer, 256);
   size = vcmnd->request_bufflen;
   if(size > 255)
   {
      size = 255;
   }

   position = 0;
   for(idx = 0; idx < size; idx++)
   {
      length = sprintf(&printBuffer[position], "%02x", buffer[idx]);
      position += length;
   }

   if (serviceAction == SCSI_PRIN_READ_RESERVATION)
   {
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "PRIn(%s:%d): ReadResvBuffer(%s)\n",
          rdacInfo->ModuleName,
          vcmnd->device->lun,
          printBuffer));
   }
   else if(serviceAction == SCSI_PRIN_READ_KEYS)
   {
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "PRIn(%s:%d): ReadKeysBuffer(%s)\n",
          rdacInfo->ModuleName,
          vcmnd->device->lun,
          printBuffer));
   }
 }
#endif
                }

                case MPPCMN_PRTN_CHECK_CONDITION:
                {
                   MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 452, 
                      "Failed to read on Lun %d - status %d\n",
                       vcmnd->device->lun,mpp_status));
                       vcmnd->result = (CHECK_CONDITION <<1);
                       return mpp_status;
                }

                case MPPCMN_PRTN_RESERVATION_CONFLICT:
                {
                   MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 453, 
                      "Failed to read on Lun %d - status %d\n",
                       vcmnd->device->lun,mpp_status));
                       vcmnd->result = (RESERVATION_CONFLICT <<1);
                       return mpp_status;
                }

                case MPPCMN_PRTN_RETURN_ERROR:
                default:
                {
                   MPP_ERRORLOGPRINT((MPP_ERR_RECOVERED, 454, 
                      "Failed to read on Lun %d - status %d\n",
                       vcmnd->device->lun,mpp_status));

                       if(retryFailedOnAnotherPath == FALSE)
                       {
                          /* 
                           * The request failed but we are not required to retry it on another
                           * path. So, we're done.
                           */
                            vcmnd->result = (DID_NO_CONNECT <<16);
                            return mpp_status;
                            break;
                       }
	
                            vcmnd->result = (DID_NO_CONNECT << 16);
                            continue;
                }
             }/* switch */

          } /* for loop */


#ifdef MPP_DEBUG

          /* 
           * Go ahead and print a debug message if we exceeded the loop count above.
           * This may be useful in troubleshooting issues.
           */
            if(path == (mppLnx_Config_Parameters.mpp_MaxPathsPerController))
            {
               MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
                  "PRIn(%s:%d;Action-0x%x): ExceededMaxPathCount\n",
                   rdacInfo->ModuleName,
                   vcmnd->device->lun,
                   serviceAction));
            }
#endif

            return mpp_status;
}


/**************************************************************************
* PROCEDURE
*
* NAME:     mppLnx_PersistentReserveOut
* SUMMARY:  Handles LnxMPP Failover Cluster PROUT commands.
* SCOPE:    local
* ARGUMENTS:
@rdacInfo 		- The object which represents a storage array
@vcmnd     		- The virtual struct scsi_cmnd's data structure
@serviceAction 		- This is the opcode which identifes what function to be performed
@proutParamData		- This is the Buffer address where the where Service Action Key is present
*
* DESCRIPTION:
*      This routine will determine which devices to send the PROUT request to based on the
*      Service Action code.  This routine sends the orginal I/O request down unlike the
*      common code s2tos3 code that creates its own requests.
*
*
*      On REGISTER/REGISTER_AND_IGNORE_EXISTING the command is sent down all
*      paths.  If any path succeeds the PR key will be stored.  If the command
*      fails down any path, return failure.
*
*      On REGISTER/REGISTER_AND_IGNORE_EXISTING w/PR Key NULL (ie. UNREGISTER) the
*      request is sent down every path.  Failure is returned if the request fails
*      down any path (but error is ignored if the path happens to be one where
*      prior REGISTER/REGISTER_AND_IGNORE_EXISTING had failed in the first place).
*      The stored PR key is cleared irrespective of success/failure being returned.
*
*      On RESERVE/RELEASE the command is sent down one path.  If it fails another path
*      is tried.  Failure is returned only if none succeed.
*
*      ON CLEAR the command is sent down one path.  If it fails another path is tried.
*      Failure is returned only if none succeed.  The stored PR key is cleared
*      irrespective of success/failure being returned.
*
*      On PREEMPT the command is sent down one path.  If it fails another path is tried.
*      Failure is returned only if none succeed.
*
*      On PREEMPT_AND_ABORT the command is sent down one path.  Failed request is
*      not retried.
*
*      NOTE: If a path shows up later a REGISTER_AND_IGNORE_EXISTING request is built
*      by the mppLnx_pathvalidate_regcheck() routine using the saved PR key and sent down the new path.
*
* RESTRICTIONS:
*      None.
*
* RETURNS:The return value can be:
*	MPPCMN_PRTN_GOOD_STATUS             
*	MPPCMN_PRTN_RESERVATION_CONFLICT    
*	MPPCMN_PRTN_RETURN_ERROR
*	MPPCMN_PRTN_CHECK_CONDITION 
*	MPPCMN_PRTN_COMMAND_NOT_SUPPORTED
* 
*************************************************************************/


LWORD mppLnx_PersistentReserveOut(RdacDeviceInformation_t *rdacInfo,
                                  struct scsi_cmnd *vcmnd,
                                  BYTE serviceAction,
                                  unsigned char *proutParamData)

{
   LunPathInfo_t            *lunPathInfo;
   mppLnx_PDeviceEntry_t    *pde = NULL;
   mppLnx_VDeviceEntry_t    *vde = NULL;
   SenseData_t               sensedata;
   TEXT                      prKey[8] = {0};
   TEXT                      tmpPRKey[8] = {0};
   int                       j=0, z=0;
   BOOL                      sendDownAllPaths = FALSE;
   BOOL                      clearPRKey = FALSE;
   BOOL                      passOnlyIfAllSucceed = FALSE;
   BOOL                      ignoreIfPreviousFailed = FALSE;
   BOOL                      savePRKeyIfAnySucceed = FALSE;
   BOOL                      onlyOnSinglePathResv = FALSE;
   BOOL                      retryOnAnotherPath = FALSE;
   BOOL                      enableCC = TRUE;
   BYTE                      controllerIndex, cIdx;
   BYTE                      pathIndex, pIdx;
   LWORD                     stateIndex;
   MPP_HANDLE                pSDp = NULL;
   LWORD                     mpp_status = MPPCMN_PRTN_GOOD_STATUS;
 

   vde = rdacInfo->RdacLuns[vcmnd->device->lun].PlatformPrivateData;
   if(vde == NULL)
   {
      MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
         "PROut(%s:%d;Action-0x%x):NoVirtualDeviceInfo\n",
          vcmnd->device->lun,
          serviceAction));
   }

   MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
      "PROut(%s:%d;Action-0x%x;): NewRequest\n",
       (char *)rdacInfo->ModuleName,
       vcmnd->device->lun,
       serviceAction));

   switch(serviceAction)
   {
      case SCSI_PROUT_REGISTER:
      case SCSI_PROUT_REGISTER_AND_IGNORE_EXISTING_KEY:
      {

          /* Copying the SrvActCode to use it in the mppLnx_pathvalidate_regcheck() routine */
          vde->srvActCode = serviceAction;

          /* 
           * Checking if it's register/unregister command with Srv_action byte
           * If it's register command,then Src_act reservation key should be present in "--param-sark"
           * which is from byte 8-15
           * ex: sg_persist -vv --out --register --param-sark=123abc /dev/sdh
           * If it's unregister command, then srv_act key will be zero (byte 8-15) & instead key will 
           * be in "--param-rk" (byte 0-7)
           */

          /* Checking the first 8 bytes to be NULL for Register command */
          if(mpp_IsNull(proutParamData, 8))
          {
             /* Copying the Service Action Key (bytes 8-15)*/
             for (z= 8; z <=15; z++)
             {
                prKey[j] = proutParamData[z];
                j++;
             }
             bcopy(&prKey, &tmpPRKey, 8);

          }
          else
          {
             /* Copying the param-rk key (bytes 0-7) */
             for (z= 0; z <=7; z++)
             {
                prKey[j] = proutParamData[z];
                j++;
             }
          }

          MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
             "PROut_REG: (%s:%d;Key-%02x%02x%02x%02x%02x%02x%02x%02x)\n",
              (char *) rdacInfo->ModuleName, vcmnd->device->lun, prKey[0], prKey[1], prKey[2], prKey[3], prKey[4], prKey[5], prKey[6], prKey[7]));

          //Command must be send down all paths
          sendDownAllPaths = TRUE;

          //Failure if it fails down even one of the paths
          passOnlyIfAllSucceed = TRUE;

          if(mpp_IsNull((BYTE *)&tmpPRKey, 8))
          {
             MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
                "PROut(Action-0x%x;): NULLPRKey\n",
                 serviceAction));

             /* If the unregister fails don't report it back as an error
              * If the previous register/register_ignore_existing request
              * down that path has failed too.
              */
              ignoreIfPreviousFailed = TRUE;

             /* Clear the PR key regardless of the status that is going
              * to be returned to ClusterDisk
              */
              clearPRKey = TRUE;
          }
          else
          {
             /* If the register/register_ignore_existing request succeeds
              * down any path save off the PR key
              */
              savePRKeyIfAnySucceed = TRUE;
          }

          break;
      }

      case SCSI_PROUT_RESERVE:
      {
         retryOnAnotherPath = TRUE;
         onlyOnSinglePathResv = TRUE ;
         break;
      }

      case SCSI_PROUT_CLEAR:
      /* Clear the stored PR Key regardless of the status returned back */
      clearPRKey = TRUE;
      case SCSI_PROUT_RELEASE:
      case SCSI_PROUT_PREEMPT:
      /* Apart from preempt_abort all other requests must be retried
       * down another path
       */
       retryOnAnotherPath = TRUE;
       break;
      case SCSI_PROUT_PREEMPT_AND_ABORT:
      break;
      default:
      {
         /* Unknown action. Check this out with debug build */
         vcmnd->result = (DID_ERROR <<16);
         return MPPCMN_PRTN_COMMAND_NOT_SUPPORTED;
      }
   }

      /* Send the request. Some of the PROUT commands require us to attempt the request to
       * each path, where other don't. In order to make sure we attempt each path we can't
       * use mppCmn_NewSelectPath() because it controls the path selection process.
       */

       controllerIndex = 0;
       pathIndex = 0;
       for(;;)
       {
          if(controllerIndex >= 2)
          {
             /* Both 'A' and 'B' controllers have been completed. We're done */
             break;
          }

          if(pathIndex >= mppLnx_Config_Parameters.mpp_MaxPathsPerController)
          {
             /* Go to the next controller & reset the path */
             pathIndex = 0;
             controllerIndex++;
             continue;
          }

          if(!sendDownAllPaths && !retryOnAnotherPath)
          {
             /* If the request doesn't need to be retried down another path on
              * failure then use mppCmn_NewSelectPath() which will maximize our
              * chances of getting a good path to use
              */

              lunPathInfo = mppCmn_NewSelectPath(rdacInfo,vcmnd->device->lun, &cIdx, &pIdx);
              if(lunPathInfo == NULL)
              {
                 vcmnd->result = (DID_NO_CONNECT <<16);
                 return MPPCMN_PRTN_RETURN_ERROR;
              }

              pSDp = lunPathInfo->LunPathDevice;
          }
          else
          {
             stateIndex = rdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].PathStateIndex;
             if(rdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].Present &&
             (rdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL ||
             rdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL_NEED_CHECK ||
             rdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].PathState[stateIndex] == MPPCMN_OPTIMAL_CHECKING) &&
             rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice &&
             !rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].PathRemoveState)
             {
                pSDp = rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice;
             }
             else
             {
                /* Path is not present in a non-OPTIMAL state, or has no LunPathDevice. Go to the next one */
                pathIndex++;
                continue;
             }
           }

           mpp_status = mppCmn_SynchronousPRIo(pSDp, vcmnd->cmnd, vcmnd->cmd_len, proutParamData, BUFFER_LENGTH, MPPCMN_XFER_HOST_DEV, enableCC, &sensedata);

           switch(mpp_status)
           {
              case MPPCMN_PRTN_GOOD_STATUS:
              {
                 if(!passOnlyIfAllSucceed)
                 {
                    /* Success down any one path means success */
                    vcmnd->result = 0x00;
                    return MPPCMN_PRTN_GOOD_STATUS;
                 }

                 if(savePRKeyIfAnySucceed)
                 {
                    MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
                       "PROut: SavePRKey(%s:%d;Key-%02x%02x%02x%02x%02x%02x%02x%02x)\n",
                        rdacInfo->ModuleName, vcmnd->device->lun, prKey[0], prKey[1], prKey[2], prKey[3], prKey[4], prKey[5], prKey[6], prKey[7]));

                    /* Copy the PRKey and store it in vde*/
                    bcopy(&prKey, &vde->PRKey, 8);
	
                    /* Set the Virtual Device flag */
                    vde->PRKeyValid = TRUE;

                    pSDp = rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice;
                    pde = rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].PlatformPrivateData;

                    /* Set the Physical Device flag */
                    pde->PRKeyRegistered = TRUE;
                 }

                 if(!(sendDownAllPaths))
                 {
                    /* The need for retrying the request on another path
                     * is only necessary if the request fails down the
                     * chosen path. Since, it succeed we're done
                     */
                     vcmnd->result = 0x00;
                     return MPPCMN_PRTN_GOOD_STATUS;
                 }
              } 
              break;

              case MPPCMN_PRTN_RESERVATION_CONFLICT:
              {
                 MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 455,
                    "%s:0/1:any:%d: mppVhba failure : RESERVATION CONFLICT vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                 if(onlyOnSinglePathResv)
                 {
                    vcmnd->result = (RESERVATION_CONFLICT <<1);
                    return MPPCMN_PRTN_RESERVATION_CONFLICT;
                 }

                    vcmnd->result = (RESERVATION_CONFLICT <<1);
              }

              break;

              case MPPCMN_PRTN_CHECK_CONDITION:
              {
                 MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 456,
                    "%s:0/1:any:%d: mppVhba failure : Check-Condition. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                     vcmnd->result = (CHECK_CONDITION <<1);
              }

              break;

              case MPPCMN_PRTN_RETURN_ERROR:
              default:
              {   
                 /* return an error. It will be translated to selection time-out*/
                 BOOL    recordFailure = TRUE;
                 MPP_ERRORLOGPRINT((MPP_ERR_FATAL, 457,
                    "%s:0/1:any:%d: mppVhba failure : Cannot talk to the deivice. vcmnd SN %ld \n",
                     (char *)rdacInfo->ModuleName,
                     vcmnd->device->lun,
                     vcmnd->serial_number));

                 if(ignoreIfPreviousFailed && !pde->PRKeyRegistered)
                 {
                    /* Okay to ignore this failure if the previous failed */
                    recordFailure = FALSE;
                 }

                 if(passOnlyIfAllSucceed && recordFailure)
                 {
                    /* If the registration fails, send the Error on this path
                     * but it needs to tried on alternate
                     */
                     vcmnd->result = DID_ERROR << 16;
                     return MPPCMN_PRTN_RETURN_ERROR;
                 }

                 if(!(sendDownAllPaths || retryOnAnotherPath))
                 {
                    vcmnd->result = (DID_NO_CONNECT << 16);
                    return MPPCMN_PRTN_RETURN_ERROR;
                 }

                    vcmnd->result = (DID_NO_CONNECT << 16);
              }

              break;

           }/* switch */

           /* If we made it this far either the request needst to be tried down another
            * path or sent down all paths
            */
            pathIndex++;

       }/* for loop */

       if(clearPRKey)
       {
          // Reuse controllerIndex and pathIndex
           for(controllerIndex = 0; controllerIndex < 2; controllerIndex++)
           {
              for(pathIndex = 0; pathIndex < mppLnx_Config_Parameters.mpp_MaxPathsPerController; pathIndex++)
              {
                 if(rdacInfo->ControllerInfo[controllerIndex]->ControllerPath[pathIndex].Present
                    && rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice &&
                    !rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].PathRemoveState)
                    {
                       pSDp = rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].LunPathDevice;
                       pde = rdacInfo->RdacLuns[vcmnd->device->lun].LunControllers[controllerIndex]->LunPaths[pathIndex].PlatformPrivateData;

                       /* Since the PRKey is cleared, Unset the flag */
                       pde->PRKeyRegistered = FALSE;	
                    }
              }
           }


           MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_4,
              "PROut: ClearPRKey(%s:%d)\n",
               rdacInfo->ModuleName, vcmnd->device->lun));

           /* We are coming here after the PRKey is cleared, so clear the PRKey */
           bzero(vde->PRKey, sizeof(vde->PRKey));

           /* Unset the Virtual Device flag */
           vde->PRKeyValid = FALSE;
       }

#ifdef MPP_DEBUG
 {
    LWORD     size;
    LWORD     idx;
    TINY      printBuffer[256];
    BYTE      *buffer;
    LINT      position;
    LINT      length;

    buffer = (BYTE *)vcmnd->request_buffer;
    bzero(&printBuffer, 256);

    size = vcmnd->request_bufflen;
    if(size > 255)
    {
       size = 255;
    }

    position = 0;
    for(idx = 0; idx < size; idx++)
    {
       length = sprintf(&printBuffer[position], "%02x", buffer[idx]);
       position += length;
    }

    MPP_DEBUGPRINT((MPP_RESERVE_DEBUG+MPP_DEBUG_LEVEL_2,
       "PROut(%s:%d;Action-0x%x): Buffer(%s)\n",
        rdacInfo->ModuleName, vcmnd->device->lun, serviceAction, printBuffer));
 }
#endif
        return mpp_status;

}
