/*******************************************************************************
* Copyright 2004-2007 LSI Logic Corporation.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation,  version 2 of the License.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
********************************************************************************/
/*******************************************************************************

NAME            mppUtil26p_sysdep.c
SUMMARY         %description%
VERSION         %version: 4 %
UPDATE DATE     %date_modified: Fri Jan 25 23:10:26 2008 %
PROGRAMMER      %created_by:    bmoger %

DESCRIPTION:

INCLUDE FILES:

NOTES:

RESTRICTIONS:

SEE ALSO:

REFERENCE:     Linux MPP Driver utility OLBS - 349-1034730

IMPLEMENTATION:

MODIFICATION HISTORY:

*******************************************************************************/

/***  INCLUDES  ***/  
#include        "mppUtil_Sysdep.h"
#include        "MPP_Sysdep.h"
#include        "MPP_Common.h"
#include        "MPP_RdacInfo.h"
#include        "mppUtil_Common.h"
#include        "mppLnx_version.h"
#include        "mppLnx_rls_version.h"
#include        "mppLnx_bld_version.h"

/***  VARIABLE DEFINITIONS  ***/

extern  HANDLE   utilVirtualFd;
extern  char	 *progname;
char temp_string[MPP_MAX_PATH];
extern	CHAR	hostName[HOST_NAME_LEN];
extern	CHAR	domainName[DOMAIN_NAME_LEN];
extern	CHAR	timeStr[TIME_STR_LEN];
time_t	currentTime;


/***  TYPE DEFINITIONS  ***/
#define MPP_BUS_NODE "/dev/mppUpper"


/***  LOCALS  ***/


/***  PROCEDURES  ***/
static void mppLnxUtil_reportSrcAttributes();

/*******************************************************************************
* PROCEDURE
*
* NAME:      utilDetermineBusFd
* SUMMARY:   open the MPP "control node" 
* SCOPE:     private 
    local = Internal procedure (static)
    public = Procedure is visible to all components throughout system
    private = Procedure is visible to a limited set of modules, within
        one component (default)
*
*
* DESCRIPTION: Open the MPP "control node" and store the value of the open file
*   descriptor in the global variable utilVirtualFd. 
*
* RESTRICTIONS:
* 
* RETURNS: 
*   TRUE    - A valid file descriptor can be determined.
*   FALSE   - A valid file descriptor cannot be determined.

* ERRNO:
*
* NOTES:
*
* EXAMPLES:
*
* SEE ALSO:
*
*/

BOOL
utilDetermineBusFd ( void )
{
	int mppFd = 0;
	int rtn = 0;

   rtn = utilMakeMppDevNode();
   if(rtn != 0 )
   {
      return(FALSE);
   }

	if( ( mppFd = open(MPP_BUS_NODE, O_RDWR ) ) >= 0 )
	{
		utilVirtualFd = mppFd;
		return(TRUE);
	}
	else
	{
		return(FALSE);
	}
    
}


/***************************************************************************
* PROCEDURE:    utilFormatHostName
* SUMMARY:      utilFormatHostName - get the hostname of the machine.
*
* DESCRIPTION:
*		The function returns the HostName of the machine.
*
* NOTES:
*
* INPUT:
*
* OUTPUT:
*   buffer - location to fill the hostname
*
* RETURNS:
*   None
*
****************************************************************************/
VOID 
utilFormatHostName (MPP_STRING buffer)
{

	gethostname( buffer, HOST_NAME_LEN );
}
/***************************************************************************
* PROCEDURE:    utilFormatDomainName
* SUMMARY:      utilFormatDomainName - get the Domain Name of the machine.
*
* DESCRIPTION:
*		The function returns the Domain Name of the machine.
*
* NOTES:
*
* INPUT:
*
* OUTPUT:
*   buffer - location to fill the Domain Name
*
* RETURNS:
*   None
*
****************************************************************************/
VOID 
utilFormatDomainName (MPP_STRING buffer)
{
	getdomainname( buffer, DOMAIN_NAME_LEN );
}
/***************************************************************************
* PROCEDURE:    utilFormatCurrentGMTTime
* SUMMARY:      utilFormatCurrentGMTTime - get the Current GMT Time of the machine.
*
* DESCRIPTION:
*		The function returns the current GMT time of the machine.
*
* NOTES:
*
* INPUT:
*
* OUTPUT:
*   buffer - location to fill the current GMT time of the system.
*
* RETURNS:
*   None
*
****************************************************************************/
VOID 
utilFormatCurrentGMTTime (MPP_STRING buffer)
{

	time(&currentTime);
	utilFormatTime(currentTime, buffer);
}

/***************************************************************************
* PROCEDURE:    utilFormatHandle
* SUMMARY:      utilFormatHandle - supply a string that is an appropriate
*               representation of an MPP handle.
*
* DESCRIPTION:
*   The function receives an MPP-HANDLE as input and returns a displayable
*   string representation of the handle as is appropriate.  On Linux platform
*   it is not applicable. It simply formats the string to "none". 
*
* NOTES:
*
*
* INPUT:
*   handle -
*
* OUTPUT:
*   buffer - representation of an MPP Handle
*
* RETURNS:
*   None
*
****************************************************************************/
VOID 
utilFormatHandle (MPP_HANDLE handle, MPP_STRING buffer)
{

	sprintf( buffer, "%s", "none");
}

/***************************************************************************
* PROCEDURE:    utilFormatVertex
* SUMMARY:      utilFormatVertex - supply a string that is an appropriate
*               representation of an MPP vertex.
*
* DESCRIPTION:
*   The function receives an MPP-HANDLE representing a "vertex" as input
*   and returns a displayable string representation of the handle. Linux 
*   does not have vertices as in SGI IRIX, so this routine can just return 
*   the string "none".
*
* NOTES:
*
*
* INPUT:
*   handle - representing the "vertex"
*
* OUTPUT:
*   buffer - string of "none"
*
* RETURNS:
*   None
*
****************************************************************************/
VOID 
utilFormatVertex (MPP_HANDLE handle, MPP_STRING buffer)
{
       if (handle)
               sprintf(buffer, "%s", "present");
       else
               sprintf(buffer, "%s", "none");

        return;
}


/***************************************************************************
* PROCEDURE:    utilFormatControllerAddress
* SUMMARY:      utilFormatControllerAdderess - supply a string that is an appropriate
*               representation of a ControllerAdderess.
*
* DESCRIPTION:
*   This function is invoked from utilPrintRdacInfo() . It is a part of the
*   system dependent code. It receives am argument of type controllerAddress_t
*   and a buffer and returns a displayable string representation. 
*   On Linux, controllerAddress_t has the following structure:
*    struct controllerAddress_t {
*       Int hostId;
*	Int targetId; 
*	Int channel;
*    } Address;
* NOTES:
*
*
* INPUT:
*   handle - representing the "vertex"
* OUTPUT:
*   buffer - string of "none"
*
* RETURNS:
*   None
*
*
****************************************************************************/
void 
utilFormatControllerAddress(ControllerAddress_t address, MPP_STRING buffer, LWORD PathId)
{
	sprintf(buffer, "%x (hostId: %d, channelId: %d, targetId: %d)", PathId, address.HostId, address.ChannelId, address.TargetId);
}

/***************************************************************************
* PROCEDURE:    utilFormatDevState
* SUMMARY:      - supply a string that is an appropriate representation of Lun State.
*
* DESCRIPTION:
*
* NOTES:
*
*
* INPUT:
*  mpp_device_state 
* OUTPUT:
*
* RETURNS:
*   string of ""
*
*
****************************************************************************/
/*
char *utilFormatDevState (MPP_DEVICE_STATE PseudoDevState) {
	return "";

}
*/




/***************************************************************************
* PROCEDURE:    utilGetPhysicalHostMaxTargetId
* SUMMARY:      utilGetPhysicalHostMaxTargetId - 
*
* DESCRIPTION: This function takes a hostid and sends an ioctl to the 
*  Upper level mpp driver that will cause the driver to go through the 
*  list of hosts maintained by the scsi middle level and return the 
*  max number of TargetIds supported by the host.
*
* NOTES:
*
*
* INPUT:
*     The HostId  of the host bus adaptor driver we are interested in
*
* OUTPUT:
*     Prints the Max number of Targets handled by the host bus adaptor driver 
*
* RETURNS:
*     True or False depending on the outcome of the ioctl
*
****************************************************************************/
BOOL
utilGetPhysicalHostMaxTargetId(MPP_STRING hostid)
{
	struct host_target_t host_target_id;

	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	if(!isNum(hostid)) {
		fprintf(stderr, "Invalid option!\n");
		return(INVALID);
	}

	host_target_id.host_id = atoi(hostid);
	utilDebug("Send physical host max target parameter request\n");
	if(ioctl(utilVirtualFd, MPP_IOCTL_GET_PHYSICALHOST_MAX_TARGET, &host_target_id ) == INVALID)
	{
		fprintf(stderr, "Unable to send ioctl request - %s!\n", strerror(errno));
		return(FALSE);
	}

	printf("MaxTargetIdPerHostId = %d\n", host_target_id.max_target_id);
	return(TRUE);

}


/***************************************************************************
* PROCEDURE:    utilGetBusFd
* SUMMARY:      utilGetBusFd
*
* DESCRIPTION: This function is intended to step through a list of installed
*  MPP-based multipath drivers and determine the next file descriptor to use.
*  For Linux only one MPP instance will be loaded only one descriptor will be
*  returned.  On Windows this function steps through a list of installed drivers
*  and returns the appropriate descriptor based on the 'whichDriver' argument.
*
* NOTES:
*
*
* INPUT:
*     Which driver to obtain a file descriptor for:
*	FIRST_DRIVER : Return the first driver in the list.
*	NEXT_DRIVER : Return the next driver in the list.
*
* OUTPUT:
*  None
*
* RETURNS:
*  TRUE if a file descriptor was obtained.
*  DONE if no other MPP instances exist.
*  FALSE if an error occurs.
****************************************************************************/
LINT
utilGetBusFd(LINT whichDriver)
{
	switch(whichDriver)
	{
		case FIRST_DRIVER:
			if(!utilDetermineBusFd())
			{
				fprintf(stderr, "Unable to determine virtual bus!!\n");
				return(FALSE);
			}

			return(TRUE);

		case NEXT_DRIVER:
			return(DONE);

		default:
			return(FALSE);
	}
}


/***********************************************************************
* PROCEDURE:    mppLuxUtil_reportSrcAttributes
* SUMMARY:      Report mpp driver source file attributes at the source release
*               time and driver build time 
*
*	DESCRIPTION: This function performs the following two operations:
*  1.	Prints out each file's versions and digests at the source package release time and
*     the driver build time
*  2.	Compare an attribute of its release time with its build time. Report true or false 
*     of the comparison result
* NOTES:
*
* INPUT: N/A
*     
* OUTPUT:
*     Reports file attributes to the standard output
* RETURNS:
*     Void
*
***********************************************************************/

void mppLnxUtil_reportSrcAttributes()
{
   int numOfModifiedFiles = 0;
   int i;
   int isSame = 0;
   printf("Linux mpp driver version: %s\n",MPPLNX_VERSION);
   printf("Linux mpp driver release time stamp: %s\n\n", MPPLNX_BUILDTIMESTAMP);
   printf("File Name            Version  Digest                              Changed\n");

   for( i = 0; i < MPPLNX_NUM_FILE_ENTRY; i++)
   {
      isSame = 1;
      if( strcmp(mppLnx_rls_version_attributes[i].version,
                 mppLnx_bld_version_attributes[i].version) != 0)
      {
         numOfModifiedFiles++;
         isSame = 0;
      }
      if( strcmp(mppLnx_rls_version_attributes[i].digest,
                 mppLnx_bld_version_attributes[i].digest) != 0)
      {
         if(isSame == 1)
         {  
            numOfModifiedFiles++;
         }
         isSame = 0;
      }

      printf("%-20.20s rls %-4.4s %-35.35s\n",
         mppLnx_rls_version_attributes[i].fileName,
         mppLnx_rls_version_attributes[i].version,
         mppLnx_rls_version_attributes[i].digest);
      printf("%-20.20s bld %-4.4s %-35.35s %s\n",
         " ",
         mppLnx_bld_version_attributes[i].version,
         mppLnx_bld_version_attributes[i].digest,
         isSame == 1?"No":"Yes");
      printf("\n\n");
   }

   if(numOfModifiedFiles == 0)
   {
      printf("No source files have been changed.\n");
   }
   else
   {
      printf("%d source file(s) have been modified.\n",numOfModifiedFiles);
   }

}

BOOL
mppLnxUtil_getUTMInfo( )
{
	
	if(!utilDetermineBusFd())
	{
		fprintf(stderr, "Unable to determine virtual bus!!\n");
		return(INVALID);
	}

	utilDebug("Send UTM Lun information request\n");
	if(ioctl(utilVirtualFd, MPP_IOCTL_GET_UTMLUN_INFO ) == INVALID)
	{
		fprintf(stderr, "Unable to send ioctl request - %s!\n", strerror(errno));
		return(FALSE);
	}

	return(TRUE);

}
/***************************************************************************
* PROCEDURE:    mppSys_ProcessCmdLineArgs
* SUMMARY:      Process platform specific mppUtil command line options
*
	DESCRIPTION: The design of the mppUtil allows each platform to specify platform
   specific command line options. This gives a platform MPP driver freedom to add command
   line options that are only applicable to the specific platform. This function is 
   the entry pointer for processing platform specific function.
*
* NOTES:
	This function is invoked by the main() function of the mppUtil. The main() function
   of the mppUtil handles all common command line options. When this function is invoked, 
   the command line options should be the options that are specific to the Linux platform.
	A platform specific option should be in capital letter
*
* INPUT:
	@option - an option's ASCII value
	@argument the operand of the option. Null if the option has no argument
* 
*
* RETURNS:
*     True or False depending on the outcome of the operation
*
****************************************************************************/
BOOL
mppSys_ProcessCmdLineArgs(LINT option, MPP_STRING argument)
{
	BOOL status = FALSE;

	switch( option )
	{
		case 'I':
			status = utilGetPhysicalHostMaxTargetId(argument);
			break;
		case 'V':
			printf("Linux MPP Driver Version: %s\n", MPPLNX_VERSION);
			status = TRUE;
			break;
		case 'C':
			mppLnxUtil_reportSrcAttributes();
			status = TRUE;
			break;
		case 'U':
			status = mppLnxUtil_getUTMInfo();
			break;
		default:
			utilUsage(progname);
			return(INVALID);
	}	

	return(status);

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_FormatVirtualTarget
* SUMMARY:      supply a string that is an appropriate representation of the VirtualTarget Path.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	utilSysStatusInfo_t and a string to be filled in
* OUTPUT:
*
* RETURNS: None
*
****************************************************************************/
void
mppUtilSys_FormatVirtualTarget(mpp_utilSysStatusInfo_t  *utilSysStatusInfo, MPP_STRING buffer) {

        sprintf(buffer, "%s%d%s%d", "H", utilSysStatusInfo->mpp_VirtualHostId,"C0T", utilSysStatusInfo->mpp_VirtualTargetId);

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_ReadVirtualTarget
* SUMMARY:      read from a string that is a representation of the VirtualTarget Path.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	utilSysStatusInfo_t and a string to read from
* OUTPUT:
*
* RETURNS: None
*
****************************************************************************/
void
mppUtilSys_ReadVirtualTarget(mpp_utilSysStatusInfo_t *utilSysStatusInfo,  MPP_STRING buffer) {

         sscanf(buffer, "H%2dC0T%2d:", &utilSysStatusInfo->mpp_VirtualHostId, &utilSysStatusInfo->mpp_VirtualTargetId  );

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_FormatPhysicalTarget
* SUMMARY:      supply a string that is an appropriate representation of the PhysicalTarget Path.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	ControllerAddress_t and a string to be filled in
* OUTPUT:
*
* RETURNS: None
*
****************************************************************************/
void
mppUtilSys_FormatPhysicalTarget(ControllerAddress_t *ctlAddress, MPP_STRING buffer) {

        sprintf(buffer, "%s%d%s%d%s%d", "H", ctlAddress->HostId,"C",ctlAddress->ChannelId,"T", ctlAddress->TargetId);

}


/***************************************************************************
* PROCEDURE:    mppUtilSys_ReadPhysicalTarget
* SUMMARY:      read from a string that is a representation of the PhysicalTarget Path.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	utilSysStatusInfo_t and a string to read from
* OUTPUT:
*
* RETURNS: None
*
****************************************************************************/
void
mppUtilSys_ReadPhysicalTarget(ControllerAddress_t *ctlAddress, MPP_STRING buffer) {

         sscanf(buffer, "H%2dC%2dT%2d:", &ctlAddress->HostId,&ctlAddress->ChannelId, &ctlAddress->TargetId);

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_CompareCtlAddr
* SUMMARY:	Compare fields of two Ctl Address structures       
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	Two Controller Address structures
* OUTPUT:
*	BOOLEAN value, TRUE if the two Addresses are the same, FALSE if not
* RETURNS: None
*
****************************************************************************/
BOOL
mppUtilSys_CompareCtlAddr( ControllerAddress_t *utilCtlAddress, ControllerAddress_t *ctlAddress) {
        if ((ctlAddress->HostId == utilCtlAddress->HostId) && (ctlAddress->ChannelId == utilCtlAddress->ChannelId) && (ctlAddress->TargetId ==  utilCtlAddress->TargetId))
                return TRUE;
        else
                return FALSE;

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_setWwnDirPath()
* SUMMARY:	set the directory path where wwn files will be stored.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	NULL
* OUTPUT:
*	MPP_STRING - string 
* RETURNS: None
*
****************************************************************************/
MPP_STRING
mppUtilSys_setWwnDirPath()
{

   if( strcmp(temp_string,"\0") == 0  ) {

        strcpy(temp_string, MPPSYS_UTIL_WWN_DIR_NAME);

   }
   return &temp_string[0];

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_DoesWwnDirPathFileExist
* SUMMARY:	Check if a file exists in the WWN file directory
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_STRING - name of file name to be checked for
* OUTPUT:
*	BOOLEAN value - TRUE if the file exists, FALSE if not
*  RETURNS: None
*
****************************************************************************/
BOOL
mppUtilSys_DoesWwnDirPathFileExist(MPP_STRING file_name)
{
        struct dirent *direntP;
        DIR* dirStructP;

                if ((dirStructP = opendir(MPPSYS_UTIL_WWN_DIR_NAME)) != NULL )
                {
                        while (( direntP = readdir(dirStructP)) != NULL )
                        {
                                if(strstr(direntP->d_name, ".wwn")) {
                                        if(strcmp(direntP->d_name, file_name) == 0)
                                                return TRUE;
                                }
                        }
                        closedir(dirStructP);
                        return FALSE;
                }else {
                        fprintf(stderr, "Unable to open directory: %s\n", strerror(errno));
                        return FALSE;
                }

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_OpenFile
* SUMMARY:	Open a system file	
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_STRING - name of file name to be checked for, flags to open the file, mode to open the file
* OUTPUT:
*	MPP_FILE_HANDLE - On Linux, this is an INT , file descriptor
*  RETURNS: None
*
****************************************************************************/
MPP_FILE_HANDLE
mppUtilSys_OpenFile(MPP_STRING fileName, INT flags, INT mode) {
        int ret;

        ret = open( fileName, flags, mode );

        if ( ret < 0 )
                fprintf(stderr, "mpp_OpenFile: %s\n", strerror(errno));

        return (ret);

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_CloseFile
* SUMMARY:	Close a system file	
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_FILE_HANDLE - On Linux, this is an INT , file descriptor
* OUTPUT:
*	int, if < 0 signifies an error
*  RETURNS: None
*
****************************************************************************/
INT
mppUtilSys_CloseFile(MPP_FILE_HANDLE fd) {
        int ret;

        ret = close( fd);

        if ( ret < 0 )
                fprintf(stderr, "mpp_CloseFile: %s\n", strerror(errno));

        return (ret);
}

/***************************************************************************
* PROCEDURE:    mppUtilSys_ReadFile
* SUMMARY:	Read a system file	
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_FILE_HANDLE - On Linux, this is an INT , file descriptor, MPP_STRING to read from, and number of bytes to read
* OUTPUT:
*	int, if < 0 signifies an error
*  RETURNS: None
*
****************************************************************************/

INT
mppUtilSys_ReadFile(MPP_FILE_HANDLE fd, MPP_STRING rdBuffer,INT bytesToRead) {
        unsigned long ret;

        ret = read(fd, rdBuffer, bytesToRead );

        if ( ret < 0 )
                fprintf(stderr, "mpp_ReadFile: %s\n", strerror(errno));

        return ret;

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_WriteFile
* SUMMARY:	Write a system file	
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_FILE_HANDLE - On Linux, this is an INT , file descriptor, MPP_STRING to write into, and number of bytes to write
* OUTPUT:
*	int, if < 0 signifies an error
*  RETURNS: None
*
****************************************************************************/
INT
mppUtilSys_WriteFile(MPP_FILE_HANDLE fd, MPP_STRING wrtBuffer, INT bytesToWrite) {
        unsigned long ret;

        ret = write(fd, wrtBuffer, bytesToWrite );

        if ( ret < 0 )
                fprintf(stderr, "mpp_WriteFile: %s\n", strerror(errno));
        return ret;

}

/***************************************************************************
* PROCEDURE:    mppUtilSys_SetFilePointer
* SUMMARY:	Set the File Pointer of a File
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_FILE_HANDLE - On Linux, this is an INT , file descriptor, INT number of bytes to move the file pointer by, 
*			  INT seekStart, the position from which to start
* OUTPUT:
*	int, if < 0 signifies an error
*  RETURNS: None
*
****************************************************************************/
INT
mppUtilSys_SetFilePointer(MPP_FILE_HANDLE fd, INT bytesToMove, INT seekStart ) {

        unsigned long ret;

        ret = lseek(fd, bytesToMove, seekStart);

        if ( ret < 0 )
                fprintf(stderr, "mpp_SetFilePointer: %s\n", strerror(errno));
        return ret;
}

/***************************************************************************
* PROCEDURE:    mppUtilSys_DeleteFile
* SUMMARY:	Delete a system file	
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	MPP_STRING - name of file name to delete
* OUTPUT:
*	int, if < 0 signifies an error
*  RETURNS: None
*
****************************************************************************/
INT
mppUtilSys_DeleteFile(MPP_STRING filePathName)
{
        int ret;

        ret = unlink(filePathName);

        if ( ret < 0 )
                fprintf(stderr, "mpp_DeleteFile: %s\n", strerror(errno));
        return ret;
}


/***************************************************************************
* PROCEDURE:    mppUtilSys_PrintWWNList
* SUMMARY:	Get the list of WWN files in the directory into a data structure.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	PWWN_FILES - pointer to structure containing information about wwn files	
* OUTPUT:
*	BOOLEAN value, TRUE if successful, FALSE if error occurs
*  RETURNS: None
*
****************************************************************************/
BOOL
mppUtilSys_PrintWWNList(PWWN_FILES *wwnFileData) {

        struct dirent *direntP;
        DIR* dirStructP;
        int cnt=0, cnt2= 0 ;
        char *fileP = NULL;
        PWWN_FILES pwwnFileData = NULL;


        if ((dirStructP = opendir(MPPSYS_UTIL_WWN_DIR_NAME)) != NULL )
        {
           while (( direntP = readdir(dirStructP)) != NULL )
           {
                if(strstr(direntP->d_name, ".wwn")) {
                        cnt ++;
                }
           }
           closedir(dirStructP);
        }

        if( cnt == 0 ) {
                pwwnFileData = (PWWN_FILES)malloc(sizeof(WWN_FILES ));
                pwwnFileData->numWwnFiles = cnt;
                *wwnFileData = pwwnFileData;
                return TRUE;
        }


        pwwnFileData = (PWWN_FILES)malloc(sizeof(WWN_FILES) + (cnt * sizeof(WWN_FILE_DATA)));
        if(pwwnFileData == NULL) {
                fprintf(stderr, "Unable to malloc memory for WWN_FILES - %s\n", strerror(errno));
                return FALSE;
        }
        pwwnFileData->numWwnFiles = cnt;
        *wwnFileData = pwwnFileData;

        cnt2 = 0;
        if ((dirStructP = opendir(MPPSYS_UTIL_WWN_DIR_NAME)) != NULL )
        {
           while (( direntP = readdir(dirStructP)) != NULL )
           {
                if(strstr(direntP->d_name, ".wwn")) {
                   strncpy(&pwwnFileData->arrayDataList[cnt2].fileName[0], direntP->d_name, MAX_ARRAY_STR_LEN+1);
                   cnt2 ++;
                }
           }
           closedir(dirStructP);
           if( pwwnFileData->numWwnFiles > cnt2 ) {
                fprintf(stderr, "Count differs from intial estimate\n");
           }
           pwwnFileData->numWwnFiles = cnt2;
           return TRUE;
        }else {
           fprintf(stderr, "err returned errno %d %s\n", errno, strerror(errno));
           return FALSE;
        }


}

/***************************************************************************
* PROCEDURE:    mppUtilSys_DeleteWWNList
* SUMMARY:	Delete the list of WWN files in the directory.
*
* DESCRIPTION:
*
* NOTES:
*
* INPUT:
*	NULL
* OUTPUT:
*	BOOLEAN value, TRUE if successful, FALSE if error occurs
*  RETURNS: None
*
****************************************************************************/
BOOL
mppUtilSys_DeleteWWNList() {

        struct dirent *direntP;
        DIR* dirStructP;
        char temp_string[MPP_MAX_PATH];

        if ((dirStructP = opendir(MPPSYS_UTIL_WWN_DIR_NAME)) != NULL )
        {
           while (( direntP = readdir(dirStructP)) != NULL )
           {
                if(strstr(direntP->d_name, ".wwn")) {
                   utilDebug("Deleting file %s in %s\n", direntP->d_name, MPPSYS_UTIL_WWN_DIR_NAME);
                   strcpy(temp_string, MPPSYS_UTIL_WWN_DIR_NAME);
                   strcat(temp_string, direntP->d_name);
                   unlink(temp_string);
                }
           }
           closedir(dirStructP);
        }else {
           fprintf(stderr, "err returned errno %d %s\n", errno, strerror(errno));
	   return FALSE;
        }
        return TRUE;

}




/***************************************************************************
* PROCEDURE:    utilMakeMppDevNode
* SUMMARY:      Make the mppUpper device node for ioctl operation at run-time
*
* DESCRIPTION:
* The device node /dev/mppUpper is used for the mppUtil to communicate with the 
mppUpper driver. The device node could not be created if the mppUpper level driver 
is loaded from the ramdisk. This is because the udev and hotplug agents is not available
 during the driver loading time. This is a workaround to solve this issue. This 
 function will be invoked before open the /dev/mppUpper device node. This function 
 will check whether or not the /dev/mppUpper node exists. If it exists, it will 
 remove the device node. Then this function reads the major and minor numbers of 
 the mppUpper driver's character device from the /sys/class/mppUpper/mppUpper/dev 
 attribute file. This function finally creates a character device node /dev/mppUpper 
 with the major and the minor number returned from the sysfs file.
*
* NOTES:
*
* INPUT:
* 
* OUTPUT:
*
* RETURNS: 0 if the operation successes. Otherwise, non 0.
*
****************************************************************************/
int utilMakeMppDevNode(void)
{
   struct stat       mpp_dev_node_stat;
   int               rtn;
   int               mppSysFd;
   char              dataBuffer[STR_LENGTH];
   char              majorString[STR_LENGTH];
   char              minorString[STR_LENGTH];
   int               major;
   int               minor;
   dev_t             mppDev;
   int               i;
   int               minorIndex;
   
   /*
   * check if the device node file exists. Remove it if it does.
   */
   rtn = stat(MPP_BUS_NODE, &mpp_dev_node_stat);
   if(rtn == 0)
   {
      /*
      * remove the device node
      */
      rtn = unlink(MPP_BUS_NODE);
      if(rtn != 0)
      {
         fprintf(stderr,"Could not remove %s error:%s\n",
            MPP_BUS_NODE,strerror(errno));
         return rtn;
      }
   }
   
   /*
   * Open the mppUppper's dev file
   */
   mppSysFd = open(MPPUPPER_SYS_FILE,O_RDONLY);
   if(mppSysFd < 0)
   {
      fprintf(stderr,"The mppUpper driver module is not loaded. Cannot continue... %s error:%s\n",
         MPPUPPER_SYS_FILE,strerror(errno));
      return mppSysFd;
   }
   
   /*
   * clear-up buffer memory
   */
   memset(dataBuffer,0,STR_LENGTH);
   memset(majorString,0,STR_LENGTH);
   memset(minorString,0,STR_LENGTH);


   /*
   * Read the major and minor numbers. It is in the format of 254:0
   */
   rtn = read(mppSysFd,dataBuffer, STR_LENGTH);
   if(!rtn)
   {
      fprintf(stderr,"Could not read %s. Cannot continue... error %s\n",
         MPPUPPER_SYS_FILE,strerror(errno));
      return rtn;
   }

   /*
   * close the fd. Don't care the return status
   */
   rtn = close(mppSysFd);
   /*
   *dataBuffer - major-number:minor-numer
   */
   minorIndex = -1;
   for(i =0; i<STR_LENGTH; i++)
   {
      if(dataBuffer[i] == 0)
      {
         break;
      }

      if(dataBuffer[i] == ':')
      {
         minorIndex = 0;
         continue;
      }else
      {
         if(minorIndex == -1)
         {
            majorString[i]=dataBuffer[i];
         }else
         {
            minorString[minorIndex] = dataBuffer[i];
            minorIndex++;
         }
      }
   }   
   major = atoi(majorString);
   minor = atoi(minorString);
   

   /*
   * generate the dev_t by using the major and minor numbers.
   */
   mppDev = MKDEV(major,minor);
   /*
   * Make the character device node with read-write permissions
   */
   rtn=mknod(MPP_BUS_NODE,S_IRUSR|S_IWUSR|S_IFCHR,mppDev);
   if(rtn != 0)
   {
      fprintf(stderr,"Could not make device node %s. Cannot continue...error:%s\n",
            MPP_BUS_NODE,strerror(errno));
      return rtn;
   }
   return 0;
}
